<?php
/**
 * Pagination layout
 *
 * @package Xpro
 */

// Exit if accessed directly.
defined( 'ABSPATH' ) || exit;

if ( ! function_exists( 'xpro_pagination' ) ) {
	/**
	 * Displays the navigation to next/previous set of posts.
	 *
	 * @param string|array $args {
	 *     (Optional) Array of arguments for generating paginated links for archives.
	 *
	 * @type string $base Base of the paginated url. Default empty.
	 * @type string $format Format for the pagination structure. Default empty.
	 * @type int $total The total amount of pages. Default is the value WP_Query's
	 *                                      `max_num_pages` or 1.
	 * @type int $current The current page number. Default is 'paged' query var or 1.
	 * @type string $aria_current The value for the aria-current attribute. Possible values are 'page',
	 *                                      'step', 'location', 'date', 'time', 'true', 'false'. Default is 'page'.
	 * @type bool $show_all Whether to show all pages. Default false.
	 * @type int $end_size How many numbers on either the start and the end list edges.
	 *                                      Default 1.
	 * @type int $mid_size How many numbers to either side of the current pages. Default 2.
	 * @type bool $prev_next Whether to include the previous and next links in the list. Default true.
	 * @type bool $prev_text The previous page text. Default '&laquo;'.
	 * @type bool $next_text The next page text. Default '&raquo;'.
	 * @type string $type Controls format of the returned value. Possible values are 'plain',
	 *                                      'array' and 'list'. Default is 'array'.
	 * @type array $add_args An array of query args to add. Default false.
	 * @type string $add_fragment A string to append to each link. Default empty.
	 * @type string $before_page_number A string to appear before the page number. Default empty.
	 * @type string $after_page_number A string to append after the page number. Default empty.
	 * @type string $screen_reader_text Screen reader text for the nav element. Default 'Posts navigation'.
	 * }
	 *
	 * @param string $class (Optional) Classes to be added to the <ul> element. Default 'pagination'.
	 */
	function xpro_pagination( $args = array(), $class = 'pagination' ) {

		if ( ! isset( $args['total'] ) && $GLOBALS['wp_query']->max_num_pages <= 1 ) {
			return;
		}

		$svg_icon = '<svg xmlns="http://www.w3.org/2000/svg" xmlns:xlink="http://www.w3.org/1999/xlink" viewBox="0 0 100 54"><g id="xpro-chevron">
                    <path d="M 50 43.402344 C 50.289062 42.976562 50.601562 42.570312 50.9375 42.179688 C 64.570312 28.667969 78.207031 15.152344 91.847656 1.636719 C 93.1875 0.308594 94.710938 -0.328125 96.558594 0.15625 C 98.066406 0.546875 99.265625 1.675781 99.730469 3.144531 C 100.199219 4.613281 99.867188 6.21875 98.855469 7.390625 C 98.675781 7.605469 98.472656 7.804688 98.269531 8.003906 L 53.507812 52.335938 C 52.121094 53.710938 50.527344 54.410156 48.609375 53.738281 C 47.8125 53.449219 47.09375 52.988281 46.496094 52.394531 C 31.503906 37.597656 16.527344 22.777344 1.566406 7.9375 C -0.375 6.019531 -0.421875 3.246094 1.375 1.445312 C 3.238281 -0.414062 6.003906 -0.355469 8.007812 1.625 C 21.65625 15.132812 35.300781 28.640625 48.941406 42.15625 C 49.242188 42.453125 49.488281 42.800781 50 43.402344 Z M 50 43.402344 "/>
                    </g></svg>';

		$args = wp_parse_args(
			$args,
			array(
				'mid_size'           => 2,
				'prev_next'          => true,
				'prev_text'          => '<i class="xpro-icon xpro-icon-chevron-left">' . $svg_icon . '</i><span class="pagination-prev">' . __( 'Prev', 'xpro' ) . '</span>',
				'next_text'          => '<span class="pagination-next">' . __( 'Next', 'xpro' ) . '</span><i class="xpro-icon xpro-icon-chevron-right">' . $svg_icon . '</i>',
				'type'               => 'array',
				'current'            => max( 1, get_query_var( 'paged' ) ),
				'screen_reader_text' => __( 'Posts Navigation', 'xpro' ),
			)
		);

		$links = paginate_links( $args );
		if ( ! $links ) {
			return;
		}

		?>

		<nav class="xpro-post-navigation" aria-labelledby="posts-nav-label">

			<h2 id="posts-nav-label" class="sr-only">
				<?php echo esc_html( $args['screen_reader_text'] ); ?>
			</h2>

			<ul class="<?php echo esc_attr( $class ); ?>">

				<?php
				foreach ( $links as $key => $link ) {
					?>
					<li class="page-item <?php echo strpos( $link, 'current' ) ? 'active' : ''; ?>">
						<?php echo str_replace( 'page-numbers', 'page-link', $link ); // phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped ?>
					</li>
					<?php
				}
				?>

			</ul>

		</nav>

		<?php
	}
}
