<?php

/**
 * Admin Notice for XBlog Pro Promotion
 *
 * @package x-blog
 */

// Exit if accessed directly.
if (! defined('ABSPATH')) {
  exit;
}

/**
 * Class XBlog_Pro_Notice
 * Handles the admin notice for XBlog Pro promotion with 20% discount
 */
class XBlog_Pro_Notice
{

  /**
   * Constructor
   */
  public function __construct()
  {
    // Only proceed if user is an admin
    if (! current_user_can('manage_options')) {
      return;
    }

    // Add the notice
    add_action('admin_notices', array($this, 'display_admin_notice'));

    // Register scripts and styles
    add_action('admin_enqueue_scripts', array($this, 'enqueue_scripts'));

    // Ajax handler for dismissing the notice
    add_action('wp_ajax_xblog_dismiss_pro_notice', array($this, 'dismiss_notice_ajax'));
  }

  /**
   * Enqueue necessary scripts and styles
   */
  public function enqueue_scripts()
  {
    // Enqueue the script
    wp_enqueue_script(
      'xblog-notice-js',
      get_template_directory_uri() . '/inc/info/js/notice.js',
      array('jquery'),
      X_BLOG_VERSION,
      true
    );

    // Enqueue the CSS file
    wp_enqueue_style(
      'xblog-notice-css',
      get_template_directory_uri() . '/inc/info/css/notice.css',
      array(),
      X_BLOG_VERSION
    );

    // Localize the script with the ajax url
    wp_localize_script(
      'xblog-notice-js',
      'xblog_notice_params',
      array(
        'ajax_url' => admin_url('admin-ajax.php'),
        'nonce'    => wp_create_nonce('xblog_dismiss_pro_notice'),
      )
    );
  }



  /**
   * Display the admin notice
   */
  /**
   * Display the admin notice
   * 
   * The notice will be hidden for a configurable number of days (default: 25)
   * after a user dismisses it. After this period, the notice will appear again.
   */
  public function display_admin_notice()
  {
    // Get the date when the notice was dismissed
    $hide_date = get_option('xblog_pro_notice_dismissed_info');

    if (!empty($hide_date)) {
      // Calculate days passed since dismissal (convert seconds to days)
      $days_passed = round((time() - strtotime($hide_date)) / 24 / 60 / 60);

      // Get the number of days to hide the notice (default: 25)
      // This filter allows theme extensions to modify the hide duration
      $hide_duration = apply_filters('xblog_pro_notice_hide_duration', 25);

      // Don't show the notice if the hide duration hasn't passed yet
      if ($days_passed < $hide_duration) {
        return;
      }
    }

    // URLs for the buttons
    $upgrade_url = 'https://wpthemespace.com/product/xblog-pro/?add-to-cart=11011';
    $demo_url = 'https://blogs.wpcolors.net/x-demos/';

    // Output the notice HTML
?>
    <div class="notice xblog-pro-notice">
      <div class="xblog-pro-notice-content">
        <div class="xblog-pro-notice-text">
          <h3>
            <span class="xblog-discount-badge"><?php esc_html_e('50% OFF', 'x-blog'); ?></span>
            <?php esc_html_e('Wow! Big Savings: 50% OFF For You– Limited Time Only!', 'x-blog'); ?>
          </h3>
          <p>
            <?php esc_html_e('Take your website to the next level with powerful customization, fast support, and premium features. For a limited time, you can grab XBlog Pro at 50% OFF. Just use the coupon!', 'x-blog'); ?>
            <span class="xblog-coupon-code"><?php esc_html_e('wpstxc50', 'x-blog'); ?></span>
          </p>
          <ul class="xblog-features-list">
            <li><?php esc_html_e('8 Posts layout options', 'x-blog'); ?></li>
            <li><?php esc_html_e('One click demo import', 'x-blog'); ?></li>
            <li><?php esc_html_e('8 Homepage design variations', 'x-blog'); ?></li>
            <li><?php esc_html_e('Word count & Reading time calculator', 'x-blog'); ?></li>
            <li><?php esc_html_e('Unlimited Customize options', 'x-blog'); ?></li>
            <li><?php esc_html_e('Related Posts options', 'x-blog'); ?></li>
            <li><?php esc_html_e('Built-in carousel and slider', 'x-blog'); ?></li>
            <li><?php esc_html_e('QR code generator', 'x-blog'); ?></li>
            <li><?php esc_html_e('WooCommerce Shop Support', 'x-blog'); ?></li>
            <li><?php esc_html_e('Shop Tools Available', 'x-blog'); ?></li>
          </ul>
        </div>
        <div class="xblog-pro-notice-buttons">
          <a href="<?php echo esc_url($upgrade_url); ?>" target="_blank" class="xblog-pro-notice-btn xblog-pro-notice-btn-primary">
            <?php esc_html_e('Get Offer Now', 'x-blog'); ?>
          </a>
          <a href="<?php echo esc_url($demo_url); ?>" target="_blank" class="xblog-pro-notice-btn xblog-pro-notice-btn-secondary">
            <?php esc_html_e('View Demo', 'x-blog'); ?>
          </a>
          <a href="#" class="xblog-pro-notice-btn xblog-pro-notice-btn-dismiss xblog-dismiss-notice">
            <?php esc_html_e('Maybe Later', 'x-blog'); ?>
          </a>
        </div>
      </div>
    </div>
<?php
  }

  /**
   * Ajax handler for dismissing the notice
   * 
   * This method handles the AJAX request when a user clicks the 'Maybe Later' button.
   * It stores the current date in the database, which is used to calculate when
   * to show the notice again (after the hide duration has passed).
   */
  public function dismiss_notice_ajax()
  {
    // Verify nonce for security
    if (! isset($_POST['nonce']) || ! wp_verify_nonce(sanitize_text_field(wp_unslash($_POST['nonce'])), 'xblog_dismiss_pro_notice')) {
      wp_send_json_error('Invalid nonce');
      exit;
    }

    // Store the current time as the dismissal date
    // The format is sanitized by WordPress core function
    $current_time = current_time('mysql');
    update_option('xblog_pro_notice_dismissed_info', $current_time);

    // Send success response back to the AJAX request
    wp_send_json_success('Notice dismissed');
  }
}

// Initialize the notice
new XBlog_Pro_Notice();
