<?php
/**
 * functions and hooks for wpopus
 *
 * @package wpopus-studio
 */

if (!function_exists('wpopus_studio_theme_constant')):
    /**
     * theme details for wpopus 
     *
     * @since 1.0
     * @return void
     */
    function wpopus_studio_theme_constant()
    {
        $theme = wp_get_theme();
        $args = array(
            'theme_name' => $theme->get('Name'),
            'theme_author' => $theme->get('Author'),
            'theme_quick_links' => array(
                array(
                    'label' => esc_html__('Theme Documentation', 'wpopus-studio'),
                    'url' => 'https://wpopus.com/docs-category/themes/',
                ),
                array(
                    'label' => esc_html__('All Themes', 'wpopus-studio'),
                    'url' => 'https://wpopus.com/themes/',
                ),
            ),
        );
        return $args;
    }
endif;
add_filter('wpopus_theme_constant_filter', 'wpopus_studio_theme_constant');

// show admin notice
if (!function_exists('wpopus_studio_admin_notice')) {
    function wpopus_studio_admin_notice()
    {
        // Check if we're on the dashboard/index.php
        $screen = get_current_screen();
        if ($screen->id !== 'dashboard') {
            return;
        }

        if (!current_user_can('manage_options')) {
            return;
        }

        // return if plugin exists
        if (function_exists('wpopus_pro') || function_exists('wpopus')) {
            return;
        }

        $theme = wp_get_theme();
        ?>
        <div class="notice notice-info is-dismissible">
            <h2><?php esc_html_e('Transform your website with the power of wpOpus - The Ultimate Gutenberg Toolkit and Site Builder.', 'wpopus-studio'); ?></h2>
            <p><?php printf('<b>%1$s</b> %2$s', esc_html($theme->get('Name')), esc_html__('is supported by wpOpus plugin, making it easy to create stunning websites with advanced blocks, pre-built templates, demo imports and powerful features!', 'wpopus-studio')); ?>
            </p>
            <p>
                <button id="install-wpopus"
                    class="button button-primary"><?php esc_html_e('Install and Activate', 'wpopus-studio'); ?></button>
                <a href="https://wpopus.com/" target="_blank"
                    class="button button-secondary"><?php esc_html_e('Learn More', 'wpopus-studio'); ?></a>
            </p>
        </div>
        <?php
    }
}
add_action('admin_notices', 'wpopus_studio_admin_notice');

function wpopus_studio_admin_enqueue_scripts($hook)
{
    if ('index.php' !== $hook) {
        return;
    }

    if (!current_user_can('manage_options')) {
        return;
    }

    // return if plugin exists
    if (function_exists('wpopus_pro') || function_exists('wpopus')) {
        return;
    }

    wp_enqueue_script(
        'wpopus-studio-admin-js',
        get_template_directory_uri() . '/assets/js/admin.js',
        ['jquery'],
        '1.0',
        true
    );

    wp_localize_script('wpopus-studio-admin-js', 'wpopusAjax', [
        'ajax_url' => admin_url('admin-ajax.php'),
        'nonce' => wp_create_nonce('install_wpopus_nonce'),
        'label' => esc_html__('Install and Activate', 'wpopus-studio'),
        'success_label' => esc_html__('Installed and Activated', 'wpopus-studio'),
    ]);
}
add_action('admin_enqueue_scripts', 'wpopus_studio_admin_enqueue_scripts');

// install and activate wpopus
function wpopus_studio_install_and_activate_plugin()
{
    // Check if the current user has the capability to manage options
    if (!current_user_can('manage_options')) {
        wp_send_json_error(['message' => esc_html__('You are not authorized to perform this action.', 'wpopus-studio')]);
    }

    // Verify if the nonce is set and valid
    if (!isset($_POST['nonce']) || !wp_verify_nonce($_POST['nonce'], 'install_wpopus_nonce')) {
        wp_send_json_error(['message' => esc_html__('Nonce verification failed.', 'wpopus-studio')]);
    }

    // Start output buffering to prevent HTML output in JSON
    ob_start();

    include_once ABSPATH . 'wp-admin/includes/plugin-install.php';
    include_once ABSPATH . 'wp-admin/includes/class-wp-upgrader.php';
    include_once ABSPATH . 'wp-admin/includes/plugin.php';

    $plugin_slug = 'wpopus';
    $plugin_file = $plugin_slug . '/' . $plugin_slug . '.php';

    // Check if the plugin already exists
    if (file_exists(WP_PLUGIN_DIR . '/' . $plugin_file)) {
        $activate = activate_plugin($plugin_file);
        if (ob_get_length()) ob_end_clean(); // Clean buffer if active

        if (is_wp_error($activate)) {
            wp_send_json_error(['message' => $activate->get_error_message()]);
        }

        wp_send_json_success(['message' => esc_html__('wpOpus activated successfully!', 'wpopus-studio')]);
    }

    // Fetch plugin info
    $api = plugins_api('plugin_information', ['slug' => $plugin_slug]);
    if (is_wp_error($api) || empty($api->download_link)) {
        if (ob_get_length()) ob_end_clean();
        $error_message = is_wp_error($api) ? $api->get_error_message() : esc_html__('Failed to retrieve plugin information.', 'wpopus-studio');
        wp_send_json_error(['message' => $error_message]);
    }

    // Install the plugin
    $upgrader = new Plugin_Upgrader();
    $install_result = $upgrader->install($api->download_link);
    if (ob_get_length()) ob_end_clean();

    if (is_wp_error($install_result)) {
        wp_send_json_error(['message' => $install_result->get_error_message()]);
    }

    // Activate the plugin
    $activate = activate_plugin($plugin_file);
    if (is_wp_error($activate)) {
        wp_send_json_error(['message' => $activate->get_error_message()]);
    }

    wp_send_json_success(['message' => esc_html__('wpOpus installed and activated successfully!', 'wpopus-studio')]);
}
add_action('wp_ajax_install_wpopus_plugin', 'wpopus_studio_install_and_activate_plugin');
