<?php
/**
 * The class dealing with Google Fonts functionality for the theme
 * Maintaining the cache, typography controls and updating font files
 */
class ITRE_Google_Fonts {

    public function __construct() {
        add_action('customize_controls_enqueue_scripts', array($this, 'itre_localize_google_fonts'));
        add_action('init', array($this, 'itre_get_selected_fonts'));
    }

    public static function itre_get_fonts($amount = 50) {
        // Check if the fonts data is already in the cache (transient)
        $google_fonts = get_transient('google_fonts_data');

        if (false === $google_fonts) {
            // Data not cached, fetch it
            $googleApi = 'https://www.googleapis.com/webfonts/v1/webfonts?sort=popularity&capability=WOFF2&key=AIzaSyA9-9K8wV9KWKWY84Sp5TLSS7p9GguLRh4';
            $fontContent = wp_remote_get($googleApi, array('sslverify' => false));

            if (is_wp_error($fontContent) || empty($fontContent['body'])) {
                error_log('Failed to fetch Google Fonts API or empty response body');
                return [];
            }

            $google_fonts_json = json_decode($fontContent['body']);
            if (json_last_error() !== JSON_ERROR_NONE) {
                error_log('Invalid JSON response from Google Fonts API');
                return [];
            }

            // Extract the font-family and their respective weights in a key => value associative array
            foreach ($google_fonts_json->items as $font) {
                $google_fonts[$font->family] = [
                    'variants' => $font->variants,
                    'category' => $font->category,
                ];
            }

            // Store the data in a transient for 7 days
            set_transient('google_fonts_data', $google_fonts, 86400 * 7);
        }

        if ($amount === 'all') {
            return $google_fonts;
        } else {
            return array_slice($google_fonts, 0, absint($amount));
        }
    }

    public function itre_localize_google_fonts() {
        $fonts = self::itre_get_fonts();
        wp_localize_script('itre-typography-js', 'itre', $fonts);
    }

    public static function itre_get_font_settings() {
        // Array containing slugs of all the typography settings
        $defs = [
            'heading' => [
                'family' => 'League Spartan',
                'weight' => 700,
                'cat'    => 'sans-serif',
            ],
            'body' => [
                'family' => 'League Spartan',
                'weight' => 400,
                'cat'    => 'sans-serif',
            ],
        ];

        $f_prefix = 'itre_gfonts_';
        $w_prefix = 'itre_gweights_';
        $c_prefix = 'itre_gcat_';
        $settings = [];

        foreach ($defs as $key => $value) {
            $setting['family'] = get_theme_mod("{$f_prefix}{$key}", $value['family']);
            $setting['weight'] = intval(get_theme_mod("{$w_prefix}{$key}", $value['weight']));
            $setting['cat']    = get_theme_mod("{$c_prefix}{$key}", $value['cat']);

            $settings[$key] = $setting;
        }
        return $settings;
    }

    public function itre_get_selected_fonts() {
        global $wp_filesystem;
        if (empty($wp_filesystem)) {
            require_once ABSPATH . 'wp-admin/includes/file.php';
            WP_Filesystem();
        }

        $fonts = self::itre_get_font_settings();

        $fileDir = get_template_directory() . '/assets/cache/fontFiles/';
        if (!is_dir($fileDir)) {
            wp_mkdir_p($fileDir);
        }

        $cache_files = glob($fileDir . '*.woff2');
        $cache_array = [];
        foreach ($cache_files as $file) {
            if (strpos($file, 'heading')) {
                $cache_array['heading'] = basename($file);
            }

            if (strpos($file, 'body')) {
                $cache_array['body'] = basename($file);
            }
        }

        foreach ($fonts as $key => $value) {
            $font_slug = sanitize_file_name(str_replace(' ', '-', strtolower($value['family'])));
            $weight = $value['weight'];
            $file = $fileDir . "font-${key}-${font_slug}-${weight}.woff2";

            // If the setting has changed, get the new file
            if (!file_exists($file)) {
                $fontFile = get_template_directory() . '/assets/cache/google-web-fonts.txt';
                $fonts_json = json_decode(file_get_contents($fontFile));
                if (json_last_error() !== JSON_ERROR_NONE) {
                    error_log('Invalid JSON in local font file');
                    continue;
                }

                $items = $fonts_json->items;
                foreach ($items as $item) {
                    if ($item->family === $value['family']) {
                        $woff2_files = $item->files;

                        $woff2_file = $value['weight'] === 400 ? $woff2_files->regular : $woff2_files->{$value['weight']};
                        break;
                    }
                }

                $response = wp_remote_get(esc_url_raw($woff2_file));
                if (is_wp_error($response)) {
                    error_log('Error fetching font file: ' . $response->get_error_message());
                    continue;
                }

                $body = wp_remote_retrieve_body($response);

                // Write the file using WP_Filesystem API
                if ($wp_filesystem->put_contents($file, $body, FS_CHMOD_FILE)) {
                    // Delete the old file
                    if (isset($cache_array[$key])) {
                        $old_file = $fileDir . $cache_array[$key];
                        if (file_exists($old_file)) {
                            $wp_filesystem->delete($old_file);
                        }
                    }
                } else {
                    error_log("Failed to write font file: $file");
                }
            }
        }
    }
}

// Initialize the class properly
add_action('init', function () {
    if (class_exists('ITRE_Google_Fonts')) {
        new ITRE_Google_Fonts();
    }
});
