<?php
/**
 * Theme Information for W3Construct
 *
 * @package     W3Construct
 * @author      W3ITEXPERTS
 * @copyright   Copyright (c) 2023, W3Construct
 * @link        http://w3construct-wordpress.wprdx.come/demo/
 * @since       W3Construct 1.0.1
 */

if ( ! class_exists( 'W3Construct_DZ_Base' ) ) {
	get_template_part( '/dz-inc/class-w3construct-dz-base' );
}

if ( ! function_exists( 'w3construct_set' ) ) {
	/**
	 * Retrieve a value from an object or an array, with an optional default value.
	 *
	 * This function checks if the input variable is an object or an array and attempts
	 * to retrieve a value using the provided key. If the key exists in the object or array,
	 * it returns that value. If the key doesn't exist, and a default value is provided,
	 * it returns the default value. If neither the key exists nor a default value is
	 * provided, it returns false.
	 *
	 * @param mixed  $var The input object or array from which to retrieve the value.
	 * @param string $key The key used to access the value within the object or array. (Optional).
	 * @param mixed  $def The default value to return if the key doesn't exist. (Optional).
	 * @return mixed|false The retrieved value, default value, or false if the key doesn't exist.
	 */
	function w3construct_set( $var, $key = '', $def = '' ) {
		if ( is_object( $var ) && isset( $var->$key ) ) {
			return $var->$key;
		} elseif ( is_array( $var ) && isset( $var[ $key ] ) ) {
			return $var[ $key ];
		} elseif ( ! is_array( $var ) && ! empty( $var ) && empty( $key ) ) { /* variable handling with blank key */
			return $var;
		} elseif ( $def ) {
			return $def;
		} else {
			return false;
		}
	}
}

if ( ! function_exists( 'w3construct_admin_page_init' ) ) {
	add_action( 'admin_init', 'w3construct_admin_page_init' );
	/**
	 * Initialize the admin page for plugin activation and deactivation.
	 *
	 * This function is responsible for handling plugin activation and deactivation actions
	 * in the admin panel. It checks the user's capabilities, and if the user can edit theme
	 * options, it processes activation and deactivation requests.
	 *
	 * @return void
	 */
	function w3construct_admin_page_init() {
		if ( current_user_can( 'edit_theme_options' ) ) {

			if ( isset( $_GET['dz-deactivate'] ) && 'deactivate-plugin' === $_GET['dz-deactivate'] ) {
				check_admin_referer( 'dz-deactivate', 'dz-deactivate-nonce' );
				$plugins_w3 = TGM_Plugin_Activation::$instance->plugins;
				foreach ( $plugins_w3 as $plugin ) {
					$plugin_slug = isset( $_GET['plugin'] ) ? sanitize_text_field( wp_unslash( $_GET['plugin'] ) ) : '';
					if ( $plugin_slug === $plugin['slug'] ) {
						deactivate_plugins( $plugin['file_path'] );
					}
				}

				wp_safe_redirect( 'admin.php?page=dz-plugins' );
			}

			if ( isset( $_GET['dz-activate'] ) && 'activate-plugin' === $_GET['dz-activate'] ) {
				check_admin_referer( 'dz-activate', 'dz-activate-nonce' );
				$plugins_w3 = TGM_Plugin_Activation::$instance->plugins;
				foreach ( $plugins_w3 as $plugin ) {
					$plugin_slug = isset( $_GET['plugin'] ) ? sanitize_text_field( wp_unslash( $_GET['plugin'] ) ) : '';
					if ( $plugin_slug === $plugin['slug'] ) {
						activate_plugin( $plugin['file_path'] );
					} elseif ( in_array( $plugin['slug'], array( 'kingcomposer', 'js_composer', 'elementor' ), true ) ) {
						deactivate_plugins( $plugin['file_path'] );
					}
				}

				wp_safe_redirect( 'admin.php?page=dz-plugins' );
			}
		}
	}
}

if ( ! function_exists( 'w3construct_plugin_link' ) ) {
	/**
	 * Generate plugin action links based on the plugin's status.
	 *
	 * This function generates action links (e.g., Install, Activate, Deactivate, Update) for a given plugin
	 * based on its current status and version. It checks if the plugin is installed, active, and up to date,
	 * and creates the appropriate action links.
	 *
	 * @param array $item An array representing the plugin data.
	 *
	 * @return array An array of action links to be displayed for the plugin.
	 */
	function w3construct_plugin_link( $item ) {
		$installed_plugins        = get_plugins();
		$item['sanitized_plugin'] = $item['name'];
		$is_plug_act              = 'is_plugin_active';
		if ( ! isset( $installed_plugins[ $item ['file_path'] ] ) ) {
			$actions = array(
				'install' => sprintf(
					'<a href="%1$s" class="button button-primary" title="Install %2$s">Install</a>',
					esc_url(
						wp_nonce_url(
							add_query_arg(
								array(
									'page'          => rawurlencode( TGM_Plugin_Activation::$instance->menu ),
									'plugin'        => rawurlencode( $item['slug'] ),
									'plugin_name'   => rawurlencode( $item['sanitized_plugin'] ),
									'plugin_source' => rawurlencode( $item['source'] ),
									'tgmpa-install' => 'install-plugin',
									'return_url'    => 'dz-plugins',
								),
								admin_url( TGM_Plugin_Activation::$instance->parent_slug )
							),
							'tgmpa-install',
							'tgmpa-nonce'
						)
					),
					$item['sanitized_plugin']
				),
			);
		} elseif ( is_plugin_inactive( $item['file_path'] ) ) {
			if ( version_compare( $item['version'], $installed_plugins[ $item ['file_path'] ]['Version'], '>' ) ) {
				$actions = array(
					'update' => sprintf(
						'<a href="%1$s" class="button button-primary" title="Update %2$s">Update</a>',
						wp_nonce_url(
							add_query_arg(
								array(
									'page'          => rawurlencode( TGM_Plugin_Activation::$instance->menu ),
									'plugin'        => rawurlencode( $item['slug'] ),
									'plugin_name'   => rawurlencode( $item['sanitized_plugin'] ),
									'plugin_source' => rawurlencode( $item['source'] ),
									'tgmpa-update'  => 'update-plugin',
									'version'       => rawurlencode( $item['version'] ),
									'return_url'    => 'dz-plugins',
								),
								admin_url( TGM_Plugin_Activation::$instance->parent_slug )
							),
							'tgmpa-update',
							'tgmpa-nonce'
						),
						$item['sanitized_plugin']
					),
				);
			} else {

				$actions = array(
					'activate' => sprintf(
						'<a href="%1$s" class="button button-primary" title="Activate %2$s">Activate</a>',
						esc_url(
							add_query_arg(
								array(
									'plugin'            => rawurlencode( $item['slug'] ),
									'plugin_name'       => rawurlencode( $item['sanitized_plugin'] ),
									'plugin_source'     => rawurlencode( $item['source'] ),
									'dz-activate'       => 'activate-plugin',
									'dz-activate-nonce' => wp_create_nonce( 'dz-activate' ),
								),
								admin_url( 'admin.php?page=dz-plugins' )
							)
						),
						$item['sanitized_plugin']
					),
				);
			}
		} elseif ( version_compare( $item['version'], $installed_plugins[ $item ['file_path'] ]['Version'], '>' ) ) {

			$actions = array(
				'update' => sprintf(
					'<a href="%1$s" class="button button-primary" title="Update %2$s">Update</a>',
					wp_nonce_url(
						add_query_arg(
							array(
								'page'          => rawurlencode( TGM_Plugin_Activation::$instance->menu ),
								'plugin'        => rawurlencode( $item['slug'] ),
								'plugin_name'   => rawurlencode( $item['sanitized_plugin'] ),
								'plugin_source' => rawurlencode( $item['source'] ),
								'tgmpa-update'  => 'update-plugin',
								'version'       => rawurlencode( $item['version'] ),
								'return_url'    => 'dz-plugins',
							),
							admin_url( TGM_Plugin_Activation::$instance->parent_slug )
						),
						'tgmpa-update',
						'tgmpa-nonce'
					),
					$item['sanitized_plugin']
				),
			);
		} elseif ( $is_plug_act( $item['file_path'] ) ) {

			$actions = array(
				'deactivate' => sprintf(
					'<a href="%1$s" class="button button-primary" title="Deactivate %2$s">Deactivate</a>',
					esc_url(
						add_query_arg(
							array(
								'plugin'              => rawurlencode( $item['slug'] ),
								'plugin_name'         => rawurlencode( $item['sanitized_plugin'] ),
								'plugin_source'       => rawurlencode( $item['source'] ),
								'dz-deactivate'       => 'deactivate-plugin',
								'dz-deactivate-nonce' => wp_create_nonce( 'dz-deactivate' ),
							),
							admin_url( 'admin.php?page=dz-plugins' )
						)
					),
					$item['sanitized_plugin']
				),
			);
		}
		return $actions;
	}
}
get_template_part( '/dz-inc/library/functions' );
get_template_part( '/dz-inc/library/widgets/autorun' );
get_template_part( '/dz-inc/helpers/class-bootstrap-walker' );

if ( ! class_exists( 'W3Construct_Menu_Handle' ) && class_exists( 'DzThemeSupport' ) && defined( 'DZ_TH_ROOT' ) ) {
	require_once DZ_TH_ROOT . 'includes/library/classes/class-dz-menu-handle.php';
}

get_template_part( '/dz-inc/dz-redux/functions' );
get_template_part( '/dz-inc/dz-redux/template-functions' );
get_template_part( '/dz-inc/dz-redux/class-w3construct-redux-framework-config' );
get_template_part( '/dz-inc/dz-redux/dz-page-options' );


if ( ! class_exists( 'CSS_Generator' ) ) {
	get_template_part( '/dz-inc/dz-redux/classes/class-css-generator' );
}

$bootstrap_walker = new Bootstrap_Walker();

if ( is_admin() ) {
	/** Plugin Activation */
	include_once get_template_directory() . '/dz-inc/thirdparty/tgm-plugin-activation/plugins.php';
}
