<?php
/**
 * W3Construct_DZ_Recent_Post widget class
 *
 * @package W3Construct
 * @subpackage Widgets
 * @since 1.0.3
 */

if ( ! class_exists( 'W3Construct_DZ_Recent_Post' ) ) {
	/** Latest Post */
	class W3Construct_DZ_Recent_Post extends WP_Widget {
		/** Constructor */
		public function __construct() {
			parent::__construct( /* Base ID */'W3Construct_DZ_Recent_Post', /* Name */esc_html__( 'W3Construct Recent Post', 'w3construct' ), array( 'description' => esc_html__( 'Show the footer recent posts sidebar', 'w3construct' ) ) );
		}

		/**
		 * Widget Function: widget
		 *
		 * This function is a part of a WordPress widget and is responsible for rendering the widget's content
		 * on the front end of the website. It displays a list of recent posts with a specified category (if provided).
		 *
		 * @param array $args     The arguments for rendering the widget (HTML tags and titles).
		 * @param array $instance The widget's instance settings.
		 *
		 * @since 1.0.3
		 */
		public function widget( $args, $instance ) {
			$before_widget = $args['before_widget'];
			$after_widget  = $args['after_widget'];
			$before_title  = $args['before_title'];
			$after_title   = $args['after_title'];
			$title         = apply_filters( 'widget_title', $instance['title'] );

			$allowed_html_tags = w3construct_allowed_html_tag();

			echo wp_kses( $before_widget, $allowed_html_tags ); ?>
			<div class="widget-column">
				<?php echo wp_kses( $before_title . $title . $after_title, $allowed_html_tags ); ?>

				<div class="recent-posts-entry style-1">
					<?php
					$query_string = 'posts_per_page=' . $instance['number'];
					if ( $instance['cat'] ) {
						$query_string .= '&cat=' . $instance['cat'];
						$this->posts( $query_string );
					} else {
						$this->posts( $query_string );
					}
					?>
				</div>
			</div>
			<?php
			echo wp_kses( $after_widget, $allowed_html_tags );
		}
		/**
		 * Update Function: update
		 *
		 * This function is part of a WordPress widget and is responsible for updating the widget's settings
		 * when changes are made in the WordPress admin interface. It sanitizes and validates the new instance
		 * settings and returns the updated instance to be saved.
		 *
		 * @param array $new_instance The new widget settings provided by the user.
		 * @param array $old_instance The previous widget settings.
		 * @return array The updated and sanitized widget instance settings.
		 *
		 * @since 1.0.3
		 */
		public function update( $new_instance, $old_instance ) {
			$instance = $old_instance;

			$instance['title']  = wp_strip_all_tags( $new_instance['title'] );
			$instance['number'] = $new_instance['number'];
			$instance['cat']    = $new_instance['cat'];
			return $instance;
		}
		/**
		 * Form Function: form
		 *
		 * This function is part of a WordPress widget and is responsible for rendering the widget's settings form
		 * in the WordPress admin interface. It provides input fields for configuring the widget's title, the number
		 * of posts to display, and the category to filter by.
		 *
		 * @param array $instance The current widget settings.
		 *
		 * @since 1.0.3
		 */
		public function form( $instance ) {

			/**
			 * Resets properties to default values
			 *
			 * @return void
			 */

			$title  = ( $instance ) ? esc_attr( $instance['title'] ) : esc_html__( 'Recent Post', 'w3construct' );
			$number = ( $instance ) ? esc_attr( $instance['number'] ) : 3;
			$cat    = ( $instance ) ? esc_attr( $instance['cat'] ) : '';
			?>

			<p>
				<label for="<?php echo esc_attr( $this->get_field_id( 'title' ) ); ?>">
					<?php esc_html_e( 'Title: ', 'w3construct' ); ?>
				</label>
				<input class="widefat" id="<?php echo esc_attr( $this->get_field_id( 'title' ) ); ?>"
					name="<?php echo esc_attr( $this->get_field_name( 'title' ) ); ?>" type="text"
					value="<?php echo esc_attr( $title ); ?>" />
			</p>
			<p>
				<label for="<?php echo esc_attr( $this->get_field_id( 'number' ) ); ?>">
					<?php esc_html_e( 'No. of Posts:', 'w3construct' ); ?>
				</label>
				<input class="widefat" id="<?php echo esc_attr( $this->get_field_id( 'number' ) ); ?>"
					name="<?php echo esc_attr( $this->get_field_name( 'number' ) ); ?>" type="text"
					value="<?php echo esc_attr( $number ); ?>" />
			</p>
			<p>
				<label for="<?php echo esc_attr( $this->get_field_id( 'cat' ) ); ?>">
					<?php esc_html_e( 'Category', 'w3construct' ); ?>
				</label>
				<?php
				wp_dropdown_categories(
					array(
						'show_option_all' => esc_html__( 'All Categories', 'w3construct' ),
						'selected'        => $cat,
						'class'           => 'widefat',
						'name'            => $this->get_field_name( 'cat' ),
					)
				);
				?>
			</p>
			<?php
		}

		/**
		 * Posts Function: posts
		 *
		 * This function is responsible for querying and displaying a list of posts based on the provided query string.
		 * It retrieves posts and generates HTML markup to display each post's thumbnail, title, and date in a widget.
		 *
		 * @param string $query_string The query string used to fetch posts.
		 *
		 * @since 1.0.3
		 */
		public function posts( $query_string ) {
			/* Showing the HTML code for frontend */
			$query_w3 = new WP_Query( $query_string );
			if ( $query_w3->have_posts() ) :
				?>
				<!-- Title -->
				<?php
				while ( $query_w3->have_posts() ) :
					$query_w3->the_post();
					?>
					<!-- Widget Post -->
					<div class="widget-post-bx">
						<div class="widget-post clearfix">
							<div class="dz-media">
								<?php the_post_thumbnail( 'thumbnail' ); ?>
							</div>
							<div class="dz-info">
								<div class="dz-post-header">
									<h5 class="title"><a href="<?php echo esc_url( get_permalink( get_the_id() ) ); ?>">
											<?php echo wp_kses( w3construct_trim( get_the_title(), 5 ), w3construct_allowed_html_tag() ); ?>
										</a></h5>
								</div>
								<div class="dz-meta">
									<ul>
										<li class="post-date">
											<?php echo esc_html( get_the_date() ); ?>
										</li>
									</ul>
								</div>
							</div>
						</div>
					</div>
					<?php
				endwhile;
			endif;
			wp_reset_postdata();
		}
	}
}
