<?php
/**
 * W3Construct_DZ_Project widget class
 *
 * @package YourThemeName
 * @subpackage Widgets
 * @since 1.0.3
 */

if ( ! class_exists( 'W3Construct_DZ_Project' ) ) {
	/** Our Portfolio */
	class W3Construct_DZ_Project extends WP_Widget {
		/** Constructor */
		public function __construct() {
			parent::__construct( 'W3Construct_DZ_project', esc_html__( 'W3Construct Our Portfolio', 'w3construct' ), array( 'description' => esc_html__( 'Show the Our Projects', 'w3construct' ) ) );
		}

		/**
		 * Widget Function: widget
		 *
		 * This function is part of a custom WordPress widget that displays a gallery of projects or portfolio items.
		 * It renders the widget's content on the front end of the website, showing a list of projects.
		 *
		 * @param array $args     The arguments for rendering the widget (HTML tags and titles).
		 * @param array $instance The widget's instance settings, including the title, number of projects, and category.
		 *
		 * @since 1.0.3
		 */
		public function widget( $args, $instance ) {
			$before_widget = $args['before_widget'];
			$after_widget  = $args['after_widget'];
			$before_title  = $args['before_title'];
			$after_title   = $args['after_title'];
			$title         = apply_filters( 'widget_title', $instance['title'] );

			if ( strpos( $before_widget, 'footer-sidebar-1' ) ) {
				$before_widget = str_replace( 'col-lg-3 col-md-6', 'col-lg-6 col-md-12', $before_widget );
			}

			$allowed_html_tags = w3construct_allowed_html_tag();

			echo wp_kses( $before_widget, $allowed_html_tags );

			/**
			 * Showing the HTML code for frontend
			 */
			?>
			<!-- Our Projects -->
			<div class="widget_gallery me-0 me-xl-5 mb-0">
				<ul class="lightgallery">
					<?php
					$args = array(
						'post_type'      => 'dz-portfolio',
						'posts_per_page' => $instance['number'],
					);
					if ( ! empty( $instance['cat'] ) ) {
						$args['tax_query'] = array(
							array(
								'taxonomy' => 'portfolio-category',
								'field'    => 'id',
								'terms'    => (array) $instance['cat'],
							),
						);
					}
					$query = new WP_Query( $args );
					if ( $query->have_posts() ) {
						while ( $query->have_posts() ) {
							$query->the_post();
							global $post;
							?>
							<li>
								<div class="dlab-post-thum dlab-img-effect"><span
										data-exthumbimage="<?php echo esc_attr( get_the_post_thumbnail_url( get_the_id(), 'thumbnail' ) ); ?>"
										data-src="<?php echo esc_attr( get_the_post_thumbnail_url( get_the_id(), 'full' ) ); ?>"
										class="lightimg" title="<?php echo the_title_attribute(); ?>">
										<?php the_post_thumbnail( 'thumbnail' ); ?>
									</span>
								</div>
							</li>
							<?php
						}
						wp_reset_postdata();
					}
					?>
				</ul>
			</div>
			<?php
			echo wp_kses( $after_widget, $allowed_html_tags );
		}

		/**
		 * Update Function: update
		 *
		 * This function is part of a custom WordPress widget and is responsible for updating the widget's settings
		 * when changes are made in the WordPress admin interface. It sanitizes and validates the new instance
		 * settings, including the title, number of projects, and category, and returns the updated instance to be saved.
		 *
		 * @param array $new_instance The new widget settings provided by the user.
		 * @param array $old_instance The previous widget settings.
		 * @return array The updated and sanitized widget instance settings.
		 *
		 * @since 1.0.3
		 */
		public function update( $new_instance, $old_instance ) {
			$instance = $old_instance;

			$instance['title']  = wp_strip_all_tags( $new_instance['title'] );
			$instance['number'] = $new_instance['number'];
			$instance['cat']    = $new_instance['cat'];

			return $instance;
		}

		/**
		 * Form Function: form
		 *
		 * This function is part of a custom WordPress widget and is responsible for rendering the widget's settings form
		 * in the WordPress admin interface. It provides input fields for configuring the widget's title, the number of
		 * projects to display, and the category to filter by.
		 *
		 * @param array $instance The current widget settings, including title, number of projects, and category.
		 *
		 * @since 1.0.3
		 */
		public function form( $instance ) {
			/**
			 * Resets properties to default values
			 *
			 * @return void
			 */
			$title  = ( $instance ) ? esc_attr( $instance['title'] ) : esc_html__( 'Our Projects', 'w3construct' );
			$number = ( $instance ) ? esc_attr( $instance['number'] ) : 6;
			$cat    = ( $instance ) ? esc_attr( $instance['cat'] ) : '';
			?>

			<p>
				<label for="<?php echo esc_attr( $this->get_field_id( 'title' ) ); ?>">
					<?php esc_html_e( 'Title: ', 'w3construct' ); ?>
				</label>
				<input class="widefat" id="<?php echo esc_attr( $this->get_field_id( 'title' ) ); ?>"
					name="<?php echo esc_attr( $this->get_field_name( 'title' ) ); ?>" type="text"
					value="<?php echo esc_attr( $title ); ?>" />
			</p>
			<p>
				<label for="<?php echo esc_attr( $this->get_field_id( 'number' ) ); ?>">
					<?php esc_html_e( 'Number of posts: ', 'w3construct' ); ?>
				</label>
				<input class="widefat" id="<?php echo esc_attr( $this->get_field_id( 'number' ) ); ?>"
					name="<?php echo esc_attr( $this->get_field_name( 'number' ) ); ?>" type="text"
					value="<?php echo esc_attr( $number ); ?>" />
			</p>
			<p>
				<label for="<?php echo esc_attr( $this->get_field_id( 'cat' ) ); ?>">
					<?php esc_html_e( 'Category', 'w3construct' ); ?>
				</label>
				<?php
					wp_dropdown_categories(
						array(
							'show_option_all' => esc_html__( 'All Categories', 'w3construct' ),
							'selected'        => $cat,
							'taxonomy'        => 'portfolio-category',
							'class'           => 'widefat',
							'name'            => $this->get_field_name( 'cat' ),
						)
					);
				?>
			</p>

			<?php
		}
	}
}
