<?php
/**
 * W3Construct_DZ_Navigation widget class
 *
 * @package W3Construct
 * @subpackage Widgets
 * @since 1.0.3
 */

if ( ! class_exists( 'W3Construct_DZ_Navigation' ) ) {
	/* Footer Nevigation */
	/**
	 * W3Construct_DZ_Navigation Widget Class
	 *
	 * This class defines a custom WordPress widget for displaying navigation menus with different styles.
	 * It allows users to select a menu, set a title, and choose a style for displaying the menu in the widget area.
	 *
	 * @since 1.0.3
	 */
	class W3Construct_DZ_Navigation extends WP_Widget {

		/** Constructor */
		public function __construct() {
			parent::__construct( /* Base ID */'W3Construct_DZ_Navigation', /* Name */esc_html__( 'W3Construct Navigation', 'w3construct' ), array( 'description' => esc_html__( 'Show the Navigation', 'w3construct' ) ) );
		}

		/**
		 * Widget Function: widget
		 *
		 * This function is part of a custom WordPress widget that displays a navigation menu with various styles.
		 * It renders the widget's content on the front end of the website, showing the selected navigation menu.
		 *
		 * @param array $args     The arguments for rendering the widget (HTML tags and titles).
		 * @param array $instance The widget's instance settings, including the title, selected menu, and menu style.
		 *
		 * @since 1.0.3
		 */
		public function widget( $args, $instance ) {
			$before_widget = $args['before_widget'];
			$after_widget  = $args['after_widget'];
			$before_title  = $args['before_title'];
			$after_title   = $args['after_title'];

			/**
			 *
			 * Apply Filter On widget title for hook
			 */
			$title = apply_filters( 'widget_title', $instance['title'] );

			$allowed_html_tags = w3construct_allowed_html_tag();

			echo wp_kses( $before_widget, $allowed_html_tags );

			/**
			 *
			 * Showing the HTML code for frontend
			 */
			?>

			<div class="sticky-top">
				<div class="widget_categories ext-sidebar-menu">
					<?php echo wp_kses( $before_title . $title . $after_title, $allowed_html_tags ); ?>
					<?php

					$menu_items = wp_get_nav_menu_items( $instance['selected_menu'] );
					?>
					<?php if ( 'style_1' === $instance['menu_style'] ) { ?>
						<ul>
						<?php } else { ?>
							<ul class="list-2">
							<?php } ?>
							<?php foreach ( $menu_items as $menu_item ) { ?>
								<li class="cat-item">
									<a href="<?php echo esc_url( $menu_item->url ); ?>">
										<?php echo esc_html( $menu_item->title ); ?>
									</a>
								</li>
							<?php } ?>
						</ul>
				</div>
			</div>
			<?php
			echo wp_kses( $after_widget, $allowed_html_tags );
		}


		/**
		 * Update Function: update
		 *
		 * This function is part of a custom WordPress widget and is responsible for updating the widget's settings
		 * when changes are made in the WordPress admin interface. It sanitizes and validates the new instance
		 * settings, including the title, menu style, and selected menu, and returns the updated instance to be saved.
		 *
		 * @param array $new_instance The new widget settings provided by the user.
		 * @param array $old_instance The previous widget settings.
		 * @return array The updated and sanitized widget instance settings.
		 *
		 * @since 1.0.3
		 */
		public function update( $new_instance, $old_instance ) {
			$instance = $old_instance;

			$instance['title']         = wp_strip_all_tags( $new_instance['title'] );
			$instance['menu_style']    = wp_strip_all_tags( $new_instance['menu_style'] );
			$instance['selected_menu'] = wp_strip_all_tags( $new_instance['selected_menu'] );
			return $instance;
		}

		/**
		 * Form Function: form
		 *
		 * This function is part of a custom WordPress widget and is responsible for rendering the widget's settings form
		 * in the WordPress admin interface. It provides input fields for configuring the widget's title, selecting a menu,
		 * and choosing a menu style.
		 *
		 * @param array $instance The current widget settings, including title, selected menu, and menu style.
		 *
		 * @since 1.0.3
		 */
		public function form( $instance ) {
			/**
			 * Resets properties to default values
			 *
			 * @return void
			 */

			$title         = ( $instance ) ? esc_attr( $instance['title'] ) : esc_html__( 'Navigation', 'w3construct' );
			$selected_menu = ! empty( $instance['selected_menu'] ) ? $instance['selected_menu'] : '';
			$menu_style    = ! empty( $instance['menu_style'] ) ? $instance['menu_style'] : '';
			?>

			<p>
				<label for="<?php echo esc_attr( $this->get_field_id( 'title' ) ); ?>">
					<?php esc_html_e( 'Title:', 'w3construct' ); ?>
				</label>
				<input placeholder="<?php esc_attr_e( 'Menu', 'w3construct' ); ?>" class="widefat"
					id="<?php echo esc_attr( $this->get_field_id( 'title' ) ); ?>"
					name="<?php echo esc_attr( $this->get_field_name( 'title' ) ); ?>" type="text"
					value="<?php echo esc_attr( $title ); ?>" />
			</p>
			<p>
				<label for="<?php echo esc_attr( $this->get_field_id( 'Select Menu' ) ); ?>">
					<?php esc_html_e( 'Select Menu:', 'w3construct' ); ?>
				</label>
				<select class="widefat" id="<?php echo esc_attr( $this->get_field_id( 'selected_menu' ) ); ?>"
					name="<?php echo esc_attr( $this->get_field_name( 'selected_menu' ) ); ?>">
					<?php

					$all_menus = get_terms( 'nav_menu', array( 'hide_empty' => true ) );
					foreach ( $all_menus as $menu ) {
						$selected = ( $selected_menu === $menu->slug ) ? ' selected="selected" ' : '';
						?>
						<option value="<?php echo esc_attr( $menu->slug ); ?>" <?php echo esc_attr( $selected ); ?>>
							<?php echo esc_html( $menu->name ); ?>
						</option>
						<?php
					}
					?>
				</select>
			</p>
			<p>
				<label for="<?php echo esc_attr( $this->get_field_id( 'Select Menu Style' ) ); ?>">
					<?php esc_html_e( 'Select Menu Style:', 'w3construct' ); ?>
				</label>
				<?php
				$style_arr = array(
					'style_1' => esc_html__( 'Style 1', 'w3construct' ),
					'style_2' => esc_html__( 'Style 2', 'w3construct' ),
				);
				?>
				<select data="<?php echo esc_attr( $menu_style ); ?>" class="widefat"
					id="<?php echo esc_attr( $this->get_field_id( 'menu_style' ) ); ?>"
					name="<?php echo esc_attr( $this->get_field_name( 'menu_style' ) ); ?>">
					<option value="">
						<?php echo esc_html__( 'Choose Style', 'w3construct' ); ?>
					</option>
					<?php
					foreach ( $style_arr as $style_key => $style_value ) {
						$style_selected = ( $style_key === $menu_style ) ? 'selected="selected"' : '';
						?>
						<option value="<?php echo esc_attr( sanitize_title( $style_key ) ); ?>" <?php echo esc_attr( $style_selected ); ?>>
							<?php echo esc_html( $style_value ); ?>
						</option>
					<?php } ?>
				</select>
			</p>
			<?php
		}

	}
}
