<?php
/**
 * Walker_Nav_Menu_Edit class.
 *
 * Create HTML list of nav menu input items.
 *
 * @package WordPress
 * @since 3.0.0
 * @uses Walker_Nav_Menu
 */

/**
 * W3Construct_Walker_Nav_Mega_Menu_Edit
 */
class W3Construct_Walker_Nav_Mega_Menu_Edit extends Walker_Nav_Menu {

	/**
	 * Start rendering a submenu level.
	 *
	 * This method is called when a submenu is encountered, and it's responsible for generating
	 * the opening part of a submenu level, typically used in dropdown menus.
	 *
	 * @since 3.0.0
	 *
	 * @param string $output Passed by reference. Used to append additional content.
	 * @param int    $depth  Depth of the submenu level. Used for indentation.
	 * @param array  $args   An array containing various arguments and settings.
	 */
	public function start_lvl( &$output, $depth = 0, $args = array() ) { }

	/**
	 * End rendering a submenu level.
	 *
	 * This method is called when the rendering of a submenu level is completed, and it's responsible for generating
	 * the closing part of a submenu level, typically used in dropdown menus.
	 *
	 * @since 3.0.0
	 *
	 * @param string $output Passed by reference. Used to append additional content.
	 * @param int    $depth  Depth of the submenu level. Used for indentation.
	 * @param array  $args   An array containing various arguments and settings.
	 */
	public function end_lvl( &$output, $depth = 0, $args = array() ) { }

	/**
	 * Generates the HTML output for a menu item in a custom Walker class.
	 *
	 * This method is called when rendering a menu item in a custom Walker class. It outputs the HTML markup for the menu item,
	 * including its title, link, and additional settings like CSS classes and attributes. This method can be customized to
	 * control the appearance and behavior of menu items in your theme.
	 *
	 * @see Walker::start_el()
	 * @since 3.0.0
	 *
	 * @param string $output Passed by reference. Used to append additional content.
	 * @param object $item Menu item data object.
	 * @param int    $depth Depth of menu item. Used for padding.
	 * @param object $args Additional arguments for rendering.
	 * @param int    $id (Optional) The menu item's ID.
	 */
	public function start_el( &$output, $item, $depth = 0, $args = array(), $id = 0 ) {
		global $_wp_nav_menu_max_depth;
		$_wp_nav_menu_max_depth = $depth > $_wp_nav_menu_max_depth ? $depth : $_wp_nav_menu_max_depth;
		$indent                 = ( $depth ) ? str_repeat( "\t", $depth ) : '';
		ob_start();
		$item_id      = esc_attr( $item->ID );
		$removed_args = array(
			'action',
			'customlink-tab',
			'edit-menu-item',
			'menu-item',
			'page-tab',
			'_wpnonce',
		);

		$original_title = '';
		if ( 'taxonomy' === $item->type ) {
			$original_title = get_term_field( 'name', $item->object_id, $item->object, 'raw' );
			if ( is_wp_error( $original_title ) ) {
				$original_title = false;
			} elseif ( 'post_type' === $item->type ) {
				$original_object = get_post( $item->object_id );
				$original_title  = $original_object->post_title;
			}
			$menu_status_class = ( isset( $_GET['edit-menu-item'] ) && $item_id === $_GET['edit-menu-item'] || check_admin_referer( sanitize_text_field( wp_unslash( $_GET['edit-menu-item'], 'edit-menu-item' ) ) ) ) ? 'active' : 'inactive';
			$classes           = array(
				'menu-item menu-item-depth-' . $depth,
				'menu-item-' . esc_attr( $item->object ),
				'menu-item-edit-' . esc_attr( $menu_status_class ),
			);

			$title = $item->title;

			if ( ! empty( $item->_invalid ) ) {
				$classes[] = 'menu-item-invalid';
				/* translators: %s: title of menu item which is invalid */
				$title = sprintf( esc_attr_e( '%s ( Invalid )', 'w3construct' ), $item->title );
			} elseif ( isset( $item->post_status ) && 'draft' === $item->post_status ) {
				$classes[] = 'pending';
				/* translators: %s: title of menu item in draft status */
				$title = sprintf( '%s ( Pending )', $item->title );
			}

			$title             = empty( $item->label ) ? $title : $item->label;
			$menu_item_classes = implode( ' ', $classes );
			?>
			<li id="menu-item-<?php echo esc_attr( $item_id ); ?>" class="<?php echo esc_attr( $menu_item_classes ); ?>">
				<dl class="menu-item-bar">
					<dt class="menu-item-handle">
						<span class="item-title">
							<?php echo esc_html( $title ); ?>
						</span>
						<span class="item-controls">
							<span class="item-type">
								<?php echo esc_html( $item->type_label ); ?>
							</span>
							<span class="item-order hide-if-js">
							<a href="
								<?php
									echo esc_url(
										wp_nonce_url(
											add_query_arg(
												array(
													'action' => 'move-up-menu-item',
													'menu-item' => $item_id,
												),
												remove_query_arg( $removed_args, admin_url( 'nav-menus.php' ) )
											),
											'move-menu_item'
										)
									);
								?>
								" class="item-move-up">
								<abbr title="<?php esc_attr_e( 'Move up', 'w3construct' ); ?>">&#8593;</abbr>
							</a> |
							<a href="
								<?php
									echo esc_url(
										wp_nonce_url(
											add_query_arg(
												array(
													'action' => 'move-down-menu-item',
													'menu-item' => $item_id,
												),
												remove_query_arg( $removed_args, admin_url( 'nav-menus.php' ) )
											),
											'move-menu_item'
										)
									);
								?>
								" class="item-move-down">
								<abbr title="<?php esc_attr_e( 'Move down', 'w3construct' ); ?>">&#8595;</abbr>
							</a>
							</span>
							<a class="item-edit" id="edit-<?php echo esc_attr( $item_id ); ?>" title="<?php esc_attr_e( 'Edit Menu Item', 'w3construct' ); ?>" href="<?php echo esc_url( ( isset( $_GET['edit-menu-item'] ) && $item_id === $_GET['edit-menu-item'] || check_admin_referer( sanitize_text_field( wp_unslash( $_GET['edit-menu-item'], 'edit-menu-item' ) ) ) ) ? admin_url( 'nav-menus.php' ) : add_query_arg( 'edit-menu-item', $item_id, remove_query_arg( $removed_args, admin_url( 'nav-menus.php#menu-item-settings-' . $item_id ) ) ) ); ?>">
							</a>
						</span>
					</dt>
				</dl>

				<div class="menu-item-settings wp-clearfix" id="menu-item-settings-<?php echo esc_attr( $item_id ); ?>">
					<?php if ( 'custom' === $item->type ) : ?>
						<p class="field-url description description-wide">
							<label for="edit-menu-item-url-<?php echo esc_attr( $item_id ); ?>">
								<?php esc_html_e( 'URL', 'w3construct' ); ?><br />
								<input type="text" id="edit-menu-item-url-<?php echo esc_attr( $item_id ); ?>"
									class="widefat code edit-menu-item-url" name="menu-item-url[<?php echo esc_attr( $item_id ); ?>]"
									value="<?php echo esc_attr( $item->url ); ?>" />
							</label>
						</p>
					<?php endif; ?>
					<p class="description description-thin">
						<label for="edit-menu-item-title-<?php echo esc_attr( $item_id ); ?>">
							<?php esc_html_e( 'Navigation Label', 'w3construct' ); ?><br />
							<input type="text" id="edit-menu-item-title-<?php echo esc_attr( $item_id ); ?>"
								class="widefat edit-menu-item-title" name="menu-item-title[<?php echo esc_attr( $item_id ); ?>]"
								value="<?php echo esc_attr( $item->title ); ?>" />
						</label>
					</p>
					<p class="description description-thin">
						<label for="edit-menu-item-attr-title-<?php echo esc_attr( $item_id ); ?>">
							<?php esc_html_e( 'Title Attribute', 'w3construct' ); ?><br />
							<input type="text" id="edit-menu-item-attr-title-<?php echo esc_attr( $item_id ); ?>"
								class="widefat edit-menu-item-attr-title"
								name="menu-item-attr-title[<?php echo esc_attr( $item_id ); ?>]"
								value="<?php echo esc_attr( $item->post_excerpt ); ?>" />
						</label>
					</p>
					<p class="field-link-target description">
						<label for="edit-menu-item-target-<?php echo esc_attr( $item_id ); ?>">
							<input type="checkbox" id="edit-menu-item-target-<?php echo esc_attr( $item_id ); ?>" value="_blank"
								name="menu-item-target[<?php echo esc_attr( $item_id ); ?>]" <?php checked( $item->target, '_blank' ); ?> />
							<?php esc_html_e( 'Open link in a new window/tab', 'w3construct' ); ?>
						</label>
					</p>
					<p class="field-css-classes description description-thin">
						<label for="edit-menu-item-classes-<?php echo esc_attr( $item_id ); ?>">
							<?php esc_html_e( 'CSS Classes ( optional )', 'w3construct' ); ?><br />
							<input type="text" id="edit-menu-item-classes-<?php echo esc_attr( $item_id ); ?>"
								class="widefat code edit-menu-item-classes"
								name="menu-item-classes[<?php echo esc_attr( $item_id ); ?>]"
								value="<?php echo esc_attr( implode( ' ', $item->classes ) ); ?>" />
						</label>
					</p>
					<p class="field-xfn description description-thin">
						<label for="edit-menu-item-xfn-<?php echo esc_attr( $item_id ); ?>">
							<?php esc_html_e( 'Link Relationship ( XFN )', 'w3construct' ); ?><br />
							<input type="text" id="edit-menu-item-xfn-<?php echo esc_attr( $item_id ); ?>"
								class="widefat code edit-menu-item-xfn" name="menu-item-xfn[<?php echo esc_attr( $item_id ); ?>]"
								value="<?php echo esc_attr( $item->xfn ); ?>" />
						</label>
					</p>
					<p class="field-description description description-wide">
						<label for="edit-menu-item-description-<?php echo esc_attr( $item_id ); ?>">
							<?php esc_html_e( 'Description', 'w3construct' ); ?><br />
							<textarea id="edit-menu-item-description-<?php echo esc_attr( $item_id ); ?>"
								class="widefat edit-menu-item-description" rows="3" cols="20"
								name="menu-item-description[<?php echo esc_attr( $item_id ); ?>]"><?php echo esc_html( $item->description ); ?></textarea>
							<span class="description">
								<?php esc_html_e( 'The description will be displayed in the menu if the current theme supports it.', 'w3construct' ); ?>
							</span>
						</label>
					</p>
					<?php
					/* Read the menu setting from post meta ( menu id, key, single ) */
					$mega_menu_cat         = get_post_meta( $item->ID, 'mega_menu_cat', true );
					$mega_menu_no_of_post  = get_post_meta( $item->ID, 'mega_menu_no_of_post', true );
					$mega_menu_images_only = get_post_meta( $item->ID, 'mega_menu_images_only', true );
					$mega_menu_no_of_post  = ! empty( $mega_menu_no_of_post ) ? $mega_menu_no_of_post : 6;

					$mega_menu_images_only = ! empty( $mega_menu_images_only ) ? $mega_menu_images_only : 'no';
					$mega_menu_cat         = explode( ',', $mega_menu_cat );

					/* Make the tree */
					$category_tree = array_merge(
						array( ' - Not mega menu - ' => '0' ),
						Mega_Menu_Util::get_category_to_id_array( false )
					);
					/* New fields insertion starts here */
					?>
					<p class="description description-wide"><br><br>
						<label>
							<?php esc_html_e( 'Make this a category mega menu', 'w3construct' ); ?>
						</label>
						<select name="mega_menu_cat_<?php echo esc_attr( $item->ID ); ?>[]" size="8" multiple="multiple"
							class="widefat code edit-menu-item-url">
							<?php
							foreach ( $category_tree as $category => $category_id ) {
								?>
								<option value="
								<?php
								echo esc_attr( $category_id );
								?>
								"
								<?php
								if ( in_array( $category_id, $mega_menu_cat, true ) ) {
									echo 'selected="selected"';
								}
								?>
								>
								<?php
								echo wp_kses( $category, 'string' );
								?>
								</option>
								<?php
							}
							?>
						</select>
					</p>
					<p class="field-no-of-post description description-wide">
						<label>
							<?php esc_html_e( 'No. of post per category ( Default 6 )', 'w3construct' ); ?><br />
							<input type="text" name="mega_menu_no_of_post_<?php echo esc_attr( $item->ID ); ?>"
								value="<?php echo esc_attr( $mega_menu_no_of_post ); ?>">
						</label>
					</p>
					<?php
					$is_mega_menu_images_only = ( 'yes' === $mega_menu_images_only ) ? 'checked="checked"' : '';
					?>
					<p class="field-no-of-post description description-wide">
						<label>
							<?php esc_html_e( 'Posts With Images Only: ( Default No )', 'w3construct' ); ?><br />
							<input type="checkbox" <?php echo esc_attr( $is_mega_menu_images_only ); ?>
								name="mega_menu_images_only_<?php echo esc_attr( $item->ID ); ?>" value="yes">
						</label>
					</p>
					<?php
					/* New fields insertion ends here */
					?>
					<div class="menu-item-actions description-wide submitbox">
						<?php if ( 'custom' !== $item->type && false !== $original_title ) : ?>
							<p class="link-to-original">
							<?php
							/* translators: %s is a placeholder for the original title */
							printf( esc_html_e( 'Original: %s', 'w3construct' ), '<a href="' . esc_attr( $item->url ) . '">' . esc_html( $original_title ) . '</a>' );
							?>
							</p>
							<?php
							endif;
						?>
						<a class="item-delete submitdelete deletion" id="delete-
						<?php echo esc_attr( $item_id ); ?>" href="
						<?php
						echo esc_url(
							wp_nonce_url(
								add_query_arg(
									array(
										'action'    => 'delete-menu-item',
										'menu-item' => $item_id,
									),
									remove_query_arg( $removed_args, admin_url( 'nav-menus.php' ) )
								),
								'delete-menu_item_' . $item_id
							)
						);
						?>
						">
							<?php esc_html_e( 'Remove', 'w3construct' ); ?>
							</a> <span class="meta-sep"> | </span> <a class="item-cancel submitcancel"
								id="cancel-<?php echo esc_attr( $item_id ); ?>"
								href="
								<?php
								echo esc_url(
									add_query_arg(
										array(
											'edit-menu-item' => $item_id,
											'cancel' => time(),
										),
										remove_query_arg( $removed_args, admin_url( 'nav-menus.php' ) )
									)
								);
								?>
								#menu-item-settings-
								<?php
								echo esc_attr( $item_id );
								?>
								">
								<?php
								esc_html_e( 'Cancel', 'w3construct' );
								?>
							</a>
					</div>

					<input class="menu-item-data-db-id" type="hidden" name="menu-item-db-id[<?php echo esc_attr( $item_id ); ?>]"
						value="<?php echo esc_attr( $item_id ); ?>" />
					<input class="menu-item-data-object-id" type="hidden"
						name="menu-item-object-id[<?php echo esc_attr( $item_id ); ?>]"
						value="<?php echo esc_attr( $item->object_id ); ?>" />
					<input class="menu-item-data-object" type="hidden" name="menu-item-object[<?php echo esc_attr( $item_id ); ?>]"
						value="<?php echo esc_attr( $item->object ); ?>" />
					<input class="menu-item-data-parent-id" type="hidden"
						name="menu-item-parent-id[<?php echo esc_attr( $item_id ); ?>]"
						value="<?php echo esc_attr( $item->menu_item_parent ); ?>" />
					<input class="menu-item-data-position" type="hidden"
						name="menu-item-position[<?php echo esc_attr( $item_id ); ?>]"
						value="<?php echo esc_attr( $item->menu_order ); ?>" />
					<input class="menu-item-data-type" type="hidden" name="menu-item-type[<?php echo esc_attr( $item_id ); ?>]"
						value="<?php echo esc_attr( $item->type ); ?>" />
				</div><!-- .menu-item-settings-->
				<ul class="menu-item-transport"></ul>
				<?php
				$output .= ob_get_clean();
		}
	}
}
get_template_part( 'dz-inc/library/mega_menu/class-mega-menu-util' );
get_template_part( 'dz-inc/library/mega_menu/class-category-to-id-array-walker' );
