<?php
/**
 * Theme Information for W3Construct
 *
 * @package     W3Construct
 * @author      W3ITEXPERTS
 * @copyright   Copyright (c) 2023, W3Construct
 * @link        http://w3construct-wordpress.wprdx.come/demo/
 * @since       W3Construct 1.0.1
 */

if ( ! function_exists( 'w3construct_dzbase' ) ) {
	/** Set the global base to get meta data */
	function w3construct_dzbase() {
		return $GLOBALS['_dz_base'];
	}
}

if ( ! function_exists( 'w3construct_get_categories' ) ) {
	/**
	 * A function to fetch the categories from WordPress.
	 *
	 * @param mixed $arg An optional argument to customize the category retrieval.
	 */
	function w3construct_get_categories( $arg = false ) {
		global $wp_taxonomies;
		$w3construct_categories_default_args = array(
			'orderby'      => 'ID',
			'order'        => 'ASC',
			'hide_empty'   => 1,
			'hierarchical' => true,
		);
		if ( ! empty( $arg['taxonomy'] ) && isset( $wp_taxonomies[ $arg['taxonomy'] ] ) ) {
			$w3construct_categories_default_args['taxonomy'] = $arg['taxonomy'];
		}

		$categories = get_categories( $w3construct_categories_default_args );
		$cats       = array();

		if ( ! is_wp_error( $categories ) ) {
			foreach ( $categories as $category ) {
				$cats[ $category->slug ] = wp_kses( $category->name, 'string' );
			}
		}
		return $cats;
	}
}

if ( ! function_exists( 'w3construct_get_the_breadcrumb' ) ) {
	/**  Show the breadcrum for frontend */
	function w3construct_get_the_breadcrumb() {
		global $wp_query;
		$queried_object = get_queried_object();
		$breadcrumb     = '';
		$delimiter      = ' ';
		$before         = '<li class="breadcrumb-item">';
		$after          = '</li>';
		if ( ! is_home() ) {
			$breadcrumb .= '<li class="breadcrumb-item"><a href="' . esc_url( home_url( '/' ) ) . '"> ' . esc_html__( 'Home', 'w3construct' ) . '</a></li>';

			/** If category or single post */
			if ( is_category() ) {
				$cat_obj       = $wp_query->get_queried_object();
				$this_category = get_category( $cat_obj->term_id );

				if ( 0 !== $this_category->parent ) {
					$parent_category = get_category( $this_category->parent );
					$parent_string   = get_category_parents( $parent_category, true, '@' );
					$parent_tree     = explode( '@', $parent_string );
					$parent_tree     = array_filter( $parent_tree );
					foreach ( $parent_tree as $cat_link ) {
						$breadcrumb .= '<li class="breadcrumb-item">' . $cat_link . '</li>';
					}
				}
				$single_cat_title = single_cat_title( '', false );
				$breadcrumb      .= '<li class="breadcrumb-item">' . $single_cat_title . '</li>';
			} elseif ( is_tax() ) {
				$term_link   = get_term_link( $queried_object );
				$term_name   = $queried_object->name;
				$breadcrumb .= '<li class="breadcrumb-item"><a href = "' . esc_url( $term_link ) . '">' . $term_name . '</a></li>';
			} elseif ( is_page() ) {/** If WP pages */
				global $post;
				if ( $post->post_parent ) {
					$anc = get_post_ancestors( $post->ID );
					foreach ( $anc as $ancestor ) {
						$perma_link     = get_permalink( $ancestor );
						$ancestor_title = get_the_title( $ancestor );
						$post_title     = get_the_title( $post->ID );
						$breadcrumb    .= '<li class="breadcrumb-item"><a href="' . esc_url( $perma_link ) . '">' . $ancestor_title . '</a></li>';
					}
					$breadcrumb .= '<li class="breadcrumb-item">' . wp_kses( $post_title, 'string' ) . '</li>';

				} else {
					$post_title  = get_the_title();
					$breadcrumb .= '<li class="breadcrumb-item">' . wp_kses( $post_title, 'string' ) . '</li>';
				}
			} elseif ( is_singular() ) {
				$category = wp_get_object_terms( get_the_ID(), 'category' );
				if ( $category ) {
					if ( ! is_wp_error( $category ) ) {
						$term_link   = get_term_link( w3construct_set( $category, '0' ) );
						$name        = w3construct_set( w3construct_set( $category, '0' ), 'name' );
						$title       = get_the_title();
						$breadcrumb .= '<li class = "breadcrumb-item"><a href = "' . esc_url( $term_link ) . '">' . $name . '</a></li>';
						$breadcrumb .= '<li class = "breadcrumb-item">' . wp_kses( $title, 'string' ) . '</li>';
					} else {
						$title       = get_the_title();
						$breadcrumb .= '<li class = "breadcrumb-item">' . wp_kses( $title, 'string' ) . '</li>';
					}
				} else {
					$title = wp_get_document_title();

					$breadcrumb .= '<li class="breadcrumb-item">' . wp_kses( $title, 'string' ) . '</li>';
				}
			} elseif ( is_tag() ) {
				$term_link   = get_term_link( $queried_object );
				$title       = single_tag_title( '', false );
				$breadcrumb .= '<li class="breadcrumb-item"><a href="' . esc_url( $term_link ) . '">' . wp_kses( $title, 'string' ) . '</a></li>'; /**If tag template*/
			} elseif ( is_day() ) {
				$time        = get_the_time( 'F jS, Y' );
				$breadcrumb .= '<li class="breadcrumb-item"><a href="#">' . esc_html__( 'Archive for ', 'w3construct' ) . wp_kses( $time, 'string' ) . '</a></li>'; /** If daily Archives */
			} elseif ( is_month() ) {
				$link        = get_month_link( get_the_time( 'Y' ), get_the_time( 'm' ) );
				$time        = get_the_time( 'F, Y' );
				$breadcrumb .= '<li class="breadcrumb-item"><a href="' . esc_url( $link ) . '">' . esc_html__( 'Archive for ', 'w3construct' ) . wp_kses( $time, 'string' ) . '</a></li>'; /** If montly Archives */
			} elseif ( is_year() ) {
				$link        = get_year_link( get_the_time( 'Y' ) );
				$time        = get_the_time( 'Y' );
				$breadcrumb .= '<li class="breadcrumb-item"><a href="' . esc_url( $link ) . '">' . esc_html__( 'Archive for ', 'w3construct' ) . wp_kses( $time, 'string' ) . '</a></li>'; /** If year Archives */
			} elseif ( is_author() ) {
				$link        = get_author_posts_url( get_the_author_meta( 'ID' ) );
				$author      = get_the_author();
				$breadcrumb .= '<li class="breadcrumb-item"><a href="' . esc_url( $link ) . '">' . esc_html__( 'Archive for ', 'w3construct' ) . wp_kses( $author, 'string' ) . '</a></li>'; /** If author Archives */
			} elseif ( is_search() ) {
				$search_query = get_search_query();
				$breadcrumb  .= '<li class="breadcrumb-item">' . esc_html__( 'Search Results for ', 'w3construct' ) . $search_query . '</li>'; /** If search template */
			} elseif ( is_404() ) {
				$breadcrumb .= '<li class="breadcrumb-item">' . esc_html__( '404 - Not Found', 'w3construct' ) . '</li>'; /** If search template */
			} elseif ( is_shop() && w3construct_is_woocommerce_active() ) {
				$term_name   = w3construct_get_opt( 'woocommerce_page_title' );
				$breadcrumb .= '<li class="breadcrumb-item"><a href="javascript:void( 0 )">' . wp_kses( $term_name, 'string' ) . '</a></li>'; /** If shop template */
			} elseif ( is_archive() ) {
				$link        = get_permalink( get_the_ID() );
				$breadcrumb .= '<li class="breadcrumb-item"><a href="' . esc_url( $link ) . '">' . wp_get_document_title() . '</a></li>';
			} else {
				$link  = get_permalink();
				$title = get_the_title();
				if ( ! empty( $title ) ) {
					$breadcrumb .= '<li class="breadcrumb-item"><a href="' . esc_url( $link ) . '">' . esc_html( $title ) . '</a></li>'; /** Default value */
				}
			}

			/** To hide if only one li is available */
			if ( substr_count( $breadcrumb, '<li class="breadcrumb-item">' ) <= 1 ) {
				$breadcrumb = '';
			}
		}
		return '<ul class="breadcrumb">' . $breadcrumb . '</ul>';
	}
}

if ( ! function_exists( 'w3construct_bunch_list_comments' ) ) {
	/**
	 * Show the comment list.
	 *
	 * @param WP_Comment $comment The comment object.
	 * @param array      $args An array of arguments.
	 * @param int        $depth The depth of the comment.
	 */
	function w3construct_bunch_list_comments( $comment, $args, $depth ) {

		/** $GLOBALSW3['comment'] = $comment; */
		$g_l_o_b_a_l_s_w3['comment'] = $comment;
		$email                       = w3construct_set( $comment, 'comment_author_email' );
		$avatar                      = get_avatar( $email, 130 );
		$comment_id                  = get_comment_ID();
		$author_link                 = get_comment_author( $comment_id );
		$comment_date_link           = get_month_link( get_the_date( 'Y' ), get_the_date( 'm' ) );
		$comment_date                = get_comment_date( 'F j, Y', $comment_id );

		$allowed_html_tags = w3construct_allowed_html_tag();

		?>
		<li <?php comment_class( 'comment' ); ?> id="comment-<?php comment_ID(); ?>">
			<div class="comment-body" id="div-comment-<?php comment_ID(); ?>">
				<div class="comment-author vcard">
					<?php echo wp_kses( $avatar, $allowed_html_tags ); ?>
					<cite class="fn">
						<?php echo wp_kses( $author_link, $allowed_html_tags ); ?>
					</cite> <span class="says">
						<?php esc_html_e( 'says:', 'w3construct' ); ?>
					</span>
					<div class="comment-meta">
						<a href="<?php echo esc_url( $comment_date_link ); ?>">
							<?php echo wp_kses( $comment_date, $allowed_html_tags ); ?>
							<?php esc_html_e( 'at', 'w3construct' ); ?>
							<?php the_time( 'g:i a' ); ?>
						</a>
					</div>
				</div>
				<div class="comment-content dz-page-text">
					<?php comment_text(); ?>
				</div>
				<div class="reply">
					<?php
					comment_reply_link(
						array_merge(
							$args,
							array(
								'depth'      => $depth,
								'max_depth'  => $args['max_depth'],
								'reply_text' => '<i class = "fa fa-reply"></i> Reply',
								'w3construct',
								'add_below'  => 'div-comment',
							)
						)
					);
					?>
				</div>
			</div>
			<?php
	}
}

if ( ! function_exists( 'w3construct_comment_form' ) ) {
	/**
	 * Returns the formatted form of the comments.
	 *
	 * @param array $args       An array of arguments to be filtered.
	 * @param int   $post_id_w3 If the form is called within the loop, then post_id is optional.
	 * @param bool  $review     Whether this is a review.
	 */
	function w3construct_comment_form( $args = array(), $post_id_w3 = null, $review = false ) {
		$allowed_html_tags = w3construct_allowed_html_tag();

		if ( null === $post_id_w3 ) {
			$post_id_w3 = get_the_ID();
		} else {
			$id = $post_id_w3;
		}
		$commenter     = wp_get_current_commenter();
		$user          = wp_get_current_user();
		$user_identity = $user->exists() ? $user->display_name : '';
		$args          = wp_parse_args( $args );
		if ( ! isset( $args['format'] ) ) {
			$args['format'] = current_theme_supports( 'html5', 'comment-form' ) ? 'html5' : 'xhtml';
		}
		$req      = get_option( 'require_name_email' );
		$aria_req = ( $req ? " aria-required = 'true'" : '' );
		$html5    = 'html5' === $args['format'];
		$fields   = array(
			'author' => '<p class="comment-form-author"><input id="name" placeholder="' . esc_attr__( 'Author', 'w3construct' ) . '"  name="author" type="text" value="' . esc_attr( $commenter['comment_author'] ) . '" ' . $aria_req . ' /></p>',
			'email'  => ' <p class="comment-form-email"><input id="email" required placeholder="' . esc_attr__( 'Email', 'w3construct' ) . '" name="email" ' . ( $html5 ? 'type="email"' : 'type="text"' ) . ' value="' . esc_attr( $commenter['comment_author_email'] ) . '" ' . $aria_req . ' /></p>',
		);

		/* translators: %s: Required Sign */
		$required_text = sprintf( ' ' . esc_html__( 'Required fields are marked %s', 'w3construct' ), '<span class="required">*</span>' );

		/**
		 * Filter the default comment form fields.
		 *
		 * @since 3.0.0
		 *
		 * @param array $fields The default comment fields.
		 */
		$w3construct_col = ( is_user_logged_in() ) ? 'col-md-12' : '';
		$fields          = apply_filters( 'comment_form_default_fields', $fields );
		$defaults        = array(
			'fields'               => $fields,
			'comment_field'        => '<p class = "comment-form-comment"><textarea id                          = "comments" placeholder   = "' . esc_attr__( 'Type Comment Here', 'w3construct' ) . '" class = "form-control4" name = "comment" cols = "45" rows = "3" aria-required = "true" required = "required"></textarea></p>',
			/* translators: %s is a placeholder for the original title */
			'must_log_in'          => '<p class = "col-md-12 col-sm-12">' . sprintf( __( 'You must be <a href  = "%s">logged in</a> to post a comment.', 'w3construct' ), esc_url( wp_login_url( apply_filters( 'the_permalink', get_permalink( $post_id_w3 ) ) ) ) ) . '</p>',
			/* translators: %s is a placeholder for the original title */
			'logged_in_as'         => '<p class = "col-md-12 col-sm-12">' . sprintf( __( 'Logged in as <a href = "%1$s">%2$s</a>. <a href = "%3$s" title                                                  = "Log out of this account">Log out?</a>', 'w3construct' ), get_edit_user_link(), $user_identity, wp_logout_url( apply_filters( 'the_permalink', get_permalink( $post_id_w3 ) ) ) ) . '</p>',
			'comment_notes_before' => '',
			'comment_notes_after'  => '',
			'id_form'              => 'comments_form',
			'id_submit'            => 'submit',
			'title_reply'          => esc_html__( 'Leave Your Comment', 'w3construct' ),
			/* translators: %s is a placeholder for the original title */
			'title_reply_to'       => esc_html__( 'Leave a Reply to %s', 'w3construct' ),
			'cancel_reply_link'    => esc_html__( 'Cancel reply', 'w3construct' ),
			'label_submit'         => esc_html__( 'Submit', 'w3construct' ),
			'format'               => 'xhtml',
		);
		/**
		 * Filter the comment form default arguments.
		 *
		 * Use 'comment_form_default_fields' to filter the comment fields.
		 *
		 * @since 3.0.0
		 *
		 * @param array $defaults The default comment form arguments.
		 */
		$args = wp_parse_args( $args, apply_filters( 'comment_form_defaults', $defaults ) );

		if ( comments_open( $post_id_w3 ) ) :
			?>
				<?php
				/**
				 * Fires before the comment form.
				 *
				 * @since 3.0.0
				 */
				do_action( 'comment_form_before' );
				?>
				<div class="default-form comment-respond style-1" id="respond">

					<?php if ( get_option( 'comment_registration' ) && ! is_user_logged_in() ) : ?>
						<?php echo wp_kses( $args['must_log_in'], $allowed_html_tags ); ?>
						<?php
						/**
						 * Fires after the HTML-formatted 'must log in after' message in the comment form.
						 *
						 * @since 3.0.0
						 */
						do_action( 'comment_form_must_log_in_after' );
						?>
					<?php else : ?>
						<div class="clearfix">
							<?php echo esc_html( comment_form( $args, $post_id_w3 ) ); ?>
						</div>
					<?php endif; ?>
				</div><!-- #respond -->
				<?php
				/**
				 * Fires after the comment form.
				 *
				 * @since 3.0.0
				 */
				do_action( 'comment_form_after' );
		else :
			/**
			 * Fires after the comment form if comments are closed.
			 *
			 * @since 3.0.0
			 */
			do_action( 'comment_form_comments_closed' );
		endif;
	}
}

if ( ! function_exists( 'w3construct_the_pagination' ) ) {
	/**
	 * Showing the default pagination for frontend.
	 *
	 * @param array $query_w3 An array of query parameters.
	 * @param array $args An array of arguments.
	 * @param int   $echo Whether to echo the pagination or not.
	 */
	function w3construct_the_pagination( $query_w3 = array(), $args = array(), $echo = 1 ) {
		global $wp_query;

		/*
		We have deleted query_w3 related code to resolve the error of wp_query variable override problem.
		*/

		$paged = get_query_var( 'paged' );

		$default = array(
			'base'      => str_replace( 99999, '%#%', esc_url( get_pagenum_link( 99999 ) ) ),
			'format'    => '?paged = %#%',
			'current'   => max( 1, $paged ),
			'total'     => $wp_query->max_num_pages,
			'prev_text' => esc_html__( 'Prev', 'w3construct' ),
			'next_text' => esc_html__( 'Next', 'w3construct' ),
			'type'      => 'list',
			'add_args'  => false,
		);

		$args            = wp_parse_args( $args, $default );
		$pagination_link = ! empty( paginate_links( $args ) ) ? paginate_links( $args ) : '';
		$paginations     = str_replace( "<ul class='page-numbers'", '<ul class = "pagination text-center p-t20 mb-lg-0 mb-5"', $pagination_link );
		$pagination      = $paginations;

		echo wp_kses( $pagination, w3construct_allowed_html_tag() );
	}
}

if ( ! function_exists( 'w3construct_trim' ) ) {
	/**
	 * Get WordPress content without spaces.
	 *
	 * @param string $text The text to trim.
	 * @param int    $len  The maximum length of the trimmed text.
	 * @param string $more (Optional) The ellipsis to append to the trimmed text.
	 */
	function w3construct_trim( $text, $len, $more = null ) {
		$text           = strip_shortcodes( $text );
		$text           = str_replace( ']]>', ']]&gt;', $text );
		$excerpt_length = apply_filters( 'excerpt_length', $len ); /** Issue Remaining */
		$excerpt_more   = apply_filters( 'excerpt_more', ' [&hellip;]' );
		$excerpt_more   = ( $more ) ? $more : '';
		$text           = wp_trim_words( $text, $len, $excerpt_more );

		return $text;
	}
}

if ( ! function_exists( 'w3construct_excerpt_more' ) ) {
	/**
	 * Excerpt More Settings
	 */
	function w3construct_excerpt_more() {
		return ' ';
	}
	add_filter( 'excerpt_more', 'w3construct_excerpt_more' );
}

if ( ! function_exists( 'w3construct_get_social_icons' ) ) {
	/**
	 * Get social icons to display in the footer and header areas.
	 *
	 * @param string $show_position (Optional) Where to display the icons (e.g., 'all', 'footer', 'header').
	 * @param string $default_class (Optional) Additional CSS class to apply to the icons.
	 * @param string $social_icons_title (Optional) Additional social name to apply to the icons.
	 * @return string HTML code for the social icons.
	 */
	function w3construct_get_social_icons( $show_position = 'all', $default_class = null, $social_icons_title = null ) {

		$options             = w3construct_dzbase()->option();
		$output              = ' ';
		$target              = $options['social_link_target'];
		$social_link_options = w3construct_social_link_options();

		if ( 'header' === $show_position ) {

			/*reCheck function and rebuild it for fast performance */

			global $w3construct_option;
			$header_social_links  = w3construct_set( $w3construct_option, 'header_social_links' );
			$header_show_links    = [];
			$updated_social_links = array();
			if ( ! empty( $header_social_links ) ) {
				foreach ( $header_social_links as $key => $value ) {
					if ( '1' === $value ) {
						$header_show_links[] = $key;
					}
				}

				foreach ( $social_link_options as $social_key => $social_link ) {
					if ( in_array( $social_link['id'], $header_show_links, true ) ) {
						$updated_social_links[ $social_key ] = $social_link;
					}
				}
			}
			$social_link_options = $updated_social_links;
		}
		foreach ( $social_link_options as $social_link ) {
			$id    = $social_link['id'];
			$sl_id = 'social_' . $id . '_url';

			$sl_title   = $social_link['title'];
			$social_url = $options[ $sl_id ];

			if ( filter_var( $social_url, FILTER_VALIDATE_URL ) && ! empty( $social_url ) ) {
				if ( ! empty( $social_icons_title ) ) {
					$output .= '<li><a target="' . esc_attr( $target ) . '" href="' . esc_url( $options[ $sl_id ] ) . '"  class="' . esc_attr( $default_class ) . '">' . $sl_title . '</a></li>' . "\n";
				} else {
					$id      = ( 'facebook' === $id ) ? 'facebook-f' : $id;
					$output .= '<li><a target="' . esc_attr( $target ) . '" href="' . esc_url( $options[ $sl_id ] ) . '"  class="' . esc_attr( $default_class . ' fab fa-' . $id ) . '"></a></li>' . "\n";
				}
			}
		}

		return $output;
	}
}

if ( ! function_exists( 'w3construct_short_description' ) ) {
	/**
	 * Generate a short description for frontend use.
	 *
	 * @param string $excerpt (Optional) The excerpt text to use.
	 * @param string $content (Optional) The content to use if excerpt is not available.
	 * @param int    $limit (Optional) The character limit for the short description.
	 * @param string $more (Optional) The text to append if the content is truncated.
	 * @return string The generated short description.
	 */
	function w3construct_short_description( $excerpt = '', $content = '', $limit = 250, $more = null ) {

		if ( empty( $excerpt ) && empty( $content ) ) {
			return false;
		}

		$short_description = '';

		if ( ! empty( $excerpt ) ) {
			$short_description = $excerpt;
		} else {
			if ( has_blocks( $content ) ) {
				$blocks = parse_blocks( $content );

				foreach ( $blocks as $k ) {
					if ( 'core/paragraph' === $k['blockName'] ) {
						if ( ! empty( $k['innerHTML'] ) ) {
							$short_description = $k['innerHTML'];
							break;
						}
					}
				}
			} else {
				$short_description = $content;
			}
		}
		$short_description = w3construct_trim( $short_description, $limit, $more );

		return $short_description;
	}
}

if ( ! function_exists( 'w3construct_get_request_data' ) ) {
	/**
	 *
	 * Receive GET - POST data
	 */
	function w3construct_get_request_data() {
		check_ajax_referer( 'ajax_security_key', 'security' );
		$data = array();
		if ( isset( $_POST ) ) {
			foreach ( $_POST as $key => $value ) {
				$data[ $key ] = sanitize_text_field( wp_unslash( $value ) );
			}
		}

		return $data;
	}
}

if ( ! function_exists( 'w3construct_load_posts_by_ajax_callback' ) ) {
	/**
	 *
	 * Ajax - Home pages listing
	 */
	function w3construct_load_posts_by_ajax_callback() {

		check_ajax_referer( 'ajax_security_key', 'security' );
		global $w3construct_query_result;

		$data                   = w3construct_get_request_data();
		$post_type_w3           = ! empty( $data['post_type'] ) ? $data['post_type'] : 'post';
		$posts_per_page         = ! empty( $data['posts_per_page'] ) ? $data['posts_per_page'] : 5;
		$page                   = ! empty( $data['page'] ) ? $data['page'] : 'post';
		$post_order             = ! empty( $data['post_order'] ) ? $data['post_order'] : 'asc';
		$post_order_by          = ! empty( $data['post_order_by'] ) ? $data['post_order_by'] : 'asc';
		$posts_image_preference = ! empty( $data['posts_image_preference'] ) ? $data['posts_image_preference'] : 'default';
		$posts_in_categories    = ! empty( $data['posts_in_categories'] ) ? $data['posts_in_categories'] : array();
		$post_by_label          = ! empty( $data['post_by_label'] ) ? $data['post_by_label'] : ' ';
		$only_featured_post     = ! empty( $data['only_featured_post'] ) ? $data['only_featured_post'] : ' ';

		/** WP Query Argument  */
		$query_args = array(
			'post_type'           => $post_type_w3,
			'post_status'         => 'publish',
			'posts_per_page'      => $posts_per_page,
			'paged'               => $page,
			'order'               => $post_order,
			'ignore_sticky_posts' => true,
		);

		if ( 'image_post_only' === $posts_image_preference ) {
			$query_args['meta_query'] = array(
				array(
					'key'     => '_thumbnail_id',
					'compare' => 'EXISTS',
				),
			);
		} elseif ( 'text_post_only' === $posts_image_preference ) {
			$query_args['meta_query'] = array(
				array(
					'key'     => '_thumbnail_id',
					'compare' => 'Not EXISTS',
				),
			);
		}

		if ( $posts_in_categories ) {

			if ( ! empty( $_POST['post_type'] ) && 'post' !== $_POST['post_type'] ) {
				/** This is category searching only for custom post type */
				$cat_arr = explode( ',', $posts_in_categories );

				$taxonomy = get_object_taxonomies( $post_type_w3 );

				$taxonomy = ! empty( $taxonomy[0] ) ? $taxonomy[0] : 'category';

				$query_args['tax_query'][] = array(
					'taxonomy' => $taxonomy,
					'field'    => 'slug',
					'terms'    => $cat_arr,
					'operator' => 'IN',
				);

			} else {
				$query_args['category_name'] = $posts_in_categories;
			}
		}

		if ( 'sticky_only' === $post_by_label ) {
			$query_args['post__in']            = get_option( 'sticky_posts' );
			$query_args['ignore_sticky_posts'] = true;
		}

		if ( 'yes' === $only_featured_post ) {
			$query_args['meta_key']     = 'featured_post';
			$query_args['meta_value']   = 1;
			$query_args['meta_compare'] = 'LIKE';
		}

		if ( 'views_count' === $post_order_by ) {
			$query_args['meta_key'] = '_views_count';
		} else {
			$query_args['orderby'] = $post_order_by;
		}

		$query_w3 = new WP_Query( $query_args );

		$w3construct_query_result['posts']                  = $query_w3->posts;
		$w3construct_query_result['posts_per_page']         = $posts_per_page;
		$w3construct_query_result['current_page']           = $page;
		$w3construct_query_result['side_bar']               = ! empty( $_POST['side_bar'] ) ? sanitize_text_field( wp_unslash( $_POST['side_bar'] ) ) : '';
		$w3construct_query_result['title_text_limit']       = ! empty( $_POST['title_text_limit'] ) ? sanitize_text_field( wp_unslash( $_POST['title_text_limit'] ) ) : '';
		$w3construct_query_result['description_text_limit'] = ! empty( $_POST['description_text_limit'] ) ? sanitize_text_field( wp_unslash( $_POST['description_text_limit'] ) ) : 20;

		$w3construct_query_result['show_date']      = ! empty( $_POST['show_date'] ) ? sanitize_text_field( wp_unslash( $_POST['show_date'] ) ) : '';
		$w3construct_query_result['show_author']    = ! empty( $_POST['show_author'] ) ? sanitize_text_field( wp_unslash( $_POST['show_author'] ) ) : '';
		$w3construct_query_result['show_comment']   = ! empty( $_POST['show_comment'] ) ? sanitize_text_field( wp_unslash( $_POST['show_comment'] ) ) : '';
		$w3construct_query_result['show_permalink'] = ! empty( $_POST['show_permalink'] ) ? sanitize_text_field( wp_unslash( $_POST['show_permalink'] ) ) : '';
		$w3construct_query_result['show_share']     = ! empty( $_POST['show_share'] ) ? sanitize_text_field( wp_unslash( $_POST['show_share'] ) ) : '';
		$w3construct_query_result['show_column']    = ! empty( $_POST['show_column'] ) ? sanitize_text_field( wp_unslash( $_POST['show_column'] ) ) : 'column_2';
		$w3construct_query_result['element_style']  = ! empty( $_POST['element_style'] ) ? sanitize_text_field( wp_unslash( $_POST['element_style'] ) ) : 'style_1';
		$w3construct_query_result['social_show']    = ! empty( $_POST['social_show'] ) ? sanitize_text_field( wp_unslash( $_POST['social_show'] ) ) : 'yes';
		$blog_view                                  = ! empty( $_POST['blog_view'] ) ? sanitize_text_field( wp_unslash( $_POST['blog_view'] ) ) : '';
		switch ( $blog_view ) {
			case 'post_listing_1':
				get_template_part( 'dz-inc/elementor/ajax/post-listing-1-ajax' );
				break;
			case 'post_listing_2':
				get_template_part( 'dz-inc/elementor/ajax/post-listing-2-ajax' );
				break;
			case 'team_listing_1':
				get_template_part( 'dz-inc/elementor/ajax/team-listing-1-ajax' );
				break;
			case 'portfolio_listing_1':
				get_template_part( 'dz-inc/elementor/ajax/portfolio-listing-1-ajax' );
				break;
			case 'service_listing_1':
				get_template_part( 'dz-inc/elementor/ajax/service-listing-1-ajax' );
				break;
		}

		unset( $GLOBALS['w3construct_query_result'] );
		wp_reset_postdata();
		wp_die();
	}
	add_action( 'wp_ajax_load_posts_by_ajax', 'w3construct_load_posts_by_ajax_callback' );
	add_action( 'wp_ajax_nopriv_load_posts_by_ajax', 'w3construct_load_posts_by_ajax_callback' );
}

if ( ! function_exists( 'w3construct_load_mega_menu_posts_ajax_callback' ) ) {
	/**
	 *
	 * AJAX - Mega menu
	 */
	function w3construct_load_mega_menu_posts_ajax_callback() {
		check_ajax_referer( 'ajax_security_key', 'security' );
		global $w3construct_query_result;

		$data                  = w3construct_get_request_data();
		$posts_per_page        = ! empty( $data['posts_per_page'] ) ? $data['posts_per_page'] : 5;
		$page                  = ! empty( $data['page'] ) ? $data['page'] : 1;
		$posts_in_categories   = ! empty( $data['posts_in_categories'] ) ? $data['posts_in_categories'] : array();
		$mega_menu_images_only = ! empty( $data['mega_menu_images_only'] ) ? $data['mega_menu_images_only'] : ' ';

		$query_args = array(
			'post_type'           => 'post',
			'post_status'         => 'publish',
			'posts_per_page'      => $posts_per_page,
			'paged'               => $page,
			'ignore_sticky_posts' => true,
			'orderby'             => 'date',
			'order'               => 'DESC',
		);

		if ( $posts_in_categories ) {
			$query_args['cat'] = $posts_in_categories;
		}

		if ( 'yes' === $mega_menu_images_only ) {
			$query_args['meta_query'] = array(
				array(
					'key'     => '_thumbnail_id',
					'compare' => 'EXISTS',
				),
			);
		}

		$query_w3 = new WP_Query( $query_args );
		set_query_var( 'query', $query_w3 );
		get_template_part( 'dz-inc/ajax/mega-menu-ajax' );
		wp_reset_postdata();
		wp_die();
	}
	add_action( 'wp_ajax_load_mega_menu_posts_by_ajax', 'w3construct_load_mega_menu_posts_ajax_callback' );
	add_action( 'wp_ajax_nopriv_load_mega_menu_posts_by_ajax', 'w3construct_load_mega_menu_posts_ajax_callback' );
}


if ( ! function_exists( 'w3construct_load_common_page_posts_ajax_callback' ) ) {
	/**
	 *
	 * AJAX - Categories, Search, Tags, Archive, Author
	 */
	function w3construct_load_common_page_posts_ajax_callback() {
		check_ajax_referer( 'ajax_security_key', 'security' );

		$w3construct_query_result = array();
		$data                     = w3construct_get_request_data();
		$posts_per_page           = ! empty( $data['posts_per_page'] ) ? $data['posts_per_page'] : 5;
		$page                     = ! empty( $data['page'] ) ? $data['page'] : 'post';
		$post_order               = ! empty( $data['post_order'] ) ? $data['post_order'] : 'asc';
		$page_view                = ! empty( $data['page_view'] ) ? $data['page_view'] : '';
		$post_order_by            = ! empty( $data['post_order_by'] ) ? $data['post_order_by'] : '';
		$object_data_page         = ! empty( $data['object_data'] ) ? $data['object_data'] : '';

		/** WP Query Argument  */
		$query_args = array(
			'post_type'           => 'post',
			'post_status'         => 'publish',
			'posts_per_page'      => $posts_per_page,
			'paged'               => $page,
			'ignore_sticky_posts' => true,
			'order'               => 'DESC',
		);

		$orderby_w3 = isset( $post_order_by ) ? $post_order_by : 'date';
		$order_w3   = isset( $post_order ) ? $post_order : 'DESC';

		if ( '_views_count' === $orderby_w3 ) {
			$query_args['meta_key'] = '_views_count';
			$query_args['order']    = 'DESC';
			$query_args['orderby']  = '_views_count';
		} else {
			$query_args['orderby'] = $orderby_w3;
		}

		$query_args['order'] = $order_w3;

		$template = '';

		if ( 'author' === $page_view ) {
			$query_args['author'] = $object_data_page;
			$template             = 'authors-ajax';
		}
		if ( 'category' === $page_view ) {
			$query_args['cat'] = $object_data_page;
			$template          = 'categories-ajax';
		}
		if ( 'search' === $page_view ) {
			$query_args['s'] = $object_data_page;
			$template        = 'searches-ajax';
		}
		if ( 'tag' === $page_view ) {
			$query_args['tag_id'] = $object_data_page;
			$template             = 'tags-ajax';
		}
		if ( 'archive' === $page_view ) {
			$object_data        = explode( '/', $object_data_page );
			$query_args['year'] = $object_data[0];

			if ( isset( $object_data[1] ) && ! empty( $object_data[1] ) ) {
				$query_args['monthnum'] = $object_data[1];
			}

			$template = 'archives-ajax';
		}

		$query_w3 = new WP_Query( $query_args );
		set_query_var( 'w3construct_query_result', $query_w3 );
		get_template_part( 'dz-inc/ajax/' . $template );
		wp_reset_postdata();
		wp_die();
	}
	add_action( 'wp_ajax_load_common_page_posts_ajax', 'w3construct_load_common_page_posts_ajax_callback' );
	add_action( 'wp_ajax_nopriv_load_common_page_posts_ajax', 'w3construct_load_common_page_posts_ajax_callback' );
}

if ( ! function_exists( 'w3construct_load_latest_posts_ajax_callback' ) ) {
	/**
	 *
	 * AJAX - index page :-
	 */
	function w3construct_load_latest_posts_ajax_callback() {

		check_ajax_referer( 'ajax_security_key', 'security' );
		$w3construct_query_result = array();
		$data                     = w3construct_get_request_data();
		$posts_per_page           = ! empty( $data['posts_per_page'] ) ? $data['posts_per_page'] : 5;
		$page                     = ! empty( $data['page'] ) ? $data['page'] : 'post';
		$query_args               = array(
			'post_type'           => 'post',
			'post_status'         => 'publish',
			'posts_per_page'      => $posts_per_page,
			'paged'               => $page,
			'orderby'             => 'post_date',
			'ignore_sticky_posts' => true,
			'order'               => 'DESC',
		);

		$query_w3 = new WP_Query( $query_args );
		set_query_var( 'w3construct_query_result', $query_w3 );
		get_template_part( 'dz-inc/ajax/index-ajax' );
		wp_reset_postdata();
		wp_die();
	}
	add_action( 'wp_ajax_load_latest_posts_ajax', 'w3construct_load_latest_posts_ajax_callback' );
	add_action( 'wp_ajax_nopriv_load_latest_posts_ajax', 'w3construct_load_latest_posts_ajax_callback' );
}

if ( ! function_exists( 'w3construct_shortcode' ) ) {
	/**
	 * Run code for a shortcode.
	 *
	 * @param string $output The output to be processed by the shortcode.
	 * @return array $output.
	 */
	function w3construct_shortcode( $output ) {
		return $output;
	}
	/** Run Code END */
}


if ( ! function_exists( 'w3construct_share_us' ) ) {
	/**
	 * Share functionality for post details pages.
	 *
	 * @param int    $post_id_w3 (Optional) The ID of the post.
	 * @param string $post_title (Optional) The title of the post.
	 * @param string $share_on (Optional) The social media or platform to share on.
	 */
	function w3construct_share_us( $post_id_w3 = '', $post_title = '', $share_on = '' ) {

		$social_shaing = w3construct_get_opt( 'social_shaing_on_post' );

		/** Control Post Sharing */
		if ( ! $social_shaing ) {
			return false;
		}
		/** Control Post Sharing END */

		$response          = '';
		$options           = get_option( 'w3construct_theme_options' );
		$share_sort_links  = w3construct_set( $options, 'share_sort_link' );
		$social_share_link = array(
			'facebook'    => 'http ://www.facebook.com/sharer.php?u        = ' . esc_url( get_permalink( $post_id_w3 ) ),
			'twitter'     => 'https://twitter.com/share?url                = ' . esc_url( get_permalink( $post_id_w3 ) ) . '&text  = ' . esc_attr( $post_title ),
			'google-plus' => 'https://plus.google.com/share?url            = ' . esc_url( get_permalink( $post_id_w3 ) ),
			'linkedin'    => 'http ://www.linkedin.com/shareArticle?url    = ' . esc_url( get_permalink( $post_id_w3 ) ) . '&title = ' . esc_attr( $post_title ),
			'pinterest'   => 'http ://pinterest.com/pin/create/button/?url = ' . esc_url( get_permalink( $post_id_w3 ) ) . '&media = ' . esc_url( get_the_post_thumbnail_url( $post_id_w3 ) ) . '&description = ' . esc_attr( $post_title ),
			'reddit'      => 'http ://reddit.com/submit?url                = ' . esc_url( get_permalink( $post_id_w3 ) ) . '&title = ' . esc_attr( $post_title ),
			'tumblr'      => 'http ://www.tumblr.com/share/link?url        = ' . esc_url( get_permalink( $post_id_w3 ) ) . '&name  = ' . esc_attr( $post_title ),
			'digg'        => 'http ://digg.com/submit?url                  = ' . esc_url( get_permalink( $post_id_w3 ) ) . '&title = ' . esc_attr( $post_title ),
		);

		if ( 'PostSingle' === $share_on ) {
			$response = '<div class="dz-share-post"><h5 class="title">' . esc_html__( 'Share :', 'w3construct' ) . '</h5><ul class="dz-social-icon">';
		} else {
			$response = '<div class="dz-share-post"><h5 class="title">' . esc_html__( 'Share :', 'w3construct' ) . '</h5><ul class="dz-social-icon">';

		}

		if ( ! empty( $share_sort_links ) ) {
			foreach ( $share_sort_links as $icon_key => $icon_value ) {

				$anchor_class = '';
				if ( 'PostSingle' === $share_on ) {
					$icon_key_name = ( 'facebook' === $icon_key ) ? 'facebook-f' : $icon_key;
					$anchor_class  = 'fab fa-' . $icon_key_name;

				} else {
					$icon_key_name = ( 'facebook' === $icon_key ) ? 'facebook-f' : $icon_key;
					$anchor_class  = 'fab fa-' . $icon_key_name;
				}
				if ( $icon_value ) {

					$response .= '<li><a class="' . $anchor_class . '" href="' . esc_url( $social_share_link[ $icon_key ] ) . '" target="_blank" ></a></li> ';
				}
			}
		}
		if ( 'PostSingle' === $share_on ) {
			$response .= '</ul></div>';
		} else {
			$response .= '</ul></div>';
		}

		return $response;
	}
}

if ( ! function_exists( 'w3construct_author_social_arr' ) ) {
	/**
	 *
	 * Used this function for Author page
	 */
	function w3construct_author_social_arr() {
		$author_social_arr = array(
			'url'         => array(
				'icon' => 'fas fa-globe',
				'text' => 'Globe',
			),
			'facebook'    => array(
				'icon' => 'fab fa-facebook-f',
				'text' => 'Facebook',
			),
			'twitter'     => array(
				'icon' => 'fab fa-twitter',
				'text' => 'Twitter',
			),
			'linkedin'    => array(
				'icon' => 'fab fa-linkedin-in',
				'text' => 'Linkedin',
			),
			'dribbble'    => array(
				'icon' => 'fab fa-dribbble',
				'text' => 'Dribble',
			),
			'github'      => array(
				'icon' => 'fab fa-github',
				'text' => 'Github',
			),
			'flickr'      => array(
				'icon' => 'fab fa-flickr',
				'text' => 'Flickr',
			),
			'google-plus' => array(
				'icon' => 'fab fa-google-plus',
				'text' => 'Google Plus',
			),
			'youtube'     => array(
				'icon' => 'fab fa-youtube',
				'text' => 'Youtube',
			),
		);
		return $author_social_arr;
	}
}
if ( ! function_exists( 'w3construct_get_website_logo' ) ) {
	/**
	 * Get the website logo for the header and footer.
	 *
	 * @param string $logo_key The key identifying the type of logo (e.g., 'header_logo' or 'footer_logo').
	 */
	function w3construct_get_website_logo( $logo_key = 'header_logo' ) {
		global $w3construct_option;

		$logo_title        = w3construct_set( $w3construct_option, 'logo_title' );
		$site_other_logo   = w3construct_set( $w3construct_option, 'site_other_logo' );
		$site_logo         = w3construct_set( $w3construct_option, 'site_logo' );
		$logo_type         = w3construct_set( $w3construct_option, 'logo_type' );
		$logo_alt          = w3construct_set( $w3construct_option, 'logo_alt' );
		$page_logo_setting = '';
		$output            = '';

		/** Logo Class  */
		$class = '';
		if ( 'site_other_logo' === $logo_key ) {
			$class = 'light-logo';
		} elseif ( 'site_logo' === $logo_key ) {
			$class = 'dark-logo';
		}

		$class = ! empty( $class ) ? 'class="' . $class . '"' : '';

		/**
		 * Get the logo URL.
		 *
		 * @return string The URL of the logo.
		 */

		$logo_url = $$logo_key;

		if ( 'text_logo' === $logo_type ) {
			
			$site_title   = get_option( 'blogname' ); /*  For Site Name */
			$site_tagline = get_option( 'blogdescription' ); /*  For Tag line or description */
			if ( empty( $site_title ) && empty( $site_tagline ) ) {
				$site_title   = esc_html__( 'W3Construct', 'w3construct' );
				$site_tagline = esc_html__( 'Construction Theme', 'w3construct' );
			}
			
			$logo_text = w3construct_get_opt( 'logo_text', $site_title );
			$logo_tag  = w3construct_get_opt( 'tag_line', $site_tagline );
			$logo_tag  = ! empty( $logo_tag ) ? w3construct_trim( $logo_tag, 2 ) : '';

			$output .= '<div class="text-logo">';
			if ( ! empty( $logo_text ) || ! empty( $logo_tag ) ) {
				$output .= '<a href="' . esc_url( home_url( '/' ) ) . '" title="' . esc_attr( $logo_title ) . '">';
				if ( ! empty( $logo_text ) ) {
					$output .= '<h1 class="site-title">';
					$output .= esc_html( $logo_text );
					$output .= '</h1>';
				}
				if ( ! empty( $logo_tag ) ) {
					$output .= '<p class="site-description">';
					$output .= esc_html( $logo_tag );
					$output .= '</p>';
				}
				$output .= '</a>';
			}
			$output .= '</div>';
		} else {
			$output .= '<a href="' . esc_url( home_url( '/' ) ) . '" ' . $class . ' title="' . esc_attr( $logo_title ) . '">';
			$output .= '<img src="' . esc_url( $logo_url ) . '" alt="' . esc_attr( $logo_alt ) . '"/>';
			$output .= '</a>';
		}
		echo wp_kses( $output, w3construct_allowed_html_tag() );
	}
	add_action( 'w3construct_get_logo', 'w3construct_get_website_logo', 2 );
}

if ( ! function_exists( 'w3construct_get_super_user_data' ) ) {
	/**
	 * Get user meta data.
	 *
	 * @param string $user_meta (Optional) The user meta key to retrieve.
	 * @return       $user_meta_value The user meta data or null if not found.
	 */
	function w3construct_get_super_user_data( $user_meta = '' ) {
		$admin_email     = get_option( 'admin_email' );
		$admin_detail    = get_user_by( 'email', $admin_email );
		$user_meta_value = get_the_author_meta( $user_meta, $admin_detail->data->ID );

		return $user_meta_value;
	}
}

if ( ! function_exists( 'w3construct_get_super_user_displayname' ) ) {
	/**
	 * Get the display name of the super user.
	 *
	 * @param string $user_meta (Optional) The user meta key to retrieve.
	 * @return string The display name of the super user.
	 */
	function w3construct_get_super_user_displayname( $user_meta = '' ) {

		$admin_email  = get_option( 'admin_email' );
		$admin_detail = get_user_by( 'email', $admin_email );

		return $admin_detail->data->display_name;
	}
}

if ( ! function_exists( 'w3construct_get_super_user_description' ) ) {
	/**
	 *
	 * Login user description
	 */
	function w3construct_get_super_user_description() {

		$admin_email  = get_option( 'admin_email' );
		$admin_detail = get_user_by( 'email', $admin_email );
		$meta         = get_user_meta( $admin_detail->data->ID );
		$description  = ! empty( $meta['description'][0] ) ? $meta['description'][0] : '';
		return $description;
	}
}

if ( ! function_exists( 'w3construct_get_domain' ) ) {
	/**
	 * W3construct_get_domain
	 *
	 * @param mixed $url here is the .
	 * @return false
	 */
	function w3construct_get_domain( $url ) {
		$pieces = wp_parse_url( $url );
		$domain = isset( $pieces['host'] ) ? $pieces['host'] : '';
		if ( preg_match( '/( ?P<domain>[a-z0-9][a-z0-9\-]{1,63}\.[a-z\.]{2,6} )$/i', $domain, $regs ) ) {
			return $regs['domain'];
		}
		return false;
	}
}

if ( ! function_exists( 'w3construct_get_youtube_video_id' ) ) {
	/**
	 * W3construct_get_youtube_video_id
	 *
	 * @param mixed $video_url for the function.
	 * @return false
	 */
	function w3construct_get_youtube_video_id( $video_url ) {
		$res = 0;
		if ( preg_match( '/( youtube.com|youtu.be )\/( watch )?( \?v= )?( \S+ )?/', $video_url, $res ) ) {
			return $res[4];
		} else {
			return 0;
		}
	}
}
if ( ! function_exists( 'w3construct_get_the_category_list' ) ) {
	/**
	 * W3construct_get_the_category_list
	 *
	 * @param string $cat_list for the function parameters.
	 * @param string $b for the function parameters.
	 * @return $category;
	 */
	function w3construct_get_the_category_list( $cat_list, $b ) {
		$category   = array();
		$category[] = $cat_list[0];
		return $category;
	}
	/** Show Only One Category */
}

if ( ! function_exists( 'w3construct_exclude_widget_categories' ) ) {
	/**
	 * W3construct_exclude_widget_categories
	 *
	 * @param string $args Hide Some Category From Widget : only for demo.
	 * @return $args
	 */
	function w3construct_exclude_widget_categories( $args ) {
		$hide_selected_cat = w3construct_get_opt( 'hide_selected_cat' );
		if ( ! empty( $hide_selected_cat ) ) {
			$hide_selected_cat = implode( ',', $hide_selected_cat );
			$args['exclude']   = $hide_selected_cat;
			return $args;
		}
	}
	/** Hide Some Category From Widget : only for demo END */
}

if ( ! function_exists( 'w3construct_custom_columns' ) ) {
	/**
	 * W3construct_custom_columns
	 *
	 * @param string $columns Show Feature Image In Post Listing.
	 * @return $columns
	 */
	function w3construct_custom_columns( $columns ) {
		$columns = array(
			'cb'             => '<input type = "checkbox" />',
			'featured_image' => 'Image',
			'title'          => 'Title',
			'comments'       => '<span class = "vers"><div title = "Comments" class = "comment-grey-bubble"></div></span>',
			'author'         => 'Author',
			'categories'     => 'Cateogies',
			'tags'           => 'Tags',
			'date'           => 'Date',
		);
		return $columns;
	}
	/** Show Feature Image In Post Listing Start */
}

if ( ! function_exists( 'w3construct_custom_columns_data' ) ) {
	/**
	 * W3construct_custom_columns_data
	 *
	 * @param string $column here is the parameter.
	 * @param string $post_id_w3 here is the parameter.
	 */
	function w3construct_custom_columns_data( $column, $post_id_w3 ) {
		switch ( $column ) {
			case 'featured_image':
				the_post_thumbnail( 'thumbnail' );
				break;
		}
	}
}

if ( ! function_exists( 'w3construct_ext_options' ) ) {
	/**
	 *
	 * Get extra option
	 */
	function w3construct_ext_options() {
		$show_only_one_cat       = w3construct_get_opt( 'show_only_one_cat' );
		$hide_cat_from_widget    = w3construct_get_opt( 'hide_cat_from_widget' );
		$show_image_on_post_list = w3construct_get_opt( 'show_image_on_post_list' );

		if ( $show_only_one_cat ) {
			add_action( 'the_category_list', 'w3construct_get_the_category_list', 10, 2 );
		}

		if ( $hide_cat_from_widget ) {
			add_filter( 'widget_categories_args', 'w3construct_exclude_widget_categories' );
		}
		if ( $show_image_on_post_list ) {
			add_filter( 'manage_post_posts_columns', 'w3construct_custom_columns' );
			add_action( 'manage_post_posts_custom_column', 'w3construct_custom_columns_data', 10, 2 );
		}
	}
	add_action( 'init', 'w3construct_ext_options' );
	/** Show Feature Image In Post Listing END */
}

if ( ! function_exists( 'w3construct_is_theme_sidebar_active' ) ) {
	/**
	 * Check if the theme sidebar is active.
	 *
	 * @return bool Whether the sidebar is active or not.
	 */
	function w3construct_is_theme_sidebar_active() {
		return w3construct_get_opt( 'show_sidebar', true );
	}
}

if ( ! function_exists( 'w3construct_show_post_view_count_view' ) ) {
	/**
	 * Customize post views in WordPress.
	 *
	 * @param array $views An array of post views.
	 * @return array Modified post views array.
	 */
	function w3construct_show_post_view_count_view( $views ) {
		$post_view_on = w3construct_get_opt( 'post_view_on', false );

		$view_html = '';
		if ( $post_view_on ) {
			$post_start_view = w3construct_get_opt( 'post_start_view', 0 );
			$views           = $views + $post_start_view;

			$view_html = '<li class="post-view"><a href="javascript:void( 0 );"><span>' . wp_kses( $views, 'string' ) . '</span></a></li>';
		}

		return $view_html;
	}
	/** Show view In Post pages End */
}

if ( ! function_exists( 'w3construct_get_banner' ) ) {
	/**
	 *
	 * Showing the Banner for Page
	 */
	function w3construct_get_banner() {
		global $w3construct_option;

		$header_style = isset( $w3construct_option['header_style'] ) ? $w3construct_option['header_style'] : 'header-1';

		$theme_options = w3construct_get_theme_option();

		$template_name = 'page_general';
		if ( is_page() ) {
			$template_name       = 'page_general';
			$page_banner_setting = w3construct_dzbase()->get_meta( 'page_banner_setting' );
		}

		$page_banner_setting = ! empty( $page_banner_setting ) ? $page_banner_setting : 'theme_default';
		$title_prefix        = '';

		if ( is_author() ) {
			$template_name = 'author_page';
			$title_prefix  = esc_html__( 'Author :', 'w3construct' );
		} elseif ( is_search() ) {
			$template_name = 'search_page';
			$title_prefix  = esc_html__( 'Search :', 'w3construct' );
		} elseif ( is_category() ) {
			$template_name = 'category_page';
			$title_prefix  = esc_html__( 'Category :', 'w3construct' );
		} elseif ( w3construct_is_woocommerce_active() && ( ( function_exists( 'is_shop' ) && is_shop() ) || ( function_exists( 'is_product_category' ) && is_product_category() ) ) ) {
			$template_name = 'woocommerce_page';
			$title_prefix  = esc_html__( 'Product :', 'w3construct' );
		} elseif ( is_tag() ) {
			$template_name = 'tag_page';
			$title_prefix  = esc_html__( 'Tag :', 'w3construct' );
		} elseif ( function_exists( 'tribe_is_event' ) && tribe_is_event() ) {
			$template_name = 'archive_page';
			$title_prefix  = '';
		} elseif ( is_archive() ) {
			$template_name = 'archive_page';
			$title_prefix  = esc_html__( 'Archive :', 'w3construct' );
		}

		$page_banner_title     = '';
		$page_banner_sub_title = '';
		if ( 'custom' === $page_banner_setting ) {
			$show_banner           = w3construct_dzbase()->get_meta( 'page_banner_on' );
			$banner_type           = w3construct_dzbase()->get_meta( 'page_banner_type' );
			$banner_layout         = w3construct_dzbase()->get_meta( 'page_banner_layout' );
			$banner_height         = w3construct_dzbase()->get_meta( 'page_banner_height' );
			$custom_height         = w3construct_dzbase()->get_meta( 'page_banner_custom_height' );
			$banner_image          = w3construct_dzbase()->get_meta( 'page_banner' );
			$banner_hide           = w3construct_dzbase()->get_meta( 'page_banner_hide' );
			$page_banner_title     = w3construct_dzbase()->get_meta( 'page_banner_title' );
			$page_banner_sub_title = w3construct_dzbase()->get_meta( 'page_banner_sub_title' );
			$show_breadcrumb       = w3construct_dzbase()->get_meta( 'page_breadcrumb' );
			$banner_image          = w3construct_set( $banner_image, 'url' );
		} else {
			$title_prefix    = w3construct_set( $theme_options, $template_name . '_title', $title_prefix );
			$show_banner     = w3construct_set( $theme_options, $template_name . '_banner_on', true );
			$banner_type     = w3construct_set( $theme_options, $template_name . '_banner_type', 'image' );
			$banner_layout   = w3construct_set( $theme_options, $template_name . '_banner_layout', 'banner_layout_1' );
			$banner_height   = w3construct_set( $theme_options, $template_name . '_banner_height', 'page_banner_small' );
			$custom_height   = w3construct_set( $theme_options, $template_name . '_banner_custom_height', '100' );
			$banner_image    = w3construct_set( $theme_options, $template_name . '_banner' );
			$banner_image    = w3construct_set( $banner_image, 'url' );
			$show_breadcrumb = w3construct_set( $theme_options, 'show_breadcrumb', true );
			$banner_hide     = w3construct_set( $theme_options, $template_name . '_banner_hide' );

		}
		$page_heading_classes = 'dz-bnr-inr-entry';

		$banner_class = ( empty( $banner_image ) ) ? 'bnr-no-img ' : '';

		if ( 'page_banner_big' === $banner_height ) {
			$banner_class        .= 'dz-bnr-inr style-1 overlay-black-middle  dz-bnr-inr-lg';
			$page_heading_classes = 'dz-bnr-inr-entry';
		} elseif ( 'page_banner_medium' === $banner_height ) {
			$banner_class .= 'dz-bnr-inr style-1 overlay-black-middle dz-bnr-inr-md';
		} elseif ( 'page_banner_small' === $banner_height ) {
			$banner_class .= 'dz-bnr-inr style-1 overlay-black-middle dz-bnr-inr-sm';
		} elseif ( 'page_banner_custom' === $banner_height ) {
			/** But you can't add height attribute here as per requirement */
			$banner_class .= 'dz-bnr-inr style-1 overlay-black-middle ' . $custom_height;
		}
		if ( $show_banner ) {
			if ( 'image' === $banner_type ) {
				?>
				<!-- Banner  -->
					<div class="<?php echo esc_attr( $banner_class ); ?>" <?php if ( ! empty( $banner_image ) && empty( $banner_hide ) ) { ?>
																	style="background-image : url( <?php echo esc_url( $banner_image ); ?> );
																	<?php if ( 'page_banner_custom' === $banner_height ) { ?>
																		height:<?php echo esc_attr( $custom_height ); ?>px;
																	<?php } ?>
						<?php } ?> ">
						<div class="container">
							<div class="<?php echo esc_attr( $page_heading_classes ); ?>">
								<h1>
									<?php
									if ( is_page() || is_single() ) {
										global $post;
										$title = isset( $post->post_title ) ? $post->post_title : '';
									} else {
										$title = wp_get_document_title();
									}

									if ( $title ) {
										if (
											w3construct_is_woocommerce_active() &&
											(
												( function_exists( 'is_shop' ) && is_shop() ) ||
												( function_exists( 'is_product_category' ) && is_product_category() )
											)
										) {
											$title_prefix = ''; /** To remove extra product word from title */

											$page_banner_title = w3construct_set( $theme_options, 'woocommerce_page_title' );
										}

										if ( ! empty( $page_banner_title ) ) {
											$title = $page_banner_title;
										} else {
											$title = $title_prefix . ' ' . $title;
										}
										$title = ! empty( $title ) ? w3construct_trim( $title, 8 ) : '';
										echo wp_kses( $title, 'string' );
									} else {
										echo esc_html( w3construct_set( $theme_options, 'blog_page_title', esc_html__( 'Blog', 'w3construct' ) ) );
									}
									?>
								</h1>

								<?php
								$breadcrumb_class = 'breadcrumb-row';
								?>

								<!-- Breadcrumb row -->
								<?php if ( $show_breadcrumb ) { ?>
									<nav aria-label="breadcrumb" class="<?php echo esc_attr( $breadcrumb_class ); ?>">
										<?php echo wp_kses( w3construct_get_the_breadcrumb(), w3construct_allowed_html_tag() ); ?>
									</nav>
								<?php } ?>
								<!-- Breadcrumb row END -->
							</div>
						</div>
					</div>
				<?php
			}
		}

	}
}

if ( ! function_exists( 'w3construct_get_post_banner' ) ) {
	/**
	 *
	 * Showing the Banner for Post,service,portfolio
	 */
	function w3construct_get_post_banner() {
		if ( ! is_single() ) {
			return false;
		}

		global $w3construct_option;
		$theme_options = w3construct_get_theme_option();

		$post_key = 'post_general';

		if ( is_singular( 'dz-service' ) ) {

			$post_key            = 'service_post';
			$post_banner_setting = 'custom';

		} elseif ( is_singular( 'dz-portfolio' ) ) {

			$post_key            = 'portfolio_post';
			$post_banner_setting = 'custom';

		} else {
			$post_banner_setting = w3construct_dzbase()->get_meta( 'post_banner_setting' );
			$post_banner_setting = ! empty( $post_banner_setting ) ? $post_banner_setting : 'theme_default';
		}

		$site_email        = w3construct_set( $theme_options, 'site_email' );
		$site_phone_number = w3construct_set( $theme_options, 'site_phone_number' );
		if ( 'custom' === $post_banner_setting ) {
			$show_banner     = w3construct_dzbase()->get_meta( $post_key . '_banner_on' );
			$banner_image    = w3construct_dzbase()->get_meta( $post_key . '_banner' );
			$banner_height   = w3construct_dzbase()->get_meta( $post_key . '_banner_height' );
			$custom_height   = w3construct_dzbase()->get_meta( $post_key . '_banner_custom_height' );
			$banner_image    = w3construct_set( $banner_image, 'url' );
			$show_breadcrumb = w3construct_dzbase()->get_meta( $post_key . '_breadcrumb' );
		} else {
			$show_banner     = w3construct_set( $theme_options, 'post_general_banner_on', false );
			$banner_height   = w3construct_set( $theme_options, 'post_general_banner_height', 'page_banner_small' );
			$custom_height   = w3construct_set( $theme_options, 'post_general_banner_custom_height', '100' );
			$banner_image    = w3construct_set( $theme_options, 'post_general_banner' );
			$banner_image    = w3construct_set( $banner_image, 'url' );
			$show_breadcrumb = w3construct_set( $theme_options, 'show_breadcrumb', true );
		}

		$page_heading_classes = 'dlab-bnr-inr-entry';
		$banner_class         = ( empty( $banner_image ) ) ? 'bnr-no-img ' : '';

		if ( 'page_banner_big' === $banner_height ) {
			$banner_class        .= 'dz-bnr-inr style-1 overlay-black-middle dz-bnr-inr-lg';
			$page_heading_classes = 'dz-bnr-inr-entry';
		} elseif ( 'page_banner_medium' === $banner_height ) {
			$banner_class .= 'dz-bnr-inr style-1 overlay-black-middle dz-bnr-inr-md';
		} elseif ( 'page_banner_small' === $banner_height ) {
			$banner_class .= 'dz-bnr-inr style-1 overlay-black-middle dz-bnr-inr-sm';
		} elseif ( 'page_banner_custom' === $banner_height ) {
			/** But you can't add height attribute here as per requirement */
			$banner_class .= 'dz-bnr-inr style-1 overlay-black-middle ' . $custom_height;
		}

		if ( $show_banner ) {

			$banner_class = 'dz-bnr-inr style-1 overlay-black-middle ' . $banner_class;

			?>

				<!-- Banner  -->
				<div class="<?php echo esc_attr( $banner_class ); ?>
					<?php
					if ( empty( $banner_image ) ) {
						echo 'bnr-no-img';
					}
					?>
					"
					<?php if ( ! empty( $banner_image ) ) { ?>
					style="background-image: url(<?php echo esc_url( $banner_image ); ?>);
							<?php if ( 'page_banner_custom' === $banner_height ) { ?>
							height:<?php echo esc_attr( $custom_height ); ?>px;<?php } ?>">
					<?php } ?>
					<div class="container">
						<div class="dz-bnr-inr-entry">
							<h1>
								<?php
								if ( is_page() || is_single() ) {
									global $post;
									$title = isset( $post->post_title ) ? $post->post_title : '';
								} else {
									$title = wp_get_document_title();
								}

								if ( $title ) {
									$title = ! empty( $title ) ? w3construct_trim( $title, 8 ) : '';
									echo wp_kses( $title, 'string' );
								} else {
									echo esc_html( w3construct_set( $theme_options, 'blog_page_title', esc_html__( 'Blog', 'w3construct' ) ) );
								}
								?>
							</h1>

							<!-- Breadcrumb row -->
							<?php if ( $show_breadcrumb && ! is_front_page() ) { ?>
								<nav aria-label="breadcrumb" class="breadcrumb-row">
									<?php echo wp_kses( w3construct_get_the_breadcrumb(), w3construct_allowed_html_tag() ); ?>
								</nav>
							<?php } ?>
							<!-- Breadcrumb row END -->
						</div>
					</div>
				</div>
				<?php

		}

	}
}

if ( ! function_exists( 'w3construct_is_post_banner_enable' ) ) {
	/**
	 *
	 * Showing the Banner for Check Enable and Disable
	 */
	function w3construct_is_post_banner_enable() {
		if ( ! is_single() ) {
			return false;
		}

		$post_banner_setting = w3construct_dzbase()->get_meta( 'post_banner_setting' );
		$post_banner_setting = ! empty( $post_banner_setting ) ? $post_banner_setting : 'theme_default';

		if ( 'custom' === $post_banner_setting ) {
			$show_banner = w3construct_dzbase()->get_meta( 'post_banner_on' );
		} else {
			$show_banner = w3construct_get_opt( 'post_general_banner_on' );
		}
		$show_banner = ( class_exists( 'ReduxFramework' ) ) ? $show_banner : false;
		return $show_banner;
	}
}

if ( ! function_exists( 'w3construct_get_loader' ) ) {
	/**
	 *
	 * Showing the loader for forntend
	 */
	function w3construct_get_loader() {
		$theme_options     = w3construct_get_theme_option();
		$page_loading_on   = w3construct_set( $theme_options, 'page_loading_on' );
		$page_loader_image = '';
		if ( '1' === $page_loading_on ) {
			$page_loader_type = w3construct_set( $theme_options, 'page_loader_type', 'loading_image' );
			if ( 'loading_image' === $page_loader_type ) {
				$custom_preloader = w3construct_set( $theme_options['custom_page_loader_image'], 'url' );
				if ( $custom_preloader ) {
					$preloader = $custom_preloader;
				} else {
					$page_loader_image = w3construct_set( $theme_options, 'page_loader_image', 'loading1' );
					$preloader         = get_template_directory_uri() . '/dz-inc/assets/images/loading-images/' . $page_loader_image . '.svg';
				}
			} elseif ( 'advanced_loader' === $page_loader_type ) {
				$page_loader = w3construct_set( $theme_options, 'advanced_page_loader_image', 'loading1' );
			}
		}
		?>
			<?php if ( '1' === $page_loading_on ) { ?>

				<?php
				if ( 'loading_image' === $page_loader_type ) {

					$loading_class = '';
					if ( 'loader_1' === $page_loader_image ) {
						$loading_class = 'loading-01';
					} elseif ( 'loader_2' === $page_loader_image ) {
						$loading_class = 'loading-02';
					} else {
						$loading_class = 'loading-03';
					}

					?>

					<?php if ( 'loading1' === $page_loader_image ) { ?>
						<div id="loading-area" class="loading-page-1">
							<div class="spinner">
								<div class="ball"></div>
								<p>
									<?php echo esc_html__( 'LOADING', 'w3construct' ); ?>
								</p>
								<span></span>
							</div>
						</div>
					<?php } else { ?>


						<!-- Preloader -->
						<div id="loading-area" class="<?php echo esc_attr( $loading_class ); ?>"
							style="background-image: url( <?php echo esc_url( $preloader ); ?> );background-repeat: no-repeat; background-position: center;">
						</div>
						<?php
					}
				}
				?>
				<?php
				if ( 'advanced_loader' === $page_loader_type && 'loading1' === $page_loader ) {
					wp_enqueue_style( 'w3construct-loading1', get_template_directory_uri() . '/assets/css/loader/loading1.css', array(), '1.0' );
					?>
					<div id="loading-area" class="loader2">
						<div class="box-load">
							<div class="load-logo">
								<?php do_action( 'w3construct_get_logo', 'site_other_logo' ); ?>
							</div>
							<h1 class="ml12">
								<?php echo esc_html( $theme_options['advanced_page_loader_title'] ); ?>
							</h1>
						</div>
					</div>
					<?php
					wp_enqueue_script( 'w3construct-anime', get_template_directory_uri() . '/assets/js/loading/anime.js', array( 'jquery' ), '1.0', true );
					wp_enqueue_script( 'w3construct-anime-app3', get_template_directory_uri() . '/assets/js/loading/anime-app3.js', array( 'jquery' ), '1.0', true );
				} elseif ( 'advanced_loader' === $page_loader_type && 'loading2' === $page_loader ) {
					wp_enqueue_style( 'w3construct-loading2', get_template_directory_uri() . '/assets/css/loader/loading2.css', array(), '1.0' );
					?>
					<div id="loading-area" class="line-loader">
						<svg viewBox="0 0 960 300">
							<symbol id="s-text">
								<text text-anchor="middle" x="50%" y="80%">
									<?php echo esc_html( get_bloginfo( 'name' ) ); ?>
								</text>
							</symbol>
							<g class="g-ants">
								<use xlink:href="#s-text" class="text-copy"></use>
								<use xlink:href="#s-text" class="text-copy"></use>
								<use xlink:href="#s-text" class="text-copy"></use>
								<use xlink:href="#s-text" class="text-copy"></use>
								<use xlink:href="#s-text" class="text-copy"></use>
							</g>
						</svg>
					</div>
					<?php
				}
			}
	}
}
if ( ! function_exists( 'w3construct_is_a_multiple_of4' ) ) {
	/**
	 * To use in Portfolio/Gallery Cobble Structure.
	 *
	 * @param int $n The number to be checked.
	 * @return bool True if 'n' is a multiple of 4, false otherwise.
	 */
	function w3construct_is_a_multiple_of4( $n ) {
		/** If true, then 'n' is a multiple of 4 */
		if ( 0 === ( $n & 3 ) ) {
			return true;
		} else {
			return false;
		}
		/** Else 'n' is not a multiple of 4 */
	}
}

if ( ! function_exists( 'w3construct_change_default_pages_post_order' ) ) {
	/**
	 * Change the default post order based on template-specific settings.
	 *
	 * @param WP_Query $query_w3 The WordPress query object.
	 */
	function w3construct_change_default_pages_post_order( $query_w3 ) {
		$template_name = '';

		if ( $query_w3->is_author() && $query_w3->is_main_query() ) {
			$template_name = 'author_page';
		} elseif ( $query_w3->is_search() && $query_w3->is_main_query() ) {
			$template_name = 'search_page';
		} elseif ( $query_w3->is_category() && $query_w3->is_main_query() ) {
			$template_name = 'category_page';
		} elseif ( $query_w3->is_tag() && $query_w3->is_main_query() ) {
			$template_name = 'tag_page';
		} elseif ( $query_w3->is_archive() && $query_w3->is_main_query() ) {
			$template_name = 'archive_page';
		}

		if ( ! empty( $template_name ) && ! is_admin() && function_exists( 'is_shop' ) && ! is_shop() ) {
			$sorting_on = w3construct_get_opt( $template_name . '_sorting_on' );
			if ( $sorting_on ) {
				$sorting = w3construct_get_opt( $template_name . '_sorting' );

				if ( 'most_visited' === $sorting ) {
					$order_w3 = 'asc';
					$query_w3->set( 'meta_key', '_views_count' );
					$query_w3->set( 'orderby', 'meta_value_num' );

				} else {
					$sort_arr   = explode( '_', $sorting );
					$orderby_w3 = $sort_arr[0];
					$order_w3   = $sort_arr[1];

					$query_w3->set( 'orderby', $orderby_w3 );
				}

				$query_w3->set( 'order', $order_w3 );
			}
		}
	}
	add_action( 'pre_get_posts', 'w3construct_change_default_pages_post_order' );
	/** Change Default WordPress Pages Sorting END */
}

if ( ! function_exists( 'w3construct_body_layout_class' ) ) {
	/**
	 * Change Body Layout.
	 *
	 * @param array $classes An array of existing body classes.
	 * @return array An array of modified body classes.
	 */
	function w3construct_body_layout_class( $classes ) {
		global $w3construct_option;

		$class = '';

		$theme_layout = w3construct_get_opt( 'theme_layout' );

		$theme_corner = '';

		/** Theme Layout */
		if ( ! empty( $theme_layout ) ) {

			if ( 'theme_layout_2' === $theme_layout ) {
				/** Boxed Layout */
				$class .= ' boxed';
			} elseif ( 'theme_layout_3' === $theme_layout ) {
				/** Frame Layout */
				$class .= 'frame';
			}
		}

		if ( 'rounded' === $theme_corner ) {
			$class .= ' theme-rounded';
		} elseif ( 'sharped' === $theme_corner ) {
			$class .= ' theme-sharped';
		} else {
			$class .= '';
		}

		return array_merge( $classes, array( $class ) );

	}
	add_filter( 'body_class', 'w3construct_body_layout_class' );
}

if ( ! function_exists( 'w3construct_body_layout_style' ) ) {
	/** Change Body Layout Style */
	function w3construct_body_layout_style() {
		$theme_options = w3construct_get_theme_option();
		$theme_layout  = w3construct_set( $theme_options, 'theme_layout', 'theme_layout_1' );
		$style         = '';
		if ( ! empty( $theme_layout ) && 'theme_layout_1' !== $theme_layout ) {
			$output  = '';
			$bg_type = w3construct_set( $theme_options, 'body_boxed_bg_type' );
			if ( 'bg_type_color' === $bg_type ) {
				$bg_color        = w3construct_set( $theme_options, 'boxed_layout_bg_color' );
				$custom_bg_color = w3construct_set( $theme_options, 'boxed_layout_custom_bg_color' );

				if ( ! empty( $custom_bg_color['color'] ) ) {
					$output = 'background-color:' . $custom_bg_color['color'] . ';';
				} elseif ( ! empty( $bg_color ) ) {
					$output = 'background-color:' . $bg_color . ';';
				}
			} elseif ( 'bg_type_image' === $bg_type ) {
				$bg_image        = w3construct_set( $theme_options, 'boxed_layout_bg_image' );
				$custom_bg_image = w3construct_set( $theme_options, 'boxed_layout_custom_bg_image' );

				if ( ! empty( $custom_bg_image['url'] ) ) {
					$output = 'background-image:url(  ' . $custom_bg_image['url'] . '  ); background-size: auto;';
				} elseif ( ! empty( $bg_image ) ) {
					$bg_image = get_template_directory_uri() . '/assets/images/switcher/background/' . $bg_image . '.jpg';
					$output   = 'background-image: url(  ' . $bg_image . '  ); background-size: auto;';
				}
			} elseif ( 'bg_type_pattern' === $bg_type ) {
				$bg_pattern        = w3construct_set( $theme_options, 'boxed_layout_bg_pattern' );
				$custom_bg_pattern = w3construct_set( $theme_options, 'boxed_layout_custom_bg_pattern' );

				$custom_bg_pattern_padding = w3construct_set( $theme_options, 'boxed_layout_bg_pattern_padding' );

				if ( ! empty( $custom_bg_pattern['url'] ) ) {
					$output = 'background-image:url(  ' . $custom_bg_pattern['url'] . '  ); background-size: auto;';
				} elseif ( ! empty( $bg_pattern ) ) {
					$bg_pattern = get_template_directory_uri() . '/assets/images/switcher/pattern/' . $bg_pattern . '.jpg';
					$output     = 'background-image: url(  ' . $bg_pattern . '  ); background-size: auto;';
				}
			}
			if ( 'theme_layout_3' === $theme_layout && ! empty( $custom_bg_pattern_padding ) ) {
				$output .= 'padding:' . $custom_bg_pattern_padding . 'px;';
			}
			$style = 'style="' . $output . '"';
		}

		echo wp_kses( $style, w3construct_allowed_html_tag() );
	}
	/** Change Body Layout Style END */
}

if ( ! function_exists( 'w3construct_get_post_meta' ) ) {
	/**
	 * W3construct_get_post_meta
	 *
	 * @param string $post_id_w3  pass argument for the functioin.
	 * @param string $meta_key pass argument for the functioin.
	 * @return $value
	 */
	function w3construct_get_post_meta( $post_id_w3, $meta_key ) {
		$value = get_post_meta( $post_id_w3, $meta_key );
		$value = ! empty( $value[0] ) ? $value[0] : '';
		return $value;
	}
	/** Get Post Meta Data END */
}

if ( ! function_exists( 'w3construct_vc_build_link' ) ) {
	/**
	 * W3construct_get_post_meta Get VC Build Link
	 *
	 * @param string $btn_arr pass argument for the functioin.
	 * @return $btn_data
	 */
	function w3construct_vc_build_link( $btn_arr ) {
		$btn_data = array();
		if ( function_exists( 'vc_build_link' ) ) {
			$btn_data = vc_build_link( $btn_arr );
		}

		return $btn_data;
	}
	/** Get VC Build Link END */
}

if ( ! function_exists( 'w3construct_post_single_background' ) ) {
	/**
	 *
	 * Get the default background image for post pages
	 */
	function w3construct_post_single_background() {
		$bg_img        = '';
		$image_url     = '';
		$post_bg_image = w3construct_get_opt( 'post_bg_image' );
		if ( ! empty( $post_bg_image ) ) {
			if ( 'skin_1' === $post_bg_image ) {
				$image_url = get_template_directory_uri() . '/assets/images/background/bg1.png';
			} elseif ( 'skin_2' === $post_bg_image ) {
				$image_url = get_template_directory_uri() . '/assets/images/background/post_bg_2.png';
			} elseif ( 'skin_3' === $post_bg_image ) {
				$image_url = get_template_directory_uri() . '/assets/images/background/post_bg_3.png';
			} elseif ( 'custom' === $post_bg_image ) {
				$image_url = w3construct_get_opt( 'post_bg_image_custom' );
				$image_url = ! empty( $image_url['url'] ) ? $image_url['url'] : '';
			}

			if ( ! empty( $image_url ) ) {
				$bg_img = 'style="background-image: url(  ' . esc_url( $image_url ) . '  ); background-size: contain;"';
			}
		}

		return $bg_img;
	}
}

if ( ! function_exists( 'w3construct_get_category_by_post_id' ) ) {
	/**
	 * Get category by post id
	 *
	 * @param string $post_id_w3 pass value to function.
	 * @return $cats
	 */
	function w3construct_get_category_by_post_id( $post_id_w3 ) {
		$cats     = '';
		$cat_list = get_the_category( $post_id_w3 );
		if ( ! empty( $cat_list ) ) {
			foreach ( $cat_list as $cat ) {
				$cats .= '<a href="' . esc_url( get_category_link( $cat->term_id ) ) . '" class="post-link-in">' . esc_html( $cat->name ) . '</a>';
			}
		}

		return $cats;
	}
}

if ( ! function_exists( 'w3construct_get_cpt_category' ) ) {
	/**
	 * W3construct_get_cpt_category
	 *
	 * @param string $cat_list pass the value to the function.
	 * @param bool   $linked pass the value to the function.
	 * @return $cats
	 */
	function w3construct_get_cpt_category( $cat_list, $linked = true ) {
		$cats = '';
		if ( ! empty( $cat_list ) ) {
			foreach ( $cat_list as $cat ) {

				if ( ! empty( $linked ) ) {
					$cats .= '<a href="' . esc_url( get_category_link( $cat->term_id ) ) . '">' . esc_html( $cat->name ) . '</a>';
				} else {
					$cats .= '<a href="javascript:void(0);">' . esc_html( $cat->name ) . '</a>';
				}
			}
		}
		return $cats;
	}
}

if ( ! function_exists( 'w3construct_filtered_output' ) ) {
	/**
	 * W3construct_filtered_output
	 *
	 * @param string $output here is the variable of function.
	 * @return apply_filters
	 */
	function w3construct_filtered_output( $output ) {
		return apply_filters( 'w3construct_filtered_output', $output );
	}
}
if ( ! function_exists( 'w3construct_generate_uniq_id' ) ) {
	/**
	 * W3construct_generate_uniq_id
	 *
	 * @param string $atts here is the variable of function.
	 * @return 'a' . md5( implode( $atts ) )
	 */
	function w3construct_generate_uniq_id( $atts ) {
		$atts = ( gettype( $atts ) === 'object' ) ? json_decode( wp_json_encode( $atts ), true ) : $atts;
		return 'a' . md5( implode( $atts ) );
	}
}

if ( ! function_exists( 'w3construct_generate_rand_number' ) ) {
	/**
	 * W3construct_generate_rand_number
	 *
	 * @param string $digit here is the variable of function.
	 * @return $no
	 */
	function w3construct_generate_rand_number( $digit = 6 ) {
		$no = substr( strtoupper( md5( uniqid( wp_rand() ) ) ), 0, $digit );
		return $no;
	}
}

if ( ! function_exists( 'w3construct_get_container' ) ) {
	/**
	 * W3construct_get_container
	 *
	 * @param string $is_sidebar here is the variable of function.
	 * @return $container
	 */
	function w3construct_get_container( $is_sidebar ) {
		$container = ( $is_sidebar ) ? 'container' : 'min-container';
		return $container;
	}
}



if ( ! function_exists( 'w3construct_get_post_tags' ) ) {
	/**
	 * Showing the Tags for frontend
	 *
	 * @param string $post_id_w3 parameter of function.
	 * @return void
	 */
	function w3construct_get_post_tags( $post_id_w3 ) {
		$tag_arr = get_the_tags( $post_id_w3 );

		$output = '';
		if ( ! empty( $tag_arr ) ) {
			$tag_count = count( $tag_arr );
			$output    = '<strong>' . esc_html__( 'Tags:', 'w3construct' ) . '</strong> ';
			foreach ( $tag_arr as $tag_index => $tag ) {
				$tag_name = ( $tag_index + 1 === $tag_count ) ? $tag->name : $tag->name . ',';
				$output  .= '<a href="' . esc_url( get_tag_link( $tag->term_id ) ) . '">' . esc_html( $tag_name ) . '</a>';
			}
			$output .= '';
		}

		echo wp_kses( $output, w3construct_allowed_html_tag() );
	}
}

if ( ! function_exists( 'w3construct_get_cat_id_by_slug' ) ) {
	/**
	 * W3construct_get_cat_id_by_slug Slugs may be array or comma seperated string
	 *
	 * @param string $slugs parameter commment here.
	 * @param string $taxonomy parameter commment here.
	 * @return $cat_id_w3
	 */
	function w3construct_get_cat_id_by_slug( $slugs, $taxonomy = 'category' ) {
		$cat_id_w3 = array();
		if ( ! is_array( $slugs ) ) {
			$slugs = explode( ',', $slugs );
		}

		if ( ! empty( $slugs ) ) {
			foreach ( $slugs as $slug ) {
				$category = get_term_by( 'slug', $slug, $taxonomy );
				if ( ! empty( $category->term_id ) ) {
					$cat_id_w3[] = $category->term_id;
				}
			}
		}
		return $cat_id_w3;
	}
}
if ( ! function_exists( 'dz_get_lazy_loading_bg_img' ) ) {
	/**
	 * Generate a lazy loading background image style.
	 *
	 * @param int    $attachment_id The ID of the image attachment.
	 * @param string $size          (Optional) The image size. Default is 'thumbnail'.
	 * @param array  $attr          (Optional) Additional attributes for the image.
	 * @return string The background image style attribute.
	 */
	function dz_get_lazy_loading_bg_img( $attachment_id, $size = 'thumbnail', $attr = null ) {
		$lazy_load_image = wp_get_attachment_image_url( $attachment_id, $size );
		echo 'style="background-image: url(' . esc_url( $lazy_load_image ) . ');"';
		return false;
	}
}
if ( ! function_exists( 'dz_get_image' ) ) {
	/**
	 * Get and optionally echo an image by attachment ID.
	 *
	 * @param int    $attachment_id The ID of the image attachment.
	 * @param string $size          (Optional) The image size. Default is 'thumbnail'.
	 * @param array  $attr          (Optional) Additional attributes for the image.
	 * @param bool   $icon          (Optional) Whether to use an icon.
	 * @param bool   $echo          (Optional) Whether to echo the image. Default is true.
	 * @return string|null If $echo is false, returns the image HTML as a string. Otherwise, returns null.
	 */
	function dz_get_image( $attachment_id, $size = 'thumbnail', $attr = null, $icon = false, $echo = true ) {

		$allowed_html_tag = w3construct_allowed_html_tag();
		$img_tag          = wp_get_attachment_image( $attachment_id, $size, $icon, $attr );

		if ( $echo ) {
			echo wp_kses( $img_tag, $allowed_html_tag );
		} else {
			return wp_kses( $img_tag, $allowed_html_tag );
		}
	}
}

if ( ! function_exists( 'w3construct_filter_search_result' ) ) {
	/** Deafault Search Filter: remove pages from search */
	add_filter( 'register_post_type_args', 'w3construct_filter_search_result', 10, 2 );

	/**
	 * Filter search results for a specific post type.
	 *
	 * @param array  $args          The search query arguments.
	 * @param string $post_type_w3  The post type for which search results are being filtered.
	 * @return array Modified search query arguments.
	 */
	function w3construct_filter_search_result( $args, $post_type_w3 ) {
		if ( ! is_admin() && 'page' === $post_type_w3 ) {
			$args['exclude_from_search'] = true;
		}
		return $args;
	}
}

if ( ! function_exists( 'w3construct_allowed_html_tag' ) ) {
	/** WordPress allowed html tag */
	function w3construct_allowed_html_tag() {
		global $w3construct_option;

		if ( ! empty( $w3construct_option ) ) {
			$allowed_html_tags = w3construct_set( $w3construct_option, 'allowed_html_tags' );
		} else {
			$allowed_html_tags = wp_kses_allowed_html( 'post' );
		}
		$allowed_html_tags = ! empty( $allowed_html_tags ) ? $allowed_html_tags : 'string';
		return $allowed_html_tags;
	}
}
if ( ! function_exists( 'w3construct_elementor_get_anchor_attribute' ) ) {
	/** Elementor Function */
	/**
	 * Generate anchor tag attributes based on button link settings.
	 *
	 * @param array $btn_link The button link settings.
	 * @return string A string containing anchor tag attributes.
	 */
	function w3construct_elementor_get_anchor_attribute( $btn_link ) {
		$anchor_attribute = '';
		if ( isset( $btn_link['is_external'] ) && 'on' === $btn_link['is_external'] ) {
			$anchor_attribute .= ' target=_blank ';
		}

		if ( isset( $btn_link['nofollow'] ) && 'on' === $btn_link['nofollow'] ) {
			$anchor_attribute .= ' rel=nofollow';
		}

		if ( ! empty( $btn_link['custom_attributes'] ) ) {
			$attr_arr = explode( ',', $btn_link['custom_attributes'] );

			if ( ! empty( $attr_arr ) ) {
				foreach ( $attr_arr as $key => $value ) {
					$attr_data         = explode( '|', $value );
					$anchor_attribute .= ' ' . $attr_data[0] . '=' . $attr_data[1] . '';
				}
			}
		}

		return $anchor_attribute;
	}
}

if ( ! function_exists( 'w3construct_get_team_social_link' ) ) {
	/** CPT Team Social Links */
	/**
	 * Get and organize social media links for a team member.
	 *
	 * @param int $post_id_w3 The ID of the team member's post.
	 * @return array An array containing social media link data.
	 */
	function w3construct_get_team_social_link( $post_id_w3 ) {
		$team_social_data = array(
			'any_fill' => false,
			'data'     => array(),
		);
		$team_socials     = array(
			'facebook'  => array(
				'key'   => 'team_social_facebook',
				'class' => 'fab fa-facebook-f',
				'url'   => '',
			),
			'twitter'   => array(
				'key'   => 'team_social_twitter',
				'class' => 'fab fa-twitter',
				'url'   => '',
			),
			'instagram' => array(
				'key'   => 'team_social_instagram',
				'class' => 'fab fa-instagram',
				'url'   => '',
			),
			'youtube'   => array(
				'key'   => 'team_social_youtube',
				'class' => 'fab fa-youtube',
				'url'   => '',
			),
		);
		foreach ( $team_socials as $key => $value ) {
			$team_socials[ $key ]['url'] = w3construct_get_post_meta( $post_id_w3, $value['key'] );

			if ( ! empty( $team_socials[ $key ]['url'] ) ) {
				$team_social_data['any_fill'] = true;
			}
		}

		$team_social_data['data'] = $team_socials;

		return $team_social_data;
	}
}
if ( ! function_exists( 'theme_direction' ) ) {
	/** Change Theme Direction LTR - RTL */
	function theme_direction() {
		global $w3construct_option;

		$rtl = w3construct_set( $w3construct_option, 'rtl_on' );

		if ( $rtl ) {
			echo 'dir="rtl"';
		}

		return false;
	}
}
if ( ! function_exists( 'is_website_ready_for_visitor' ) ) {
	/**
	 * Check if the website is ready for visitors based on its status.
	 *
	 * @param string $website_status The current status of the website.
	 * @return bool True if the website is ready for visitors, false otherwise.
	 */
	function is_website_ready_for_visitor( $website_status ) {
		if ( 'live_mode' === $website_status || ( in_array( $website_status, array( 'comingsoon_mode', 'maintenance_mode' ), true ) && is_user_logged_in() ) ) {
			return true;
		} else {
			return false;
		}
	}
}

if ( ! function_exists( 'w3construct_is_woocommerce_active' ) ) {
	/**
	 * Check if WooCommerce is active
	 **/
	function w3construct_is_woocommerce_active() {
		if ( in_array( 'woocommerce/woocommerce.php', apply_filters( 'active_plugins', get_option( 'active_plugins' ) ), true ) ) {
			return true;
		} else {
			return false;
		}
	}
}

if ( ! function_exists( 'w3construct_wp_kses_allowed_html' ) ) {
	/** Add Svg tags for wp hook*/
	add_filter( 'wp_kses_allowed_html', 'w3construct_wp_kses_allowed_html' );
	/**
	 * W3construct_wp_kses_allowed_html
	 *
	 * @param string $allowed_html this variable take the value for this function.
	 * @return $allowed_tags
	 */
	function w3construct_wp_kses_allowed_html( $allowed_html ) {

		$svg_args = array(
			'svg'      => array(
				'fill'            => true,
				'class'           => true,
				'aria-hidden'     => true,
				'aria-labelledby' => true,
				'role'            => true,
				'xmlns'           => true,
				'width'           => true,
				'height'          => true,
				'viewbox'         => true,
				// <= Must be lower case!
			),
			'iframe'   => array(
				'src'             => true,
				'width'           => true,
				'height'          => true,
				'frameborder'     => true,
				'allowfullscreen' => true,
			),
			'title'    => array( 'title' => true ),
			'path'     => array(
				'd'    => true,
				'fill' => true,
			),
			'defs'     => array(
				'id'       => true,
				'path'     => true,
				'clipPath' => array(
					'id'   => true,
					'path' => true,
				),
			),
			'clipPath' => array(
				'id' => true,
			),
			'g'        => array(
				'id'           => true,
				'stroke'       => true,
				'stroke-width' => true,
				'fill'         => true,
				'fill-rule'    => true,
				'transform'    => true,
				'g'            => array(
					'line'     => array(
						'id' => true,
						'x1' => true,
						'y1' => true,
						'x2' => true,
						'y2' => true,
					),
					'polyline' => array(
						'id'     => true,
						'points' => true,
					),
				),
			),
		);
		$allowed_tags = array_merge( $allowed_html, $svg_args );
		return $allowed_tags;
	}
}
/**
 * Fix skip link focus in IE11.
 *
 * This does not enqueue the script because it is tiny and because it is only for IE11,
 * thus it does not warrant having an entire dedicated blocking script being loaded.
 *
 * @link https://git.io/vWdr2
 */
function w3construct_skip_link_focus_fix() {
	// The following is minified via `terser --compress --mangle -- js/skip-link-focus-fix.js`.
	?>
	<script>
	/(trident|msie)/i.test(navigator.userAgent)&&document.getElementById&&window.addEventListener&&window.addEventListener("hashchange",function(){var t,e=location.hash.substring(1);/^[A-z0-9_-]+$/.test(e)&&(t=document.getElementById(e))&&(/^(?:a|select|input|button|textarea)$/i.test(t.tagName)||(t.tabIndex=-1),t.focus())},!1);
	</script>
	<?php
}
add_action( 'wp_print_footer_scripts', 'w3construct_skip_link_focus_fix' );