<?php
/**
 * Register a metabox for posts based on Redux Framework.
 *
 * This code registers a metabox for posts using the Redux Framework. It provides methods to
 * configure and manage the metabox for different post types.
 *
 * @package   W3Construct
 * @author    W3ITEXPERTS
 * @copyright Copyright (c) 2023, W3Construct
 * @link      http://w3construct-wordpress.wprdx.come/demo/
 * @since     W3Construct 1.0.1
 *
 * Supported methods:
 * - isset_args( $post_type_w3 )
 * - set_args( $post_type_w3, $redux_args, $metabox_args )
 * - add_section($post_type_w3, $sections )
 *
 * Please note that each field ID begins with an underscore sign ( _ ) to prevent it from
 * appearing in the Custom Field Metabox of the WordPress core feature.
 */

if ( ! function_exists( 'w3construct_get_default_option' ) ) {
	/**
	 * Retrieve the default option value for a given key.
	 *
	 * This function retrieves the default option value for the specified key. If the key is not
	 * found in the options, the provided default value is returned.
	 *
	 * @param string $key     The key of the option to retrieve.
	 * @param mixed  $default The default value to return if the key is not found (optional).
	 * @return mixed The default option value or the provided default value if the key is not found.
	 */
	function w3construct_get_default_option( $key, $default = '' ) {
		if ( empty( $key ) ) {
			return '';
		}
		$options = get_option( w3construct_get_opt_name(), array() );
		$value   = isset( $options[ $key ] ) ? $options[ $key ] : $default;

		return $value;
	}
}
if ( ! function_exists( 'w3construct_page_options_register' ) ) {
	/**
	 * Register a metabox for page options using Redux Framework.
	 *
	 * This function is used to register a metabox for page options using the Redux Framework.
	 * It defines the configuration for the metabox, such as sections, fields, and options.
	 *
	 * @param array $metabox The configuration for the metabox, including sections, fields, and options.
	 */
	function w3construct_page_options_register( $metabox ) {
		$post_layouts_options       = w3construct_get_post_layouts_options();
		$sidebar_layout_options     = w3construct_get_sidebar_layout_options();
		$header_style_options       = w3construct_get_header_style_options();
		$page_banner_options        = w3construct_get_page_banner_options();
		$page_banner_layout_options = w3construct_get_page_banner_layout_options();
		$post_banner_options        = w3construct_get_post_banner_options();
		$footer_style_options       = w3construct_get_footer_style_options();
		$banner_type                = w3construct_banner_type();
		/**
		 * Option for single posts.
		 *
		 * This section defines options for single posts. If the metabox configuration for 'post'
		 * does not exist, it will be added here.
		 */
		if ( ! $metabox->isset_args( 'post' ) ) {
			$metabox->set_args(
				'post',
				array(
					'opt_name'            => w3construct_get_post_opt_name(),
					'display_name'        => esc_html__( 'Post Settings', 'w3construct' ),
					'show_options_object' => false,
					'hints'               => array(
						'icon'          => 'el el-question-sign',
						'icon_position' => 'right',
						'icon_color'    => '#1085e4',
						'icon_size'     => '10',
						'tip_style'     => array(
							'color'   => '#1085e4',
							'shadow'  => true,
							'rounded' => false,
							'style'   => '',
						),
						'tip_position'  => array(
							'my' => 'top left',
							'at' => 'bottom right',
						),
						'tip_effect'    => array(
							'show' => array(
								'effect'   => 'slide',
								'duration' => '500',
								'event'    => 'mouseover',
							),
							'hide' => array(
								'effect'   => 'slide',
								'duration' => '500',
								'event'    => 'click mouseleave',
							),
						),
					),
				),
				array(
					'context'  => 'advanced',
					'priority' => 'default',
				)
			);
		}
		$metabox->add_section(
			'post',
			array(
				'title'  => esc_html__( 'General', 'w3construct' ),
				'icon'   => 'fa fa-newspaper-o',
				'fields' => array(
					array(
						'id'      => 'featured_post',
						'type'    => 'checkbox',
						'title'   => esc_html__( 'Featured Post?', 'w3construct' ),
						'desc'    => esc_html__( 'Check if you want to make this post as a featured post', 'w3construct' ),
						'default' => '',
					),
				),
			)
		);

		$metabox->add_section(
			'post',
			array(
				'title'  => esc_html__( 'Post Header', 'w3construct' ),
				'desc'   => esc_html__( 'Header settings for the post.', 'w3construct' ),
				'icon'   => 'fa fa-window-maximize',
				'fields' => array(
					array(
						'id'      => 'post_header_setting',
						'type'    => 'button_set',
						'title'   => esc_html__( 'Post Header Settings', 'w3construct' ),
						'options' => array(
							'theme_default' => esc_html__( 'Theme Default', 'w3construct' ),
							'custom'        => esc_html__( 'Custom Setting', 'w3construct' ),
						),
						'default' => 'theme_default',
					),
					array(
						'id'           => 'post_header_style',
						'type'         => 'image_select',
						'title'        => esc_html__( 'Header Style', 'w3construct' ),
						'subtitle'     => esc_html__( 'Choose header style. White header is set as the default header for this post.', 'w3construct' ),
						'options'      => $header_style_options,
						'default'      => w3construct_get_opt( 'header_style' ),
						'force_output' => true,
						'required'     => array(
							0 => 'post_header_setting',
							1 => 'equals',
							2 => 'custom',
						),
					),
					array(
						'id'       => 'post_header_theme',
						'type'     => 'button_set',
						'title'    => esc_html__( 'Post Header Theme', 'w3construct' ),
						'options'  => array(
							'dark'  => esc_html__( 'Dark', 'w3construct' ),
							'light' => esc_html__( 'Light', 'w3construct' ),
						),
						'default'  => 'light',
						'required' => array(
							array(
								0 => 'post_header_setting',
								1 => 'equals',
								2 => 'custom',
							),
							array(
								0 => 'post_header_style',
								1 => 'equals',
								2 => 'header-2',
							),
						),
					),
				),
			)
		);

		$metabox->add_section(
			'post',
			array(
				'title'  => esc_html__( 'Post Layout', 'w3construct' ),
				'icon'   => 'fa fa-file-text-o',
				'fields' => array(
					array(
						'id'       => 'post_layout',
						'type'     => 'image_select',
						'height'   => '55',
						'title'    => esc_html__( 'Layout', 'w3construct' ),
						'subtitle' => esc_html__( 'Select a template.', 'w3construct' ),
						'desc'     => esc_html__( 'Click on the template icon to select.', 'w3construct' ),
						'options'  => $post_layouts_options,
						'default'  => w3construct_get_opt( 'post_general_layout' ),
						'hint'     => array(
							'title'   => esc_html__( 'How it Works?', 'w3construct' ),
							'content' => esc_html__( 'Once you select the template from here, the template will apply for this page only.', 'w3construct' ),
						),
					),
					array(
						'id'       => 'post_type_gallery1',
						'type'     => 'gallery',
						'title'    => esc_html__( 'Gallery', 'w3construct' ),
						'subtitle' => esc_html__( 'Select the gallery images', 'w3construct' ),
						'desc'     => esc_html__( 'For better layout, Image size width greater than 1000 and height greater than 600', 'w3construct' ),
						'default'  => '',
						'required' => array(
							0 => 'post_layout',
							1 => 'equals',
							2 => 'slider_post_1',
						),
					),
					array(
						'id'       => 'post_type_gallery2',
						'type'     => 'gallery',
						'title'    => esc_html__( 'Gallery', 'w3construct' ),
						'subtitle' => esc_html__( 'Select the gallery images', 'w3construct' ),
						'desc'     => esc_html__( 'For better layout, Image size width greater than 1000 and height greater than 600', 'w3construct' ),
						'default'  => '',
						'required' => array(
							0 => 'post_layout',
							1 => 'equals',
							2 => 'slider_post_2',
						),
					),
					array(
						'id'       => 'post_type_gallery3',
						'type'     => 'gallery',
						'title'    => esc_html__( 'Gallery', 'w3construct' ),
						'subtitle' => esc_html__( 'Select the gallery images', 'w3construct' ),
						'desc'     => esc_html__( 'For better layout, Image size width greater than 1000 and height greater than 600', 'w3construct' ),
						'default'  => '',
						'required' => array(
							0 => 'post_layout',
							1 => 'equals',
							2 => 'slider_post_3',
						),
					),
					array(
						'id'       => 'post_type_link',
						'type'     => 'text',
						'title'    => esc_html__( 'External Link', 'w3construct' ),
						'default'  => '',
						'validate' => 'url',
						'required' => array(
							0 => 'post_layout',
							1 => 'equals',
							2 => 'link_post',
						),
					),
					array(
						'id'       => 'post_type_quote_author',
						'type'     => 'text',
						'title'    => esc_html__( 'Author Name', 'w3construct' ),
						'default'  => esc_html__( 'Author Name', 'w3construct' ),
						'required' => array(
							0 => 'post_layout',
							1 => 'equals',
							2 => 'quote_post',
						),
					),
					array(
						'id'       => 'post_type_quote_text',
						'type'     => 'textarea',
						'title'    => esc_html__( 'Quote Text', 'w3construct' ),
						'default'  => esc_html__( 'Quote Text', 'w3construct' ),
						'required' => array(
							0 => 'post_layout',
							1 => 'equals',
							2 => 'quote_post',
						),
					),
					array(
						'id'       => 'post_type_audio',
						'type'     => 'text',
						'title'    => esc_html__( 'Sound Cloud Link', 'w3construct' ),
						'default'  => '',
						'validate' => 'url',
						'required' => array(
							0 => 'post_layout',
							1 => 'equals',
							2 => 'audio_post',
						),
					),
					array(
						'id'       => 'post_type_video',
						'type'     => 'text',
						'title'    => esc_html__( 'Video Link', 'w3construct' ),
						'default'  => '',
						'validate' => 'url',
						'required' => array(
							0 => 'post_layout',
							1 => 'equals',
							2 => 'video_post',
						),
					),
					array(
						'id'       => 'post_show_sidebar',
						'type'     => 'switch',
						'title'    => esc_html__( 'Sidebar', 'w3construct' ),
						'desc'     => esc_html__( 'Show / hide sidebar from this post\'s detail page.', 'w3construct' ),
						'on'       => esc_html__( 'Enabled', 'w3construct' ),
						'off'      => esc_html__( 'Disabled', 'w3construct' ),
						'default'  => w3construct_get_opt( 'show_sidebar' ),
						'required' => array(
							array(
								0 => 'post_layout',
								1 => '!=',
								2 => 'gutenberg',
							),
						),
					),
					array(
						'id'       => 'post_sidebar_layout',
						'type'     => 'image_select',
						'title'    => esc_html__( 'Sidebar Layout', 'w3construct' ),
						'subtitle' => esc_html__( 'Choose the layout for the page. ( Default : Right Side ).', 'w3construct' ),
						'options'  => $sidebar_layout_options,
						'default'  => 'right',
						'required' => array(
							0 => 'post_show_sidebar',
							1 => 'equals',
							2 => '1',
						),
					),
					array(
						'id'       => 'post_sidebar',
						'type'     => 'select',
						'data'     => 'sidebars',
						'title'    => esc_html__( 'Sidebar', 'w3construct' ),
						'subtitle' => esc_html__( 'Select sidebar.', 'w3construct' ),
						'default'  => 'dz_blog_sidebar',
						'required' => array(
							0 => 'post_sidebar_layout',
							1 => 'equals',
							2 => array(
								'right',
								'left',
							),
						),
					),
					array(
						'id'      => 'featured_image',
						'type'    => 'switch',
						'title'   => esc_html__( 'Show Feature Image', 'w3construct' ),
						'on'      => esc_html__( 'Enabled', 'w3construct' ),
						'off'     => esc_html__( 'Disabled', 'w3construct' ),
						'default' => 1,
					),
					array(
						'id'      => 'post_pagination',
						'type'    => 'switch',
						'title'   => esc_html__( 'Show Post Pagination', 'w3construct' ),
						'on'      => esc_html__( 'Enabled', 'w3construct' ),
						'off'     => esc_html__( 'Disabled', 'w3construct' ),
						'default' => 0,
					),
				),
			)
		);
		$metabox->add_section(
			'post',
			array(
				'title'  => esc_html__( 'Post Banner', 'w3construct' ),
				'desc'   => esc_html__( 'Settings for post banner.', 'w3construct' ),
				'icon'   => 'fa fa-television',
				'fields' => array(
					array(
						'id'      => 'post_banner_setting',
						'type'    => 'button_set',
						'title'   => esc_html__( 'Post Banner Settings', 'w3construct' ),
						'options' => array(
							'theme_default' => esc_html__( 'Theme Default', 'w3construct' ),
							'custom'        => esc_html__( 'Custom Setting', 'w3construct' ),
						),
						'default' => 'theme_default',
					),
					array(
						'id'       => 'post_general_banner_on',
						'type'     => 'switch',
						'title'    => esc_html__( 'Post Banner', 'w3construct' ),
						'on'       => esc_html__( 'Enabled', 'w3construct' ),
						'off'      => esc_html__( 'Disabled', 'w3construct' ),
						'default'  => w3construct_get_opt( 'post_general_banner_on' ),
						'required' => array(
							0 => 'post_banner_setting',
							1 => 'equals',
							2 => 'custom',
						),
					),
					array(
						'id'       => 'post_general_banner_height',
						'type'     => 'image_select',
						'title'    => esc_html__( 'Post Banner Height', 'w3construct' ),
						'subtitle' => esc_html__( 'Choose the height for all tag page banner. Default : Big Banner', 'w3construct' ),
						'options'  => $page_banner_options,
						'height'   => '20',
						'default'  => w3construct_get_opt( 'post_general_banner_height' ),
						'required' => array(
							0 => 'post_general_banner_on',
							1 => 'equals',
							2 => 1,
						),
					),
					array(
						'id'            => 'post_general_banner_custom_height',
						'type'          => 'slider',
						'title'         => esc_html__( 'Post Banner Custom Height', 'w3construct' ),
						'desc'          => esc_html__( 'Height description. Min: 100, max: 800', 'w3construct' ),
						'default'       => w3construct_get_opt( 'post_general_banner_height' ),
						'min'           => 100,
						'max'           => 800,
						'display_value' => 'text',
						'required'      => array(
							0 => 'post_general_banner_height',
							1 => 'equals',
							2 => 'page_banner_custom',
						),
					),

					array(
						'id'       => 'post_general_banner',
						'type'     => 'media',
						'url'      => true,
						'title'    => esc_html__( 'Post Banner Image', 'w3construct' ),
						'subtitle' => esc_html__( 'Enter page banner image. It will work as default banner image for all pages', 'w3construct' ),
						'desc'     => esc_html__( 'Upload banner image.', 'w3construct' ),
						'default'  => w3construct_get_opt( 'post_general_banner' ),
						'required' => array(
							0 => 'post_general_banner_on',
							1 => 'equals',
							2 => 1,
						),
					),
					array(
						'id'       => 'post_general_breadcrumb',
						'type'     => 'switch',
						'title'    => esc_html__( 'Breadcrumb Area', 'w3construct' ),
						'subtitle' => esc_html__( 'Click on the tab to Enable / Disable the website breadcrumb.', 'w3construct' ),
						'desc'     => esc_html__( 'This setting affects only on this page.', 'w3construct' ),
						'on'       => esc_html__( 'Enabled', 'w3construct' ),
						'off'      => esc_html__( 'Disabled', 'w3construct' ),
						'default'  => w3construct_get_opt( 'show_breadcrumb' ),
						'required' => array(
							0 => 'post_general_banner_on',
							1 => 'equals',
							2 => 1,
						),
					),
				),
			)
		);
		$metabox->add_section(
			'post',
			array(
				'title'  => esc_html__( 'Post Footer', 'w3construct' ),
				'desc'   => esc_html__( 'Settings for footer area.', 'w3construct' ),
				'icon'   => 'fa fa-home',
				'fields' => array(
					array(
						'id'      => 'post_footer_setting',
						'type'    => 'button_set',
						'title'   => esc_html__( 'Post Footer Settings', 'w3construct' ),
						'options' => array(
							'theme_default' => esc_html__( 'Theme Default', 'w3construct' ),
							'custom'        => esc_html__( 'Custom Setting', 'w3construct' ),
						),
						'default' => 'theme_default',
					),
					array(
						'id'       => 'post_footer_on',
						'type'     => 'switch',
						'title'    => esc_html__( 'Footer', 'w3construct' ),
						'on'       => esc_html__( 'Enabled', 'w3construct' ),
						'off'      => esc_html__( 'Disabled', 'w3construct' ),
						'default'  => w3construct_get_opt( 'footer_on' ),
						'required' => array(
							0 => 'post_footer_setting',
							1 => 'equals',
							2 => 'custom',
						),
					),
					array(
						'id'       => 'post_footer_style',
						'type'     => 'image_select',
						'height'   => '80',
						'title'    => esc_html__( 'Footer Template', 'w3construct' ),
						'subtitle' => esc_html__( 'Choose a template for footer.', 'w3construct' ),
						'options'  => $footer_style_options,
						'default'  => w3construct_get_opt( 'footer_style' ),
						'required' => array(
							array(
								0 => 'post_footer_setting',
								1 => 'equals',
								2 => 'custom',
							),
							array(
								0 => 'post_footer_on',
								1 => 'equals',
								2 => 1,
							),
						),
					),
				),
			)
		);

		/**
		 * Option for single pages.
		 *
		 * This section defines options for single pages. If the metabox configuration for 'page'
		 * does not exist, it will be added here.
		 */
		if ( ! $metabox->isset_args( 'page' ) ) {
			$metabox->set_args(
				'page',
				array(
					'opt_name'            => w3construct_get_page_opt_name(),
					'display_name'        => esc_html__( 'Page Settings', 'w3construct' ),
					'show_options_object' => false,
					'hints'               => array(
						'icon'          => 'el el-question-sign',
						'icon_position' => 'right',
						'icon_color'    => '#1085e4',
						'icon_size'     => '10',
						'tip_style'     => array(
							'color'   => '#1085e4',
							'shadow'  => true,
							'rounded' => false,
							'style'   => '',
						),
						'tip_position'  => array(
							'my' => 'top left',
							'at' => 'bottom right',
						),
						'tip_effect'    => array(
							'show' => array(
								'effect'   => 'slide',
								'duration' => '500',
								'event'    => 'mouseover',
							),
							'hide' => array(
								'effect'   => 'slide',
								'duration' => '500',
								'event'    => 'click mouseleave',
							),
						),
					),
				),
				array(
					'context'  => 'advanced',
					'priority' => 'default',
				)
			);
		}
		$metabox->add_section(
			'page',
			array(
				'title'  => esc_html__( 'Page Header', 'w3construct' ),
				'desc'   => esc_html__( 'Header settings for the page.', 'w3construct' ),
				'icon'   => 'fa fa-window-maximize',
				'fields' => array(
					array(
						'id'      => 'page_header_setting',
						'type'    => 'button_set',
						'title'   => esc_html__( 'Page Header Settings', 'w3construct' ),
						'options' => array(
							'theme_default' => esc_html__( 'Theme Default', 'w3construct' ),
							'custom'        => esc_html__( 'Custom Setting', 'w3construct' ),
						),
						'default' => 'theme_default',
					),
					array(
						'id'           => 'page_header_style',
						'type'         => 'image_select',
						'title'        => esc_html__( 'Header Style', 'w3construct' ),
						'subtitle'     => esc_html__( 'Choose header style. White header is set as default header for this page.', 'w3construct' ),
						'options'      => $header_style_options,
						'default'      => w3construct_get_opt( 'header_style' ),
						'force_output' => true,
						'required'     => array(
							0 => 'page_header_setting',
							1 => 'equals',
							2 => 'custom',
						),
					),
					array(
						'id'       => 'page_header_theme',
						'type'     => 'button_set',
						'title'    => esc_html__( 'Page Header Theme', 'w3construct' ),
						'options'  => array(
							'dark'  => esc_html__( 'Dark', 'w3construct' ),
							'light' => esc_html__( 'Light', 'w3construct' ),
						),
						'default'  => 'light',
						'required' => array(
							array(
								0 => 'page_header_setting',
								1 => 'equals',
								2 => 'custom',
							),
							array(
								0 => 'page_header_style',
								1 => 'equals',
								2 => 'header-2',
							),
						),
					),
				),
			)
		);

		$metabox->add_section(
			'page',
			array(
				'title'  => esc_html__( 'Page Banner', 'w3construct' ),
				'desc'   => esc_html__( 'Settings for page banner.', 'w3construct' ),
				'icon'   => 'fa fa-television',
				'fields' => array(
					array(
						'id'      => 'page_banner_setting',
						'type'    => 'button_set',
						'title'   => esc_html__( 'Page Banner Settings', 'w3construct' ),
						'options' => array(
							'theme_default' => esc_html__( 'Theme Default', 'w3construct' ),
							'custom'        => esc_html__( 'Custom Setting', 'w3construct' ),
						),
						'default' => 'theme_default',
					),
					array(
						'id'       => 'page_banner_on',
						'type'     => 'switch',
						'title'    => esc_html__( 'Page Banner', 'w3construct' ),
						'on'       => esc_html__( 'Enabled', 'w3construct' ),
						'off'      => esc_html__( 'Disabled', 'w3construct' ),
						'default'  => w3construct_get_opt( 'page_general_banner_on' ),
						'required' => array(
							0 => 'page_banner_setting',
							1 => 'equals',
							2 => 'custom',
						),
					),
					array(
						'id'       => 'page_banner_type',
						'type'     => 'button_set',
						'title'    => esc_html__( 'Page Banner Type', 'w3construct' ),
						'options'  => $banner_type,
						'default'  => w3construct_get_opt( 'page_general_banner_type' ),
						'required' => array(
							0 => 'page_banner_on',
							1 => 'equals',
							2 => '1',
						),
					),
					array(
						'id'       => 'page_banner_height',
						'type'     => 'image_select',
						'title'    => esc_html__( 'Page Banner Height', 'w3construct' ),
						'subtitle' => esc_html__( 'Choose the height for all tag page banner. Default : Big Banner', 'w3construct' ),
						'options'  => $page_banner_options,
						'height'   => '20',
						'default'  => w3construct_get_opt( 'page_general_banner_height' ),
						'required' => array(
							0 => 'page_banner_type',
							1 => 'equals',
							2 => 'image',
						),
					),
					array(
						'id'            => 'page_banner_custom_height',
						'type'          => 'slider',
						'title'         => esc_html__( 'Page Banner Custom Height', 'w3construct' ),
						'desc'          => esc_html__( 'Height description. Min: 100, max: 800', 'w3construct' ),
						'default'       => w3construct_get_opt( 'page_general_banner_height' ),
						'min'           => 100,
						'max'           => 800,
						'display_value' => 'text',
						'required'      => array(
							0 => 'page_banner_height',
							1 => 'equals',
							2 => 'page_banner_custom',
						),
					),
					array(
						'id'       => 'page_banner',
						'type'     => 'media',
						'url'      => true,
						'title'    => esc_html__( 'Page Banner Image', 'w3construct' ),
						'subtitle' => esc_html__( 'Enter page banner image. It will work as a default banner image for all pages', 'w3construct' ),
						'desc'     => esc_html__( 'Upload banner image.', 'w3construct' ),
						'default'  => w3construct_get_opt( 'page_general_banner' ),
						'required' => array(
							0 => 'page_banner_type',
							1 => 'equals',
							2 => 'image',
						),
					),
					array(
						'id'       => 'page_banner_title',
						'type'     => 'text',
						'url'      => true,
						'title'    => esc_html__( 'Page Banner Title', 'w3construct' ),
						'subtitle' => esc_html__( 'Enter page banner title.', 'w3construct' ),
						'default'  => '',
						'required' => array(
							0 => 'page_banner_type',
							1 => 'equals',
							2 => 'image',
						),
					),
					array(
						'id'       => 'page_banner_sub_title',
						'type'     => 'text',
						'url'      => true,
						'title'    => esc_html__( 'Page Banner Sub Title', 'w3construct' ),
						'subtitle' => esc_html__( 'Enter page banner Sub title.', 'w3construct' ),
						'default'  => '',
						'required' => array(
							0 => 'page_banner_type',
							1 => 'equals',
							2 => 'image',
						),
					),
					array(
						'id'       => 'page_breadcrumb',
						'type'     => 'switch',
						'title'    => esc_html__( 'Breadcrumb Area', 'w3construct' ),
						'subtitle' => esc_html__( 'Click on the tab to Enable / Disable the website breadcrumb.', 'w3construct' ),
						'desc'     => esc_html__( ' This setting affects only on this page.', 'w3construct' ),
						'on'       => esc_html__( 'Enabled', 'w3construct' ),
						'off'      => esc_html__( 'Disabled', 'w3construct' ),
						'default'  => w3construct_get_opt( 'show_breadcrumb' ),
						'required' => array(
							0 => 'page_banner_type',
							1 => 'equals',
							2 => 'image',
						),
					),
					array(
						'id'       => 'page_banner_hide',
						'type'     => 'checkbox',
						'title'    => esc_html__( 'Don`t use banner image for this page', 'w3construct' ),
						'default'  => w3construct_get_opt( 'page_general_banner_hide' ),
						'desc'     => esc_html__( 'Check if you don`t want to use a banner image', 'w3construct' ),
						'required' => array(
							0 => 'page_banner_type',
							1 => 'equals',
							2 => 'image',
						),
						'hint'     => array(
							'content' => 'If we don`t have a suitable image then we can hide the current or default banner images and show only the banner container with the theme default color.',
						),
					),
					array(
						'id'       => 'page_no_of_post',
						'type'     => 'text',
						'title'    => esc_html__( 'Number of Posts', 'w3construct' ),
						'subtitle' => esc_html__( 'Enter the number of posts. Default : 3', 'w3construct' ),
						'default'  => w3construct_get_opt( 'page_general_no_of_post' ),
						'required' => array(
							0 => 'page_banner_type',
							1 => 'equals',
							2 => 'post',
						),
					),
					array(
						'id'       => 'page_post_banner_layout',
						'type'     => 'image_select',
						'title'    => esc_html__( 'Post Banner Layout', 'w3construct' ),
						'subtitle' => esc_html__( 'Select banner layout. Default : Full Banner', 'w3construct' ),
						'options'  => $post_banner_options,
						'default'  => w3construct_get_opt( 'general_post_banner_layout' ),
						'required' => array(
							0 => 'page_banner_type',
							1 => 'equals',
							2 => 'post',
						),
					),
					array(
						'id'       => 'page_banner_cat',
						'type'     => 'select',
						'multi'    => true,
						'data'     => 'terms',
						'args'     => array(
							'taxonomies' => 'category',
						),
						'title'    => esc_html__( 'Post Category', 'w3construct' ),
						'subtitle' => esc_html__( 'Select post category. It will work as a default banner for all pages', 'w3construct' ),
						'default'  => w3construct_get_opt( 'page_general_banner_cat' ),
						'required' => array(
							0 => 'page_banner_type',
							1 => 'equals',
							2 => 'post',
						),
					),
					array(
						'id'           => 'page_post_type',
						'type'         => 'button_set',
						'title'        => esc_html__( 'Post Type', 'w3construct' ),
						'options'      => array(
							'all'      => esc_html__( 'All', 'w3construct' ),
							'featured' => esc_html__( 'Featured', 'w3construct' ),
						),
						'default'      => w3construct_get_opt( 'page_general_post_type' ),
						'force_output' => true,
						'required'     => array(
							0 => 'page_banner_type',
							1 => 'equals',
							2 => 'post',
						),
					),
					array(
						'id'       => 'page_items_with',
						'type'     => 'button_set',
						'title'    => esc_html__( 'Items With', 'w3construct' ),
						'options'  => array(
							'with_any_type'          => esc_html__( 'Any Type', 'w3construct' ),
							'with_featured_image'    => esc_html__( 'With Featured Image', 'w3construct' ),
							'without_featured_image' => esc_html__( 'Without Featured Image', 'w3construct' ),
						),
						'default'  => w3construct_get_opt( 'page_general_items_with' ),
						'required' => array(
							0 => 'page_banner_type',
							1 => 'equals',
							2 => 'post',
						),
					),
				),
			)
		);
		$metabox->add_section(
			'page',
			array(
				'title'  => esc_html__( 'Page Sidebar', 'w3construct' ),
				'desc'   => esc_html__( 'Settings for sidebar area.', 'w3construct' ),
				'icon'   => 'fa fa-server',
				'fields' => array(
					array(
						'id'      => 'page_show_sidebar_information',
						'type'    => 'info',
						'style'   => 'warning',
						'title'   => esc_html__( 'Sidebar Information!', 'w3construct' ),
						'icon'    => 'el-icon-info-sign',
						'desc'    => esc_html__( 'These settings only working with default template in page attribute.', 'w3construct' ),
						'default' => w3construct_get_opt( 'page_general_show_sidebar' ),
					),
					array(
						'id'      => 'page_show_sidebar',
						'type'    => 'switch',
						'title'   => esc_html__( 'Sidebar', 'w3construct' ),
						'on'      => esc_html__( 'Enabled', 'w3construct' ),
						'off'     => esc_html__( 'Disabled', 'w3construct' ),
						'default' => w3construct_get_opt( 'page_general_show_sidebar' ),
					),
					array(
						'id'       => 'page_sidebar_layout',
						'type'     => 'image_select',
						'title'    => esc_html__( 'Sidebar Layout', 'w3construct' ),
						'subtitle' => esc_html__( 'Choose the layout for page. ( Default: Right Side ).', 'w3construct' ),
						'options'  => $sidebar_layout_options,
						'default'  => w3construct_get_opt( 'page_general_sidebar_layout' ),
						'required' => array(
							0 => 'page_show_sidebar',
							1 => 'equals',
							2 => '1',
						),
					),
					array(
						'id'       => 'page_sidebar',
						'type'     => 'select',
						'data'     => 'sidebars',
						'title'    => esc_html__( 'Sidebar', 'w3construct' ),
						'subtitle' => esc_html__( 'Select sidebar for the page', 'w3construct' ),
						'default'  => w3construct_get_opt( 'page_general_sidebar' ),
						'required' => array(
							0 => 'page_sidebar_layout',
							1 => 'equals',
							2 => array(
								'left',
								'right',
							),
						),
					),
				),
			)
		);
		$metabox->add_section(
			'page',
			array(
				'title'  => esc_html__( 'Page Footer', 'w3construct' ),
				'desc'   => esc_html__( 'Settings for footer area.', 'w3construct' ),
				'icon'   => 'fa fa-home',
				'fields' => array(
					array(
						'id'      => 'page_footer_setting',
						'type'    => 'button_set',
						'title'   => esc_html__( 'Page Footer Settings', 'w3construct' ),
						'options' => array(
							'theme_default' => esc_html__( 'Theme Default', 'w3construct' ),
							'custom'        => esc_html__( 'Custom Setting', 'w3construct' ),
						),
						'default' => 'theme_default',
					),
					array(
						'id'       => 'page_footer_on',
						'type'     => 'switch',
						'title'    => esc_html__( 'Footer', 'w3construct' ),
						'on'       => esc_html__( 'Enabled', 'w3construct' ),
						'off'      => esc_html__( 'Disabled', 'w3construct' ),
						'default'  => w3construct_get_opt( 'footer_on' ),
						'required' => array(
							0 => 'page_footer_setting',
							1 => 'equals',
							2 => 'custom',
						),
					),
					array(
						'id'       => 'page_footer_style',
						'type'     => 'image_select',
						'height'   => '80',
						'title'    => esc_html__( 'Footer Template', 'w3construct' ),
						'subtitle' => esc_html__( 'Choose a template for footer.', 'w3construct' ),
						'options'  => $footer_style_options,
						'default'  => w3construct_get_opt( 'footer_style' ),
						'required' => array(
							array(
								0 => 'page_footer_setting',
								1 => 'equals',
								2 => 'custom',
							),
							array(
								0 => 'page_footer_on',
								1 => 'equals',
								2 => 1,
							),
						),
					),
				),
			)
		);

		/**
		 * Option for single pages.
		 *
		 * This section defines options for single pages. If the metabox configuration for 'page'
		 * does not exist, it will be added here.
		 */
		if ( ! $metabox->isset_args( 'dz_team' ) ) {
			$metabox->set_args(
				'dz_team',
				array(
					'opt_name'            => w3construct_get_page_opt_name(),
					'display_name'        => esc_html__( 'Team Settings', 'w3construct' ),
					'show_options_object' => false,
					'hints'               => array(
						'icon'          => 'el el-question-sign',
						'icon_position' => 'right',
						'icon_color'    => '#1085e4',
						'icon_size'     => '10',
						'tip_style'     => array(
							'color'   => '#1085e4',
							'shadow'  => true,
							'rounded' => false,
							'style'   => '',
						),
						'tip_position'  => array(
							'my' => 'top left',
							'at' => 'bottom right',
						),
						'tip_effect'    => array(
							'show' => array(
								'effect'   => 'slide',
								'duration' => '500',
								'event'    => 'mouseover',
							),
							'hide' => array(
								'effect'   => 'slide',
								'duration' => '500',
								'event'    => 'click mouseleave',
							),
						),
					),
				),
				array(
					'context'  => 'advanced',
					'priority' => 'default',
				)
			);
		}
		$metabox->add_section(
			'dz_team',
			array(
				'title'  => esc_html__( 'Featured Post', 'w3construct' ),
				'icon'   => 'fa fa-newspaper-o',
				'fields' => array(
					array(
						'id'      => 'featured_post',
						'type'    => 'checkbox',
						'title'   => esc_html__( 'Featured Post?', 'w3construct' ),
						'desc'    => esc_html__( 'Check if you want to make this post as featured post', 'w3construct' ),
						'default' => '',
					),
				),
			)
		);
		$metabox->add_section(
			'dz_team',
			array(
				'title'  => esc_html__( 'General', 'w3construct' ),
				'desc'   => esc_html__( 'Header settings for the page.', 'w3construct' ),
				'icon'   => 'fa fa-window-maximize',
				'fields' => array(
					array(
						'id'      => 'team_designation',
						'type'    => 'text',
						'title'   => esc_html__( 'Designation', 'w3construct' ),
						'desc'    => esc_html__( 'Enter client designation', 'w3construct' ),
						'default' => 'Director',
					),
				),
			)
		);

		$metabox->add_section(
			'dz_team',
			array(
				'title'  => esc_html__( 'Social Settings', 'w3construct' ),
				'desc'   => esc_html__( 'Settings for page banner.', 'w3construct' ),
				'icon'   => 'fa fa-television',
				'fields' => array(
					array(
						'id'      => 'team_social_facebook',
						'type'    => 'text',
						'url'     => true,
						'title'   => esc_html__( 'Facebook Link', 'w3construct' ),
						'desc'    => esc_html__( 'Enter Facebook Url', 'w3construct' ),
						'default' => 'https://www.facebook.com',
					),
					array(
						'id'      => 'team_social_twitter',
						'type'    => 'text',
						'url'     => true,
						'title'   => esc_html__( 'Twitter Link', 'w3construct' ),
						'desc'    => esc_html__( 'Enter Twitter Url', 'w3construct' ),
						'default' => 'https://www.twitter.com',
					),
					array(
						'id'      => 'team_social_instagram',
						'type'    => 'text',
						'url'     => true,
						'title'   => esc_html__( 'Instagram Link', 'w3construct' ),
						'desc'    => esc_html__( 'Enter Instagram Url', 'w3construct' ),
						'default' => 'https://www.instagram.com',
					),
					array(
						'id'      => 'team_social_whatsapp',
						'type'    => 'text',
						'url'     => true,
						'title'   => esc_html__( 'WhatsApp Link', 'w3construct' ),
						'desc'    => esc_html__( 'Enter WhatsApp Url', 'w3construct' ),
						'default' => 'https://www.whatsapp.com/ ',
					),
					array(
						'id'      => 'team_social_youtube',
						'type'    => 'text',
						'url'     => true,
						'title'   => esc_html__( 'Youtube Link', 'w3construct' ),
						'desc'    => esc_html__( 'Enter Youtube Url', 'w3construct' ),
						'default' => 'https://www.youtube.com',
					),

				),
			)
		);

		/**
		 * Option for single pages.
		 *
		 * This section defines options for single pages. If the metabox configuration for 'page'
		 * does not exist, it will be added here.
		 */
		if ( ! $metabox->isset_args( 'dz_testimonial' ) ) {
			$metabox->set_args(
				'dz_testimonial',
				array(
					'opt_name'            => w3construct_get_post_opt_name(),
					'display_name'        => esc_html__( 'Testimonial Settings', 'w3construct' ),
					'show_options_object' => false,
					'hints'               => array(
						'icon'          => 'el el-question-sign',
						'icon_position' => 'right',
						'icon_color'    => '#1085e4',
						'icon_size'     => '10',
						'tip_style'     => array(
							'color'   => '#1085e4',
							'shadow'  => true,
							'rounded' => false,
							'style'   => '',
						),
						'tip_position'  => array(
							'my' => 'top left',
							'at' => 'bottom right',
						),
						'tip_effect'    => array(
							'show' => array(
								'effect'   => 'slide',
								'duration' => '500',
								'event'    => 'mouseover',
							),
							'hide' => array(
								'effect'   => 'slide',
								'duration' => '500',
								'event'    => 'click mouseleave',
							),
						),
					),
				),
				array(
					'context'  => 'advanced',
					'priority' => 'default',
				)
			);
		}

		$metabox->add_section(
			'dz_testimonial',
			array(
				'title'  => esc_html__( 'Feature Post', 'w3construct' ),
				'icon'   => 'fa fa-newspaper-o',
				'fields' => array(
					array(
						'id'      => 'featured_post',
						'type'    => 'checkbox',
						'title'   => esc_html__( 'Featured Post?', 'w3construct' ),
						'desc'    => esc_html__( 'Check if you want to make this post as featured post', 'w3construct' ),
						'default' => '',
					),
				),
			)
		);

		$metabox->add_section(
			'dz_testimonial',
			array(
				'title'  => esc_html__( 'General', 'w3construct' ),
				'icon'   => 'fa fa-newspaper-o',
				'fields' => array(
					array(
						'id'      => 'testimonial_designation',
						'type'    => 'text',
						'title'   => esc_html__( 'Designation', 'w3construct' ),
						'desc'    => esc_html__( 'Enter client designation', 'w3construct' ),
						'default' => 'CEO & Founder',
					),
					array(
						'id'      => 'testimonial_rating',
						'type'    => 'select',
						'title'   => esc_html__( 'Rating', 'w3construct' ),
						'desc'    => esc_html__( 'Select Rating.', 'w3construct' ),
						'options' => array(
							'1' => '1 Star',
							'2' => '2 Star',
							'3' => '3 Star',
							'4' => '4 Star',
							'5' => '5 Star',
						),
						'default' => '5',
					),
					array(
						'id'      => 'testimonial_rating_title',
						'type'    => 'text',
						'title'   => esc_html__( 'Rating Title', 'w3construct' ),
						'desc'    => esc_html__( 'Enter Rating Title', 'w3construct' ),
						'default' => 'Awesome',
					),
				),
			)
		);

		/**
		 * Option for single pages.
		 *
		 * This section defines options for single pages. If the metabox configuration for 'page'
		 * does not exist, it will be added here.
		 */
		if ( ! $metabox->isset_args( 'dz-service' ) ) {
			$metabox->set_args(
				'dz-service',
				array(
					'opt_name'            => w3construct_get_post_opt_name(),
					'display_name'        => esc_html__( 'Service Settings', 'w3construct' ),
					'show_options_object' => false,
					'hints'               => array(
						'icon'          => 'el el-question-sign',
						'icon_position' => 'right',
						'icon_color'    => '#1085e4',
						'icon_size'     => '10',
						'tip_style'     => array(
							'color'   => '#1085e4',
							'shadow'  => true,
							'rounded' => false,
							'style'   => '',
						),
						'tip_position'  => array(
							'my' => 'top left',
							'at' => 'bottom right',
						),
						'tip_effect'    => array(
							'show' => array(
								'effect'   => 'slide',
								'duration' => '500',
								'event'    => 'mouseover',
							),
							'hide' => array(
								'effect'   => 'slide',
								'duration' => '500',
								'event'    => 'click mouseleave',
							),
						),
					),
				),
				array(
					'context'  => 'advanced',
					'priority' => 'default',
				)
			);
		}
		$metabox->add_section(
			'dz-service',
			array(
				'title'  => esc_html__( 'Feature Post', 'w3construct' ),
				'icon'   => 'fa fa-newspaper-o',
				'fields' => array(
					array(
						'id'      => 'featured_post',
						'type'    => 'checkbox',
						'title'   => esc_html__( 'Featured Post?', 'w3construct' ),
						'desc'    => esc_html__( 'Check if you want to make this post as featured post', 'w3construct' ),
						'default' => '',
					),
				),
			)
		);
		$metabox->add_section(
			'dz-service',
			array(
				'title'  => esc_html__( 'Service Page Header', 'w3construct' ),
				'desc'   => esc_html__( 'Header settings for the page.', 'w3construct' ),
				'icon'   => 'fa fa-window-maximize',
				'fields' => array(
					array(
						'id'      => 'service_page_header_setting',
						'type'    => 'button_set',
						'title'   => esc_html__( 'Header Settings', 'w3construct' ),
						'options' => array(
							'theme_default' => esc_html__( 'Theme Default', 'w3construct' ),
							'custom'        => esc_html__( 'Custom Setting', 'w3construct' ),
						),
						'default' => 'theme_default',
					),
					array(
						'id'           => 'service_page_header_style',
						'type'         => 'image_select',
						'title'        => esc_html__( 'Header Style', 'w3construct' ),
						'subtitle'     => esc_html__( 'Choose header style. White header is set as default header for this service.', 'w3construct' ),
						'options'      => $header_style_options,
						'default'      => w3construct_get_opt( 'header_style' ),
						'force_output' => true,
						'required'     => array(
							0 => 'service_page_header_setting',
							1 => 'equals',
							2 => 'custom',
						),
					),
				),
			)
		);

		$metabox->add_section(
			'dz-service',
			array(
				'title'  => esc_html__( 'General', 'w3construct' ),
				'icon'   => 'fa fa-newspaper-o',
				'fields' => array(
					array(
						'id'      => 'service_icon_type',
						'type'    => 'select',
						'title'   => esc_html__( 'Service Icon Type', 'w3construct' ),
						'options' => array(
							'fontawesome' => esc_html__( 'Font Awesome', 'w3construct' ),
							'themify'     => esc_html__( 'Themify Icon', 'w3construct' ),
							'svg'         => esc_html__( 'Svg Icon', 'w3construct' ),
						),
						'desc'    => esc_html__( 'Select Service Icon Type', 'w3construct' ),
						'default' => 'svg',
					),
					array(
						'id'       => 'service_icon_svg',
						'type'     => 'textarea',
						'title'    => esc_html__( 'Svg Icon', 'w3construct' ),
						'desc'     => esc_html__( 'Enter your svg', 'w3construct' ),
						'required' => array(
							0 => 'service_icon_type',
							1 => 'equals',
							2 => 'svg',
						),
					),
					array(
						'id'       => 'service_icon_fontawesome',
						'type'     => 'select',
						'title'    => esc_html__( 'Service Icon', 'w3construct' ),
						'options'  => w3construct_get_fontawesome_icon(),
						'desc'     => esc_html__( 'Select Service Icon', 'w3construct' ),
						'required' => array(
							0 => 'service_icon_type',
							1 => 'equals',
							2 => 'fontawesome',
						),
					),

					array(
						'id'       => 'service_icon_themify',
						'type'     => 'select',
						'title'    => esc_html__( 'Service Icon', 'w3construct' ),
						'options'  => w3construct_get_themify_icon(),
						'desc'     => esc_html__( 'Select Service Icon', 'w3construct' ),
						'required' => array(
							0 => 'service_icon_type',
							1 => 'equals',
							2 => 'themify',
						),
					),
				),
			)
		);
		$metabox->add_section(
			'dz-service',
			array(
				'title'  => esc_html__( 'Service Post Banner', 'w3construct' ),
				'desc'   => esc_html__( 'Settings for post banner.', 'w3construct' ),
				'icon'   => 'fa fa-television',
				'fields' => array(
					array(
						'id'      => 'service_post_banner_on',
						'type'    => 'switch',
						'title'   => esc_html__( 'Post Banner', 'w3construct' ),
						'on'      => esc_html__( 'Enabled', 'w3construct' ),
						'off'     => esc_html__( 'Disabled', 'w3construct' ),
						'default' => w3construct_get_opt( 'post_general_banner_on' ),
					),
					array(
						'id'       => 'service_post_banner_height',
						'type'     => 'image_select',
						'title'    => esc_html__( 'Post Banner Height', 'w3construct' ),
						'subtitle' => esc_html__( 'Choose the height for all tag page banner. Default : Big Banner', 'w3construct' ),
						'options'  => $page_banner_options,
						'height'   => '20',
						'default'  => w3construct_get_opt( 'service_post_general_banner_height' ),
						'required' => array(
							0 => 'service_post_banner_on',
							1 => 'equals',
							2 => 1,
						),
					),

					array(
						'id'       => 'service_post_banner_layout',
						'type'     => 'image_select',
						'title'    => esc_html__( 'Post Banner Layout', 'w3construct' ),
						'subtitle' => esc_html__( 'Choose the layout for all post banner. Default : Banner Layout 1', 'w3construct' ),
						'height'   => '40',
						'options'  => $page_banner_layout_options,
						'default'  => 'banner_layout_1',
						'required' => array(
							0 => 'page_banner_type',
							1 => 'equals',
							2 => 'image',
						),
					),

					array(
						'id'       => 'service_post_banner',
						'type'     => 'media',
						'url'      => true,
						'title'    => esc_html__( 'Post Banner Image', 'w3construct' ),
						'subtitle' => esc_html__( 'Enter page banner image. It will work as default banner image for all pages', 'w3construct' ),
						'desc'     => esc_html__( 'Upload banner image.', 'w3construct' ),
						'default'  => w3construct_get_opt( 'service_post_general_banner' ),
						'required' => array(
							0 => 'service_post_banner_on',
							1 => 'equals',
							2 => 1,
						),
					),
					array(
						'id'       => 'service_post_breadcrumb',
						'type'     => 'switch',
						'title'    => esc_html__( 'Breadcrumb Area', 'w3construct' ),
						'subtitle' => esc_html__( 'Click on the tab to Enable / Disable the website breadcrumb.', 'w3construct' ),
						'desc'     => esc_html__( 'This setting affects only on this page.', 'w3construct' ),
						'on'       => esc_html__( 'Enabled', 'w3construct' ),
						'off'      => esc_html__( 'Disabled', 'w3construct' ),
						'default'  => w3construct_get_opt( 'show_breadcrumb' ),
						'required' => array(
							0 => 'service_post_banner_on',
							1 => 'equals',
							2 => 1,
						),
					),
				),
			)
		);

		$metabox->add_section(
			'dz-service',
			array(
				'title'  => esc_html__( 'Service Post Footer', 'w3construct' ),
				'desc'   => esc_html__( 'Settings for footer area.', 'w3construct' ),
				'icon'   => 'fa fa-home',
				'fields' => array(
					array(
						'id'      => 'service_page_footer_setting',
						'type'    => 'button_set',
						'title'   => esc_html__( 'Service Post Footer Settings', 'w3construct' ),
						'options' => array(
							'theme_default' => esc_html__( 'Theme Default', 'w3construct' ),
							'custom'        => esc_html__( 'Custom Setting', 'w3construct' ),
						),
						'default' => 'theme_default',
					),
					array(
						'id'       => 'service_page_footer_on',
						'type'     => 'switch',
						'title'    => esc_html__( 'Footer', 'w3construct' ),
						'on'       => esc_html__( 'Enabled', 'w3construct' ),
						'off'      => esc_html__( 'Disabled', 'w3construct' ),
						'default'  => w3construct_get_opt( 'footer_on' ),
						'required' => array(
							0 => 'service_page_footer_setting',
							1 => 'equals',
							2 => 'custom',
						),
					),
					array(
						'id'       => 'service_page_footer_style',
						'type'     => 'image_select',
						'height'   => '80',
						'title'    => esc_html__( 'Footer Template', 'w3construct' ),
						'subtitle' => esc_html__( 'Choose a template for footer.', 'w3construct' ),
						'options'  => $footer_style_options,
						'default'  => w3construct_get_opt( 'footer_style' ),
						'required' => array(
							array(
								0 => 'service_page_footer_setting',
								1 => 'equals',
								2 => 'custom',
							),
							array(
								0 => 'service_page_footer_on',
								1 => 'equals',
								2 => 1,
							),
						),
					),
				),
			)
		);
		/**
		 * Option for Portfolio single posts.
		 */
		if ( ! $metabox->isset_args( 'dz-portfolio' ) ) {
			$metabox->set_args(
				'dz-portfolio',
				array(
					'opt_name'            => w3construct_get_post_opt_name(),
					'display_name'        => esc_html__( 'Portfolio Settings', 'w3construct' ),
					'show_options_object' => false,
					'hints'               => array(
						'icon'          => 'el el-question-sign',
						'icon_position' => 'right',
						'icon_color'    => '#1085e4',
						'icon_size'     => '10',
						'tip_style'     => array(
							'color'   => '#1085e4',
							'shadow'  => true,
							'rounded' => false,
							'style'   => '',
						),
						'tip_position'  => array(
							'my' => 'top left',
							'at' => 'bottom right',
						),
						'tip_effect'    => array(
							'show' => array(
								'effect'   => 'slide',
								'duration' => '500',
								'event'    => 'mouseover',
							),
							'hide' => array(
								'effect'   => 'slide',
								'duration' => '500',
								'event'    => 'click mouseleave',
							),
						),
					),
				),
				array(
					'context'  => 'advanced',
					'priority' => 'default',
				)
			);
		}

		$metabox->add_section(
			'dz-portfolio',
			array(
				'title'  => esc_html__( 'Feature Post', 'w3construct' ),
				'icon'   => 'fa fa-newspaper-o',
				'fields' => array(
					array(
						'id'      => 'featured_post',
						'type'    => 'checkbox',
						'title'   => esc_html__( 'Featured Post?', 'w3construct' ),
						'desc'    => esc_html__( 'Check if you want to make this post as a featured post', 'w3construct' ),
						'default' => '',
					),
				),
			)
		);

		$metabox->add_section(
			'dz-portfolio',
			array(
				'title'  => esc_html__( 'Portfolio Page Header', 'w3construct' ),
				'desc'   => esc_html__( 'Header settings for the page.', 'w3construct' ),
				'icon'   => 'fa fa-window-maximize',
				'fields' => array(
					array(
						'id'      => 'portfolio_page_header_setting',
						'type'    => 'button_set',
						'title'   => esc_html__( 'Header Settings', 'w3construct' ),
						'options' => array(
							'theme_default' => esc_html__( 'Theme Default', 'w3construct' ),
							'custom'        => esc_html__( 'Custom Setting', 'w3construct' ),
						),
						'default' => 'theme_default',
					),
					array(
						'id'           => 'portfolio_page_header_style',
						'type'         => 'image_select',
						'title'        => esc_html__( 'Header Style', 'w3construct' ),
						'subtitle'     => esc_html__( 'Choose header style. White header is set as the default header for this service.', 'w3construct' ),
						'options'      => $header_style_options,
						'default'      => w3construct_get_opt( 'header_style' ),
						'force_output' => true,
						'required'     => array(
							0 => 'portfolio_page_header_setting',
							1 => 'equals',
							2 => 'custom',
						),
					),
				),
			)
		);

		$metabox->add_section(
			'dz-portfolio',
			array(
				'title'  => esc_html__( 'Portfolio Post Banner', 'w3construct' ),
				'desc'   => esc_html__( 'Settings for post banner.', 'w3construct' ),
				'icon'   => 'fa fa-television',
				'fields' => array(
					array(
						'id'      => 'portfolio_post_banner_on',
						'type'    => 'switch',
						'title'   => esc_html__( 'Post Banner', 'w3construct' ),
						'on'      => esc_html__( 'Enabled', 'w3construct' ),
						'off'     => esc_html__( 'Disabled', 'w3construct' ),
						'default' => w3construct_get_opt( 'post_general_banner_on' ),
					),
					array(
						'id'       => 'portfolio_post_banner_height',
						'type'     => 'image_select',
						'title'    => esc_html__( 'Post Banner Height', 'w3construct' ),
						'subtitle' => esc_html__( 'Choose the height for all tag page banners. Default: Big Banner', 'w3construct' ),
						'options'  => $page_banner_options,
						'height'   => '20',
						'default'  => w3construct_get_opt( 'portfolio_post_general_banner_height' ),
						'required' => array(
							0 => 'portfolio_post_banner_on',
							1 => 'equals',
							2 => 1,
						),
					),
					array(
						'id'       => 'portfolio_post_banner_layout',
						'type'     => 'image_select',
						'title'    => esc_html__( 'Post Banner Layout', 'w3construct' ),
						'subtitle' => esc_html__( 'Choose the layout for all post banners. Default: Banner Layout 1', 'w3construct' ),
						'height'   => '40',
						'options'  => $page_banner_layout_options,
						'default'  => 'banner_layout_1',
						'required' => array(
							0 => 'page_banner_type',
							1 => 'equals',
							2 => 'image',
						),
					),
					array(
						'id'       => 'portfolio_post_banner',
						'type'     => 'media',
						'url'      => true,
						'title'    => esc_html__( 'Post Banner Image', 'w3construct' ),
						'subtitle' => esc_html__( 'Enter page banner image. It will work as the default banner image for all pages', 'w3construct' ),
						'desc'     => esc_html__( 'Upload banner image.', 'w3construct' ),
						'default'  => w3construct_get_opt( 'portfolio_post_general_banner' ),
						'required' => array(
							0 => 'portfolio_post_banner_on',
							1 => 'equals',
							2 => 1,
						),
					),
					array(
						'id'       => 'portfolio_post_breadcrumb',
						'type'     => 'switch',
						'title'    => esc_html__( 'Breadcrumb Area', 'w3construct' ),
						'subtitle' => esc_html__( 'Click on the tab to Enable / Disable the website breadcrumb.', 'w3construct' ),
						'desc'     => esc_html__( 'This setting affects only this page.', 'w3construct' ),
						'on'       => esc_html__( 'Enabled', 'w3construct' ),
						'off'      => esc_html__( 'Disabled', 'w3construct' ),
						'default'  => w3construct_get_opt( 'show_breadcrumb' ),
						'required' => array(
							0 => 'portfolio_post_banner_on',
							1 => 'equals',
							2 => 1,
						),
					),
				),
			)
		);

		$metabox->add_section(
			'dz-portfolio',
			array(
				'title'  => esc_html__( 'Portfolio Post Footer', 'w3construct' ),
				'desc'   => esc_html__( 'Settings for the footer area.', 'w3construct' ),
				'icon'   => 'fa fa-home',
				'fields' => array(
					array(
						'id'      => 'portfolio_page_footer_setting',
						'type'    => 'button_set',
						'title'   => esc_html__( 'Portfolio Post Footer Settings', 'w3construct' ),
						'options' => array(
							'theme_default' => esc_html__( 'Theme Default', 'w3construct' ),
							'custom'        => esc_html__( 'Custom Setting', 'w3construct' ),
						),
						'default' => 'theme_default',
					),
					array(
						'id'       => 'portfolio_page_footer_on',
						'type'     => 'switch',
						'title'    => esc_html__( 'Footer', 'w3construct' ),
						'on'       => esc_html__( 'Enabled', 'w3construct' ),
						'off'      => esc_html__( 'Disabled', 'w3construct' ),
						'default'  => w3construct_get_opt( 'footer_on' ),
						'required' => array(
							0 => 'portfolio_page_footer_setting',
							1 => 'equals',
							2 => 'custom',
						),
					),
					array(
						'id'       => 'portfolio_page_footer_style',
						'type'     => 'image_select',
						'height'   => '80',
						'title'    => esc_html__( 'Footer Template', 'w3construct' ),
						'subtitle' => esc_html__( 'Choose a template for the footer.', 'w3construct' ),
						'options'  => $footer_style_options,
						'default'  => w3construct_get_opt( 'footer_style' ),
						'required' => array(
							array(
								0 => 'portfolio_page_footer_setting',
								1 => 'equals',
								2 => 'custom',
							),
							array(
								0 => 'portfolio_page_footer_on',
								1 => 'equals',
								2 => 1,
							),
						),
					),
				),
			)
		);
	}
}
add_action( 'dsx_post_metabox_register', 'w3construct_page_options_register' );
