<?php
/**
 * Base Template
 *
 * This class serves as the base template for the W3Construct theme.
 *
 * @package     W3Construct
 * @subpackage  Base_Template
 * @author      W3ITEXPERTS
 * @copyright   Copyright (c) 2023, W3Construct
 * @link        http://w3construct-wordpress.wprdx.com/demo/
 * @since       W3Construct 1.0.1
 */

if ( ! class_exists( 'W3Construct_DZ_Base' ) ) {
	/**
	 * W3Construct_DZ_Base
	 */
	class W3Construct_DZ_Base {
		/**
		 * Default path for the template.
		 *
		 * @var string
		 */
		public $path = '';

		/**
		 * Default URL for the template.
		 *
		 * @var string
		 */
		public $url = '';

		/**
		 * Includes directory path.
		 *
		 * @var string
		 */
		public $inc = '';

		/**
		 * Includes directory URL.
		 *
		 * @var string
		 */
		public $inc_url = '';

		/**
		 * Page settings.
		 *
		 * @var mixed
		 */
		public $page_settings;

		/**
		 * Extras.
		 *
		 * @var mixed
		 */
		public $extras;

		/**
		 * Shortcodes.
		 *
		 * @var mixed
		 */
		public $shortcodes;

		/**
		 * Constructor method.
		 *
		 * Sets the default paths and URLs.
		 */
		public function __construct() {
			/* Define default path */
			$this->path    = get_template_directory() . '/';
			$this->url     = get_template_directory_uri() . '/';
			$this->inc     = $this->path . 'includes/';
			$this->inc_url = $this->url . 'includes/';
		}

		/**
		 * Set HTML attributes from an array.
		 *
		 * @param array $attr An array of attributes.
		 * @return string HTML attributes.
		 */
		public function set_attrib( $attr = array() ) {
			$res = ' ';
			foreach ( $attr as $k => $v ) {
				$res .= $k . '="' . $v . '" ';
			}
			return $res;
		}

		/**
		 * Get theme options.
		 *
		 * @param string $key The option key.
		 * @return mixed The theme options.
		 */
		public function option( $key = '' ) {
			$theme_options = get_option( 'w3construct_theme_options' );
			if ( $key ) {
				$theme_options = get_option( 'w3construct_' . $key );
			}

			return $theme_options;
		}

		/**
		 * Get URL and path to includes.
		 *
		 * @param string $path The path to the file.
		 * @param bool   $url Whether to return URL or path.
		 * @return string URL or path to the file.
		 */
		public function includes( $path = '', $url = false ) {
			$child = get_stylesheet_directory() . '/';
			if ( file_exists( $child . $path ) ) {
				if ( $url ) {
					return get_stylesheet_directory_uri() . '/' . $path;
				} else {
					return $child . $path;
				}
			}
			if ( $url ) {
				return get_template_directory_uri() . '/' . $path;
			} else {
				return $this->path . $path;
			}
		}

		/**
		 * Get meta value.
		 *
		 * @param string $key The meta key.
		 * @param int    $id The post ID.
		 * @return mixed The meta value.
		 */
		public function get_meta( $key = '', $id = '' ) {
			global $post;
			if ( ! is_object( $post ) ) {
				return false;
			}

			$id  = ( $id ) ? $id : w3construct_set( $post, 'ID' );
			$key = ( $key ) ? $key : '';

			$meta = get_post_meta( intval( $id ), $key, true );
			return ( $meta ) ? $meta : false;
		}

		/**
		 * Set meta key for post type.
		 *
		 * @param string $post_type_w3 The post type name.
		 * @return string The meta key.
		 */
		public function set_meta_key( $post_type_w3 ) {
			if ( ! $post_type_w3 ) {
				return;
			}

			return '_bunch_' . $post_type_w3 . '_settings';
		}

		/**
		 * Get term meta value.
		 *
		 * @param string $key The meta key.
		 * @return mixed The term meta value.
		 */
		public function get_term_meta( $key = '' ) {
			$object = get_queried_object();

			$key  = ( $key ) ? $key . $object->term_id : '_bunch_' . $object->taxonomy . '_settings' . $object->term_id;
			$meta = get_option( $key );

			return ( $meta ) ? $meta : false;
		}

		/**
		 * Set term key for post type.
		 *
		 * @param string $post_type_w3 The post type name.
		 * @return string The term key.
		 */
		public function set_term_key( $post_type_w3 ) {
			if ( ! $post_type_w3 ) {
				return;
			}

			return '_bunch_' . $post_type_w3 . '_settings';
		}

		/**
		 * Get page template.
		 *
		 * @param string $tpl The template name.
		 * @return mixed The page template.
		 */
		public function page_template( $tpl ) {
			$page = get_pages(
				array(
					'meta_key'   => '_wp_page_template',
					'meta_value' => $tpl,
				)
			);
			if ( $page ) {
				return current( (array) $page );
			} else {
				return false;
			}
		}

		/**
		 * Set current first name and last name.
		 *
		 * @param int $current The current index.
		 * @param int $cols The number of columns.
		 * @return string The name for the current element.
		 */
		public function first_last( $current, $cols ) {
			$current++;
			if ( 1 === $current ) {
				return ' first';
			} elseif ( ( ( $current ) % $cols ) === 0 ) {
				return ' last';
			} elseif ( ( ( $current - 1 ) % $cols ) === 0 ) {
				return ' first';
			}
		}
	}
}
$GLOBALS['_dz_base'] = new W3Construct_DZ_Base();
