<?php
/**
 * Custom functions that act independently of the theme templates
 *
 * Eventually, some of the functionality here could be replaced by core features
 *
 * @package tweaker5
 */

/**
 * Get our wp_nav_menu() fallback, wp_page_menu(), to show a home link.
 *
 * @param array $args Configuration arguments.
 * @return array
 */
function tweaker5_page_menu_args( $args ) {
	$args['show_home'] = true;
	return $args;
}
add_filter( 'wp_page_menu_args', 'tweaker5_page_menu_args' );

/**
 * Adds custom classes to the array of body classes.
 *
 * @param array $classes Classes for the body element.
 * @return array
 */
function tweaker5_body_classes( $classes ) {
	// Adds a class of group-blog to blogs with more than 1 published author.
	if ( is_multi_author() ) {
		$classes[] = 'group-blog';
	}

	return $classes;
}
add_filter( 'body_class', 'tweaker5_body_classes' );

/**
 * Filters wp_title to print a neat <title> tag based on what is being viewed.
 *
 * @param string $title Default title text for current view.
 * @param string $sep Optional separator.
 * @return string The filtered title.
 */
function tweaker5_wp_title( $title, $sep ) {
	if ( is_feed() ) {
		return $title;
	}
	
	global $page, $paged;

	// Add the blog name
	$title .= get_bloginfo( 'name', 'display' );

	// Add the blog description for the home/front page.
	$site_description = get_bloginfo( 'description', 'display' );
	if ( $site_description && ( is_home() || is_front_page() ) ) {
		$title .= " $sep $site_description";
	}

	// Add a page number if necessary:
	if ( $paged >= 2 || $page >= 2 ) {
		$title .= " $sep " . sprintf( __( 'Page %s', 'tweaker5' ), max( $paged, $page ) );
	}

	return $title;
}
add_filter( 'wp_title', 'tweaker5_wp_title', 10, 2 );

/**
 * Sets the authordata global when viewing an author archive.
 *
 * This provides backwards compatibility with
 * http://core.trac.wordpress.org/changeset/25574
 *
 * It removes the need to call the_post() and rewind_posts() in an author
 * template to print information about the author.
 *
 * @global WP_Query $wp_query WordPress Query object.
 * @return void
 */
function tweaker5_setup_author() {
	global $wp_query;

	if ( $wp_query->is_author() && isset( $wp_query->post ) ) {
		$GLOBALS['authordata'] = get_userdata( $wp_query->post->post_author );
	}
}
add_action( 'wp', 'tweaker5_setup_author' );

// If more than one page exists, return TRUE.
// Used to show pagination only when required
function tweaker5_show_posts_nav() {
    global $wp_query;
    return ($wp_query->max_num_pages > 1);
}

// Paginate Function.
function tweaker5_paginate() {
	global $wp_query, $wp_rewrite;
	$wp_query->query_vars['paged'] > 1 ? $current = $wp_query->query_vars['paged'] : $current = 1;
	
	$pagination = array(
		'base' => @add_query_arg('page','%#%'),
		'format' => '',
		'total' => $wp_query->max_num_pages,
		'current' => $current,
		'show_all' => true,
		'type' => 'list',
		'next_text' => 'Next &raquo;',
		'prev_text' => '&laquo; Previous'
		);
	
	if( $wp_rewrite->using_permalinks() )
		$pagination['base'] = user_trailingslashit( trailingslashit( remove_query_arg( 's', get_pagenum_link( 1 ) ) ) . 'page/%#%/', 'paged' );
	
	if( !empty($wp_query->query_vars['s']) )
		$pagination['add_args'] = array( 's' => get_query_var( 's' ) );
	
	echo paginate_links( $pagination );
}
// Get first image in a post, used on image post formats
function tweaker5_first_image() {
  global $post, $posts;
  $first_img = '';
  ob_start();
  ob_end_clean();
  $output = preg_match_all('/<img.+src=[\'"]([^\'"]+)[\'"].*>/i', $post->post_content, $matches);
  {
$first_img = $matches[1][0];
}
  return $first_img;
}
//Function to display copyright notice in footer
function tweaker5_copyright_notice() {
	$notice = 'Copyright &copy; ';
	$firstdate = intval( get_theme_mod( 'tweaker5_first_date_setting' ) );
	if ( $firstdate < date('Y') && $firstdate != 0 ){
		$notice .= $firstdate;
		$notice .= ' - ';
	}
	$notice .= date('Y');
	
	$owner = esc_attr( get_theme_mod( 'tweaker5_copyright_owner_setting' ) );
	if ($owner != '') {
		$notice .= ' ' . $owner;
	} else {
		$notice .= ' ' . get_bloginfo();
	}
	
	echo $notice;
}
// Function to add blockquote to format quote posts if they are not there.
add_filter( 'the_content', 'tweaker5_quote_content' );
function tweaker5_quote_content( $content ) {
	if ( has_post_format( 'quote' ) ) {
		preg_match( '/<blockquote.*?>/', $content, $matches );
		if ( empty( $matches ) )
			$content = "<blockquote>{$content}</blockquote>";
	}
	return $content;
}
//Function to add post format images to posts
function tweaker5_format_images($format) {
	if ( is_sticky() ){
      	$images = '<h1><a href="' . get_permalink() . '" title=""><span class="sticky-post"></span><span class="'. $format . '"></span>' . get_the_title() . '</a></h1>';
   	} else {
      	$images = '<h1><a href="' . get_permalink() . '" title=""><span class="'. $format . '"></span>' . get_the_title() . '</a></h1>';
    }
	echo $images;
}
//Create a shortcode for buttons in posts and pages
function tweaker5_button($atts, $content = null) {
   extract(shortcode_atts(array('url' => '#'), $atts));
   return '<a class="button" href="'.$url.'"><span>' . do_shortcode($content) . '</span></a>';
}
add_shortcode('button', 'tweaker5_button');
// Clean up social custom menu
function tweaker5_social_custom_menus() {
	$menu_name = 'social';
	if (($locations = get_nav_menu_locations()) && isset($locations[$menu_name])) {
		$menu = wp_get_nav_menu_object($locations[$menu_name]);
		$menu_items = wp_get_nav_menu_items($menu->term_id);

		$menu_list = '<div class="top-menu">' ."\n";
		$menu_list .= "\t\t\t\t". '<ul>' ."\n";
		foreach ((array) $menu_items as $key => $menu_item) {
			$title = $menu_item->title;
			$url = $menu_item->url;
			$menu_list .= "\t\t\t\t\t". '<li><a href="'. $url .'"><span class="screen-reader-text">'. $title .'</span>/a></li>' ."\n";
		}
		$menu_list .= "\t\t\t\t". '</ul>' ."\n";
		$menu_list .= "\t\t\t". '</div>' ."\n";
	} else {
		// $menu_list = '<!-- no list defined -->';
	}
	echo $menu_list;
}