<?php
/**
 * WooCommerce Compatibility File
 *
 * @link https://woocommerce.com/
 *
 * @package Suit Blog
 */

/**
 * WooCommerce setup function.
 *
 * @link https://docs.woocommerce.com/document/third-party-custom-theme-compatibility/
 * @link https://github.com/woocommerce/woocommerce/wiki/Enabling-product-gallery-features-(zoom,-swipe,-lightbox)
 * @link https://github.com/woocommerce/woocommerce/wiki/Declaring-WooCommerce-support-in-themes
 *
 * @return void
 */
function suit_blog_woocommerce_setup() {
	add_theme_support(
		'woocommerce',
		array(
			'thumbnail_image_width' => 150,
			'single_image_width'    => 300,
			'product_grid'          => array(
				'default_rows'    => 3,
				'min_rows'        => 1,
				'default_columns' => 4,
				'min_columns'     => 1,
				'max_columns'     => 6,
			),
		)
	);
	add_theme_support( 'wc-product-gallery-zoom' );
	add_theme_support( 'wc-product-gallery-lightbox' );
	add_theme_support( 'wc-product-gallery-slider' );
}
add_action( 'after_setup_theme', 'suit_blog_woocommerce_setup' );

/**
 * WooCommerce specific scripts & stylesheets.
 *
 * @return void
 */
function suit_blog_woocommerce_scripts() {
	wp_enqueue_style( 'suit-blog-woocommerce-style', Suit_Blog_THEME_URI . 'woocommerce.css', array(), Suit_Blog_THEME_VERSION );

	$font_path   = WC()->plugin_url() . '/assets/fonts/';
	$inline_font = '@font-face {
			font-family: "star";
			src: url("' . $font_path . 'star.eot");
			src: url("' . $font_path . 'star.eot?#iefix") format("embedded-opentype"),
				url("' . $font_path . 'star.woff") format("woff"),
				url("' . $font_path . 'star.ttf") format("truetype"),
				url("' . $font_path . 'star.svg#star") format("svg");
			font-weight: normal;
			font-style: normal;
		}';

	wp_add_inline_style( 'suit-blog-woocommerce-style', $inline_font );
}
add_action( 'wp_enqueue_scripts', 'suit_blog_woocommerce_scripts' );

/**
 * Disable the default WooCommerce stylesheet.
 *
 * Removing the default WooCommerce stylesheet and enqueing your own will
 * protect you during WooCommerce core updates.
 *
 * @link https://docs.woocommerce.com/document/disable-the-default-stylesheet/
 */
add_filter( 'woocommerce_enqueue_styles', '__return_empty_array' );

/**
 * Add 'woocommerce-active' class to the body tag.
 *
 * @param  array $classes CSS classes applied to the body tag.
 * @return array $classes modified to include 'woocommerce-active' class.
 */
function suit_blog_woocommerce_active_body_class( $classes ) {
	$classes[] = 'woocommerce-active';

	return $classes;
}
add_filter( 'body_class', 'suit_blog_woocommerce_active_body_class' );

/**
 * Related Products Args.
 *
 * @param array $args related products args.
 * @return array $args related products args.
 */
function suit_blog_woocommerce_related_products_args( $args ) {
	$defaults = array(
		'posts_per_page' => 3,
		'columns'        => 3,
	);

	$args = wp_parse_args( $defaults, $args );

	return $args;
}
add_filter( 'woocommerce_output_related_products_args', 'suit_blog_woocommerce_related_products_args' );

/**
 * Product Image Size
 *
 * @param string $size
 * @return string
 */
add_filter( 'single_product_archive_thumbnail_size', function( $size ) {
	return 'full';
} );

/**
 * Remove default WooCommerce wrapper.
 */
remove_action( 'woocommerce_before_main_content', 'woocommerce_output_content_wrapper', 10 );
remove_action( 'woocommerce_after_main_content', 'woocommerce_output_content_wrapper_end', 10 );

if ( ! function_exists( 'suit_blog_woocommerce_wrapper_before' ) ) {
	/**
	 * Before Content.
	 *
	 * Wraps all WooCommerce content in wrappers which match the theme markup.
	 *
	 * @return void
	 */
	function suit_blog_woocommerce_wrapper_before() {
		suit_blog_site_primary_content_start();
		?>
			<main id="primary"<?php suit_blog_primary_class(); ?>>
		<?php
	}
}
add_action( 'woocommerce_before_main_content', 'suit_blog_woocommerce_wrapper_before' );

if ( ! function_exists( 'suit_blog_woocommerce_wrapper_after' ) ) {
	/**
	 * After Content.
	 *
	 * Closes the wrapping divs.
	 *
	 * @return void
	 */
	function suit_blog_woocommerce_wrapper_after() {
		?>
			</main><!-- #main -->
			<?php suit_blog_site_content_sidebar(); ?>
			</div><!-- .row -->
			</div><!-- .container -->
			</section><!-- .section-post-container -->
		<?php
	}
}
add_action( 'woocommerce_after_main_content', 'suit_blog_woocommerce_wrapper_after' );

if ( ! function_exists( 'suit_blog_woocommerce_cart_link_fragment' ) ) {
	/**
	 * Cart Fragments.
	 *
	 * Ensure cart contents update when products are added to the cart via AJAX.
	 *
	 * @param array $fragments Fragments to refresh via AJAX.
	 * @return array Fragments to refresh via AJAX.
	 */
	function suit_blog_woocommerce_cart_link_fragment( $fragments ) {
		ob_start();
		?>
		<span class="cart-value"> <?php echo wp_kses_data( WC()->cart->get_cart_contents_count() ); ?></span>
		<?php
		$fragments['span.cart-value'] = ob_get_clean();
		return $fragments;
	}
}
add_filter( 'woocommerce_add_to_cart_fragments', 'suit_blog_woocommerce_cart_link_fragment' );

/**
 * Change number or products per row to 3
 */
add_filter('loop_shop_columns', 'suit_blog_wc_loop_columns', 999);
if (!function_exists('suit_blog_wc_loop_columns')) {
	function suit_blog_wc_loop_columns() {
		return 3; // 3 products per row
	}
}

add_filter('loop_shop_per_page', 'suit_blog_wc_no_of_products', 999);
if (!function_exists('suit_blog_wc_no_of_products')) {
	function suit_blog_wc_no_of_products() {
		return 9; // * products per row
	}
}

// Add extra option on header elements
add_filter( 'suit_blog_header_elements', 'suit_blog_wc_cart_option' );
function suit_blog_wc_cart_option( $options ){
	$value 			= $options;
	$value['cart'] 	= esc_html__( 'Woo Cart', 'suit-blog' );
	return $value;
}