<?php
/**
 * Functions which enhance the theme by hooking into WordPress
 *
 * @package stag-blocks
 */

/**
 * Adds custom classes to the array of body classes.
 *
 * @param array $classes Classes for the body element.
 * @return array
 */
function stagblocks_body_classes( $classes ) {
	// Adds a class of hfeed to non-singular pages.
	if ( ! is_singular() ) {
		$classes[] = 'hfeed';
	}

	if ( is_active_sidebar( 'sidebar-1' ) ) {
		global $template;
		if ( 'front-page.php' !== basename( $template ) ) {
			$classes[] = 'has-sidebar';
		}
	}

	$classes[] = 'search-off';

	return $classes;
}
add_filter( 'body_class', 'stagblocks_body_classes' );

/**
 * Add a pingback url auto-discovery header for singularly identifiable articles.
 */
function stagblocks_pingback_header() {
	if ( is_singular() && pings_open() ) {
		echo '<link rel="pingback" href="', esc_url( get_bloginfo( 'pingback_url' ) ), '">';
	}
}
add_action( 'wp_head', 'stagblocks_pingback_header' );

/**
 * Adds async/defer attributes to enqueued / registered scripts.
 *
 * If #12009 lands in WordPress, this function can no-op since it would be handled in core.
 *
 * @link https://core.trac.wordpress.org/ticket/12009
 * @param string $tag    The script tag.
 * @param string $handle The script handle.
 * @return array
 */
function stagblocks_filter_script_loader_tag( $tag, $handle ) {

	foreach ( array( 'async', 'defer' ) as $attr ) {
		if ( ! wp_scripts()->get_data( $handle, $attr ) ) {
			continue;
		}

		// Prevent adding attribute when already added in #12009.
		if ( ! preg_match( ":\s$attr(=|>|\s):", $tag ) ) {
			$tag = preg_replace( ':(?=></script>):', " $attr", $tag, 1 );
		}

		// Only allow async or defer, not both.
		break;
	}

	return $tag;
}

add_filter( 'script_loader_tag', 'stagblocks_filter_script_loader_tag', 10, 2 );

/**
 * Generate preload markup for stylesheets.
 *
 * @param object $wp_styles Registered styles.
 * @param string $handle The style handle.
 */
function stagblocks_get_preload_stylesheet_uri( $wp_styles, $handle ) {
	$preload_uri = $wp_styles->registered[ $handle ]->src . '?ver=' . $wp_styles->registered[ $handle ]->ver;
	return $preload_uri;
}

/**
 * Adds preload for in-body stylesheets depending on what templates are being used.
 * Disabled when AMP is active as AMP injects the stylesheets inline.
 *
 * @link https://developer.mozilla.org/en-US/docs/Web/HTML/Preloading_content
 */
function stagblocks_add_body_style() {

	if ( ! stagblocks_is_amp() ) {

		// Get registered styles.
		$wp_styles = wp_styles();

		$preloads = array();

		// Preload content.css.
		$preloads['stag-blocks-content'] = stagblocks_get_preload_stylesheet_uri( $wp_styles, 'stag-blocks-content' );

		// Preload sidebar.css and widget.css.
		if ( is_active_sidebar( 'sidebar-1' ) ) {
			$preloads['stag-blocks-sidebar'] = stagblocks_get_preload_stylesheet_uri( $wp_styles, 'stag-blocks-sidebar' );
			$preloads['stag-blocks-widgets'] = stagblocks_get_preload_stylesheet_uri( $wp_styles, 'stag-blocks-widgets' );
		}

		// Preload comments.css.
		if ( ! post_password_required() && is_singular() && ( comments_open() || get_comments_number() ) ) {
			$preloads['stag-blocks-comments'] = stagblocks_get_preload_stylesheet_uri( $wp_styles, 'stag-blocks-comments' );
		}

		// Preload front-page.css.
		global $template;
		if ( 'front-page.php' === basename( $template ) ) {
			$preloads['stag-blocks-front-page'] = stagblocks_get_preload_stylesheet_uri( $wp_styles, 'stag-blocks-front-page' );
		}

		// Output the preload markup in <head>.
		foreach ( $preloads as $handle => $src ) {
			echo '<link rel="preload" id="' . esc_attr( $handle ) . '-preload" href="' . esc_url( $src ) . '" as="style" />';
			echo "\n";
		}
	}

}
add_action( 'wp_head', 'stagblocks_add_body_style' );

/**
 * Add dropdown symbol to nav menu items with children.
 *
 * Adds the dropdown markup after the menu link element,
 * before the submenu.
 *
 * Javascript converts the symbol to a toggle button.
 *
 * @TODO:
 * - This doesn't work for the page menu because it
 *   doesn't have a similar filter. So the dropdown symbol
 *   is only being added for page menus if JS is enabled.
 *   Create a ticket to add to core?
 *
 * @param string   $item_output The menu item's starting HTML output.
 * @param WP_Post  $item        Menu item data object.
 * @param int      $depth       Depth of menu item. Used for padding.
 * @param stdClass $args        An object of wp_nav_menu() arguments.
 * @return string Modified nav menu HTML.
 */
function stagblocks_add_primary_menu_dropdown_symbol( $item_output, $item, $depth, $args ) {

	// Only for our primary menu location.
	if ( empty( $args->theme_location ) || 'primary' != $args->theme_location ) {
		return $item_output;
	}

	// Add the dropdown for items that have children.
	if ( ! empty( $item->classes ) && in_array( 'menu-item-has-children', $item->classes ) ) {
		return $item_output . '<span class="dropdown"><i class="dropdown-symbol"></i></span>';
	}

	return $item_output;
}
add_filter( 'walker_nav_menu_start_el', 'stagblocks_add_primary_menu_dropdown_symbol', 10, 4 );

/**
 * Filters the HTML attributes applied to a menu item's anchor element.
 *
 * Checks if the menu item is the current menu
 * item and adds the aria "current" attribute.
 *
 * @param array   $atts   The HTML attributes applied to the menu item's `<a>` element.
 * @param WP_Post $item  The current menu item.
 * @return array Modified HTML attributes
 */
function stagblocks_add_nav_menu_aria_current( $atts, $item ) {
	/*
	 * First, check if "current" is set,
	 * which means the item is a nav menu item.
	 *
	 * Otherwise, it's a post item so check
	 * if the item is the current post.
	 */
	if ( isset( $item->current ) ) {
		if ( $item->current ) {
			$atts['aria-current'] = 'page';
		}
	} elseif ( ! empty( $item->ID ) ) {
		global $post;
		if ( ! empty( $post->ID ) && $post->ID == $item->ID ) {
			$atts['aria-current'] = 'page';
		}
	}

	return $atts;
}
add_filter( 'nav_menu_link_attributes', 'stagblocks_add_nav_menu_aria_current', 10, 2 );
add_filter( 'page_menu_link_attributes', 'stagblocks_add_nav_menu_aria_current', 10, 2 );

/**
 * Replaces the excerpt "[..]" text by a custom text.
 *
 * @param string|mixed $more Read more text.
 * @return string|mixed Modified more text.
 */
function stagblocks_excerpt_more( $more ) {
	global $post;

	if ( is_admin() ) {
		return $more;
	}

	return '<a class="read-more-link" href="' . esc_url( get_permalink() ) . '">' . esc_html__( 'Continue Reading', 'stag-blocks' ) . '</a>';
}
add_filter( 'excerpt_more', 'stagblocks_excerpt_more' );

/**
 * Replaces the content "Read more" text by a cusom text.
 */
function stagblocks_modify_read_more_link() {
	if ( is_admin() ) {
		return;
	}

	return '<a class="read-more-link" href="' . esc_url( get_permalink() ) . '">' . esc_html__( 'Continue Reading', 'stag-blocks' ) . '</a>';
}
add_filter( 'the_content_more_link', 'stagblocks_modify_read_more_link' );

/**
 * Check if a post is a custom post type.
 *
 * @param  mixed $post Post object or ID.
 * @return boolean
 */
function is_custom_post_type( $post = null ) {
	$all_custom_post_types = get_post_types( array( '_builtin' => false ) );

	// there are no custom post types.
	if ( empty( $all_custom_post_types ) ) {
		return false;
	}

	$custom_types      = array_keys( $all_custom_post_types );
	$current_post_type = get_post_type( $post );

	// could not detect current type.
	if ( ! $current_post_type ) {
		return false;
	}

	return in_array( $current_post_type, $custom_types );
}
/**
 * Add Sticky badge & styles if is sticky.
 */
function stagblocks_sticky_badge() {
	if ( is_sticky() ) {
		echo '<div class="sticky-badge"><span class="sticky-text">' . esc_html__( 'Featured', 'stag-blocks' ) . '</span></div>';
	}
}
