<?php
/**
 * Stag Blocks functions and definitions
 *
 * @link https://developer.wordpress.org/themes/basics/theme-functions/
 *
 * @package stag-blocks
 */

/**
 * Sets up theme defaults and registers support for various WordPress features.
 *
 * Note that this function is hooked into the after_setup_theme hook, which
 * runs before the init hook. The init hook is too late for some features, such
 * as indicating support for post thumbnails.
 */
function stagblocks_setup() {
	/*
	* Make theme available for translation.
	* Translations can be filed in the /languages/ directory.
	* If you're building a theme based on stag-blocks, use a find and replace
	* to change 'stag-blocks' to the name of your theme in all the template files.
	*/
	load_theme_textdomain( 'stag-blocks', get_template_directory() . '/languages' );

	// Add default posts and comments RSS feed links to head.
	add_theme_support( 'automatic-feed-links' );

	/*
	* Let WordPress manage the document title.
	* By adding theme support, we declare that this theme does not use a
	* hard-coded <title> tag in the document head, and expect WordPress to
	* provide it for us.
	*/
	add_theme_support( 'title-tag' );

	/*
	* Enable support for Post Thumbnails on posts and pages.
	*
	* @link https://developer.wordpress.org/themes/functionality/featured-images-post-thumbnails/
	*/
	add_theme_support( 'post-thumbnails' );

	// This theme uses wp_nav_menu() in one location.
	register_nav_menus(
		array(
			'primary' => esc_html__( 'Primary', 'stag-blocks' ),
		)
	);

	/*
	* Switch default core markup for search form, comment form, and comments
	* to output valid HTML5.
	*/
	add_theme_support(
		'html5', array(
			'search-form',
			'comment-form',
			'comment-list',
			'gallery',
			'caption',
		)
	);

	// Set up the WordPress core custom background feature.
	add_theme_support(
		'custom-background', apply_filters(
			'stagblocks_custom_background_args', array(
				'default-color' => 'ffffff',
				'default-image' => '',
			)
		)
	);

	// Add theme support for selective refresh for widgets.
	add_theme_support( 'customize-selective-refresh-widgets' );

	/**
	 * Add support for core custom logo.
	 *
	 * @link https://codex.wordpress.org/Theme_Logo
	 */
	add_theme_support(
		'custom-logo', array(
			'height'      => 250,
			'width'       => 100,
			'flex-width'  => true,
			'flex-height' => true,
		)
	);

	/**
	 * Add support for wide aligments.
	 *
	 * @link https://wordpress.org/gutenberg/handbook/extensibility/theme-support/
	 */
	add_theme_support( 'align-wide' );

	/**
	 * Add support for block color palettes.
	 *
	 * @link https://wordpress.org/gutenberg/handbook/extensibility/theme-support/
	 */
	add_theme_support(
		'editor-color-palette', array(
			array(
				'name'  => __( 'Dusty orange', 'stag-blocks' ),
				'slug'  => 'dusty-orange',
				'color' => '#ed8f5b',
			),
			array(
				'name'  => __( 'Dusty red', 'stag-blocks' ),
				'slug'  => 'dusty-red',
				'color' => '#e36d60',
			),
			array(
				'name'  => __( 'Dusty wine', 'stag-blocks' ),
				'slug'  => 'dusty-wine',
				'color' => '#9c4368',
			),
			array(
				'name'  => __( 'Dark sunset', 'stag-blocks' ),
				'slug'  => 'dark-sunset',
				'color' => '#33223b',
			),
			array(
				'name'  => __( 'Almost black', 'stag-blocks' ),
				'slug'  => 'almost-black',
				'color' => '#0a1c28',
			),
			array(
				'name'  => __( 'Dusty water', 'stag-blocks' ),
				'slug'  => 'dusty-water',
				'color' => '#41848f',
			),
			array(
				'name'  => __( 'Dusty sky', 'stag-blocks' ),
				'slug'  => 'dusty-sky',
				'color' => '#72a7a3',
			),
			array(
				'name'  => __( 'Dusty daylight', 'stag-blocks' ),
				'slug'  => 'dusty-daylight',
				'color' => '#97c0b7',
			),
			array(
				'name'  => __( 'Dusty sun', 'stag-blocks' ),
				'slug'  => 'dusty-sun',
				'color' => '#eee9d1',
			),
		)
	);

	/**
	 * Optional: Disable custom colors in block color palettes.
	 *
	 * @link https://wordpress.org/gutenberg/handbook/extensibility/theme-support/
	 *
	 * add_theme_support( 'disable-custom-colors' );
	 */

	/**
	 * Optional: Add AMP support.
	 *
	 * Add built-in support for the AMP plugin and specific AMP features.
	 * Control how the plugin, when activated, impacts the theme.
	 *
	 * @link https://wordpress.org/plugins/amp/
	 */
	add_theme_support(
		'amp', array(
			'comments_live_list' => true,
		)
	);

	/**
	 * Add Gutenberg Support.
	 *
	 * @link https://wordpress.org/gutenberg/handbook/extensibility/theme-support/
	 */
	add_theme_support( 'wp-block-styles' );
	add_theme_support( 'align-wide' );

}
add_action( 'after_setup_theme', 'stagblocks_setup' );

/**
 * Set the embed width in pixels, based on the theme's design and stylesheet.
 *
 * @param array $dimensions An array of embed width and height values in pixels (in that order).
 * @return array
 */
function stagblocks_embed_dimensions( array $dimensions ) {
	$dimensions['width'] = 760;
	return $dimensions;
}
add_filter( 'embed_defaults', 'stagblocks_embed_dimensions' );

/**
 * Register Google Fonts
 */
function stagblocks_fonts_url() {
	$fonts_url = '';

	/**
	 * Translator: If Poppins does not support characters in your language, translate this to 'off'.
	 */
	$poppins = esc_html_x( 'on', 'Poppins font: on or off', 'stag-blocks' );

	$font_families = array();

	if ( 'off' !== $poppins ) {
		$font_families[] = 'Poppins:400,400i,500,600,600i';
	}

	if ( in_array( 'on', array( $poppins ) ) ) {
		$query_args = array(
			'family' => urlencode( implode( '|', $font_families ) ),
			'subset' => urlencode( 'latin,latin-ext' ),
		);

		$fonts_url = add_query_arg( $query_args, 'https://fonts.googleapis.com/css' );
	}

	return esc_url_raw( $fonts_url );

}

/**
 * Add preconnect for Google Fonts.
 *
 * @since Twenty Seventeen 1.0
 *
 * @param array  $urls           URLs to print for resource hints.
 * @param string $relation_type  The relation type the URLs are printed.
 * @return array $urls           URLs to print for resource hints.
 */
function stagblocks_resource_hints( $urls, $relation_type ) {
	if ( wp_style_is( 'stag-blocks-fonts', 'queue' ) && 'preconnect' === $relation_type ) {
		$urls[] = array(
			'href' => 'https://fonts.gstatic.com',
			'crossorigin',
		);
	}

	return $urls;
}
add_filter( 'wp_resource_hints', 'stagblocks_resource_hints', 10, 2 );

/**
 * Enqueue WordPress theme styles within Gutenberg.
 */
function stagblocks_gutenberg_styles() {
	// Add custom fonts, used in the main stylesheet.
	wp_enqueue_style( 'stag-blocks-fonts', stagblocks_fonts_url(), array(), '20180817' );

	// Enqueue main stylesheet.
	wp_enqueue_style( 'stag-blocks-base-style', get_theme_file_uri( '/css/editor-styles.css' ), array(), '20180514' );

	$color_accent = esc_attr( get_theme_mod( 'accent_color', '#3575ff' ) );

	wp_add_inline_style(
		'stag-blocks-base-style', "
		.edit-post-visual-editor a {
			color: {$color_accent};
		}
	"
	);
}
add_action( 'enqueue_block_editor_assets', 'stagblocks_gutenberg_styles' );

/**
 * Register widget area.
 *
 * @link https://developer.wordpress.org/themes/functionality/sidebars/#registering-a-sidebar
 */
function stagblocks_widgets_init() {
	register_sidebar(
		array(
			'name'          => esc_html__( 'Footer', 'stag-blocks' ),
			'id'            => 'sidebar-footer',
			'description'   => esc_html__( 'Add widgets here.', 'stag-blocks' ),
			'before_widget' => '<section id="%1$s" class="widget %2$s">',
			'after_widget'  => '</section>',
			'before_title'  => '<h3 class="widget-title">',
			'after_title'   => '</h3>',
		)
	);
}
add_action( 'widgets_init', 'stagblocks_widgets_init' );

/**
 * Enqueue styles.
 */
function stagblocks_styles() {
	// Add custom fonts, used in the main stylesheet.
	wp_enqueue_style( 'stag-blocks-fonts', stagblocks_fonts_url(), array(), '20180817' );

	// Enqueue main stylesheet.
	wp_enqueue_style( 'stag-blocks-base-style', get_stylesheet_uri(), array(), '20180514' );

	// Register component styles that are printed as needed.
	wp_register_style( 'stag-blocks-comments', get_theme_file_uri( '/css/comments.css' ), array(), '20180514' );
	wp_register_style( 'stag-blocks-content', get_theme_file_uri( '/css/content.css' ), array(), '20180514' );
	wp_register_style( 'stag-blocks-sidebar', get_theme_file_uri( '/css/sidebar.css' ), array(), '20180514' );
	wp_register_style( 'stag-blocks-widgets', get_theme_file_uri( '/css/widgets.css' ), array(), '20180514' );
	wp_register_style( 'stag-blocks-front-page', get_theme_file_uri( '/css/front-page.css' ), array(), '20180514' );
	wp_register_style( 'stag-blocks-404', get_theme_file_uri( '/css/page-404.css' ), array(), '20180730' );

	$color_accent = esc_attr( get_theme_mod( 'accent_color', '#3575ff' ) );
	wp_add_inline_style(
		'stag-blocks-base-style', "
		a, a:hover, a:focus,
		a:active,
		.nav-links a:hover,
		.nav-links a:focus,
		.comment-meta a:hover,
		.comment-meta a:focus {
			color: {$color_accent};
		}
		.main-navigation .menu > li.current-menu-item > a,
		.main-navigation .menu li a:hover,
		.main-navigation .menu li a:focus {
			color: {$color_accent};
			border-color: {$color_accent};
		}
		.sticky-badge .sticky-text,
		.comment-form .form-submit input {
			background: {$color_accent};
		}
	"
	);
}
add_action( 'wp_enqueue_scripts', 'stagblocks_styles' );

/**
 * Enqueue scripts.
 */
function stagblocks_scripts() {

	// If the AMP plugin is active, return early.
	if ( stagblocks_is_amp() ) {
		return;
	}

	// Enqueue the navigation script.
	wp_enqueue_script( 'stag-blocks-navigation', get_theme_file_uri( '/js/navigation.js' ), array( 'jquery' ), '20180514', false );
	wp_script_add_data( 'stag-blocks-navigation', 'async', true );
	wp_localize_script(
		'stag-blocks-navigation', 'stagBlocksScreenReaderText', array(
			'expand'   => __( 'Expand child menu', 'stag-blocks' ),
			'collapse' => __( 'Collapse child menu', 'stag-blocks' ),
		)
	);

	// Enqueue skip-link-focus script.
	wp_enqueue_script( 'stag-blocks-skip-link-focus-fix', get_theme_file_uri( '/js/skip-link-focus-fix.js' ), array(), '20180514', false );
	wp_script_add_data( 'stag-blocks-skip-link-focus-fix', 'defer', true );

	// Enqueue fitVids for fluid video wrappers.
	wp_enqueue_script( 'stag-blocks-fitVids', get_theme_file_uri( 'js/fitVids.js' ), array( 'jquery' ), null, true );

	// Default fitvids selectors.
	$selector_array = array(
		"iframe[src*='www.viddler.com']",
		"iframe[src*='money.cnn.com']",
		"iframe[src*='www.educreations.com']",
		"iframe[src*='//blip.tv']",
		"iframe[src*='//embed.ted.com']",
		"iframe[src*='//www.hulu.com']",
	);

	// Allow dev to customize the selectors.
	$fitvids_custom_selectors = apply_filters( 'stag-blocks_fitvids_custom_selectors', $selector_array );

	// Compile selectors.
	$fitvids_custom_selectors = array(
		'customSelector' => implode( ',', $fitvids_custom_selectors ),
	);
	// Send to the script.
	wp_localize_script(
		'stag-blocks-fitVids',
		'stagblocksFitvidsCustomSelectors',
		$fitvids_custom_selectors
	);

	wp_add_inline_script(
		'stag-blocks-fitVids', "!function($){function setupFitVids() {
		if (!$.fn.fitVids) {
			return;
		}
		var selectors;
		if ('object' === typeof stagblocksFitvidsCustomSelectors) {
			selectors = stagblocksFitvidsCustomSelectors.customSelector;
		}
		$('.entry-content').fitVids({ customSelector: selectors });
		$('.fluid-width-video-wrapper:nth-child(2)', '.video-container').css({ 'paddingTop': 0 });
	}
	setupFitVids();
	}(jQuery);"
	);

	// Enqueue comment script on singular post/page views only.
	if ( is_singular() && comments_open() && get_option( 'thread_comments' ) ) {
		wp_enqueue_script( 'comment-reply' );
	}

}
add_action( 'wp_enqueue_scripts', 'stagblocks_scripts' );

/**
 * Custom responsive image sizes.
 */
require get_template_directory() . '/inc/image-sizes.php';

/**
 * Implement the Custom Header feature.
 */
require get_template_directory() . '/pluggable/custom-header.php';

/**
 * Custom template tags for this theme.
 */
require get_template_directory() . '/inc/template-tags.php';

/**
 * Functions which enhance the theme by hooking into WordPress.
 */
require get_template_directory() . '/inc/template-functions.php';

/**
 * Customizer additions.
 */
require get_template_directory() . '/inc/customizer.php';

/**
 * Optional: Add theme support for lazyloading images.
 *
 * @link https://developers.google.com/web/fundamentals/performance/lazy-loading-guidance/images-and-video/
 */
require get_template_directory() . '/pluggable/lazyload/lazyload.php';

/**
 * Jetpack module additions.
 */
require get_template_directory() . '/inc/jetpack.php';
