<?php
/**
 * The core functions file for the SpringBoard framework. Functions defined here are generally
 * used across the entire framework to make various tasks faster. This file should be loaded
 * prior to any other files because its functions are needed to run the framework.
 *
 * @package SpringBoardCore
 * @subpackage Functions
 */

/**
 * Defines the theme prefix. This allows developers to infinitely change the theme. In theory,
 * one could use the SpringBoard core to create their own theme or filter 'sboard_prefix' with a 
 * plugin to make it easier to use hooks across multiple themes without having to figure out
 * each theme's hooks (assuming other themes used the same system).
 *
 * @since 0.7.0
 * @uses get_template() Defines the theme prefix based on the theme directory.
 * @global object $sboard The global SpringBoard object.
 * @return string $sboard->prefix The prefix of the theme.
 */
function sboard_get_prefix() {
	global $sboard;

	/* If the global prefix isn't set, define it. Plugin/theme authors may also define a custom prefix. */
	if ( empty( $sboard->prefix ) )
		$sboard->prefix = sanitize_key( apply_filters( 'sboard_prefix', get_template() ) );

	return $sboard->prefix;
}

/**
 * Defines the theme textdomain. This allows the framework to recognize the proper textdomain of the 
 * parent theme. Theme developers building from the framework should use this function in their templates 
 * to easily define the correct textdomain.
 *
 * @since 0.7.0
 * @uses get_template() Defines the theme textdomain based on the template directory.
 * @global object $sboard The global SpringBoard object.
 * @return string $sboard->textdomain The textdomain of the theme.
 */
function sboard_get_textdomain() {
	global $sboard;

	/* If the global textdomain isn't set, define it. Plugin/theme authors may also define a custom textdomain. */
	if ( empty( $sboard->textdomain ) )
		$sboard->textdomain = sanitize_key( apply_filters( sboard_get_prefix() . '_textdomain', get_template() ) );

	return $sboard->textdomain;
}

/**
 * Returns the textdomain for the child theme.
 *
 * @since 1.2.0
 * @uses get_stylesheet() Defines the child theme textdomain based on the stylesheet directory.
 * @global object $sboard The global SpringBoard object.
 * @return string $sboard->child_theme_textdomain The textdomain of the child theme.
 */
function sboard_get_child_textdomain() {
	global $sboard;

	/* If a child theme isn't active, return an empty string. */
	if ( !is_child_theme() )
		return '';

	/* If the global textdomain isn't set, define it. Plugin/theme authors may also define a custom textdomain. */
	if ( empty( $sboard->child_textdomain ) )
		$sboard->child_textdomain = sanitize_key( apply_filters( sboard_get_prefix() . '_child_textdomain', get_stylesheet() ) );

	return $sboard->child_textdomain;
}

/**
 * Filters the 'load_textdomain_mofile' filter hook so that we can change the directory and file name 
 * of the mofile for translations.  This allows child themes to have a folder called /languages with translations
 * of their parent theme so that the translations aren't lost on a parent theme upgrade.
 *
 * @since 0.9.0
 * @param string $mofile File name of the .mo file.
 * @param string $domain The textdomain currently being filtered.
 */
function sboard_load_textdomain( $mofile, $domain ) {

	/* If the $domain is for the parent or child theme, search for a $domain-$locale.mo file. */
	if ( $domain == sboard_get_textdomain() || $domain == sboard_get_child_textdomain() ) {

		/* Check for a $domain-$locale.mo file in the parent and child theme root and /languages folder. */
		$locale = get_locale();
		$locate_mofile = locate_template( array( "languages/{$domain}-{$locale}.mo", "{$domain}-{$locale}.mo" ) );

		/* If a mofile was found based on the given format, set $mofile to that file name. */
		if ( !empty( $locate_mofile ) )
			$mofile = $locate_mofile;
	}

	/* Return the $mofile string. */
	return $mofile;
}

/**
 * Adds contextual action hooks to the theme.  This allows users to easily add context-based content 
 * without having to know how to use WordPress conditional tags.  The theme handles the logic.
 *
 * An example of a basic hook would be 'sboard_header'.  The do_atomic() function extends that to 
 * give extra hooks such as 'sboard_singular_header', 'sboard_singular-post_header', and 
 * 'sboard_singular-post-ID_header'.
 *
 * @since 0.7.0
 * @uses sboard_get_prefix() Gets the theme prefix.
 * @uses sboard_get_context() Gets the context of the current page.
 * @param string $tag Usually the location of the hook but defines what the base hook is.
 * @param mixed $arg,... Optional additional arguments which are passed on to the functions hooked to the action.
 */
function do_atomic( $tag = '', $arg = '' ) {
	if ( empty( $tag ) )
		return false;

	/* Get the theme prefix. */
	$pre = sboard_get_prefix();

	/* Get the args passed into the function and remove $tag. */
	$args = func_get_args();
	array_splice( $args, 0, 1 );

	/* Do actions on the basic hook. */
	do_action_ref_array( "{$pre}_{$tag}", $args );

	/* Loop through context array and fire actions on a contextual scale. */
	foreach ( (array)sboard_get_context() as $context )
		do_action_ref_array( "{$pre}_{$context}_{$tag}", $args );
}

/**
 * Adds contextual filter hooks to the theme.  This allows users to easily filter context-based content 
 * without having to know how to use WordPress conditional tags.  The theme handles the logic.
 *
 * An example of a basic hook would be 'sboard_entry_meta'.  The apply_atomic() function extends 
 * that to give extra hooks such as 'sboard_singular_entry_meta', 'sboard_singular-post_entry_meta', 
 * and 'sboard_singular-post-ID_entry_meta'.
 *
 * @since 0.7.0
 * @uses sboard_get_prefix() Gets the theme prefix.
 * @uses sboard_get_context() Gets the context of the current page.
 * @param string $tag Usually the location of the hook but defines what the base hook is.
 * @param mixed $value The value on which the filters hooked to $tag are applied on.
 * @param mixed $var,... Additional variables passed to the functions hooked to $tag.
 * @return mixed $value The value after it has been filtered.
 */
function apply_atomic( $tag = '', $value = '' ) {
	if ( empty( $tag ) )
		return false;

	/* Get theme prefix. */
	$pre = sboard_get_prefix();

	/* Get the args passed into the function and remove $tag. */
	$args = func_get_args();
	array_splice( $args, 0, 1 );

	/* Apply filters on the basic hook. */
	$value = $args[0] = apply_filters_ref_array( "{$pre}_{$tag}", $args );

	/* Loop through context array and apply filters on a contextual scale. */
	foreach ( (array)sboard_get_context() as $context )
		$value = $args[0] = apply_filters_ref_array( "{$pre}_{$context}_{$tag}", $args );

	/* Return the final value once all filters have been applied. */
	return $value;
}

/**
 * Wraps the output of apply_atomic() in a call to do_shortcode(). This allows developers to use 
 * context-aware functionality alongside shortcodes. Rather than adding a lot of code to the 
 * function itself, developers can create individual functions to handle shortcodes.
 *
 * @since 0.7.0
 * @param string $tag Usually the location of the hook but defines what the base hook is.
 * @param mixed $value The value to be filtered.
 * @return mixed $value The value after it has been filtered.
 */
function apply_atomic_shortcode( $tag = '', $value = '' ) {
	return do_shortcode( apply_atomic( $tag, $value ) );
}

/**
 * The theme can save multiple things in a transient to help speed up page load times. We're
 * setting a default of 12 hours or 43,200 seconds (60 * 60 * 12).
 *
 * @since 0.8.0
 * @return int Transient expiration time in seconds.
 */
function sboard_get_transient_expiration() {
	return apply_filters( sboard_get_prefix() . '_transient_expiration', 43200 );
}

/**
 * Function for formatting a hook name if needed. It automatically adds the theme's prefix to 
 * the hook, and it will add a context (or any variable) if it's given.
 *
 * @since 0.7.0
 * @param string $tag The basic name of the hook (e.g., 'before_header').
 * @param string $context A specific context/value to be added to the hook.
 */
function sboard_format_hook( $tag, $context = '' ) {
	return sboard_get_prefix() . ( ( !empty( $context ) ) ? "_{$context}" : "" ). "_{$tag}";
}

/**
 * Function for setting the content width of a theme.  This does not check if a content width has been set; it 
 * simply overwrites whatever the content width is.
 *
 * @since 1.2.0
 * @global int $content_width The width for the theme's content area.
 * @param int $width Numeric value of the width to set.
 */
function sboard_set_content_width( $width = '' ) {
	global $content_width;

	$content_width = absint( $width );
}

/**
 * Function for getting the theme's content width.
 *
 * @since 1.2.0
 * @global int $content_width The width for the theme's content area.
 * @return int $content_width
 */
function sboard_get_content_width() {
	global $content_width;

	return $content_width;
}

/**
 * Gets theme data and stores it in the global $sboard variable.  By storing it, it can be accessed quickly without 
 * having to run through the get_theme_data() function again.
 *
 * @since 1.2.0
 * @param string $path Whether to use the template (parent theme) or stylesheet (child theme) path.
 */
function sboard_get_theme_data( $path = 'template' ) {
	global $sboard;

	/* If 'template' is requested, get the parent theme data. */
	if ( 'template' == $path ) {

		/* If the parent theme data isn't set, grab it with the get_theme_data() function. */
		if ( empty( $sboard->theme_data ) )
			$sboard->theme_data = get_theme_data( trailingslashit( TEMPLATEPATH ) . 'style.css' );

		/* Return the parent theme data. */
		return $sboard->theme_data;
	}

	/* If 'stylesheet' is requested, get the child theme data. */
	elseif ( 'stylesheet' == $path ) {

		/* If the child theme data isn't set, grab it with the get_theme_data() function. */
		if ( empty( $sboard->child_theme_data ) )
			$sboard->child_theme_data = get_theme_data( trailingslashit( STYLESHEETPATH ) . 'style.css' );

		/* Return the child theme data. */
		return $sboard->child_theme_data;
	}

	/* Return false for everything else. */
	return false;
}

?>