<?php
/**
 * Functions for dealing with theme settings on both the front end of the site and the admin.  This allows us 
 * to set some default settings and make it easy for theme developers to quickly grab theme settings from 
 * the database.  This file is only loaded if the theme adds support for the 'sboard-core-theme-settings' 
 * feature.
 *
 * @package SpringBoardCore
 * @subpackage Functions
 */

/**
 * Loads the SpringBoard theme settings once and allows the input of the specific field the user would 
 * like to show.  SpringBoard theme settings are added with 'autoload' set to 'yes', so the settings are 
 * only loaded once on each page load.
 *
 * @since 0.7.0
 * @uses get_option() Gets an option from the database.
 * @uses sboard_get_prefix() Gets the prefix of the theme.
 * @global object $sboard The global SpringBoard object.
 * @param string $option The specific theme setting the user wants.
 * @return string|int|array $settings[$option] Specific setting asked for.
 */
function sboard_get_setting( $option = '' ) {
	global $sboard;

	/* If no specific option was requested, return false. */
	if ( !$option )
		return false;

	/* If the settings array hasn't been set, call get_option() to get an array of theme settings. */
	if ( !isset( $sboard->settings ) )
		$sboard->settings = get_option( sboard_get_prefix() . '_theme_settings', sboard_get_default_theme_settings() );

	/* If the settings isn't an array or the specific option isn't in the array, return false. */
	if ( !is_array( $sboard->settings ) || empty( $sboard->settings[$option] ) )
		return false;

	/* If the specific option is an array, return it. */
	if ( is_array( $sboard->settings[$option] ) )
		return $sboard->settings[$option];

	/* Strip slashes from the setting and return. */
	else
		return wp_kses_stripslashes( $sboard->settings[$option] );
}

/**
 * Sets up a default array of theme settings for use with the theme.  Theme developers should filter the 
 * "{$prefix}_default_theme_settings" hook to define any default theme settings.  WordPress does not 
 * provide a hook for default settings at this time.
 *
 * @since 1.0.0
 */
function sboard_get_default_theme_settings() {

	/* Set up some default variables. */
	$settings = array();
	$domain = sboard_get_textdomain();
	$prefix = sboard_get_prefix();

	/* Get theme-supported meta boxes for the settings page. */
	$supports = get_theme_support( 'sboard-core-theme-settings' );

	/* If the current theme supports the footer meta box and shortcodes, add default footer settings. */
	if ( is_array( $supports[0] ) && in_array( 'footer', $supports[0] ) && current_theme_supports( 'sboard-core-shortcodes' ) ) {

		/* If there is a child theme active, add the [child-link] shortcode to the $footer_insert. */
		if ( is_child_theme() )
			$settings['footer_insert'] = '<p class="copyright">' . __( 'Copyright &#169; [the-year] [site-link].', $domain ) . '</p>' . "\n\n" . '<p class="credit">' . __( 'Powered by [wp-link], [theme-link], and [child-link].', $domain ) . '</p>';

		/* If no child theme is active, leave out the [child-link] shortcode. */
		else
			$settings['footer_insert'] = '<p class="copyright">' . __( 'Copyright &#169; [the-year] [site-link].', $domain ) . '</p>' . "\n\n" . '<p class="credit">' . __( 'Powered by [wp-link] and [theme-link].', $domain ) . '</p>';
	}

	/* Return the $settings array and provide a hook for overwriting the default settings. */
	return apply_filters( "{$prefix}_default_theme_settings", $settings );
}

?>