<?php
/**
 * The functions file is used to initialize everything in the theme.  It controls how the theme is loaded and 
 * sets up the supported features, default actions, and default filters.  If making customizations, users 
 * should create a child theme and make changes to its functions.php file (not this one).  Friends don't let 
 * friends modify parent theme files. ;)
 *
 * Child themes should do their setup on the 'after_setup_theme' hook with a priority of 11 if they want to
 * override parent theme features.  Use a priority of 9 if wanting to run before the parent theme.
 *
 * This program is free software; you can redistribute it and/or modify it under the terms of the GNU 
 * General Public License version 2, as published by the Free Software Foundation.  You may NOT assume 
 * that you can use any other version of the GPL.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without 
 * even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 *
 * You should have received a copy of the GNU General Public License along with this program; if not, write 
 * to the Free Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA
 *
 * @package SpringBoard
 * @subpackage Functions
 * @version 0.1
 * @author Semper Fi Web Design <sales@semperfiwebdesign.com>
 * @copyright Copyright (c) 2011 Semper Fi Web Design, 2008 - 2011, Justin Tadlock
 * @link http://semperthemes.com/
 * @license http://www.gnu.org/licenses/old-licenses/gpl-2.0.html
 */

/* Load the core theme framework. */
require_once( trailingslashit( TEMPLATEPATH ) . 'library/sboard.php' );
$theme = new SpringBoard();

/* Do theme setup on the 'after_setup_theme' hook. */
add_action( 'after_setup_theme', 'sboard_theme_setup_theme' );

/**
 * Theme setup function.  This function adds support for theme features and defines the default theme
 * actions and filters.
 *
 * @since 0.9.0
 */
function sboard_theme_setup_theme() {
	global $content_width;
	global $theme_settings;
	
	/* Get the theme prefix. */
	$prefix = sboard_get_prefix();
	$url = get_bloginfo('template_url');
	
	/* Add support for framework features. */
	add_theme_support( 'sboard-core-menus', array( 'primary' ) );
	add_theme_support( 'sboard-core-sidebars', array( 'primary', 'secondary', 'subsidiary', 'before-content', 'after-content', 'after-singular' ) );
	add_theme_support( 'sboard-core-widgets' );
	add_theme_support( 'sboard-core-shortcodes' );
	add_theme_support( 'sboard-core-theme-settings', array( 'about', 'logo', 'background' ) );
	add_theme_support( 'sboard-core-drop-downs' );
	add_theme_support( 'sboard-core-template-hierarchy' );
	add_theme_support( 'sboard-core-deprecated' );

	/* Add support for framework extensions. */
	add_theme_support( 'breadcrumb-trail' );
	add_theme_support( 'custom-field-series' );
	add_theme_support( 'get-the-image' );
	add_theme_support( 'post-stylesheets' );

	/* Only add cleaner gallery support if not using child theme. Eventually, all child themes should support this. */
	if ( 'sboard' == get_stylesheet() )
		add_theme_support( 'cleaner-gallery' );

	/* Add support for WordPress features. */
	add_theme_support( 'automatic-feed-links' );

	/* Register sidebars. */
	add_action( 'init', 'sboard_theme_register_sidebars', 11 );

	/* Disables widget areas. */
	add_filter( 'sidebars_widgets', 'sboard_theme_remove_sidebars' );

	/* Header actions. */
	add_action( "{$prefix}_header", 'sboard_site_title' );
	add_action( "{$prefix}_header", 'sboard_site_description' );

	/* Load the primary menu. */
	add_action( "{$prefix}_after_header", 'sboard_get_primary_menu' );

	/* Add the primary and secondary sidebars after the container. */
	add_action( "{$prefix}_after_container", 'sboard_get_primary' );
	add_action( "{$prefix}_after_container", 'sboard_get_secondary' );

	/* Add the breadcrumb trail and before content sidebar before the content. */
	add_action( "{$prefix}_before_content", 'sboard_breadcrumb' );
	add_action( "{$prefix}_before_content", 'sboard_get_utility_before_content' );

	/* Add the title, byline, and entry meta before and after the entry. */
	add_action( "{$prefix}_before_entry", 'sboard_entry_title' );
	add_action( "{$prefix}_before_entry", 'sboard_byline' );
	add_action( "{$prefix}_after_entry", 'sboard_entry_meta' );

	/* Add the after singular sidebar and custom field series extension after singular views. */
	add_action( "{$prefix}_after_singular", 'sboard_get_utility_after_singular' );
	add_action( "{$prefix}_after_singular", 'custom_field_series' );

	/* Add the after content sidebar and navigation links after the content. */
	add_action( "{$prefix}_after_content", 'sboard_get_utility_after_content' );
	add_action( "{$prefix}_after_content", 'sboard_navigation_links' );

	/* Add the subsidiary sidebar and footer insert to the footer. */
	add_action( "{$prefix}_before_footer", 'sboard_get_subsidiary' ); 
	add_action( "{$prefix}_footer", 'sboard_footer_insert' );

	/* Add the comment avatar and comment meta before individual comments. */
	add_action( "{$prefix}_before_comment", 'sboard_avatar' );
	add_action( "{$prefix}_before_comment", 'sboard_comment_meta' );

	/* Add SpringBoard theme-specific body classes. */
	add_filter( 'body_class', 'sboard_theme_body_class' );
	
	add_action( "create_markup", "sboard_produce_markup" );
	
	add_action( "create_logo", "sboard_set_logo" );

	/* Content width. */
	if ( empty( $content_width ) && !is_active_sidebar( 'primary' ) && !is_active_sidebar( 'secondary' ) )
		$content_width = 960;
	elseif ( empty( $content_width ) )
		$content_width = 620;
		
	/* Theme Options Functions */
	add_action( "{$prefix}_establish_options", 'sboard_set_options_callback' );
}
 
function sboard_set_options_callback($args){
	global $theme_settings;
	$target = $args[ 'target' ];
	if ( is_array( $args ) ){
		if( !is_array( $args[ 'values' ] ) ): 
		$args['values'] = Array( $args['values'] => Array(
						'value' => true, 'field' => 'choice' ) ); endif;
		foreach ( $args['values'] as $key => $values ){
			$theme_settings[ $values[ 'section' ] ][ $target ][ $key ][ 'id' ] = 
			sboard_settings_field_id( $key );
			$theme_settings[ $values[ 'section' ] ][ $target ][ $key ][ 'name' ] = 		
			sboard_settings_field_name( $key );
			
			if ( $values[ 'field' ] == 'checkbox'){
				//print sboard_get_setting( $key );
			}
			
				if ( $values[ 'field' ] == 'checkbox'){
					if ( !sboard_get_setting( $key ) ){
						$value = 'off';
					} elseif ( sboard_get_setting( $key ) ) {
						$value = 'on';
					}
				} elseif ( $values[ 'field' ] != 'checkbox') {
					if ( sboard_get_setting( $key ) ){
						$value = sboard_get_setting( $key );
					} elseif ( !sboard_get_setting( $key ) ){
						$value = $values['value'];
					}
				}
				
			$theme_settings[ $values[ 'section' ] ]
			[ $target ][ $key ][ 'value' ] = $value;
			
			$theme_settings[ $values[ 'section' ] ]
			[ $target ][ $key ][ 'default' ] = $values[ 'value' ];
			
			$theme_settings[ $values[ 'section' ] ]
			[ $target ][ $key ][ 'field' ] = $values[ 'field' ];
			
			if ( isset( $values['misc'] ) )
			$theme_settings[ $values[ 'section' ] ]
			[ $target ][ $key ][ 'misc' ] = $values[ 'misc' ];
			
			$theme_settings[ $values[ 'section' ] ]
			[ $target ][ $key ][ 'tag' ] = $key;
			
		}
	}
}

function sboard_header_styling(){
	
}

function sboard_admin_styling(){
	
}

function sboard_admin_callback(){
	
}


function sboard_produce_markup($args){
	if ( !is_array($args['markup']) ){ $args['markup'] = Array(0 => $args['markup']); }
	foreach ($args['markup'] as $key => $values){ 
		print '<div class=' . $args['class'] . '>' . $values . '</div>'; 
	}	
}

function sboard_set_logo($args){
	$markup = '<img src="' . $args['url'] . '" alt ="' . $args['alt'] . '"/>';
	do_action('create_markup', Array('markup' => $markup, 'class'=>'something' ) );
}

/**
 * Register additional sidebars that are not a part of the core framework and are exclusive to this
 * theme.
 *
 * @since 0.9.0
 */
function sboard_theme_register_sidebars() {

	/* Register the widgets template sidebar. */
	register_sidebar(
		array(
			'id' => 'widgets-template',
			'name' => __( 'Widgets Template', sboard_get_textdomain() ),
			'description' => __( 'Used as the content of the Widgets page template.', sboard_get_textdomain() ),
			'before_widget' => '<div id="%1$s" class="widget %2$s widget-%2$s"><div class="widget-inside">',
			'after_widget' => '</div></div>',
			'before_title' => '<h3 class="widget-title">',
			'after_title' => '</h3>'
		)
	);

	/* Register the 404 template sidebar. */
	register_sidebar(
		array(
			'id' => 'error-404-template',
			'name' => __( '404 Template', sboard_get_textdomain() ),
			'description' => __( 'Replaces the default 404 error page content.', sboard_get_textdomain() ),
			'before_widget' => '<div id="%1$s" class="widget %2$s widget-%2$s"><div class="widget-inside">',
			'after_widget' => '</div></div>',
			'before_title' => '<h3 class="widget-title">',
			'after_title' => '</h3>'
		)
	);
	
	register_sidebar(
		array(
			'id' => 'footer-widget',
			'name' => __( 'Footer', sboard_get_textdomain() ),
			'description' => __( 'Adds footer widget to the footer area', sboard_get_textdomain() ),
			'before_widget' => '<div id="%1$s" class="widget %2$s widget-%2$s"><div class="widget-inside">',
			'after_widget' => '</div></div>',
			'before_title' => '<h3 class="widget-title">',
			'after_title' => '</h3>'
		)
	);
	
	register_sidebar(
		array(
			'id' => 'header-widget',
			'name' => __( 'Header', sboard_get_textdomain() ),
			'description' => __( 'Adds footer widget to the footer area', sboard_get_textdomain() ),
			'before_widget' => '<div id="%1$s" class="widget %2$s widget-%2$s"><div class="widget-inside">',
			'after_widget' => '</div></div>',
			'before_title' => '<h3 class="widget-title">',
			'after_title' => '</h3>'
		)
	);
}

/**
 * Function for adding SpringBoard theme <body> classes.
 *
 * @since 0.9.0
 */
function sboard_theme_body_class( $classes ) {
	global $wp_query, $is_lynx, $is_gecko, $is_IE, $is_opera, $is_NS4, $is_safari, $is_chrome;

	/* Singular post classes (deprecated). */
	if ( is_singular() ) {

		if ( is_page() )
			$classes[] = "page-{$wp_query->post->ID}"; // Use singular-page-ID

		elseif ( is_singular( 'post' ) )
			$classes[] = "single-{$wp_query->post->ID}"; // Use singular-post-ID
	}

	/* Browser detection. */
	$browsers = array( 'gecko' => $is_gecko, 'opera' => $is_opera, 'lynx' => $is_lynx, 'ns4' => $is_NS4, 'safari' => $is_safari, 'chrome' => $is_chrome, 'msie' => $is_IE );
	foreach ( $browsers as $key => $value ) {
		if ( $value ) {
			$classes[] = $key;
			break;
		}
	}

	/* SpringBoard theme widgets detection. */
	foreach ( array( 'primary', 'secondary', 'subsidiary' ) as $sidebar )
		$classes[] = ( is_active_sidebar( $sidebar ) ) ? "{$sidebar}-active" : "{$sidebar}-inactive";

	if ( in_array( 'primary-inactive', $classes ) && in_array( 'secondary-inactive', $classes ) && in_array( 'subsidiary-inactive', $classes ) )
		$classes[] = 'no-widgets';

	/* Return the array of classes. */
	return $classes;
}

/**
 * Displays the breadcrumb trail extension if it's supported.
 *
 * @since 0.1.0
 */
function sboard_breadcrumb() {
	if ( current_theme_supports( 'breadcrumb-trail' ) )
		breadcrumb_trail( array( 'front_page' => false ) );
}

/**
 * Displays the post title.
 *
 * @since 0.5.0
 */
function sboard_entry_title() {
	echo apply_atomic_shortcode( 'entry_title', '[entry-title]' );
}

/**
 * Default entry byline for posts.
 *
 * @since 0.5.0
 */
function sboard_byline() {

	$byline = '';

	if ( 'post' == get_post_type() && 'link_category' !== get_query_var( 'taxonomy' ) )
		$byline = '<p class="byline">' . __( 'By [entry-author] on [entry-published] [entry-edit-link before=" | "]', sboard_get_textdomain() ) . '</p>';

	echo apply_atomic_shortcode( 'byline', $byline );
}

/**
 * Displays the default entry metadata.
 *
 * @since 0.5.0
 */
function sboard_entry_meta() {

	$meta = '';

	if ( 'post' == get_post_type() )
		$meta = '<p class="entry-meta">' . __( '[entry-terms taxonomy="category" before="Posted in "] [entry-terms taxonomy="post_tag" before="| Tagged "] [entry-comments-link before="| "]', sboard_get_textdomain() ) . '</p>';

	elseif ( is_page() && current_user_can( 'edit_page', get_the_ID() ) )
		$meta = '<p class="entry-meta">[entry-edit-link]</p>';

	echo apply_atomic_shortcode( 'entry_meta', $meta );
}

/**
 * Function for displaying a comment's metadata.
 *
 * @since 0.7.0
 */
function sboard_comment_meta() {
	echo apply_atomic_shortcode( 'comment_meta', '<div class="comment-meta comment-meta-data">[comment-author] [comment-published] [comment-permalink before="| "] [comment-edit-link before="| "] [comment-reply-link before="| "]</div>' );
}

/**
 * Loads the loop-nav.php template with backwards compability for navigation-links.php.
 *
 * @since 0.2.0
 * @uses locate_template() Checks for template in child and parent theme.
 */
function sboard_navigation_links() {
	locate_template( array( 'navigation-links.php', 'loop-nav.php' ), true );
}

/**
 * Displays the footer insert from the theme settings page.
 *
 * @since 0.2.1
 */
function sboard_footer_insert() {
	$footer_insert = sboard_get_setting( 'footer_insert' );

	if ( !empty( $footer_insert ) )
		echo '<div class="footer-insert">' . do_shortcode( $footer_insert ) . '</div>';
}

/**
 * Removes all widget areas on the x page/post template.  No widget templates should come in
 * the form of $post_type-no-widgets.php.  This function also provides backwards compatibility with the old
 * no-widgets.php template.
 *
 * @since 0.9.0
 */
function sboard_theme_remove_sidebars( $sidebars_widgets ) {
	global $wp_query;

	if ( is_singular() ) {
		$template = get_post_meta( $wp_query->post->ID, "_wp_{$wp_query->post->post_type}_template", true );

		if ( 'no-widgets.php' == $template || "{$wp_query->post->post_type}-no-widgets.php" == $template )
			$sidebars_widgets = array( true );
	}

	return $sidebars_widgets;
}

/**
 * Loads the sidebar-primary.php template.
 *
 * @since 0.2.2
 * @uses get_sidebar() Checks for the template in the child and parent theme.
 */
function sboard_get_primary() {
	get_sidebar( 'primary' );
}

/**
 * Loads the sidebar-secondary.php template.
 *
 * @since 0.2.2
 * @uses get_sidebar() Checks for the template in the child and parent theme.
 */
function sboard_get_secondary() {
	get_sidebar( 'secondary' );
}

/**
 * Loads the sidebar-subsidiary.php template.
 *
 * @since 0.3.1
 * @uses get_sidebar() Checks for the template in the child and parent theme.
 */
function sboard_get_subsidiary() {
	get_sidebar( 'subsidiary' );
}

/**
 * Loads the sidebar-before-content.php template.
 *
 * @since 0.4.0
 * @uses get_sidebar() Checks for the template in the child and parent theme.
 */
function sboard_get_utility_before_content() {
	get_sidebar( 'before-content' );
}

/**
 * Loads the sidebar-after-content.php template.
 *
 * @since 0.4.0
 * @uses get_sidebar() Checks for the template in the child and parent theme.
 */
function sboard_get_utility_after_content() {
	get_sidebar( 'after-content' );
}

/**
 * Loads the sidebar-after-singular.php template.
 *
 * @since 0.7.0
 * @uses get_sidebar() Checks for the template in the child and parent theme.
 */
function sboard_get_utility_after_singular() {
	get_sidebar( 'after-singular' );
}

/**
 * Loads the menu-primary.php template.
 *
 * @since 0.8.0
 * @uses get_template_part() Checks for template in child and parent theme.
 */
function sboard_get_primary_menu() {
	get_template_part( 'menu', 'primary' );
}

/**
 * @since 0.7.0
 * @deprecated 0.9.0
 */
function sboard_disable_styles() {
	_deprecated_function( __FUNCTION__, '0.9.0' );
}

/**
 * @since 0.4.0
 * @deprecated 0.9.0
 */
function sboard_favicon() {
	_deprecated_function( __FUNCTION__, '0.9.0' );
}

/**
 * @since 0.4.0
 * @deprecated 0.9.0
 */
function sboard_feed_link( $output, $feed ) {
	_deprecated_function( __FUNCTION__, '0.9.0' );
	return $output;
}

/**
 * @since 0.4.0
 * @deprecated 0.9.0
 */
function sboard_other_feed_link( $link ) {
	_deprecated_function( __FUNCTION__, '0.9.0' );
	return $link;
}

?>
