<?php



/**



 * Nursery functions and definitions.



 *



 * Sets up the theme and provides some helper functions, which are used in the



 * theme as custom template tags. Others are attached to action and filter



 * hooks in WordPress to change core functionality.



 *



 * When using a child theme (see http://codex.wordpress.org/Theme_Development



 * and http://codex.wordpress.org/Child_Themes), you can override certain



 * functions (those wrapped in a function_exists() call) by defining them first



 * in your child theme's functions.php file. The child theme's functions.php



 * file is included before the parent theme's file, so the child theme



 * functions would be used.



 *



 * Functions that are not pluggable (not wrapped in function_exists()) are



 * instead attached to a filter or action hook.



 *



 * For more information on hooks, actions, and filters,



 * see http://codex.wordpress.org/Plugin_API



 *



 * @package Designpromote



 * @subpackage Nursery



 * @since Twenty Nursery 1.0



 */







/**



 * Sets up the content width value based on the theme's design.



 * @see nursery_content_width() for template-specific adjustments.



 */



if ( ! isset( $content_width ) )



	$content_width = 604;







/**



 * Adds support for a custom header image.



 */



require get_template_directory() . '/inc/custom-header.php';







/**



 * Twenty Thirteen only works in WordPress 3.6 or later.



 */



if ( version_compare( $GLOBALS['wp_version'], '3.6-alpha', '<' ) )



	require get_template_directory() . '/inc/back-compat.php';







/**



 * Sets up theme defaults and registers the various WordPress features that



 * Twenty Thirteen supports.



 *



 * @uses load_theme_textdomain() For translation/localization support.



 * @uses add_editor_style() To add Visual Editor stylesheets.



 * @uses add_theme_support() To add support for automatic feed links, post



 * formats, and post thumbnails.



 * @uses register_nav_menu() To add support for a navigation menu.



 * @uses set_post_thumbnail_size() To set a custom post thumbnail size.



 *



 * @since Twenty Thirteen 1.0



 *



 * @return void



 */



function nursery_setup() {



	/*



	 * Makes Twenty Thirteen available for translation.



	 *



	 * Translations can be added to the /languages/ directory.



	 * If you're building a theme based on Twenty Thirteen, use a find and



	 * replace to change 'nursery' to the name of your theme in all



	 * template files.



	 */



	load_theme_textdomain( 'nursery', get_template_directory() . '/languages' );







	/*



	 * This theme styles the visual editor to resemble the theme style,



	 * specifically font, colors, icons, and column width.



	 */



	add_editor_style( array( 'css/editor-style.css', 'fonts/genericons.css', nursery_fonts_url() ) );







	// Adds RSS feed links to <head> for posts and comments.



	add_theme_support( 'automatic-feed-links' );







	// Switches default core markup for search form, comment form, and comments



	// to output valid HTML5.



	add_theme_support( 'html5', array( 'search-form', 'comment-form', 'comment-list' ) );







	/*



	 * This theme supports all available post formats by default.



	 * See http://codex.wordpress.org/Post_Formats



	 */



	add_theme_support( 'post-formats', array(



		'aside', 'audio', 'chat', 'gallery', 'image', 'link', 'quote', 'status', 'video'



	) );







	// This theme uses wp_nav_menu() in one location.



	register_nav_menu( 'primary', __( 'Navigation Menu', 'nursery' ) );







	/*



	 * This theme uses a custom image size for featured images, displayed on



	 * "standard" posts and pages.



	 */



	add_theme_support( 'post-thumbnails' );



	set_post_thumbnail_size( 604, 270, true );







	// This theme uses its own gallery styles.



	add_filter( 'use_default_gallery_style', '__return_false' );



}



add_action( 'after_setup_theme', 'nursery_setup' );







/**



 * Returns the Google font stylesheet URL, if available.



 *



 * The use of Source Sans Pro and Bitter by default is localized. For languages



 * that use characters not supported by the font, the font can be disabled.



 *



 * @since Twenty Thirteen 1.0



 *



 * @return string Font stylesheet or empty string if disabled.



 */



function nursery_fonts_url() {



	$fonts_url = '';







	/* Translators: If there are characters in your language that are not



	 * supported by Source Sans Pro, translate this to 'off'. Do not translate



	 * into your own language.



	 */



	$source_sans_pro = _x( 'on', 'Source Sans Pro font: on or off', 'nursery' );







	/* Translators: If there are characters in your language that are not



	 * supported by Bitter, translate this to 'off'. Do not translate into your



	 * own language.



	 */



	$bitter = _x( 'on', 'Bitter font: on or off', 'nursery' );







	if ( 'off' !== $source_sans_pro || 'off' !== $bitter ) {



		$font_families = array();







		if ( 'off' !== $source_sans_pro )



			$font_families[] = 'Source Sans Pro:300,400,700,300italic,400italic,700italic';







		if ( 'off' !== $bitter )



			$font_families[] = 'Bitter:400,700';







		$query_args = array(



			'family' => urlencode( implode( '|', $font_families ) ),



			'subset' => urlencode( 'latin,latin-ext' ),



		);



		$fonts_url = add_query_arg( $query_args, "//fonts.googleapis.com/css" );



	}







	return $fonts_url;



}







/**



 * Enqueues scripts and styles for front end.



 *



 * @since Twenty Thirteen 1.0



 *



 * @return void



 */



function nursery_scripts_styles() {



	// Adds JavaScript to pages with the comment form to support sites with



	// threaded comments (when in use).



	if ( is_singular() && comments_open() && get_option( 'thread_comments' ) )



		wp_enqueue_script( 'comment-reply' );







	// Adds Masonry to handle vertical alignment of footer widgets.



	if ( is_active_sidebar( 'sidebar-1' ) )



		wp_enqueue_script( 'jquery-masonry' );







	// Loads JavaScript file with functionality specific to Twenty Thirteen.



	wp_enqueue_script( 'nursery-script', get_template_directory_uri() . '/js/functions.js', array( 'jquery' ), '2013-07-18', true );







	// Add Open Sans and Bitter fonts, used in the main stylesheet.



	wp_enqueue_style( 'nursery-fonts', nursery_fonts_url(), array(), null );







	// Add Genericons font, used in the main stylesheet.



	wp_enqueue_style( 'genericons', get_template_directory_uri() . '/fonts/genericons.css', array(), '2.09' );







	// Loads our main stylesheet.



	wp_enqueue_style( 'nursery-style', get_stylesheet_uri(), array(), '2013-07-18' );







	// Loads the Internet Explorer specific stylesheet.



	wp_enqueue_style( 'nursery-ie', get_template_directory_uri() . '/css/ie.css', array( 'nursery-style' ), '2013-07-18' );



	wp_style_add_data( 'nursery-ie', 'conditional', 'lt IE 9' );



}



add_action( 'wp_enqueue_scripts', 'nursery_scripts_styles' );







/**



 * Creates a nicely formatted and more specific title element text for output



 * in head of document, based on current view.



 *



 * @since Twenty Thirteen 1.0



 *



 * @param string $title Default title text for current view.



 * @param string $sep Optional separator.



 * @return string The filtered title.



 */



function nursery_wp_title( $title, $sep ) {



	global $paged, $page;







	if ( is_feed() )



		return $title;







	// Add the site name.



	$title .= get_bloginfo( 'name' );







	// Add the site description for the home/front page.



	$site_description = get_bloginfo( 'description', 'display' );



	if ( $site_description && ( is_home() || is_front_page() ) )



		$title = "$title $sep $site_description";







	// Add a page number if necessary.



	if ( $paged >= 2 || $page >= 2 )



		$title = "$title $sep " . sprintf( __( 'Page %s', 'nursery' ), max( $paged, $page ) );







	return $title;



}



add_filter( 'wp_title', 'nursery_wp_title', 10, 2 );







/**



 * Registers two widget areas.



 *



 * @since Twenty Thirteen 1.0



 *



 * @return void



 */



function nursery_widgets_init() {



	register_sidebar( array(



		'name'          => __( 'Main Widget Area', 'nursery' ),



		'id'            => 'sidebar-1',



		'description'   => __( 'Appears in the footer section of the site.', 'nursery' ),



		'before_widget' => '<aside id="%1$s" class="widget %2$s">',



		'after_widget'  => '</aside>',



		'before_title'  => '<h3 class="widget-title">',



		'after_title'   => '</h3>',



	) );







	register_sidebar( array(



		'name'          => __( 'Secondary Widget Area', 'nursery' ),



		'id'            => 'sidebar-2',



		'description'   => __( 'Appears on posts and pages in the sidebar.', 'nursery' ),



		'before_widget' => '<aside id="%1$s" class="widget %2$s">',



		'after_widget'  => '</aside>',



		'before_title'  => '<h3 class="widget-title">',



		'after_title'   => '</h3>',



	) );



}



add_action( 'widgets_init', 'nursery_widgets_init' );







if ( ! function_exists( 'nursery_paging_nav' ) ) :



/**



 * Displays navigation to next/previous set of posts when applicable.



 *



 * @since Twenty Thirteen 1.0



 *



 * @return void



 */



function nursery_paging_nav() {



	global $wp_query;







	// Don't print empty markup if there's only one page.



	if ( $wp_query->max_num_pages < 2 )



		return;



	?>



	<nav class="navigation paging-navigation" role="navigation">



		<h1 class="screen-reader-text"><?php _e( 'Posts navigation', 'nursery' ); ?></h1>



		<div class="nav-links">







			<?php if ( get_next_posts_link() ) : ?>



			<div class="nav-previous"><?php next_posts_link( __( '<span class="meta-nav">&larr;</span> Older posts', 'nursery' ) ); ?></div>



			<?php endif; ?>







			<?php if ( get_previous_posts_link() ) : ?>



			<div class="nav-next"><?php previous_posts_link( __( 'Newer posts <span class="meta-nav">&rarr;</span>', 'nursery' ) ); ?></div>



			<?php endif; ?>







		</div><!-- .nav-links -->



	</nav><!-- .navigation -->



	<?php



}



endif;







if ( ! function_exists( 'nursery_post_nav' ) ) :



/**



 * Displays navigation to next/previous post when applicable.



*



* @since Twenty Thirteen 1.0



*



* @return void



*/



function nursery_post_nav() {



	global $post;







	// Don't print empty markup if there's nowhere to navigate.



	$previous = ( is_attachment() ) ? get_post( $post->post_parent ) : get_adjacent_post( false, '', true );



	$next     = get_adjacent_post( false, '', false );







	if ( ! $next && ! $previous )



		return;



	?>



	<nav class="navigation post-navigation" role="navigation">



		<h1 class="screen-reader-text"><?php _e( 'Post navigation', 'nursery' ); ?></h1>



		<div class="nav-links">







			<?php previous_post_link( '%link', _x( '<span class="meta-nav">&larr;</span> %title', 'Previous post link', 'nursery' ) ); ?>



			<?php next_post_link( '%link', _x( '%title <span class="meta-nav">&rarr;</span>', 'Next post link', 'nursery' ) ); ?>







		</div><!-- .nav-links -->



	</nav><!-- .navigation -->



	<?php



}



endif;







if ( ! function_exists( 'nursery_entry_meta' ) ) :



/**



 * Prints HTML with meta information for current post: categories, tags, permalink, author, and date.



 *



 * Create your own nursery_entry_meta() to override in a child theme.



 *



 * @since Twenty Thirteen 1.0



 *



 * @return void



 */



function nursery_entry_meta() {



	if ( is_sticky() && is_home() && ! is_paged() )



		echo '<span class="featured-post">' . __( 'Sticky', 'nursery' ) . '</span>';







	if ( ! has_post_format( 'link' ) && 'post' == get_post_type() )



		nursery_entry_date();







	// Translators: used between list items, there is a space after the comma.



	$categories_list = get_the_category_list( __( ', ', 'nursery' ) );



	if ( $categories_list ) {



		echo '<span class="categories-links">' . $categories_list . '</span>';



	}







	// Translators: used between list items, there is a space after the comma.



	$tag_list = get_the_tag_list( '', __( ', ', 'nursery' ) );



	if ( $tag_list ) {



		echo '<span class="tags-links">' . $tag_list . '</span>';



	}







	// Post author



	if ( 'post' == get_post_type() ) {



		printf( '<span class="author vcard"><a class="url fn n" href="%1$s" title="%2$s" rel="author">%3$s</a></span>',



			esc_url( get_author_posts_url( get_the_author_meta( 'ID' ) ) ),



			esc_attr( sprintf( __( 'View all posts by %s', 'nursery' ), get_the_author() ) ),



			get_the_author()



		);



	}



}



endif;







if ( ! function_exists( 'nursery_entry_date' ) ) :



/**



 * Prints HTML with date information for current post.



 *



 * Create your own nursery_entry_date() to override in a child theme.



 *



 * @since Twenty Thirteen 1.0



 *



 * @param boolean $echo Whether to echo the date. Default true.



 * @return string The HTML-formatted post date.



 */



function nursery_entry_date( $echo = true ) {



	if ( has_post_format( array( 'chat', 'status' ) ) )



		$format_prefix = _x( '%1$s on %2$s', '1: post format name. 2: date', 'nursery' );



	else



		$format_prefix = '%2$s';







	$date = sprintf( '<span class="date"><a href="%1$s" title="%2$s" rel="bookmark"><time class="entry-date" datetime="%3$s">%4$s</time></a></span>',



		esc_url( get_permalink() ),



		esc_attr( sprintf( __( 'Permalink to %s', 'nursery' ), the_title_attribute( 'echo=0' ) ) ),



		esc_attr( get_the_date( 'c' ) ),



		esc_html( sprintf( $format_prefix, get_post_format_string( get_post_format() ), get_the_date() ) )



	);







	if ( $echo )



		echo $date;







	return $date;



}



endif;







if ( ! function_exists( 'nursery_the_attached_image' ) ) :



/**



 * Prints the attached image with a link to the next attached image.



 *



 * @since Twenty Thirteen 1.0



 *



 * @return void



 */



function nursery_the_attached_image() {



	$post                = get_post();



	$attachment_size     = apply_filters( 'nursery_attachment_size', array( 724, 724 ) );



	$next_attachment_url = wp_get_attachment_url();







	/**



	 * Grab the IDs of all the image attachments in a gallery so we can get the URL



	 * of the next adjacent image in a gallery, or the first image (if we're



	 * looking at the last image in a gallery), or, in a gallery of one, just the



	 * link to that image file.



	 */



	$attachment_ids = get_posts( array(



		'post_parent'    => $post->post_parent,



		'fields'         => 'ids',



		'numberposts'    => -1,



		'post_status'    => 'inherit',



		'post_type'      => 'attachment',



		'post_mime_type' => 'image',



		'order'          => 'ASC',



		'orderby'        => 'menu_order ID'



	) );







	// If there is more than 1 attachment in a gallery...



	if ( count( $attachment_ids ) > 1 ) {



		foreach ( $attachment_ids as $attachment_id ) {



			if ( $attachment_id == $post->ID ) {



				$next_id = current( $attachment_ids );



				break;



			}



		}







		// get the URL of the next image attachment...



		if ( $next_id )



			$next_attachment_url = get_attachment_link( $next_id );







		// or get the URL of the first image attachment.



		else



			$next_attachment_url = get_attachment_link( array_shift( $attachment_ids ) );



	}







	printf( '<a href="%1$s" title="%2$s" rel="attachment">%3$s</a>',



		esc_url( $next_attachment_url ),



		the_title_attribute( array( 'echo' => false ) ),



		wp_get_attachment_image( $post->ID, $attachment_size )



	);



}



endif;







/**



 * Returns the URL from the post.



 *



 * @uses get_url_in_content() to get the URL in the post meta (if it exists) or



 * the first link found in the post content.



 *



 * Falls back to the post permalink if no URL is found in the post.



 *



 * @since Twenty Thirteen 1.0



 *



 * @return string The Link format URL.



 */



function nursery_get_link_url() {



	$content = get_the_content();



	$has_url = get_url_in_content( $content );







	return ( $has_url ) ? $has_url : apply_filters( 'the_permalink', get_permalink() );



}







/**



 * Extends the default WordPress body classes.



 *



 * Adds body classes to denote:



 * 1. Single or multiple authors.



 * 2. Active widgets in the sidebar to change the layout and spacing.



 * 3. When avatars are disabled in discussion settings.



 *



 * @since Twenty Thirteen 1.0



 *



 * @param array $classes A list of existing body class values.



 * @return array The filtered body class list.



 */



function nursery_body_class( $classes ) {



	if ( ! is_multi_author() )



		$classes[] = 'single-author';







	if ( is_active_sidebar( 'sidebar-2' ) && ! is_attachment() && ! is_404() && ! is_front_page() )



		$classes[] = 'sidebar';







	if ( ! get_option( 'show_avatars' ) )



		$classes[] = 'no-avatars';



		



	//if ( is_front_page() || is_home() )



		//$classes[] = 'template-front-page';



	$classes[] = 'template-sparkle';

	return $classes;



}
add_filter( 'body_class', 'nursery_body_class' );



/**



 * Adjusts content_width value for video post formats and attachment templates.



 *



 * @since Twenty Thirteen 1.0



 *



 * @return void



 */



function nursery_content_width() {



	global $content_width;







	if ( is_attachment() )



		$content_width = 724;



	elseif ( has_post_format( 'audio' ) )



		$content_width = 484;



}



add_action( 'template_redirect', 'nursery_content_width' );







/**



 * Add postMessage support for site title and description for the Customizer.



 *



 * @since Twenty Thirteen 1.0



 *



 * @param WP_Customize_Manager $wp_customize Customizer object.



 * @return void



 */



function nursery_customize_register( $wp_customize ) {



	$wp_customize->get_setting( 'blogname' )->transport         = 'postMessage';



	$wp_customize->get_setting( 'blogdescription' )->transport  = 'postMessage';



	$wp_customize->get_setting( 'header_textcolor' )->transport = 'postMessage';



}



add_action( 'customize_register', 'nursery_customize_register' );



/**
 * Remove customized section
 * Header Image and Static Front Page are removed because they are used in Nusery Theme
 */
function nursery_customize_remover( $wp_customize ) {
	$wp_customize->remove_section( 'header_image' );
	$wp_customize->remove_section( 'static_front_page' );
}
add_action( 'customize_register', 'nursery_customize_remover' );




/**



 * Binds JavaScript handlers to make Customizer preview reload changes



 * asynchronously.



 *



 * @since Twenty Thirteen 1.0



 */



function nursery_customize_preview_js() {



	wp_enqueue_script( 'nursery-customizer', get_template_directory_uri() . '/js/theme-customizer.js', array( 'customize-preview' ), '20130226', true );



}



add_action( 'customize_preview_init', 'nursery_customize_preview_js' );







/* Filter to add author credit to Infinite Scroll footer */



function nursery_footer_credits( $credit ) {



	$credit = sprintf( __( 'Develop by %1$s | %2$s | ', 'nursery' ), '<a href="http://designpromote.co.uk/" rel="designer">Website Design Edinburgh</a>', '<a href="' . site_url('/wp-login.php') . '" rel="login">Admin</a>' );



	//return $credit;



	echo $credit;



}



//add_filter( 'infinite_scroll_credit', 'shop_footer_credits' );



add_action( 'nursery_credits', 'nursery_footer_credits' );







//disable break line into <p> in pages and posts



remove_filter ('the_content', 'wpautop');







/**



 * new Recent_Posts widget class based on



 * Recent_Posts widget class



 *



 * @since 2.8.0



 */



class nursery_WP_Widget_Recent_Posts extends WP_Widget {







	function __construct() {



		$widget_ops = array('classname' => 'zn_widget_recent_entries', 'description' => __( "The most recent posts with images on your site") );



		parent::__construct('zn_recent-posts', __('Recent Posts and Images'), $widget_ops);



		$this->alt_option_name = 'widget_recent_entries';







		add_action( 'save_post', array($this, 'flush_widget_cache') );



		add_action( 'deleted_post', array($this, 'flush_widget_cache') );



		add_action( 'switch_theme', array($this, 'flush_widget_cache') );



	}







	function widget($args, $instance) {



		$cache = wp_cache_get('widget_recent_posts', 'widget');







		if ( !is_array($cache) )



			$cache = array();







		if ( ! isset( $args['widget_id'] ) )



			$args['widget_id'] = $this->id;







		if ( isset( $cache[ $args['widget_id'] ] ) ) {



			echo $cache[ $args['widget_id'] ];



			return;



		}







		ob_start();



		extract($args);







		$title = ( ! empty( $instance['title'] ) ) ? $instance['title'] : __( '' );



		$title = apply_filters( 'widget_title', $title, $instance, $this->id_base );



		$number = ( ! empty( $instance['number'] ) ) ? absint( $instance['number'] ) : 10;



		if ( ! $number )



 			$number = 10;



		$show_date = isset( $instance['show_date'] ) ? $instance['show_date'] : false;







		$r = new WP_Query( apply_filters( 'widget_posts_args', array( 'posts_per_page' => $number, 'no_found_rows' => true, 'post_status' => 'publish', 'ignore_sticky_posts' => true ) ) );



		



		if( $r->have_posts() ) :



			echo $before_widget;



			if ( $title ) echo $before_title . $title . $after_title; 



		echo '<ul>';



		while ( $r->have_posts() ) : $r->the_post();



			echo '<li>';







				if ( has_post_thumbnail() && ! post_password_required() ) :



					echo '<div class="entry-thumbnail">';



					the_post_thumbnail( );



					echo '</div>';



				else:







				$post = get_post(); //global $post;



				//var_dump( $post -> post_content );



				$attachment_args = array(



					'numberposts' => 1,



					'order' => 'ASC',



					'post_mime_type' => 'image',



					'post_parent' => $post->ID,



					'post_status' => null,



					'post_type' => 'attachment',



				);



				$attachments = get_children( $attachment_args );



				//var_dump( count( $attachments ) );



				if ( count( $attachments ) > 0 ) {



					foreach ( $attachments as $attachment ) {



						//$image_attributes = wp_get_attachment_image_src( $attachment->ID, 'thumbnail' )  ? wp_get_attachment_image_src( $attachment->ID, 'thumbnail' ) : wp_get_attachment_image_src( $attachment->ID, 'full' );



						//var_dump( wp_get_attachment_image( $attachment->ID ) );



						echo '<div class="entry-thumbnail">';



						echo '<img src="' . wp_get_attachment_thumb_url( $attachment->ID ) . '" class="current">';



						echo '</div>';



					}



				}



				endif; 



?>



			



				<a href="<?php the_permalink() ?>" title="<?php echo esc_attr( get_the_title() ? get_the_title() : get_the_ID() ); ?>"><?php if ( get_the_title() ) the_title(); else the_ID(); ?></a>



			<?php if ( $show_date ) : ?>



				<div class="post-date"><?php echo get_the_date(); ?></div>



			<?php endif; ?>



			</li>



		<?php endwhile; ?>



		</ul>



		<?php echo $after_widget; ?>



<?php



		// Reset the global $the_post as this query will have stomped on it



		wp_reset_postdata();







		endif;







		$cache[$args['widget_id']] = ob_get_flush();



		wp_cache_set('widget_recent_posts', $cache, 'widget');



	}







	function update( $new_instance, $old_instance ) {



		$instance = $old_instance;



		$instance['title'] = strip_tags($new_instance['title']);



		$instance['number'] = (int) $new_instance['number'];



		$instance['show_date'] = isset( $new_instance['show_date'] ) ? (bool) $new_instance['show_date'] : false;



		$this->flush_widget_cache();







		$alloptions = wp_cache_get( 'alloptions', 'options' );



		if ( isset($alloptions['widget_recent_entries']) )



			delete_option('widget_recent_entries');







		return $instance;



	}







	function flush_widget_cache() {



		wp_cache_delete('widget_recent_posts', 'widget');



	}







	function form( $instance ) {



		$title     = isset( $instance['title'] ) ? esc_attr( $instance['title'] ) : '';



		$number    = isset( $instance['number'] ) ? absint( $instance['number'] ) : 5;



		$show_date = isset( $instance['show_date'] ) ? (bool) $instance['show_date'] : false;



?>



		<p><label for="<?php echo $this->get_field_id( 'title' ); ?>"><?php _e( 'Title:' ); ?></label>



		<input class="widefat" id="<?php echo $this->get_field_id( 'title' ); ?>" name="<?php echo $this->get_field_name( 'title' ); ?>" type="text" value="<?php echo $title; ?>" /></p>







		<p><label for="<?php echo $this->get_field_id( 'number' ); ?>"><?php _e( 'Number of posts to show:' ); ?></label>



		<input id="<?php echo $this->get_field_id( 'number' ); ?>" name="<?php echo $this->get_field_name( 'number' ); ?>" type="text" value="<?php echo $number; ?>" size="3" /></p>







		<p><input class="checkbox" type="checkbox" <?php checked( $show_date ); ?> id="<?php echo $this->get_field_id( 'show_date' ); ?>" name="<?php echo $this->get_field_name( 'show_date' ); ?>" />



		<label for="<?php echo $this->get_field_id( 'show_date' ); ?>"><?php _e( 'Display post date?' ); ?></label></p>



<?php



	}



}



add_action( 'widgets_init', 'nursery_register_widgets');



function nursery_register_widgets() {



	register_widget( 'nursery_WP_Widget_Recent_Posts' );



}



function nursery_hide_title( $title, $id ) {



	$hide_titles = array( 'Gallery', 'Blogs', 'Contact' ); 



	if( ( is_front_page() || is_home() ) && in_array( $title, $hide_titles ) ) return '';



	else return $title;



}



//add_filter( 'the_title', 'nursery_hide_title', 10, 2 );







// Change based on PHP code by Chirp Internet: www.chirp.com.au



function nursery_truncate( $content, $limit, $break = '.', $more_link_text = null ) {



	// return with no change if string is shorter than $limit 



	$length = strlen( $content );



	if( $length <= $limit) return $content; 



	



	// is $break present between $limit and the end of the string? 



	if( false !== ( $breakpoint = strrpos( $content, $break, -( $length - $limit ) ) ) ) {



		if( $breakpoint < $length - 1 )



			$content = substr( $content, 0, $breakpoint + 1 ) . ' ' . $more_link_text;



	}



	return $content;	



}







function nursery_front_page_content( $content ) {



	if( 1 != preg_match( '/more.../', $content ) ) :



		$more_link_text = ' <a href="' . get_permalink() . '" class="more-link">more<span class="meta-nav">&rarr;</span></a>'; 



		$content = nursery_truncate( $content, 350, '.', $more_link_text );



		//$content = $content . ' <a href="' . get_permalink() . " class=\"more-link\">more<span class=\"meta-nav\">&rarr;</span></a>";



	endif;



	return $content;



}



//add_filter( 'the_content', 'nursery_front_page_content' );



//is_home or is_front_page doesn't work in functions.php because At the time functions.php is included during bootup, WordPress has no idea on the contents of the query, and doesn't know the nature of the page. is_home will return false.







function nursery_front_page_gallery( $id = null, $content = null ) {



	$args = array(



		'numberposts' => 1,



		'order' => 'ASC',



		'post_mime_type' => 'image',



		'post_parent' => $id,



		'post_status' => null,



		'post_type' => 'attachment',



	);



	$number_attachment = 0; 



	$attachments = get_children( $args );



	if ( $attachments ) {



		$number_attachment = 1; // found one image attached



		foreach ( $attachments as $attachment ) {



			//$number_attachment++;



			// if( $number_attachment++ > 1 ) break;



			//$image_attributes = wp_get_attachment_image_src( $attachment->ID, 'thumbnail' )  ? wp_get_attachment_image_src( $attachment->ID, 'thumbnail' ) : wp_get_attachment_image_src( $attachment->ID, 'full' );



			echo '<img src="' . wp_get_attachment_thumb_url( $attachment->ID ) . '" class="current">';



		}



	}



	/* show up to 4 images from [gallery] and attachment */



	$regex_pattern = get_shortcode_regex();



	//var_dump( $regex_pattern );



	preg_match ('/'.$regex_pattern.'/s', $content, $regex_matches);



	//var_dump( $regex_matches[3] ); 



	



	if( $regex_matches ) :



	preg_match_all( "/[0-9]+/", $regex_matches[3], $matches );



	//var_dump( $matches );



	//print_r( $matches );



	$number_image = 0;



	$total_image = 4 - $number_attachment;



	foreach( $matches as $vals ) {



		foreach( $vals as $val ) {



			if( $number_image++ >= $total_image ) break; 



			//echo wp_get_attachment_image( $val, array( 160, 160 ) );



			echo wp_get_attachment_image( $val );



		}



	}



	endif;



}







?>