<?php
/**
 * Add inline CSS for styles handled by the Theme customizer
 *
 * @package Short News
 * @since Short News 1.0
 */


/**
 * Get Contrast
 */
function short_news_get_brightness($hex) {
	// returns brightness value from 0 to 255
	// strip off any leading #
	$hex = str_replace('#', '', $hex);

	$c_r = hexdec(substr($hex, 0, 2));
	$c_g = hexdec(substr($hex, 2, 2));
	$c_b = hexdec(substr($hex, 4, 2));

	return (($c_r * 299) + ($c_g * 587) + ($c_b * 114)) / 1000;
}


/**
 * Hex 2 rgba
 *
 * Convert hexadecimal color to rgba
 */

if ( !function_exists( 'short_news_hex2rgba' ) ):
	function short_news_hex2rgba( $color, $opacity = false ) {
		$default = 'rgb(0,0,0)';

		//Return default if no color provided
		if ( empty( $color ) )
			return $default;

		//Sanitize $color if "#" is provided
		if ( $color[0] == '#' ) {
			$color = substr( $color, 1 );
		}

		//Check if color has 6 or 3 characters and get values
		if ( strlen( $color ) == 6 ) {
			$hex = array( $color[0] . $color[1], $color[2] . $color[3], $color[4] . $color[5] );
		} elseif ( strlen( $color ) == 3 ) {
			$hex = array( $color[0] . $color[0], $color[1] . $color[1], $color[2] . $color[2] );
		} else {
			return $default;
		}

		//Convert hexadec to rgb
		$rgb =  array_map( 'hexdec', $hex );

		//Check if opacity is set(rgba or rgb)
		if ( $opacity ) {
			if ( abs( $opacity ) > 1 ) { $opacity = 1.0; }
			$output = 'rgba('.implode( ",", $rgb ).','.$opacity.')';
		} else {
			$output = 'rgb('.implode( ",", $rgb ).')';
		}

		//Return rgb(a) color string
		return $output;
	}
endif;


/**
 * Set the custom CSS via Customizer options.
 */
function short_news_custom_css() {
	$logo_size = esc_attr( get_theme_mod('logo_size', 'resize') );
	$logo_width = esc_attr( get_theme_mod('logo_width') );

	$header_image_padding = esc_attr(get_theme_mod('header_image_padding') );
	$header_image_dark_overlay = esc_attr( get_theme_mod('header_image_dark_overlay') );
	$header_image_v_align = esc_attr( get_theme_mod('header_image_v_align') );

	$accent_color = esc_attr( get_theme_mod('accent_color') );
	$site_tagline_color = esc_attr( get_theme_mod('site_tagline_color') );

	$theme_css = "";

	// Custom Logo
	if ( 'fullwidth' == $logo_size ) {
		$theme_css .= ".site-logo {max-width: 100%;}";
	} else {
		if ( ! empty($logo_width) ) {
			$theme_css .= "
			@media screen and (min-width: 600px) {
			.site-logo {max-width: {$logo_width}px;}
			}";
		}
	}

	// Header Image
	if( get_header_image() ) {
		$header_image_url = esc_url( get_header_image() );
		$theme_css .= ".custom-header {background-image: url({$header_image_url});}";

		if ( ! empty($header_image_padding) ) {
			$theme_css .= ".custom-header {padding-top: {$header_image_padding}px;padding-bottom: {$header_image_padding}px;}";
		}

		if ( ! empty($header_image_v_align) ) {
			$theme_css .= ".custom-header {background-position: center {$header_image_v_align};}";
		}

		if ( ! empty($header_image_dark_overlay) ) {
			$theme_css .= "
			.custom-header:before {
			content: '';
			position: absolute;
			left: 0;
			top: 0;
			width: 100%;
			height: 100%;
			background-color: #000;
			opacity: 0.{$header_image_dark_overlay};
			}";
		}
	}

	// Site Tagline Color
	if ( ! empty($site_tagline_color) ) {
		$theme_css .= ".site-description {color: {$site_tagline_color};}";
	}

	// Accent Color
	if ( ! empty($accent_color) ) {
		$theme_css .= "
		a, .site-title a:hover, .entry-title a:hover, .main-navigation ul ul li:hover > a, .widget a:hover, .widget_recent_comments a,
		blockquote:before, .cat-links a, .comment-metadata .comment-edit-link, .standard-post .read-more:hover,
		.posts-navigation a:hover, .post-navigation a:hover .meta-nav, .post-navigation a:hover .post-title,
		.author-link a:hover, .entry-content .has-accent-color {
		color: {$accent_color};
		}
		button, input[type='button'], input[type='reset'], input[type='submit'], .main-navigation > ul > li.current-menu-item:after,
		.sidebar .widget_tag_cloud a:hover, .single .cat-links a, .entry-meta-top .comments-link > span:hover, .entry-meta-top .comments-link > a:hover,
		.standard-post .read-more:after, .newsticker .news-dot, .pagination .current, .pagination .page-numbers:hover,
		.featured-post-header .cat-links a:hover, .post-edit-link, .reply a, #sb_instagram .sbi_follow_btn a, .entry-content .has-accent-background-color {
		background-color: {$accent_color};
		}
		.entry-meta-top .comments-link > a:hover:after, .entry-meta-top .comments-link > span:hover:after {border-top-color: {$accent_color};}";
		if ( short_news_get_brightness($accent_color) > 160 ) {
			$theme_css .= "
			button, input[type='button'], input[type='reset'], input[type='submit'], .sidebar .widget_tag_cloud a:hover, .post-edit-link, .reply a, .single .cat-links a, #sb_instagram .sbi_follow_btn a {
			color: rgba(0,0,0,.7);
			}";
		}
	}

	return $theme_css;
}


/**
 * Enqueues front-end CSS for styles handled by the Theme customizer
 */
function short_news_add_inline_style() {
	$css = short_news_custom_css();
	if ( ! empty( $css ) ) {
		wp_add_inline_style( 'short-news-style', $css );
	}
}
add_action( 'wp_enqueue_scripts', 'short_news_add_inline_style' );


/**
  * Set the custom CSS via Customizer options.
  */
function short_news_editor_css() {
	$accent_color = esc_attr( get_theme_mod('accent_color') );

	$editor_css = "";

	// Accent Color
	if ( ! empty( $accent_color ) ) {
		$editor_css .= "
		.editor-styles-wrapper :where(.wp-block a),
		.editor-styles-wrapper :where(.wp-block a:hover),
		.wp-block-freeform.block-library-rich-text__tinymce a
		.wp-block-freeform.block-library-rich-text__tinymce a:hover,
		.editor-styles-wrapper .wp-block-quote:before,
		.wp-block-freeform.block-library-rich-text__tinymce blockquote:before {
			color: {$accent_color};
		}";
		$editor_css.= ".editor-styles-wrapper .wp-block-search__button {background-color: {$accent_color};}";
	}

	return $editor_css;
}


/**
 * Enqueue styles for the block-based editor.
 */
function short_news_block_editor_styles() {
	// Add Google fonts.
	wp_enqueue_style( 'short-news-fonts', short_news_fonts_url(), array(), null );
	// Add Editor style.
	wp_enqueue_style( 'short-news-block-editor-style', get_theme_file_uri( '/inc/css/editor-blocks.css' ) );
	wp_add_inline_style( 'short-news-block-editor-style', short_news_editor_css() );
}
add_action( 'enqueue_block_editor_assets', 'short_news_block_editor_styles' );
