<?php
/**
 * Custom page walker for this theme.
 *
 * @package Responsive Landing Page
 */

if (!class_exists('Responsive_Landing_Page_Walker_Page')) {
    /**
     * CUSTOM PAGE WALKER
     * A custom walker for pages.
     */
    class Responsive_Landing_Page_Walker_Page extends Walker_Page
    {

        /**
         * Outputs the beginning of the current element in the tree.
         *
         * @param string $responsive_landing_page_output Used to append additional content. Passed by reference.
         * @param WP_Post $page Page data object.
         * @param int $responsive_landing_page_depth Optional. Depth of page. Used for padding. Default 0.
         * @param array $responsive_landing_page_args Optional. Array of arguments. Default empty array.
         * @param int $current_page Optional. Page ID. Default 0.
         * @since 2.1.0
         *
         * @see Walker::start_el()
         */

        public function start_lvl( &$responsive_landing_page_output, $responsive_landing_page_depth = 0, $responsive_landing_page_args = array() ) {
            $responsive_landing_page_indent  = str_repeat( "\t", $responsive_landing_page_depth );
            $responsive_landing_page_output .= "$responsive_landing_page_indent<ul class='sub-menu'>\n";
        }

        public function start_el(&$responsive_landing_page_output, $page, $responsive_landing_page_depth = 0, $responsive_landing_page_args = array(), $current_page = 0)
        {

            if (isset($responsive_landing_page_args['item_spacing']) && 'preserve' === $responsive_landing_page_args['item_spacing']) {
                $t = "\t";
                $n = "\n";
            } else {
                $t = '';
                $n = '';
            }
            if ($responsive_landing_page_depth) {
                $responsive_landing_page_indent = str_repeat($t, $responsive_landing_page_depth);
            } else {
                $responsive_landing_page_indent = '';
            }

            $responsive_landing_page_css_class = array('page_item', 'page-item-' . $page->ID);

            if (isset($responsive_landing_page_args['pages_with_children'][$page->ID])) {
                $responsive_landing_page_css_class[] = 'page_item_has_children';
            }

            if (!empty($current_page)) {
                $_current_page = get_post($current_page);
                if ($_current_page && in_array($page->ID, $_current_page->ancestors, true)) {
                    $responsive_landing_page_css_class[] = 'current_page_ancestor';
                }
                if ($page->ID === $current_page) {
                    $responsive_landing_page_css_class[] = 'current_page_item';
                } elseif ($_current_page && $page->ID === $_current_page->post_parent) {
                    $responsive_landing_page_css_class[] = 'current_page_parent';
                }
            } elseif (get_option('page_for_posts') === $page->ID) {
                $responsive_landing_page_css_class[] = 'current_page_parent';
            }

            /** This filter is documented in wp-includes/class-walker-page.php */
            $responsive_landing_page_css_classes = implode(' ', apply_filters('page_css_class', $responsive_landing_page_css_class, $page, $responsive_landing_page_depth, $responsive_landing_page_args, $current_page));
            $responsive_landing_page_css_classes = $responsive_landing_page_css_classes ? ' class="' . esc_attr($responsive_landing_page_css_classes) . '"' : '';

            if ('' === $page->post_title) {
                /* translators: %d: ID of a post. */
                $page->post_title = sprintf(__('#%d (no title)', 'responsive-landing-page'), $page->ID);
            }

            $responsive_landing_page_args['link_before'] = empty($responsive_landing_page_args['link_before']) ? '' : $responsive_landing_page_args['link_before'];
            $responsive_landing_page_args['link_after'] = empty($responsive_landing_page_args['link_after']) ? '' : $responsive_landing_page_args['link_after'];

            $responsive_landing_page_atts = array();
            $responsive_landing_page_atts['href'] = get_permalink($page->ID);
            $responsive_landing_page_atts['aria-current'] = ($page->ID === $current_page) ? 'page' : '';

            /** This filter is documented in wp-includes/class-walker-page.php */
            $responsive_landing_page_atts = apply_filters('page_menu_link_attributes', $responsive_landing_page_atts, $page, $responsive_landing_page_depth, $responsive_landing_page_args, $current_page);

            $responsive_landing_page_attributes = '';
            foreach ($responsive_landing_page_atts as $attr => $responsive_landing_page_value) {
                if (!empty($responsive_landing_page_value)) {
                    $responsive_landing_page_value = ('href' === $attr) ? esc_url($responsive_landing_page_value) : esc_attr($responsive_landing_page_value);
                    $responsive_landing_page_attributes .= ' ' . $attr . '="' . $responsive_landing_page_value . '"';
                }
            }

            $responsive_landing_page_args['list_item_before'] = '';
            $responsive_landing_page_args['list_item_after'] = '';
            $responsive_landing_page_args['icon_rennder'] = '';
            // Wrap the link in a div and append a sub menu toggle.
            if (isset($responsive_landing_page_args['show_toggles']) && true === $responsive_landing_page_args['show_toggles']) {
                // Wrap the menu item link contents in a div, used for positioning.
                $responsive_landing_page_args['list_item_after'] = '';
            }


            // Add icons to menu items with children.
            if (isset($responsive_landing_page_args['show_sub_menu_icons']) && true === $responsive_landing_page_args['show_sub_menu_icons']) {
                if (isset($responsive_landing_page_args['pages_with_children'][$page->ID])) {
                    $responsive_landing_page_args['icon_rennder'] = '';
                }
            }

            // Add icons to menu items with children.
            if (isset($responsive_landing_page_args['show_toggles']) && true === $responsive_landing_page_args['show_toggles']) {
                if (isset($responsive_landing_page_args['pages_with_children'][$page->ID])) {

                    $toggle_target_string = '.page_item.page-item-' . $page->ID . ' > .sub-menu';

                    $responsive_landing_page_args['list_item_after'] = '<button type="button" class="theme-aria-button submenu-toggle" data-toggle-target="' . $toggle_target_string . '" data-toggle-type="slidetoggle" data-toggle-duration="250"><span class="btn__content" tabindex="-1"><span class="screen-reader-text">' . __( 'Show sub menu', 'responsive-landing-page' ) . '</span>' . responsive_landing_page_get_theme_svg( 'chevron-down' ) . '</span></button>';
                }
            }

            if (isset($responsive_landing_page_args['show_toggles']) && true === $responsive_landing_page_args['show_toggles']) {

                $responsive_landing_page_output .= $responsive_landing_page_indent . sprintf(
                        '<li%s>%s%s<a%s>%s%s%s</a>%s%s',
                        $responsive_landing_page_css_classes,
                        '<div class="submenu-wrapper">',
                        $responsive_landing_page_args['list_item_before'],
                        $responsive_landing_page_attributes,
                        $responsive_landing_page_args['link_before'],
                        /** This filter is documented in wp-includes/post-template.php */
                        apply_filters('the_title', $page->post_title, $page->ID),
                        $responsive_landing_page_args['link_after'],
                        $responsive_landing_page_args['list_item_after'],
                        '</div>'
                    );

            }else{

                $responsive_landing_page_output .= $responsive_landing_page_indent . sprintf(
                        '<li%s>%s<a%s>%s%s%s%s</a>%s',
                        $responsive_landing_page_css_classes,
                        $responsive_landing_page_args['list_item_before'],
                        $responsive_landing_page_attributes,
                        $responsive_landing_page_args['link_before'],
                        /** This filter is documented in wp-includes/post-template.php */
                        apply_filters('the_title', $page->post_title, $page->ID),
                        $responsive_landing_page_args['icon_rennder'],
                        $responsive_landing_page_args['link_after'],
                        $responsive_landing_page_args['list_item_after']
                    );

            }

            if (!empty($responsive_landing_page_args['show_date'])) {
                if ('modified' === $responsive_landing_page_args['show_date']) {
                    $responsive_landing_page_time = $page->post_modified;
                } else {
                    $responsive_landing_page_time = $page->post_date;
                }

                $responsive_landing_page_date_format = empty($responsive_landing_page_args['date_format']) ? '' : $responsive_landing_page_args['date_format'];
                $responsive_landing_page_output .= ' ' . mysql2date($responsive_landing_page_date_format, $responsive_landing_page_time);
            }
        }
    }
}