<?php
/**
 * Custom Functions
 * @package Responsive Landing Page
 * @since 1.0.0
 */

if( !function_exists('responsive_landing_page_site_logo') ):

    /**
     * Logo & Description
     */
    /**
     * Displays the site logo, either text or image.
     *
     * @param array $responsive_landing_page_args Arguments for displaying the site logo either as an image or text.
     * @param boolean $responsive_landing_page_echo Echo or return the HTML.
     *
     * @return string $responsive_landing_page_html Compiled HTML based on our arguments.
     */
    function responsive_landing_page_site_logo( $responsive_landing_page_args = array(), $responsive_landing_page_echo = true ){
        $responsive_landing_page_logo = get_custom_logo();
        $responsive_landing_page_site_title = get_bloginfo('name');
        $responsive_landing_page_contents = '';
        $responsive_landing_page_classname = '';
        $responsive_landing_page_defaults = array(
            'logo' => '%1$s<span class="screen-reader-text">%2$s</span>',
            'logo_class' => 'site-logo site-branding',
            'title' => '<a href="%1$s" class="custom-logo-name">%2$s</a>',
            'title_class' => 'site-title',
            'home_wrap' => '<h1 class="%1$s">%2$s</h1>',
            'single_wrap' => '<div class="%1$s">%2$s</div>',
            'condition' => (is_front_page() || is_home()) && !is_page(),
        );
        $responsive_landing_page_args = wp_parse_args($responsive_landing_page_args, $responsive_landing_page_defaults);
        /**
         * Filters the arguments for `responsive_landing_page_site_logo()`.
         *
         * @param array $responsive_landing_page_args Parsed arguments.
         * @param array $responsive_landing_page_defaults Function's default arguments.
         */
        $responsive_landing_page_args = apply_filters('responsive_landing_page_site_logo_args', $responsive_landing_page_args, $responsive_landing_page_defaults);
        
        $responsive_landing_page_show_logo  = get_theme_mod('responsive_landing_page_display_logo', false);
        $responsive_landing_page_show_title = get_theme_mod('responsive_landing_page_display_title', true);

        if ( has_custom_logo() && $responsive_landing_page_show_logo ) {
            $responsive_landing_page_contents .= sprintf($responsive_landing_page_args['logo'], $responsive_landing_page_logo, esc_html($responsive_landing_page_site_title));
            $responsive_landing_page_classname = $responsive_landing_page_args['logo_class'];
        }

        if ( $responsive_landing_page_show_title ) {
            $responsive_landing_page_contents .= sprintf($responsive_landing_page_args['title'], esc_url(get_home_url(null, '/')), esc_html($responsive_landing_page_site_title));
            // If logo isn't shown, fallback to title class for wrapper.
            if ( !$responsive_landing_page_show_logo ) {
                $responsive_landing_page_classname = $responsive_landing_page_args['title_class'];
            }
        }

        // If nothing is shown (logo or title both disabled), exit early
        if ( empty($responsive_landing_page_contents) ) {
            return;
        }

        $responsive_landing_page_wrap = $responsive_landing_page_args['condition'] ? 'home_wrap' : 'single_wrap';
        // $responsive_landing_page_wrap = 'home_wrap';
        $responsive_landing_page_html = sprintf($responsive_landing_page_args[$responsive_landing_page_wrap], $responsive_landing_page_classname, $responsive_landing_page_contents);
        /**
         * Filters the arguments for `responsive_landing_page_site_logo()`.
         *
         * @param string $responsive_landing_page_html Compiled html based on our arguments.
         * @param array $responsive_landing_page_args Parsed arguments.
         * @param string $responsive_landing_page_classname Class name based on current view, home or single.
         * @param string $responsive_landing_page_contents HTML for site title or logo.
         */
        $responsive_landing_page_html = apply_filters('responsive_landing_page_site_logo', $responsive_landing_page_html, $responsive_landing_page_args, $responsive_landing_page_classname, $responsive_landing_page_contents);
        if (!$responsive_landing_page_echo) {
            return $responsive_landing_page_html;
        }
        echo $responsive_landing_page_html; //phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped

    }

endif;

if( !function_exists('responsive_landing_page_site_description') ):

    /**
     * Displays the site description.
     *
     * @param boolean $responsive_landing_page_echo Echo or return the html.
     *
     * @return string $responsive_landing_page_html The HTML to display.
     */
    function responsive_landing_page_site_description($responsive_landing_page_echo = true){

        if ( get_theme_mod('responsive_landing_page_display_header_text', false) == true ) :
        $responsive_landing_page_description = get_bloginfo('description');
        if (!$responsive_landing_page_description) {
            return;
        }
        $responsive_landing_page_wrapper = '<div class="site-description"><span>%s</span></div><!-- .site-description -->';
        $responsive_landing_page_html = sprintf($responsive_landing_page_wrapper, esc_html($responsive_landing_page_description));
        /**
         * Filters the html for the site description.
         *
         * @param string $responsive_landing_page_html The HTML to display.
         * @param string $responsive_landing_page_description Site description via `bloginfo()`.
         * @param string $responsive_landing_page_wrapper The format used in case you want to reuse it in a `sprintf()`.
         * @since 1.0.0
         *
         */
        $responsive_landing_page_html = apply_filters('responsive_landing_page_site_description', $responsive_landing_page_html, $responsive_landing_page_description, $responsive_landing_page_wrapper);
        if (!$responsive_landing_page_echo) {
            return $responsive_landing_page_html;
        }
        echo $responsive_landing_page_html; //phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped
        endif;
    }

endif;

if( !function_exists('responsive_landing_page_posted_on') ):

    /**
     * Prints HTML with meta information for the current post-date/time.
     */
    function responsive_landing_page_posted_on( $responsive_landing_page_icon = true, $responsive_landing_page_animation_class = '' ){

        $responsive_landing_page_default = responsive_landing_page_get_default_theme_options();
        $responsive_landing_page_post_date = absint( get_theme_mod( 'responsive_landing_page_post_date',$responsive_landing_page_default['responsive_landing_page_post_date'] ) );

        if( $responsive_landing_page_post_date ){

            $responsive_landing_page_time_string = '<time class="entry-date published updated" datetime="%1$s">%2$s</time>';
            if (get_the_time('U') !== get_the_modified_time('U')) {
                $responsive_landing_page_time_string = '<time class="entry-date published" datetime="%1$s">%2$s</time><time class="updated" datetime="%3$s">%4$s</time>';
            }

            $responsive_landing_page_time_string = sprintf($responsive_landing_page_time_string,
                esc_attr(get_the_date(DATE_W3C)),
                esc_html(get_the_date()),
                esc_attr(get_the_modified_date(DATE_W3C)),
                esc_html(get_the_modified_date())
            );

            $responsive_landing_page_year = get_the_date('Y');
            $responsive_landing_page_month = get_the_date('m');
            $responsive_landing_page_day = get_the_date('d');
            $responsive_landing_page_link = get_day_link($responsive_landing_page_year, $responsive_landing_page_month, $responsive_landing_page_day);

            $responsive_landing_page_posted_on = '<a href="' . esc_url($responsive_landing_page_link) . '" rel="bookmark">' . $responsive_landing_page_time_string . '</a>';

            echo '<div class="entry-meta-item entry-meta-date">';
            echo '<div class="entry-meta-wrapper '.esc_attr( $responsive_landing_page_animation_class ).'">';

            if( $responsive_landing_page_icon ){

                echo '<span class="entry-meta-icon calendar-icon"> ';
                responsive_landing_page_the_theme_svg('calendar');
                echo '</span>';

            }

            echo '<span class="posted-on">' . $responsive_landing_page_posted_on . '</span>'; // phpcs:ignore Standard.Category.SniffName.ErrorCode
            echo '</div>';
            echo '</div>';

        }

    }

endif;

if( !function_exists('responsive_landing_page_posted_by') ) :

    /**
     * Prints HTML with meta information for the current author.
     */
    function responsive_landing_page_posted_by( $responsive_landing_page_icon = true, $responsive_landing_page_animation_class = '' ){   

        $responsive_landing_page_default = responsive_landing_page_get_default_theme_options();
        $responsive_landing_page_post_author = absint( get_theme_mod( 'responsive_landing_page_post_author',$responsive_landing_page_default['responsive_landing_page_post_author'] ) );

        if( $responsive_landing_page_post_author ){

            echo '<div class="entry-meta-item entry-meta-author">';
            echo '<div class="entry-meta-wrapper '.esc_attr( $responsive_landing_page_animation_class ).'">';

            if( $responsive_landing_page_icon ){
            
                echo '<span class="entry-meta-icon author-icon"> ';
                responsive_landing_page_the_theme_svg('user');
                echo '</span>';
                
            }

            $responsive_landing_page_byline = '<span class="author vcard"><a class="url fn n" href="' . esc_url( get_author_posts_url( get_the_author_meta('ID') ) ) . '">' . esc_html(get_the_author()) . '</a></span>';
            echo '<span class="byline"> ' . $responsive_landing_page_byline . '</span>'; // phpcs:ignore Standard.Category.SniffName.ErrorCode
            echo '</div>';
            echo '</div>';

        }

    }

endif;


if( !function_exists('responsive_landing_page_posted_by_avatar') ) :

    /**
     * Prints HTML with meta information for the current author.
     */
    function responsive_landing_page_posted_by_avatar( $responsive_landing_page_date = false ){

        $responsive_landing_page_default = responsive_landing_page_get_default_theme_options();
        $responsive_landing_page_post_author = absint( get_theme_mod( 'responsive_landing_page_post_author',$responsive_landing_page_default['responsive_landing_page_post_author'] ) );

        if( $responsive_landing_page_post_author ){



            echo '<div class="entry-meta-left">';
            echo '<div class="entry-meta-item entry-meta-avatar">';
            echo wp_kses_post( get_avatar( get_the_author_meta( 'ID' ) ) );
            echo '</div>';
            echo '</div>';

            echo '<div class="entry-meta-right">';

            $responsive_landing_page_byline = '<span class="author vcard"><a class="url fn n" href="' . esc_url( get_author_posts_url( get_the_author_meta('ID') ) ) . '">' . esc_html(get_the_author()) . '</a></span>';

            echo '<div class="entry-meta-item entry-meta-byline"> ' . $responsive_landing_page_byline . '</div>';

            if( $responsive_landing_page_date ){
                responsive_landing_page_posted_on($responsive_landing_page_icon = false);
            }
            echo '</div>';

        }

    }

endif;

if( !function_exists('responsive_landing_page_entry_footer') ):

    /**
     * Prints HTML with meta information for the categories, tags and comments.
     */
    function responsive_landing_page_entry_footer( $responsive_landing_page_cats = true, $responsive_landing_page_tags = true, $responsive_landing_page_edits = true){   

        $responsive_landing_page_default = responsive_landing_page_get_default_theme_options();
        $responsive_landing_page_post_category = absint( get_theme_mod( 'responsive_landing_page_post_category',$responsive_landing_page_default['responsive_landing_page_post_category'] ) );
        $responsive_landing_page_post_tags = absint( get_theme_mod( 'responsive_landing_page_post_tags',$responsive_landing_page_default['responsive_landing_page_post_tags'] ) );

        // Hide category and tag text for pages.
        if ('post' === get_post_type()) {

            if( $responsive_landing_page_cats && $responsive_landing_page_post_category ){

                /* translators: used between list items, there is a space after the comma */
                $responsive_landing_page_categories = get_the_category();
                if ($responsive_landing_page_categories) {
                    echo '<div class="entry-meta-item entry-meta-categories">';
                    echo '<div class="entry-meta-wrapper">';
                
                    /* translators: 1: list of categories. */
                    echo '<span class="cat-links">';
                    foreach( $responsive_landing_page_categories as $responsive_landing_page_category ){

                        $responsive_landing_page_cat_name = $responsive_landing_page_category->name;
                        $responsive_landing_page_cat_slug = $responsive_landing_page_category->slug;
                        $responsive_landing_page_cat_url = get_category_link( $responsive_landing_page_category->term_id );
                        ?>

                        <a class="twp_cat_<?php echo esc_attr( $responsive_landing_page_cat_slug ); ?>" href="<?php echo esc_url( $responsive_landing_page_cat_url ); ?>" rel="category tag"><?php echo esc_html( $responsive_landing_page_cat_name ); ?></a>

                    <?php }
                    echo '</span>'; // phpcs:ignore Standard.Category.SniffName.ErrorCode
                    echo '</div>';
                    echo '</div>';
                }

            }

            if( $responsive_landing_page_tags && $responsive_landing_page_post_tags ){
                /* translators: used between list items, there is a space after the comma */
                $responsive_landing_page_tags_list = get_the_tag_list('', esc_html_x(', ', 'list item separator', 'responsive-landing-page'));
                if( $responsive_landing_page_tags_list ){

                    echo '<div class="entry-meta-item entry-meta-tags">';
                    echo '<div class="entry-meta-wrapper">';

                    /* translators: 1: list of tags. */
                    echo '<span class="tags-links">';
                    echo wp_kses_post($responsive_landing_page_tags_list) . '</span>'; // phpcs:ignore Standard.Category.SniffName.ErrorCode
                    echo '</div>';
                    echo '</div>';

                }

            }

            if( $responsive_landing_page_edits ){

                edit_post_link(
                    sprintf(
                        wp_kses(
                        /* translators: %s: Name of current post. Only visible to screen readers */
                            __('Edit <span class="screen-reader-text">%s</span>', 'responsive-landing-page'),
                            array(
                                'span' => array(
                                    'class' => array(),
                                ),
                            )
                        ),
                        get_the_title()
                    ),
                    '<span class="edit-link">',
                    '</span>'
                );
            }

        }
    }

endif;

if ( ! function_exists( 'responsive_landing_page_post_thumbnail' ) ) :

    /**
     * Displays an optional post thumbnail.
     *
     * Shows background style image with height class on archive/search/front,
     * and a normal inline image on single post/page views.
     */
    function responsive_landing_page_post_thumbnail( $responsive_landing_page_image_size = 'full' ) {

        if ( post_password_required() || is_attachment() ) {
            return;
        }

        // Fallback image path
        $responsive_landing_page_default_image = get_template_directory_uri() . '/assets/images/slide-bg.png';

        // Image size → height class map
        $responsive_landing_page_size_class_map = array(
            'full'      => 'data-bg-large',
            'large'     => 'data-bg-big',
            'medium'    => 'data-bg-medium',
            'small'     => 'data-bg-small',
            'xsmall'    => 'data-bg-xsmall',
            'thumbnail' => 'data-bg-thumbnail',
        );

        $responsive_landing_page_class = isset( $responsive_landing_page_size_class_map[ $responsive_landing_page_image_size ] )
            ? $responsive_landing_page_size_class_map[ $responsive_landing_page_image_size ]
            : 'data-bg-medium';

        if ( is_singular() ) {
            the_post_thumbnail();
        } else {
            // 🔵 On archives → use background image style
            $responsive_landing_page_image = has_post_thumbnail()
                ? wp_get_attachment_image_src( get_post_thumbnail_id(), $responsive_landing_page_image_size )
                : array( $responsive_landing_page_default_image );

            $responsive_landing_page_bg_image = isset( $responsive_landing_page_image[0] ) ? $responsive_landing_page_image[0] : $responsive_landing_page_default_image;
            ?>
            <div class="post-thumbnail data-bg <?php echo esc_attr( $responsive_landing_page_class ); ?>"
                 data-background="<?php echo esc_url( $responsive_landing_page_bg_image ); ?>">
                <a href="<?php the_permalink(); ?>" class="theme-image-responsive" tabindex="0"></a>
            </div>
            <?php
        }
    }

endif;

if( !function_exists('responsive_landing_page_is_comment_by_post_author') ):

    /**
     * Comments
     */
    /**
     * Check if the specified comment is written by the author of the post commented on.
     *
     * @param object $responsive_landing_page_comment Comment data.
     *
     * @return bool
     */
    function responsive_landing_page_is_comment_by_post_author($responsive_landing_page_comment = null){

        if (is_object($responsive_landing_page_comment) && $responsive_landing_page_comment->user_id > 0) {
            $responsive_landing_page_user = get_userdata($responsive_landing_page_comment->user_id);
            $post = get_post($responsive_landing_page_comment->comment_post_ID);
            if (!empty($responsive_landing_page_user) && !empty($post)) {
                return $responsive_landing_page_comment->user_id === $post->post_author;
            }
        }
        return false;
    }

endif;

if( !function_exists('responsive_landing_page_breadcrumb') ) :

    /**
     * Responsive Landing Page Breadcrumb
     */
    function responsive_landing_page_breadcrumb($responsive_landing_page_comment = null){

        echo '<div class="entry-breadcrumb">';
        responsive_landing_page_breadcrumb_trail();
        echo '</div>';

    }

endif;