<?php

/**
 * real-estate-blocks functions and definitions
 *
 * @link https://developer.wordpress.org/themes/basics/theme-functions/
 *
 * @package real-estate-blocks
 * @since real-estate-blocks 1.0
 */
if (!defined('ABSPATH')) {
  exit; // Exit if accessed directly.
}

if (!defined('REAL_ESTATE_BLOCKS_URL')) {
  define('REAL_ESTATE_BLOCKS_URL', get_template_directory_uri());
}

/**
 * Register block pattern categories.
 */
function real_estate_blocks_categories()
{
  $categories = [
    'real-estate-blocks/page' => ['label' => __('Pages', 'real-estate-blocks')],
    'real-estate-blocks/faq' => ['label' => __('FAQ', 'real-estate-blocks')],
    'real-estate-blocks/services' => ['label' => __('Services', 'real-estate-blocks')],
    'real-estate-blocks/about' => ['label' => __('About', 'real-estate-blocks')],
    'real-estate-blocks/general' => ['label' => __('General', 'real-estate-blocks')],
  ];
  foreach ($categories as $slug => $args) {
    if (!WP_Block_Pattern_Categories_Registry::get_instance()->is_registered($slug)) {
      register_block_pattern_category($slug, $args);
    }
  }
}
add_action('init', 'real_estate_blocks_categories');


/**
 * Gets the content of a pattern file by including it, which allows PHP to be executed.
 *
 * @param string $pattern_file_path The path to the pattern file.
 * @return string The processed pattern content.
 */
function real_estate_blocks_get_pattern_content_from_file($pattern_file_path)
{
  $theme_dir = realpath(get_template_directory());
  $real_path = realpath($pattern_file_path);
  if (!$real_path || strpos($real_path, $theme_dir) !== 0) {
    if (defined('WP_DEBUG') && WP_DEBUG) {
      trigger_error(esc_html__('Invalid pattern file path: ', 'real-estate-blocks') . esc_html($pattern_file_path), E_USER_WARNING);
    }
    return '';
  }
  if (!file_exists($real_path)) {
    if (defined('WP_DEBUG') && WP_DEBUG) {
      trigger_error(esc_html__('Pattern file not found: ', 'real-estate-blocks') . esc_html($real_path), E_USER_WARNING);
    }
    return '';
  }
  ob_start();
  include $real_path;
  return ob_get_clean();
}



/**
 * Register all block patterns for the theme.
 */
function real_estate_blocks_register_all_patterns()
{
  $patterns = [
    'header-default'        => ['title' => __('Default Header', 'real-estate-blocks'), 'categories' => ['real-estate-blocks/page']],
    'footer-default'        => ['title' => __('Default Footer', 'real-estate-blocks'), 'categories' => ['real-estate-blocks/page']],
    'about'                 => ['title' => __('About Us', 'real-estate-blocks'), 'categories' => ['real-estate-blocks/about']],
    'faq'                   => ['title' => __('FAQ Section', 'real-estate-blocks'), 'categories' => ['real-estate-blocks/faq']],
    '404-content'           => ['title' => __('404 Page Content', 'real-estate-blocks'), 'categories' => ['real-estate-blocks/page']],
    'template-query-loop'   => ['title' => __('Post List', 'real-estate-blocks'), 'categories' => ['query']],
    'services-three-column' => ['title' => __('Three Column Service', 'real-estate-blocks'), 'categories' => ['real-estate-blocks/services-three-column']],
    'media-text-content'    => ['title' => __('Media and Text Content', 'real-estate-blocks'), 'categories' => ['real-estate-blocks/general']],
    'slider'                => ['title' => __('Real Estate Slider', 'real-estate-blocks'), 'categories' => ['real-estate-blocks/general']],
  ];

  foreach ($patterns as $slug => $details) {
    $pattern_name = "real-estate-blocks/{$slug}";

    // Unregister the auto-registered version first.
    if (WP_Block_Patterns_Registry::get_instance()->is_registered($pattern_name)) {
      unregister_block_pattern($pattern_name);
    }

    $content = real_estate_blocks_get_pattern_content_from_file(get_theme_file_path("/patterns/{$slug}.php"));

    if (!empty($content)) {
      register_block_pattern(
        $pattern_name,
        array_merge(
          $details,
          ['content' => $content]
        )
      );
    }
  }
}
add_action('init', 'real_estate_blocks_register_all_patterns');


/**
 * Enqueue theme stylesheet.
 */
function real_estate_blocks_theme_enqueue_styles()
{
  wp_enqueue_style(
    'real-estate-blocks-style',
    get_stylesheet_uri(),
    [],
    wp_get_theme()->get('Version')
  );
}
add_action('wp_enqueue_scripts', 'real_estate_blocks_theme_enqueue_styles');

/**
 * Theme setup.
 */
function real_estate_blocks_setup()
{
  add_theme_support('title-tag');
  add_theme_support('post-thumbnails');
  add_theme_support('custom-logo');
  add_theme_support('align-wide');
  add_theme_support('responsive-embeds');
  add_theme_support('wp-block-styles');
  add_theme_support('editor-styles');
  add_editor_style('style.css');
}
add_action('after_setup_theme', 'real_estate_blocks_setup');

// Optional: Include the dashboard panel if it exists.
$dashboard_file = get_template_directory() . '/dashboard/dashboard.php';
if (file_exists($dashboard_file)) {
  require_once $dashboard_file;
}

/**
 * Checks if the required 'Real Estate Right Now' plugin is installed.
 * If not, it displays a dismissible admin notice.
 */
function real_estate_blocks_check_required_plugin() {
    
  // Define the full path to the plugin's main file.
  // WP_PLUGIN_DIR is a WordPress constant that points directly to the plugins folder.
  $plugin_path = WP_PLUGIN_DIR . '/real-estate-right-now/realestate.php';

  // Check directly on the file system if the plugin file does NOT exist.
  // This method is reliable and not affected by WordPress cache.
  if ( ! file_exists( $plugin_path ) ) {

      // Prepare the URL to search for the plugin in the WordPress installer.
      $plugin_slug = 'real-estate-right-now';
      $install_url = admin_url( 'plugin-install.php?tab=search&s=' . $plugin_slug );
      
      // The message to be displayed in the notice.
      $message = sprintf(
          esc_html__( 'The Real Estate Blocks theme requires the free "Real Estate Right Now" plugin for full functionality. %s', 'real-estate-blocks' ),
          '<a href="' . esc_url( $install_url ) . '"><strong>' . esc_html__( 'Please install the plugin from the WordPress directory.', 'real-estate-blocks' ) . '</strong></a>'
      );

      // Output the admin notice HTML.
      printf(
          '<div class="notice notice-warning is-dismissible"><p>%s</p></div>',
          $message
      );
  }
}
// Add the function to the 'admin_notices' action hook.
add_action( 'admin_notices', 'real_estate_blocks_check_required_plugin' );

if (!function_exists('real_estate_blocks_block_styles')) :
  /**
   * Register custom block styles.
   *
   * @since real-estate-blocks 1.0
   * @return void
   */
  function real_estate_blocks_block_styles()
  {
    $block_styles = [
      'core/details' => [
        'name' => 'arrow-icon-details',
        'label' => esc_attr__('Arrow icon', 'real-estate-blocks'),
        'style_handle' => 'real-estate-blocks-arrow-icon-details',
      ],
      'core/post-terms' => [
        'name' => 'pill',
        'label' => esc_attr__('Pill', 'real-estate-blocks'),
        'style_handle' => 'real-estate-blocks-post-terms-pill',
      ],
      'core/list' => [
        'name' => 'checkmark-list',
        'label' => esc_attr__('Checkmark', 'real-estate-blocks'),
        'style_handle' => 'real-estate-blocks-checkmark-list',
      ],
      'core/navigation-link' => [
        'name' => 'arrow-link',
        'label' => esc_attr__('With arrow', 'real-estate-blocks'),
        'style_handle' => 'real-estate-blocks-arrow-link',
      ],
      'core/heading' => [
        'name' => 'asterisk',
        'label' => esc_attr__('With asterisk', 'real-estate-blocks'),
        'style_handle' => 'real-estate-blocks-heading-asterisk',
      ],
    ];
    $style_path = get_theme_file_path('assets/css/block-styles.css');
    $style_src = get_theme_file_uri('assets/css/block-styles.css');
    foreach ($block_styles as $block => $args) {
      register_block_style($block, [
        'name' => $args['name'],
        'label' => $args['label'],
      ]);
      if (file_exists($style_path)) {
        wp_enqueue_block_style($block, [
          'handle' => $args['style_handle'],
          'src' => $style_src,
          'path' => $style_path,
          'ver' => wp_get_theme()->get('Version'),
        ]);
      } elseif (defined('WP_DEBUG') && WP_DEBUG) {
        trigger_error(esc_html__('Block style file not found: ', 'real-estate-blocks') . esc_html($style_path), E_USER_WARNING);
      }
    }
  }
endif;
add_action('init', 'real_estate_blocks_block_styles');

function real_estate_blocks_load_theme_color_palette() {
  // Obter o estilo selecionado
  $selected_style = get_theme_mod('real_estate_blocks_selected_style', 'blue');
  $json_file = get_template_directory() . '/styles/colors/' . $selected_style . '.json';
  
  // Fallback para blue se o arquivo não existir
  if (!file_exists($json_file)) {
      $json_file = get_template_directory() . '/styles/colors/blue.json';
  }
  
  // Ler e decodificar o JSON
  $json_data = file_get_contents($json_file);
  $color_data = json_decode($json_data, true);
  
  return $color_data;
}

function real_estate_blocks_apply_theme_colors() {
  //$selected_style = get_theme_mod('selected_style', '');
  $user_has_chosen = get_theme_mod('real_estate_blocks_user_has_chosen_style', false);
  if ($user_has_chosen) {
      return;
  }
  $color_data = real_estate_blocks_load_theme_color_palette();
  if (!$color_data) {
      return;
  }
  $css = '';
  // 1. Variáveis CSS
  if (isset($color_data['settings']['color']['palette'])) {
      $css .= ':root {';
      foreach ($color_data['settings']['color']['palette'] as $color) {
          if (isset($color['slug']) && isset($color['color'])) {
              $css .= '--wp--preset--color--' . $color['slug'] . ': ' . $color['color'] . ';';
          }
      }
      $css .= '}';
  }
  // 2. Estilos para headings (usar valores reais)
  $contrast_color = '#000000'; // Fallback
  if (isset($color_data['settings']['color']['palette'])) {
      foreach ($color_data['settings']['color']['palette'] as $color) {
          if ($color['slug'] === 'contrast') {
              $contrast_color = $color['color'];
              break;
          }
      }
  }
  $css .= '.wp-block-heading, h1, h2, h3, h4, h5, h6 { color: ' . $contrast_color . ' !important; }';
  $css .= 'h2.wp-block-heading { color: ' . $contrast_color . ' !important; }';
  // 3. Adicionar CSS diretamente no head
  echo '<style id="theme-color-styles">' . $css . '</style>';
}
add_action('wp_head', 'real_estate_blocks_apply_theme_colors', 10);
/**
 * Adiciona uma flag quando os estilos globais são salvos via Editor de Site.
 */
add_action( 'rest_insert_wp_global_styles', 'real_estate_blocks_registrar_estilo_salvo', 10, 3 );
function real_estate_blocks_registrar_estilo_salvo( $post, $request, $creating ) {
    if ( $post->post_type === 'wp_global_styles' && strpos( $post->post_name, 'wp-global-styles-' ) === 0 ) {
        set_theme_mod( 'real_estate_blocks_user_has_chosen_style', true );
    }
}

/**
 * Enfileira o CSS e JavaScript do padrão Car Dealer Showcase.
 */

 /*
function real_estate_blocks_enqueue_assets() {
  // Verifica se o padrão está em uso na página atual
  global $post;
 // if ( has_block( 'car-dealer-show/slider', $post ) ) {
      // Enfileira o CSS
      wp_enqueue_style(
          'real-estate-blocks-style',
          get_theme_file_uri( '/assets/css/slider.css' ),
          array(),
          '1.0'
      );

      // Enfileira o JavaScript
      wp_enqueue_script(
          'real-estate-blocks-script',
          get_theme_file_uri( '/assets/js/slider.js' ),
          array(),
          '1.0',
          true // Coloca o script no rodapé
      );
  //}
}
add_action( 'wp_enqueue_scripts', 'real_estate_blocks_enqueue_assets' );
*/
function real_estate_blocks_enqueue_assets() {
  // Define os caminhos dos arquivos
  $css_path = get_theme_file_path( '/assets/css/slider.css' );
  $js_path = get_theme_file_path( '/assets/js/slider.js' );
  
  /*
  // Verifica se os arquivos existem e grava debug no rodapé
  add_action('wp_footer', function() use ($css_path, $js_path) {
      echo '<div style="background: #f0f0f0; padding: 10px; margin: 20px; border: 2px solid red;">';
      echo '<h3>DEBUG - real_estate_blocks_enqueue_assets</h3>';
      
      echo '<p>CSS Path: ' . esc_html($css_path) . '</p>';
      echo '<p>CSS Exists: ' . (file_exists($css_path) ? '✅ SIM' : '❌ NÃO') . '</p>';
      
      echo '<p>JS Path: ' . esc_html($js_path) . '</p>';
      echo '<p>JS Exists: ' . (file_exists($js_path) ? '✅ SIM' : '❌ NÃO') . '</p>';
      
      // Verifica se a função foi chamada
      echo '<p>Função executada: ✅ SIM</p>';
      echo '</div>';
  });
  */
  
  // Só enfileira se os arquivos existirem
  if ( file_exists( $css_path ) ) {
      wp_enqueue_style(
          'real-estate-blocks-slider',
          get_theme_file_uri( '/assets/css/slider.css' ),
          array(),
          filemtime( $css_path )
      );
  }
  
  if ( file_exists( $js_path ) ) {
      wp_enqueue_script(
          'real-estate-blocks-slider',
          get_theme_file_uri( '/assets/js/slider.js' ),
          array(),
          filemtime( $js_path ),
          true
      );
  }
}
add_action( 'wp_enqueue_scripts', 'real_estate_blocks_enqueue_assets' );