<?php
/**
 * Custom Functions
 * @package Plumbing Technician
 * @since 1.0.0
 */

if( !function_exists('plumbing_technician_site_logo') ):

    /**
     * Logo & Description
     */
    /**
     * Displays the site logo, either text or image.
     *
     * @param array $plumbing_technician_args Arguments for displaying the site logo either as an image or text.
     * @param boolean $plumbing_technician_echo Echo or return the HTML.
     *
     * @return string $plumbing_technician_html Compiled HTML based on our arguments.
     */
    function plumbing_technician_site_logo( $plumbing_technician_args = array(), $plumbing_technician_echo = true ){
        $plumbing_technician_logo = get_custom_logo();
        $plumbing_technician_site_title = get_bloginfo('name');
        $plumbing_technician_contents = '';
        $plumbing_technician_classname = '';
        $plumbing_technician_defaults = array(
            'logo' => '%1$s<span class="screen-reader-text">%2$s</span>',
            'logo_class' => 'site-logo site-branding',
            'title' => '<a href="%1$s" class="custom-logo-name">%2$s</a>',
            'title_class' => 'site-title',
            'home_wrap' => '<h1 class="%1$s">%2$s</h1>',
            'single_wrap' => '<div class="%1$s">%2$s</div>',
            'condition' => (is_front_page() || is_home()) && !is_page(),
        );
        $plumbing_technician_args = wp_parse_args($plumbing_technician_args, $plumbing_technician_defaults);
        /**
         * Filters the arguments for `plumbing_technician_site_logo()`.
         *
         * @param array $plumbing_technician_args Parsed arguments.
         * @param array $plumbing_technician_defaults Function's default arguments.
         */
        $plumbing_technician_args = apply_filters('plumbing_technician_site_logo_args', $plumbing_technician_args, $plumbing_technician_defaults);
        
        $plumbing_technician_show_logo  = get_theme_mod('plumbing_technician_display_logo', false);
        $plumbing_technician_show_title = get_theme_mod('plumbing_technician_display_title', true);

        if ( has_custom_logo() && $plumbing_technician_show_logo ) {
            $plumbing_technician_contents .= sprintf($plumbing_technician_args['logo'], $plumbing_technician_logo, esc_html($plumbing_technician_site_title));
            $plumbing_technician_classname = $plumbing_technician_args['logo_class'];
        }

        if ( $plumbing_technician_show_title ) {
            $plumbing_technician_contents .= sprintf($plumbing_technician_args['title'], esc_url(get_home_url(null, '/')), esc_html($plumbing_technician_site_title));
            // If logo isn't shown, fallback to title class for wrapper.
            if ( !$plumbing_technician_show_logo ) {
                $plumbing_technician_classname = $plumbing_technician_args['title_class'];
            }
        }

        // If nothing is shown (logo or title both disabled), exit early
        if ( empty($plumbing_technician_contents) ) {
            return;
        }

        $plumbing_technician_wrap = $plumbing_technician_args['condition'] ? 'home_wrap' : 'single_wrap';
        // $plumbing_technician_wrap = 'home_wrap';
        $plumbing_technician_html = sprintf($plumbing_technician_args[$plumbing_technician_wrap], $plumbing_technician_classname, $plumbing_technician_contents);
        /**
         * Filters the arguments for `plumbing_technician_site_logo()`.
         *
         * @param string $plumbing_technician_html Compiled html based on our arguments.
         * @param array $plumbing_technician_args Parsed arguments.
         * @param string $plumbing_technician_classname Class name based on current view, home or single.
         * @param string $plumbing_technician_contents HTML for site title or logo.
         */
        $plumbing_technician_html = apply_filters('plumbing_technician_site_logo', $plumbing_technician_html, $plumbing_technician_args, $plumbing_technician_classname, $plumbing_technician_contents);
        if (!$plumbing_technician_echo) {
            return $plumbing_technician_html;
        }
        echo $plumbing_technician_html; //phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped

    }

endif;

if( !function_exists('plumbing_technician_site_description') ):

    /**
     * Displays the site description.
     *
     * @param boolean $plumbing_technician_echo Echo or return the html.
     *
     * @return string $plumbing_technician_html The HTML to display.
     */
    function plumbing_technician_site_description($plumbing_technician_echo = true){

        if ( get_theme_mod('plumbing_technician_display_header_text', false) == true ) :
        $plumbing_technician_description = get_bloginfo('description');
        if (!$plumbing_technician_description) {
            return;
        }
        $plumbing_technician_wrapper = '<div class="site-description"><span>%s</span></div><!-- .site-description -->';
        $plumbing_technician_html = sprintf($plumbing_technician_wrapper, esc_html($plumbing_technician_description));
        /**
         * Filters the html for the site description.
         *
         * @param string $plumbing_technician_html The HTML to display.
         * @param string $plumbing_technician_description Site description via `bloginfo()`.
         * @param string $plumbing_technician_wrapper The format used in case you want to reuse it in a `sprintf()`.
         * @since 1.0.0
         *
         */
        $plumbing_technician_html = apply_filters('plumbing_technician_site_description', $plumbing_technician_html, $plumbing_technician_description, $plumbing_technician_wrapper);
        if (!$plumbing_technician_echo) {
            return $plumbing_technician_html;
        }
        echo $plumbing_technician_html; //phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped
        endif;
    }

endif;

if( !function_exists('plumbing_technician_posted_on') ):

    /**
     * Prints HTML with meta information for the current post-date/time.
     */
    function plumbing_technician_posted_on( $plumbing_technician_icon = true, $plumbing_technician_animation_class = '' ){

        $plumbing_technician_default = plumbing_technician_get_default_theme_options();
        $plumbing_technician_post_date = absint( get_theme_mod( 'plumbing_technician_post_date',$plumbing_technician_default['plumbing_technician_post_date'] ) );

        if( $plumbing_technician_post_date ){

            $plumbing_technician_time_string = '<time class="entry-date published updated" datetime="%1$s">%2$s</time>';
            if (get_the_time('U') !== get_the_modified_time('U')) {
                $plumbing_technician_time_string = '<time class="entry-date published" datetime="%1$s">%2$s</time><time class="updated" datetime="%3$s">%4$s</time>';
            }

            $plumbing_technician_time_string = sprintf($plumbing_technician_time_string,
                esc_attr(get_the_date(DATE_W3C)),
                esc_html(get_the_date()),
                esc_attr(get_the_modified_date(DATE_W3C)),
                esc_html(get_the_modified_date())
            );

            $plumbing_technician_year = get_the_date('Y');
            $plumbing_technician_month = get_the_date('m');
            $plumbing_technician_day = get_the_date('d');
            $plumbing_technician_link = get_day_link($plumbing_technician_year, $plumbing_technician_month, $plumbing_technician_day);

            $plumbing_technician_posted_on = '<a href="' . esc_url($plumbing_technician_link) . '" rel="bookmark">' . $plumbing_technician_time_string . '</a>';

            echo '<div class="entry-meta-item entry-meta-date">';
            echo '<div class="entry-meta-wrapper '.esc_attr( $plumbing_technician_animation_class ).'">';

            if( $plumbing_technician_icon ){

                echo '<span class="entry-meta-icon calendar-icon"> ';
                plumbing_technician_the_theme_svg('calendar');
                echo '</span>';

            }

            echo '<span class="posted-on">' . $plumbing_technician_posted_on . '</span>'; // phpcs:ignore Standard.Category.SniffName.ErrorCode
            echo '</div>';
            echo '</div>';

        }

    }

endif;

if( !function_exists('plumbing_technician_posted_by') ) :

    /**
     * Prints HTML with meta information for the current author.
     */
    function plumbing_technician_posted_by( $plumbing_technician_icon = true, $plumbing_technician_animation_class = '' ){   

        $plumbing_technician_default = plumbing_technician_get_default_theme_options();
        $plumbing_technician_post_author = absint( get_theme_mod( 'plumbing_technician_post_author',$plumbing_technician_default['plumbing_technician_post_author'] ) );

        if( $plumbing_technician_post_author ){

            echo '<div class="entry-meta-item entry-meta-author">';
            echo '<div class="entry-meta-wrapper '.esc_attr( $plumbing_technician_animation_class ).'">';

            if( $plumbing_technician_icon ){
            
                echo '<span class="entry-meta-icon author-icon"> ';
                plumbing_technician_the_theme_svg('user');
                echo '</span>';
                
            }

            $plumbing_technician_byline = '<span class="author vcard"><a class="url fn n" href="' . esc_url( get_author_posts_url( get_the_author_meta('ID') ) ) . '">' . esc_html(get_the_author()) . '</a></span>';
            echo '<span class="byline"> ' . $plumbing_technician_byline . '</span>'; // phpcs:ignore Standard.Category.SniffName.ErrorCode
            echo '</div>';
            echo '</div>';

        }

    }

endif;


if( !function_exists('plumbing_technician_posted_by_avatar') ) :

    /**
     * Prints HTML with meta information for the current author.
     */
    function plumbing_technician_posted_by_avatar( $date = false ){

        $plumbing_technician_default = plumbing_technician_get_default_theme_options();
        $plumbing_technician_post_author = absint( get_theme_mod( 'plumbing_technician_post_author',$plumbing_technician_default['plumbing_technician_post_author'] ) );

        if( $plumbing_technician_post_author ){

            echo '<div class="entry-meta-left">';
            echo '<div class="entry-meta-item entry-meta-avatar">';
            echo wp_kses_post( get_avatar( get_the_author_meta( 'ID' ) ) );
            echo '</div>';
            echo '</div>';

            echo '<div class="entry-meta-right">';

            $plumbing_technician_byline = '<span class="author vcard"><a class="url fn n" href="' . esc_url( get_author_posts_url( get_the_author_meta('ID') ) ) . '">' . esc_html(get_the_author()) . '</a></span>';

            echo '<div class="entry-meta-item entry-meta-byline"> ' . $plumbing_technician_byline . '</div>';

            if( $date ){
                plumbing_technician_posted_on($plumbing_technician_icon = false);
            }
            echo '</div>';

        }

    }

endif;

if( !function_exists('plumbing_technician_entry_footer') ):

    /**
     * Prints HTML with meta information for the categories, tags and comments.
     */
    function plumbing_technician_entry_footer( $plumbing_technician_cats = true, $plumbing_technician_tags = true, $plumbing_technician_edits = true){   

        $plumbing_technician_default = plumbing_technician_get_default_theme_options();
        $plumbing_technician_post_category = absint( get_theme_mod( 'plumbing_technician_post_category',$plumbing_technician_default['plumbing_technician_post_category'] ) );
        $plumbing_technician_post_tags = absint( get_theme_mod( 'plumbing_technician_post_tags',$plumbing_technician_default['plumbing_technician_post_tags'] ) );

        // Hide category and tag text for pages.
        if ('post' === get_post_type()) {

            if( $plumbing_technician_cats && $plumbing_technician_post_category ){

                /* translators: used between list items, there is a space after the comma */
                $plumbing_technician_categories = get_the_category();
                if ($plumbing_technician_categories) {
                    echo '<div class="entry-meta-item entry-meta-categories">';
                    echo '<div class="entry-meta-wrapper">';
                
                    /* translators: 1: list of categories. */
                    echo '<span class="cat-links">';
                    foreach( $plumbing_technician_categories as $plumbing_technician_category ){

                        $plumbing_technician_cat_name = $plumbing_technician_category->name;
                        $plumbing_technician_cat_slug = $plumbing_technician_category->slug;
                        $plumbing_technician_cat_url = get_category_link( $plumbing_technician_category->term_id );
                        ?>

                        <a class="twp_cat_<?php echo esc_attr( $plumbing_technician_cat_slug ); ?>" href="<?php echo esc_url( $plumbing_technician_cat_url ); ?>" rel="category tag"><?php echo esc_html( $plumbing_technician_cat_name ); ?></a>

                    <?php }
                    echo '</span>'; // phpcs:ignore Standard.Category.SniffName.ErrorCode
                    echo '</div>';
                    echo '</div>';
                }

            }

            if( $plumbing_technician_tags && $plumbing_technician_post_tags ){
                /* translators: used between list items, there is a space after the comma */
                $plumbing_technician_tags_list = get_the_tag_list('', esc_html_x(', ', 'list item separator', 'plumbing-technician'));
                if( $plumbing_technician_tags_list ){

                    echo '<div class="entry-meta-item entry-meta-tags">';
                    echo '<div class="entry-meta-wrapper">';

                    /* translators: 1: list of tags. */
                    echo '<span class="tags-links">';
                    echo wp_kses_post($plumbing_technician_tags_list) . '</span>'; // phpcs:ignore Standard.Category.SniffName.ErrorCode
                    echo '</div>';
                    echo '</div>';

                }

            }

            if( $plumbing_technician_edits ){

                edit_post_link(
                    sprintf(
                        wp_kses(
                        /* translators: %s: Name of current post. Only visible to screen readers */
                            __('Edit <span class="screen-reader-text">%s</span>', 'plumbing-technician'),
                            array(
                                'span' => array(
                                    'class' => array(),
                                ),
                            )
                        ),
                        get_the_title()
                    ),
                    '<span class="edit-link">',
                    '</span>'
                );
            }

        }
    }

endif;

if ( ! function_exists( 'plumbing_technician_post_thumbnail' ) ) :

    /**
     * Displays an optional post thumbnail.
     *
     * Shows background style image with height class on archive/search/front,
     * and a normal inline image on single post/page views.
     */
    function plumbing_technician_post_thumbnail( $plumbing_technician_image_size = 'medium' ) {

        if ( post_password_required() || is_attachment() ) {
            return;
        }

        // Fallback image path
        $plumbing_technician_default_image = get_template_directory_uri() . '/assets/images/service.png';

        // Image size → height class map
        $plumbing_technician_size_class_map = array(
            'full'      => 'data-bg-large',
            'large'     => 'data-bg-big',
            'medium'    => 'data-bg-medium',
            'small'     => 'data-bg-small',
            'xsmall'    => 'data-bg-xsmall',
            'thumbnail' => 'data-bg-thumbnail',
        );

        $plumbing_technician_class = isset( $plumbing_technician_size_class_map[ $plumbing_technician_image_size ] )
            ? $plumbing_technician_size_class_map[ $plumbing_technician_image_size ]
            : 'data-bg-medium';

        if ( is_singular() ) {
            the_post_thumbnail();
        } else {
            // 🔵 On archives → use background image style
            $plumbing_technician_image = has_post_thumbnail()
                ? wp_get_attachment_image_src( get_post_thumbnail_id(), $plumbing_technician_image_size )
                : array( $plumbing_technician_default_image );

            $plumbing_technician_bg_image = isset( $plumbing_technician_image[0] ) ? $plumbing_technician_image[0] : $plumbing_technician_default_image;
            ?>
            <div class="post-thumbnail data-bg <?php echo esc_attr( $plumbing_technician_class ); ?>"
                 data-background="<?php echo esc_url( $plumbing_technician_bg_image ); ?>">
                <a href="<?php the_permalink(); ?>" class="theme-image-responsive" tabindex="0"></a>
            </div>
            <?php
        }
    }

endif;

if( !function_exists('plumbing_technician_is_comment_by_post_author') ):

    /**
     * Comments
     */
    /**
     * Check if the specified comment is written by the author of the post commented on.
     *
     * @param object $plumbing_technician_comment Comment data.
     *
     * @return bool
     */
    function plumbing_technician_is_comment_by_post_author($plumbing_technician_comment = null){

        if (is_object($plumbing_technician_comment) && $plumbing_technician_comment->user_id > 0) {
            $user = get_userdata($plumbing_technician_comment->user_id);
            $post = get_post($plumbing_technician_comment->comment_post_ID);
            if (!empty($user) && !empty($post)) {
                return $plumbing_technician_comment->user_id === $post->post_author;
            }
        }
        return false;
    }

endif;

if( !function_exists('plumbing_technician_breadcrumb') ) :

    /**
     * Plumbing Technician Breadcrumb
     */
    function plumbing_technician_breadcrumb($plumbing_technician_comment = null){

        echo '<div class="entry-breadcrumb">';
        plumbing_technician_breadcrumb_trail();
        echo '</div>';

    }

endif;