<?php
/**
 * Custom template tags for this theme.
 *
 *
 * @package plug-shop
 */


/**
 * Custom Function to Include File
 *
 * Check for the template in the child theme first,
 * if there is no file there, call the one from
 * parent theme. This is done to support deeper level files
 * overriding
 *
 * @param $file
 * @return void
 * @since 1.0.0
 */
function plug_shop_include_file($file) {
	$template = locate_template($file);

	if ( '' !== $template ) {
		require_once( $template );
	}
}

/**
* Get template part from the loop folder
*
* @since 1.0.0
*/
function plug_shop_get_version() {
	$theme = wp_get_theme();
	
	if(is_child_theme()) {
		$parent  = $theme->parent();
		$version = $parent['Version'];
	} else {
		$version = $theme['Version'];
	}

	return $version;
}

if( !function_exists('plug_shop_get_content') ) {
	/**
	 * Get template part from the loop folder
	 *
	 * @since 1.0.0
	 */
	function plug_shop_get_content($content) {
		$pre = 'loop/content';

		get_template_part( $pre, $content ); 
	}
}

if( !function_exists('plug_shop_get_part') ) {
	/**
	 * Get template part from the loop folder
	 *
	 * @since 1.0.0
	 */
	function plug_shop_get_part($template) {
		$pre = 'parts/'; 
		$template = $pre.$template; 

		get_template_part( $template );
	}
}

/**
* Get sidebar template from the templates folder
*
* @since 1.0.0
*/
if( !function_exists('plug_shop_get_sidebar') )
{
	function plug_shop_get_sidebar($content)
	{ 
		$pre = 'templates/sidebar'; 

		get_template_part( $pre, $content ); 
	}
}

/**
* Retrive the nav menu
* If locations does not exists
* use the pages menu as callback
*
* @since 1.0.0
*/
if( !function_exists('plug_shop_get_menu') )
{
	function plug_shop_get_menu($args)
	{
		if ( isset( $args['theme_location'] ) && has_nav_menu( $args['theme_location'] ) ) {
			wp_nav_menu( $args );
		} else {
			wp_page_menu( $args );
		}
	}
}


if( !function_exists('plug_shop_get_banners') ) {
	/**
	 * Get the banners from options
	 *
	 * @since 1.0.0
	 */
	function plug_shop_get_banners() {
		$banners = plug_shop_options('banners', '');
		$show    = plug_shop_options('show_banners', 'shop');
		$html = '';

		if ( function_exists( 'is_shop' )  ) {

			if ( ! is_shop() && ! is_singular('product') ) {
				return;
			}

			if ( 'shop' === $show && ! is_shop() ) {
				return;
			}

			if ( 'no_single' === $show && is_singular('product' ) ) {
				return;
			}
		}

		if ( ! empty( $banners ) && ( isset( $banners[0] ) || isset( $banners[1] ) ) ) {
			$html .= '<div class="psf-banners-holder">';

			for( $n = 0; $n <= 1; $n++ ) {
				$style = '';

				if ( isset( $banners[$n] ) && ! empty( $banners[$n]['image'] ) ) {
					$html .= '<div class="psf-banner-image" ';

					if ( ! empty( $banners[$n]['image'] ) ) {
						$style .= sprintf( 'background-image: url(%s);', wp_get_attachment_image_url($banners[$n]['image'], 'full' ) );
					}

					if ( ! empty( $banners[$n]['height'] ) && ( strpos($banners[$n]['height'], 'px') !== false || strpos($banners[$n]['height'], 'em') !== false ) ) {
						$style .= sprintf( 'height; %s', esc_attr($banners[$n]['height']) );
					} else if ( ! empty( $banners[$n]['height'] ) && ( strpos($banners[$n]['height'], 'px') === false || strpos($banners[$n]['height'], 'em') === false ) ) {
						$style .= sprintf( 'height; %spx', esc_attr($banners[$n]['height']) );
					}

					$html .= sprintf( 'style="%s">', esc_attr($style) );

					if ( ! empty( $banners[$n]['link_url'] ) ) {
						$html .= sprintf( '<a href="%s"', esc_url( $banners[$n]['link_url'] ) );
					} else {
						$html .= '<a href="#"';
					}

					if ( ! empty( $banners[$n]['link_text'] ) ) {
						$html .= sprintf( ' class="psf-link-text">%s', esc_html( $banners[$n]['link_text'] ) );
					} else {
						$html .= '>';
					}

					$html .= '</a>';
					$html .= '</div>';
				}

			}

			$html .= '</div>';
		}

		return $html;
	}
}

if ( ! function_exists( 'plug_shop_cart_link' ) ) {
	/**
	 * Cart Link
	 * Displayed a link to the cart including the number of items present and the cart total
	 *
	 * @return void
	 * @since  1.0.0
	 */
	function plug_shop_cart_link() {
		if ( plug_shop_is_wc_installed() ) { ?>
			<a class="cart-contents" href="<?php echo esc_url(wc_get_cart_url()); ?>" title="<?php esc_attr_e('View your shopping cart', 'plug-shop'); ?>">
				<span class="amount"><?php echo wp_kses_data(WC()->cart->get_cart_subtotal()); ?></span>
				<span class="icon-android-cart psf-icon-cart">
					<span class="count"><?php echo wp_kses_data(WC()->cart->get_cart_contents_count()); ?></span>
				</span>
			</a>
			<?php
		}
	}
}

if ( ! function_exists( 'plug_shop_posted_on' ) ) {
	/**
	 * Prints HTML with meta information
	 * for the current post-date/time and author.
	 *
	 * @since 1.0.0
	 */
	function plug_shop_posted_on()	{
		printf( '<span class="%1$s"><i class="icon-clock-time-outline"></i> %2$s</span> %3$s <span class="meta-sep"></span><i class="icon-user-circle"></i> %4$s %5$s',
			'meta-prep meta-prep-author',
			esc_html__( 'Posted on', 'plug-shop' ),
			sprintf( '<a href="%1$s" title="%2$s" rel="bookmark"><span class="entry-date">%3$s</span></a>',
				esc_url( get_permalink() ),
				esc_attr( get_the_time() ),
				get_the_date()
			),
			esc_html__( 'by', 'plug-shop' ),
			sprintf( '<span class="author vcard"><a class="url fn n" href="%1$s" title="%2$s">%3$s</a></span>',
				get_author_posts_url( get_the_author_meta( 'ID' ) ),
				sprintf( '%s %s',
						esc_attr__( 'View all posts by', 'plug-shop' ),
						get_the_author()
				),
				get_the_author()
			)
		);
	}
}

if ( ! function_exists( 'plug_shop_posted_in' ) ) {
	/**
	 *
	 * Prints HTML with meta information for
	 * the current post (category, tags and permalink)
	 *
	 * @since 1.0.0
	 */
	function plug_shop_posted_in() {
		// Retrieves tag list of current post, separated by commas.
		$tag_list = get_the_tag_list( '', ', ' );
		if ( $tag_list ) {
			$posted_in = sprintf( '%1$s <i class="icon-ios-folder-outline"></i> %2$s %3$s <i class="icon-tag-outline"></i> %4$s. %5$s <i class="icon-link-streamline"></i> <a href="%6$s" title="%7$s %8$s" rel="bookmark">%9$s</a>.',
				esc_html__( 'This entry was posted in', 'plug-shop' ),
				get_the_category_list( ', ' ),
					esc_html__( 'and tagged', 'plug-shop' ),
				$tag_list,
					esc_html__( 'Bookmark the', 'plug-shop' ),
				esc_url( get_permalink() ),
					esc_html__( 'Permalink to', 'plug-shop' ),
				the_title_attribute( 'echo=0' ),
					esc_html__( 'permalink', 'plug-shop' )
			);
		} elseif ( is_object_in_taxonomy( get_post_type(), 'category' ) ) {
			$posted_in = sprintf( '%1$s %2$s. %3$s <a href="%4$s" title="%5$s %6$s" rel="bookmark">%7$s</a>.', 'plug-shop',
					esc_html__('This entry was posted in', 'plug-shop'),
				get_the_category_list( ', ' ),
					esc_html__('Bookmark the', 'plug-shop'),
				esc_url( get_permalink() ),
					esc_html__('Permalink to', 'plug-shop'),
				the_title_attribute( 'echo=0' ),
					esc_html__('permalink', 'plug-shop')
			 );
		} else {
			$posted_in = sprintf( '%1$s <a href="%2$s" title="%3$s %4$s" rel="bookmark">%5$s</a>.',
					esc_html__('Bookmark the', 'plug-shop'),
				esc_url( get_permalink() ),
					esc_html__('Permalink to', 'plug-shop'),
				the_title_attribute( 'echo=0' ),
					esc_html__('permalink', 'plug-shop')
			);
		}

		// Prints the string, replacing the placeholders.
		echo $posted_in;
	}
}


if( !function_exists('plug_shop_get_thumbnail') ) {
	/**
	 * Get post thumbnail
	 *
	 * @since 1.0.0
	 */
	function plug_shop_get_thumbnail($size  = 'post-thumbnail', $isForced = false)
	{ 
		if( is_page() || ( 'no' == plug_shop_options('bg_thumb') || '' == plug_shop_options('bg_thumb') ) )	{
			global $post;
			global $id;
			$size = !$isForced ? apply_filters( 'plug_shop_thumbnail_size', $size ) : $size;
			$no_image = false;

			if( is_sticky($id) ) {
				$size = 'full';
			}

			$align = 'scale-with-grid';

			$link = sprintf( '<a class="psf-thumb-wrap" href="%s">', get_the_permalink($id) );
			$image = has_post_thumbnail($id) ? get_the_post_thumbnail($id, $size, array('class' => $align)) : '';

			if( '' == $image && !is_single() && 'page' !== get_post_type() ) {
				$no_image = true;

				$image = sprintf( '<img src="%1$s" alt="thumbnail"/>', get_template_directory_uri().esc_attr('/assets/images/thumb.jpg') );
			}

			$image = $link . $image . '</a>';
			
			return apply_filters( 'plug_shop_image_thumbnail_markup', $image, $id, $no_image );
		}
	}
}


if ( ! function_exists( 'plug_shop_comments' ) ) {
	/**
	 *
	 * Comment Styles
	 *
	 * @since  1.0.0
	 */
	function plug_shop_comments($comment, $args, $depth)
	{
		$comment = $GLOBALS['comment']; ?>
		<li <?php comment_class(); ?> id="comment-<?php comment_ID() ?>" class="single-comment clearfix">
			<div class="comment-author vcard"> <?php echo get_avatar($comment,$size='80'); ?></div>
			<div class="comment-meta commentmetadata">
				<?php if ($comment->comment_approved == '0') { ?>
					<em><?php esc_html_e('Comment is awaiting moderation','plug-shop');?></em> <br />
				<?php } ?>
				<div class="psf-comment-heading">
					<div class="psf-comment-title">
							<span class="psf-comment-date">
								<?php echo get_comment_date() . '  -  ' . get_comment_time(); ?>
							</span>
						<h5><?php echo esc_html__('By','plug-shop') . ' ' . get_comment_author_link( get_comment_ID() ) . ' '; ?></h5>
					</div>
				</div>
				<?php comment_text(); ?>
				<div class="psf-comment-links">
					<?php edit_comment_link( esc_html__('Edit comment','plug-shop'),'  ','' ); ?>
					<?php comment_reply_link( array_merge( $args, array( 'reply_text' => esc_html__( 'Reply','plug-shop'),'depth' => $depth, 'max_depth' => $args['max_depth'] ) ) ); ?>
				</div>
			</div>
			<!-- </li> -->
	<?php  
	}
}


/**
 * 
 * Format the font
 *
 * @since 1.0.0
 */
function plug_shop_format_font($data)
{
	$query_args = '';

	if( is_array($data) && !empty($data) )
	{
		$font  = str_replace("+"," ", $data['font-family']);

		if( isset($data['variant']) && 'regular' != $data['variant'] )
			$font 	= $font.':'.$data['variant'];

		if( isset( $data['subsets'] ) && is_array( $data['subsets'] ) )
			$font 	= $font.':'.implode( ', ', $data['subsets'] );
		else if( isset( $data['subsets'] ) )
			 $font 	= $font.':'.$data['subsets'];

		$query_args = array( 'family' => $font );
	}

	return $query_args;
}

/**
 * 
 * Customizer fallback typography
 *
 * @since 1.0.0
 */
function plug_shop_customizer_cb($font)
{
	return array(
		'font-family'    => $font,
		'font-size'      => '',
		'letter-spacing' => '',
		'color'          => '',
		'text-align'     => '',
		'text-transform' => '',
		'line-height'    => '',
		'variant'        => ''
	);
}

/**
 * 
 * Custoizer styles join
 *
 * @since 1.0.0
 */
function plug_shop_join_styles($css)
{
	$key = key($css);

	return $css[$key];
}

/**
 * 
 * Get thumbnail image src
 *
 * @since 1.0.0
 */
function plug_shop_get_thumb_src($post)
{
	$img_id  = get_post_thumbnail_id( $post );
	$img     = wp_get_attachment_image_src($img_id, 'full');
	$img_src = is_array($img) && isset($img[0]) ? $img[0] : '';

	return $img_src;
}

/**
 * 
 * Get custom post types
 *
 * @since 1.0.0
 */
function plug_shop_dynamic_post_selection($name, $selected = '')
{
	$types = array( 'post', esc_html__('Posts','plug-shop') );
	$wc_exists = plug_shop_is_wc_installed();

	$html  = '<select name="'.$name.'" id="'.$name.'">';
	$html .= $wc_exists ? '<option '.selected('dynamic_post_type', $selected, false).' value="'.$types[0].'">'.$types[1].'</option>' : '<option value=""></option>';
	$html .= apply_filters( 'plug_shop_dynamic_post_types', '', $selected );
	$html .= '</select>';
	$html .= sprintf('<p class="description">%1$s</p>', esc_html__('Select the post type you want to integrate into the template. This tempalte will list the selected post type, keeping the original theme layout for blog list. In Pro version you will be able to select any custom post type.', 'plug-shop'));

	return $html;
}

/**
 *
 * Get custom post types
 *
 * @since 1.0.0
 */
function plug_shop_dynamic_post_args()
{
	$id = get_queried_object_id();
	$name = 'dynamic_post_type';
	$selected = get_post_meta($id, $name, true );

	$args = array(
			'post_type' => $selected,
			'posts_per_page' => get_option('posts_per_page')
	);
	$args['paged'] = get_query_var( 'paged' ) ? get_query_var( 'paged' ) : 1;

	return apply_filters('plug_shop_dynamic_posts_args', $args);
}

/**
 *
 * Check if Winning Portfolio Plugin Is Active
 *
 * @since 1.0.0
 */
function plug_shop_is_wpf_active()
{
	return class_exists('WPF_Portfolio');
}

/**
 *
 * Wrapper function for related posts class
 *
 * @since 1.0.0
 */
function plug_shop_related_posts($type)
{
	return new PF_Related_Posts($type);
}

function plug_shop_is_wc_installed() {
	/**
	 *
	 * Get thumbnail image src
	 *
	 * @since 1.0.0
	 */
	if( class_exists( 'WooCommerce' ) )	{
		return true;
	}

	return false;
}

/**
 * 
 * Check if it is woocommerce tempalte
 *
 * @since 1.0.0
 */
function plug_shop_is_wc_template()
{
	return function_exists ( "is_woocommerce" ) && is_woocommerce();
}

/**
 * Retrieve the general archive title.
 *
 * @since  3.0.0
 * @access public
 * @return string
 */
function plug_shop_get_single_archive_title() {
	return esc_html__( 'Archives', 'plug-shop' );
}

/**
 * Retrieve the author archive title.
 *
 * @since  1.0.0
 * @access public
 * @return string
 */
function plug_shop_single_author_title() {
	return get_the_author_meta( 'display_name', absint( get_query_var( 'author' ) ) );
}

/**
 * Print the year archive title.
 *
 * @since  1.0.0
 * @access public
 * @return void
 */
function plug_shop_single_year_title() {
	echo plug_shop_get_single_year_title();
}

/**
 * Retrieve the year archive title.
 *
 * @since  1.0.0
 * @access public
 * @return string
 */
function plug_shop_get_single_year_title() {
	return get_the_date( esc_html_x( 'Y', 'yearly archives date format', 'plug-shop' ) );
}

/**
 * Print the week archive title.
 *
 * @since  1.0.0
 * @access public
 * @return void
 */
function plug_shop_single_week_title() {
	echo plug_shop_get_single_week_title();
}

/**
 * Retrieve the week archive title.
 *
 * @since  1.0.0
 * @access public
 * @return string
 */
function plug_shop_get_single_week_title() {

	// Translators: 1 is the week number and 2 is the year.
	return sprintf(
			'%1$s %2$s of %3$s',
			esc_html__( 'Week', 'plug-shop' ),
			get_the_time( esc_html_x( 'W', 'weekly archives date format', 'plug-shop' ) ),
			get_the_time( esc_html_x( 'Y', 'yearly archives date format', 'plug-shop' ) )
	);
}

/**
 * Print the day archive title.
 *
 * @since  1.0.0
 * @access public
 * @return void
 */
function plug_shop_single_day_title() {
	echo plug_shop_get_single_day_title();
}

/**
 * Retrieve the day archive title.
 *
 * @since  1.0.0
 * @access public
 * @return string
 */
function plug_shop_get_single_day_title() {
	return get_the_date( esc_html_x( 'F j, Y', 'daily archives date format', 'plug-shop' ) );
}

/**
 * Print the hour archive title.
 *
 * @since  1.0.0
 * @access public
 * @return void
 */
function plug_shop_single_hour_title() {
	echo plug_shop_get_single_hour_title();
}

/**
 * Retrieve the hour archive title.
 *
 * @since  1.0.0
 * @access public
 * @return string
 */
function plug_shop_get_single_hour_title() {
	return get_the_time( esc_html_x( 'g a', 'hour archives time format', 'plug-shop' ) );
}

/**
 * Print the minute archive title.
 *
 * @since  1.0.0
 * @access public
 * @return void
 */
function plug_shop_single_minute_title() {
	echo plug_shop_get_single_minute_title();
}

/**
 * Retrieve the minute archive title.
 *
 * @since  1.0.0
 * @access public
 * @return string
 */
function plug_shop_get_single_minute_title() {

	// Translators: Minute archive title. %s is the minute time format.
	return sprintf(
			'%s %s',
			esc_html__( 'Minute', 'plug-shop' ),
			get_the_time( esc_html_x( 'i', 'minute archives time format', 'plug-shop' ) )
	);
}

/**
 * Print the minute + hour archive title.
 *
 * @since  1.0.0
 * @access public
 * @return void
 */
function plug_shop_single_minute_hour_title() {
	echo plug_shop_get_single_minute_hour_title();
}

/**
 * Retrieve the minute + hour archive title.
 *
 * @since  1.0.0
 * @access public
 * @return string
 */
function plug_shop_get_single_minute_hour_title() {
	return get_the_time( esc_html_x( 'g:i a', 'minute and hour archives time format', 'plug-shop' ) );
}

/**
 * Print the search results title.
 *
 * @since  1.0.0
 * @access public
 * @return void
 */
function plug_shop_search_title() {
	echo plug_shop_get_search_title();
}

/**
 * Retrieve the search results title.
 *
 * @since  1.0.0
 * @access public
 * @return string
 */
function plug_shop_get_search_title() {

	// Translators: %s is the search query. The HTML entities are opening and closing curly quotes.
	return sprintf(
			'%s &#8220;%s&#8221;',
			esc_html__( 'Search results for', 'plug-shop' ),
			get_search_query()
	);
}

/**
 * Retrieve the 404 page title.
 *
 * @since  1.0.0
 * @access public
 * @return void
 */
function plug_shop_404_title() {
	echo plug_shop_get_404_title();
}

/**
 * Retrieve the 404 page title.
 *
 * @since  1.0.0
 * @access public
 * @return string
 */
function plug_shop_get_404_title() {
	return esc_html__( '404 Not Found', 'plug-shop' );
}