<?php
/**
 * Core functions for initializing required theme functions
 *
 * @package Plug Shop WordPress Theme
 * @subpackage Plug Shop
 * @author Pressfore - www.pressfore.com
 *
 */


/*-----------------------------------------------------------------------------------*/
/* Register Core Stylesheets and javascript
/* These are necessary for the theme to function as intended
/*-----------------------------------------------------------------------------------*/

if ( ! function_exists( 'plug_shop_scripts' ) ) {

	function plug_shop_scripts() {
		// Set a dynamic version for cache busting.
		$version = plug_shop_get_version();

		// If dev mode is defined, load unminified js.
		if( defined('PLUG_STORE_DEV') && true === PLUG_STORE_DEV ) {
			$main = 'main.js';
		} else {
			$main = 'main.min.js';
		}

		if ( is_singular() && comments_open() && get_option( 'thread_comments' ) ) {
			wp_enqueue_script( 'comment-reply' );
		}

		// Primary theme stylesheet.
		wp_enqueue_style( 'plug-shop-core', get_template_directory_uri() . '/assets/css/theme.min.css' );
		wp_enqueue_style( 'plug-shop', get_stylesheet_uri() );

		// Javascript.
		wp_enqueue_script( 'plug-shop-plugins', get_template_directory_uri() . '/assets/js/plugins.min.js', array('jquery'), $version, true );
		wp_enqueue_script( 'plug-shop', get_template_directory_uri() . '/assets/js/'.$main, array('jquery'), $version, true );
		wp_enqueue_style( 'pressfore-icons', get_template_directory_uri() .'/assets/css/pressfore-icons.css' );

	}

	add_action( 'wp_enqueue_scripts', 'plug_shop_scripts');
}


if ( !function_exists( 'plug_shop_admin_scripts' ) ) {
	function plug_shop_admin_scripts() {
		$screen = get_current_screen();

		if( is_admin() && 'page' === $screen->post_type ) {
			wp_enqueue_style('plug-shop-admin', get_template_directory_uri() . '/assets/css/admin.css');
		}

		wp_enqueue_script('plug-shop-admin', get_template_directory_uri() . '/assets/js/admin/admin.js', array(), null);
	}

	add_action( 'admin_enqueue_scripts', 'plug_shop_admin_scripts');
}


/** Tell WordPress to run plug_shop_setup() when the 'after_setup_theme' hook is run. */

/**
 * Sets up theme defaults and registers support for various WordPress features.
 *
 * Note that this function is hooked into the after_setup_theme hook, which runs
 * before the init hook. The init hook is too late for some features, such as indicating
 * support post thumbnails.
 *
 * To override plug_shop_setup() in a child theme, add your own plug_shop_setup to your child theme's
 * functions.php file.
 *
 */
if ( ! function_exists( 'plug_shop_setup' ) ) {

	function plug_shop_setup() {
		// Translations can be filed in the /languages/ directory
		load_theme_textdomain( 'plug-shop', get_template_directory() . '/languages' );

		add_editor_style();

		add_theme_support( 'title-tag' );
		add_theme_support( 'automatic-feed-links' );
		add_theme_support( 'post-thumbnails' );

		add_theme_support( 'custom-background', array(
			'default-color'  => '#f9f9f9'
		));

		// Add custom header support.
		add_theme_support( 'custom-header', apply_filters( 'plug_shop_custom_header_args', array(
				'default-image'      => get_parent_theme_file_uri( '/assets/images/header.jpg' ),
				'width'              => 2000,
				'height'             => 1200,
				'flex-height'        => true,
				'video'              => true,
		) ) );

		add_theme_support( 'custom-logo', array(
				'height'      => 40,
				'width'       => 150,
				'flex-height' => true,
				'flex-width'  => true,
				'header-text' => array( 'site-title', 'site-description' ),
		) );

		// WooCommerce custom support
		add_theme_support( 'wc-product-gallery-zoom' );
		add_theme_support( 'wc-product-gallery-lightbox' );
		add_theme_support( 'wc-product-gallery-slider' );

		// Add theme support for selective refresh for widgets.
		add_theme_support( 'customize-selective-refresh-widgets' );

		$post_formats = apply_filters( 'plug_shop_post_formats_support' , array( 'audio', 'video', 'gallery', 'link', 'quote' ) );
		add_theme_support( 'post-formats', $post_formats );

		$crop = plug_shop_options('crop_thumb', 'no');
		if( 'no' === $crop ) $crop = false;
		else $crop = true;

	  	set_post_thumbnail_size( 320, 210, $crop );
		// 150px square
		add_image_size( 'plug-shop-squared150', 150, 150, true );
		// 250px square
		add_image_size( 'plug-shop-squared250', 250, 250, true );
		// 320px rectangle
		add_image_size( 'plug-shop-rectangle320', 320, 250, true );
		// 4:3 Video
		add_image_size( 'plug-shop-video43', 320, 240, true );
		// 16:9 Video
		add_image_size( 'plug-shop-video169', 320, 180, true );

		register_nav_menus( array(
			'primary' => esc_html__( 'Primary Navigation', 'plug-shop' ),
			'footer'  => esc_html__( 'Footer Navigation', 'plug-shop' )
		));
	}

	add_action( 'after_setup_theme', 'plug_shop_setup' );
}

/*-----------------------------------------------------------------------------------*/
//	Register widgetized areas, including two sidebars and four widget-ready columns in the footer.
//	To override plug_shop_widgets_init() in a child theme, remove the action hook and add your own
//	function tied to the init hook.
/*-----------------------------------------------------------------------------------*/

if ( !function_exists( 'plug_shop_widgets_init' ) ) {
	function plug_shop_widgets_init() {
		// Area 1, Posts Widget area
		register_sidebar( array(
			'id'            => 'sidebar-1',
			'name'          => esc_html__( 'Posts Widget Area', 'plug-shop' ),
			'description'   => esc_html__( 'Shown only in Blog Posts, Archives, Categories, etc.', 'plug-shop' ),
			'before_widget' => '<div id="%1$s" class="widget-container %2$s">',
			'after_widget'  => '</div>',
			'before_title'  => '<h4 class="widget-title">',
			'after_title'   => '</h4>',
		) );


		// Area 2, located on pages
		register_sidebar( array(
			'id'            => 'sidebar-2',
			'name'          => esc_html__( 'Pages Widget Area', 'plug-shop' ),
			'description'   => esc_html__( 'Shown only in Pages', 'plug-shop' ),
			'before_widget' => '<div id="%1$s" class="widget widget-container %2$s">',
			'after_widget'  => '</div>',
			'before_title'  => '<h4 class="widget-title">',
			'after_title'   => '</h4>',
		) );

		// Area 3, located on woocommerce pages
		if( plug_shop_is_wc_installed() )
			register_sidebar( array(
				'id'            => 'sidebar-3',
				'name'          => esc_html__( 'Shop Widget Area', 'plug-shop' ),
				'description'   => esc_html__( 'Shown only in Woocommerce pages and products', 'plug-shop' ),
				'before_widget' => '<div id="%1$s" class="widget widget-container %2$s">',
				'after_widget'  => '</div>',
				'before_title'  => '<h4 class="widget-title">',
				'after_title'   => '</h4>',
			) );

		// Area located in the header.
		register_sidebar( array(
			'id'            => 'sidebar-header',
			'name'          => esc_html__( 'Header Widget Area', 'plug-shop' ),
			'description'   => esc_html__( 'Shown in header after the main menu in the right corner.', 'plug-shop' ),
			'before_widget' => '<div id="%1$s" class="widget widget-container %2$s">',
			'after_widget'  => '</div>',
			'before_title'  => '<h4 class="widget-title">',
			'after_title'   => '</h4>',
		) );

		// Area 3, located in the footer. Empty by default.
		register_sidebar( array(
			'id'            => 'footer-widget-area-1',
			'name'          => esc_html__( 'First Footer Widget Area', 'plug-shop' ),
			'description'   => esc_html__( 'The first footer widget area', 'plug-shop' ),
			'before_widget' => '<div id="%1$s" class="widget-container %2$s">',
			'after_widget'  => '</div>',
			'before_title'  => '<h4 class="widget-title">',
			'after_title'   => '</h4>',
		) );

		// Area 4, located in the footer. Empty by default.
		register_sidebar( array(
			'id'            => 'footer-widget-area-2',
			'name'          => esc_html__( 'Second Footer Widget Area', 'plug-shop' ),
			'description'   => esc_html__( 'The second footer widget area', 'plug-shop' ),
			'before_widget' => '<div id="%1$s" class="widget-container %2$s">',
			'after_widget'  => '</div>',
			'before_title'  => '<h4 class="widget-title">',
			'after_title'   => '</h4>',
		) );

		// Area 5, located in the footer. Empty by default.
		register_sidebar( array(
			'id'            => 'footer-widget-area-3',
			'name'          => esc_html__( 'Third Footer Widget Area', 'plug-shop' ),
			'description'   => esc_html__( 'The third footer widget area', 'plug-shop' ),
			'before_widget' => '<div id="%1$s" class="widget-container %2$s">',
			'after_widget'  => '</div>',
			'before_title'  => '<h4 class="widget-title">',
			'after_title'   => '</h4>',
		) );

		// Area 6, located in the footer. Empty by default.
		register_sidebar( array(
			'id'            => 'footer-widget-area-4',
			'name'          => esc_html__( 'Fourth Footer Widget Area', 'plug-shop' ),
			'description'   => esc_html__( 'The fourth footer widget area', 'plug-shop' ),
			'before_widget' => '<div id="%1$s" class="widget-container %2$s">',
			'after_widget'  => '</div>',
			'before_title'  => '<h4 class="widget-title">',
			'after_title'   => '</h4>',
		) );
	}

	add_action( 'widgets_init', 'plug_shop_widgets_init' );
}

add_action('after_switch_theme', 'plug_shop_notice_field_remove');

function plug_shop_notice_field_remove () {
	delete_option( 'plug_shop_dismissed_notice' );
}

function plug_shop_intro_notice() {
	if ( is_admin() && current_user_can( 'manage_options' ) && ! get_option( 'plug_shop_dismissed_notice' ) ) {

		$dismiss_url = add_query_arg( array(
				'plug_shop_dismissed' => 'plug_shop_dismissed_notice'
		), admin_url() );

		?>
		<div class="plug-shop-notice notice notice-info is-dismissible" data-dismiss-url="<?php echo esc_url( $dismiss_url ); ?>">
			<?php printf( '<p style="font-size: 1.13em">%s <strong>%s</strong> %s <a href="%s">%s</a> %s <a href="%s">%s</a> %s <a href="%s" target="_blank">%s</a> %s</p>',
					esc_html__('Thank you for using', 'plug-shop'),
					esc_html__('Plug Shop!', 'plug-shop'),
					esc_html__('To get started (and get demo content), head up to the', 'plug-shop'),
					esc_url('themes.php?page=plug-shop-getting-started'),
					esc_html__('Getting Started', 'plug-shop'),
					esc_html__('page by clicking the link, or by going to Appearance -> Getting Started. If you want even more features check our', 'plug-shop'),
					esc_url('http://pressfore.com/item/plug-shop-pro/'),
					esc_html__('PRO version', 'plug-shop'),
					esc_html__('You can visit our online', 'plug-shop'),
					esc_url('http://docs.pressfore.com/plug-shop/'),
					esc_html__('documentation', 'plug-shop'),
					esc_html__('as well. We hope you will make something awesome out of our theme!', 'plug-shop') );
			?>
		</div>
		<?php
	}
}

add_action('admin_notices', 'plug_shop_intro_notice');

function plug_shop_action_admin_init() {
	$dismiss_option = filter_input( INPUT_GET, 'plug_shop_dismissed', FILTER_SANITIZE_STRING );
	if ( is_string( $dismiss_option ) ) {
		update_option( 'plug_shop_dismissed_notice', true );
		wp_die();
	}
}

add_action( 'admin_init', 'plug_shop_action_admin_init' );