<?php
/**
 * Pixilate functions and definitions
 *
 * @package Pixilate
 */

/**
 * Set the content width based on the theme's design and stylesheet.
 */
if ( ! isset( $content_width ) ) {
	$content_width = 640; /* pixels */
}

if ( ! function_exists( 'pixilate_setup' ) ) :
/**
 * Sets up theme defaults and registers support for various WordPress features.
 *
 * Note that this function is hooked into the after_setup_theme hook, which
 * runs before the init hook. The init hook is too late for some features, such
 * as indicating support for post thumbnails.
 */
function pixilate_setup() {

	/*
	 * Make theme available for translation.
	 * Translations can be filed in the /languages/ directory.
	 * If you're building a theme based on Pixilate, use a find and replace
	 * to change 'pixilate' to the name of your theme in all the template files
	 */
	load_theme_textdomain( 'pixilate', get_template_directory() . '/languages' );

	// Add default posts and comments RSS feed links to head.
	add_theme_support( 'automatic-feed-links' );

	/*
	 * Enable support for Post Thumbnails on posts and pages.
	 *
	 * @link http://codex.wordpress.org/Function_Reference/add_theme_support#Post_Thumbnails
	 */
	add_theme_support( 'post-thumbnails' );

	// This theme uses wp_nav_menu() in one location.
	register_nav_menus( array(
		'primary' => __( 'Primary Menu', 'pixilate' ),
	) );

	// Enable support for Post Formats.
	add_theme_support( 'post-formats', array( 'aside', 'image', 'video', 'quote', 'link' ) );
}
endif; // pixilate_setup
add_action( 'after_setup_theme', 'pixilate_setup' );

/**
 * Register widgetized area and update sidebar with default widgets.
 */
function pixilate_widgets_init() {
	register_sidebar( array(
		'name'          => __( 'Sidebar', 'pixilate' ),
		'description'   => __( 'The primary sidebar that appears in the footer.', 'pixilate' ),
		'id'            => 'primary-sidebar',
		'before_widget' => '<aside id="%1$s" class="widget %2$s">',
		'after_widget'  => '</aside>',
		'before_title'  => '<h2 class="widget-title">',
		'after_title'   => '</h2>',
	) );
}
add_action( 'widgets_init', 'pixilate_widgets_init' );

/**
 * Enqueue scripts and styles.
 */
function pixilate_scripts() {
	wp_enqueue_style( 'pixilate-style', get_stylesheet_uri() );

	wp_enqueue_script( 'pixilate-navigation', get_template_directory_uri() . '/js/navigation.js', array(), '20120206', true );

	wp_enqueue_script( 'pixilate-skip-link-focus-fix', get_template_directory_uri() . '/js/skip-link-focus-fix.js', array(), '20130115', true );
	
	// Call in Masonry
	wp_enqueue_script( 'jquery-masonry' );
	
	// Activate various JS
	wp_enqueue_script( 'pixilate-theme', get_template_directory_uri() . '/js/theme.js', array( 'jquery-masonry' ), null, false );
	
	if ( is_singular() && comments_open() && get_option( 'thread_comments' ) ) {
		wp_enqueue_script( 'comment-reply' );
	}
}
add_action( 'wp_enqueue_scripts', 'pixilate_scripts' );

/**
 * Arrange items into rows.
 * 
 * http://slobodanmanic.com/444/flexible-horizontal-sidebars-wordpress-twenty-thirteen-child-theme/
 *
 * This function takes an integer and returns text. Used to arrange posts on search pages, archives, others
 */
function pixilate_arrange_items( $count ) {
	$item_class = 'arranged-items arranged-items-small arranged-item-count-' . $count;
	
	if ( $count % 4 == 0 || $count > 6 ) :
		// Four items per row if there are exactly four items or more than six
		$item_class .= ' four-per-row';
	elseif ( $count >=3 ) :
		// Three items per row if there's three or more
		$item_class .= ' three-per-row';
	elseif ( $count == 2 ) :
		// Two items per row if there's only two
		$item_class .= ' two-per-row';
	elseif ( $count == 1 ) :
		// One item per row if there's only one
		$item_class .= ' one-per-row';
	endif;
	
	return $item_class;
}

/**
 * Make "read more" link to the post
 */
function pixilate_excerpt_link( $more ) {
	return ' <a class="read-more" href="'. get_permalink( get_the_ID() ) . '">[...]</a>';
}
add_filter( 'excerpt_more', 'pixilate_excerpt_link' );

/**
 * Add author's first name to comment reply link
 * Thanks to Raam Dev
 * http://raamdev.com/2013/personalizing-the-wordpress-comment-reply-link/
 */
function pixilate_comment_author_to_reply_link( $link, $args, $comment ) {
	$comment = get_comment( $comment );
 
	// If no comment author is blank, use 'Anonymous'
	if ( empty( $comment->comment_author ) ) :
        	if ( ! empty( $comment->user_id ) ) :
			$user = get_userdata( $comment->user_id );
			$author = $user->user_login;
		else :
			$author = __( 'Anonymous', 'windsorknot' );
		endif;
    else :
        $author = $comment->comment_author;
    endif;
 
    // If the user provided more than a first name, use only first name
    if ( strpos( $author, ' ' ) ) :
        $author = substr( $author, 0, strpos( $author, ' ' ) );
    endif;
 
    // Replace Reply Link with "Reply to (Author First Name)"
    $reply_link_text = $args[ 'reply_text' ];
    $link = str_replace( $reply_link_text, 'Reply to ' . $author, $link );
 
    return $link;
}
add_filter( 'comment_reply_link', 'pixilate_comment_author_to_reply_link', 10, 3 );

/**
 * Modify the query on archives/search to ignore the posts_per_page setting
 */
function pixilate_posts_per_page( $query ) {
	if ( $query->is_main_query() && !is_admin() && is_archive() || is_search() ) :
		$query->set( 'posts_per_page', '-1' );
	endif;
	
	return $query;
}
add_filter( 'pre_get_posts', 'pixilate_posts_per_page' );

/**
 * Strip out all HTML tags except for ones that are explicitly allowed
 */
function pixilate_strip_tags( $commentdata ) {
	$commentdata[ 'comment_content' ] = wp_kses_data( $commentdata[ 'comment_content' ] );
	
	return $commentdata;	
}
add_filter( 'preprocess_comment', 'pixilate_strip_tags', 10, 1 );

/**
 * Implement the custom editor styling
 */
function pixilate_custom_editor() {
	add_editor_style( 'editor-style.css' );
}
add_action( 'init', 'pixilate_custom_editor' );

/**
 * Implement the Custom Header feature.
 */
// require get_template_directory() . '/inc/custom-header.php';

/**
 * Custom template tags for this theme.
 */
require get_template_directory() . '/inc/template-tags.php';

/**
 * Custom functions that act independently of the theme templates.
 */
require get_template_directory() . '/inc/extras.php';

/**
 * Customizer additions.
 */
require get_template_directory() . '/inc/customizer.php';

/**
 * Load Jetpack compatibility file.
 */
// require get_template_directory() . '/inc/jetpack.php';