<?php
// Exit if accessed directly
if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

/**
 * Add Theme Info Page to WP Admin
 */
function photo_nook_add_about_theme_page() {
    add_theme_page(
        __( 'About Photo Nook', 'photo-nook' ),
        __( 'Photo Nook', 'photo-nook' ),
        'edit_theme_options',
        'photo-nook-about',
        'photo_nook_render_about_page'
    );
}
add_action( 'admin_menu', 'photo_nook_add_about_theme_page' );

/**
 * Enqueue admin styles and JS
 */
function photo_nook_admin_assets( $hook ) {
    if ( $hook === 'appearance_page_photo-nook-about' || $hook === 'index.php' ) {
        wp_enqueue_style(
            'photo-nook-admin-css',
            get_template_directory_uri() . '/admin/about-theme.css',
            array(),
            wp_get_theme()->get( 'Version' )
        );
    }

    // Enqueue JS for dismissing the notice
    // Enqueue JS for dismissing the notice (on all admin pages)
    if ( $hook === 'index.php' || strpos( $hook, 'appearance_page' ) !== false ) {
        wp_enqueue_script(
            'photo-nook-admin-js',
            get_template_directory_uri() . '/admin/admin-notice.js',
            array( 'jquery' ),
            wp_get_theme()->get( 'Version' ),
            true
        );

        wp_localize_script( 'photo-nook-admin-js', 'photoNookNotice', array(
            'ajax_url' => admin_url( 'admin-ajax.php' ),
            'nonce'    => wp_create_nonce( 'photo_nook_notice_nonce' )
        ) );
    }
}
add_action( 'admin_enqueue_scripts', 'photo_nook_admin_assets' );

/**
 * Output a dismissible notice unless dismissed
 */
function photo_nook_show_admin_notice() {
    if ( ! current_user_can( 'edit_theme_options' ) ) {
        return;
    }

    // Don't show on the theme info page itself
    $screen = get_current_screen();
    if ( $screen && $screen->id === 'appearance_page_photo-nook-about' ) {
        return;
    }

    $user_id = get_current_user_id();
    $dismissed_users = get_option( 'photo_nook_dismissed_users', array() );
    
    // Check if current user has dismissed
    if ( in_array( $user_id, $dismissed_users ) ) {
        return;
    }

    $about_url = admin_url( 'themes.php?page=photo-nook-about' );
    ?>
    <div id="photo-nook-notice" class="notice notice-success is-dismissible photo-nook-notice">
        <p>
            <?php esc_html_e( 'Thanks for installing Photo Nook! Get started by visiting the theme info page.', 'photo-nook' ); ?>
            <br />
            <a href="<?php echo esc_url( $about_url ); ?>" class="button button-primary" style="margin-top: 6px;">
                <?php esc_html_e( 'Learn More', 'photo-nook' ); ?>
            </a>
        </p>
    </div>
    <?php
}
add_action( 'admin_notices', 'photo_nook_show_admin_notice' );

/**
 * Handle AJAX dismissal
 */
function photo_nook_dismiss_notice() {
    check_ajax_referer( 'photo_nook_notice_nonce', 'nonce' );

    if ( ! current_user_can( 'edit_theme_options' ) ) {
        wp_send_json_error( array( 'message' => 'Unauthorized' ) );
        exit;
    }

    $user_id = get_current_user_id();
    $dismissed_users = get_option( 'photo_nook_dismissed_users', array() );
    
    // Add current user to dismissed list if not already there
    if ( ! in_array( $user_id, $dismissed_users ) ) {
        $dismissed_users[] = $user_id;
        update_option( 'photo_nook_dismissed_users', $dismissed_users );
    }
    
    wp_send_json_success();
}
add_action( 'wp_ajax_photo_nook_dismiss_notice', 'photo_nook_dismiss_notice' );

/**
 * Reset dismissal flag on theme switch
 */
function photo_nook_reset_notice_on_switch() {
    delete_option( 'photo_nook_dismissed_users' );
}
add_action( 'after_switch_theme', 'photo_nook_reset_notice_on_switch' );

/**
 * Render About Page
 */
function photo_nook_render_about_page() {
    $pro_url   = 'https://thethemenook.com/products/photo-nook-pro/';
    $demo_url  = 'http://thethemenook.com/photo-nook-pro/';
    $image_url = get_template_directory_uri() . '/admin/images/photo_nook_promo_collage.jpg';
    ?>
    <div class="photo-nook-about-wrap">
        <div class="photo-nook-promo-layout">
            <div class="photo-nook-promo-text">
                <h2><?php esc_html_e( 'Elevate your portfolio with Photo Nook Pro', 'photo-nook' ); ?></h2>

                <p><?php esc_html_e( 'A soft, elegant WordPress theme designed for wedding photographers who want to showcase their work beautifully.', 'photo-nook' ); ?></p>
                <p><?php esc_html_e( 'Photo Nook Pro is built for photographers who value timeless design, intentional layouts, and a portfolio-focused website that attracts dream clients. It’s refined, easy to customize, and crafted to let your images shine — no coding required.', 'photo-nook' ); ?></p>
                <p><?php esc_html_e( 'It’s simple to set up, mobile-friendly, and created to grow with your business — whether you’re updating your first photography site or refreshing a seasoned brand.', 'photo-nook' ); ?></p>

                <h3><?php esc_html_e( 'Photo Nook Pro comes with:', 'photo-nook' ); ?></h3>
                <ul class="photo-nook-feature-list">
                    <li><?php esc_html_e( 'Extra page templates: About, Services, Portfolio, and Contact.', 'photo-nook' ); ?></li>
                    <li><?php esc_html_e( 'Thoughtfully designed patterns for weddings, elopements, and destination galleries.', 'photo-nook' ); ?></li>
                    <li><?php esc_html_e( 'Multiple color palettes + curated font pairings for different stylistic moods.', 'photo-nook' ); ?></li>
                    <li><?php esc_html_e( '20+ extra patterns that make creating new pages a breeze.', 'photo-nook' ); ?></li>
                    <li><?php esc_html_e( 'Easy demo import so you can launch a polished site in minutes.', 'photo-nook' ); ?></li>
                </ul>

                <p class="photo-nook-buttons">
                    <a href="<?php echo esc_url( $demo_url ); ?>" class="button button-secondary button-hero" target="_blank" style="margin-left: 10px;">
                        <?php esc_html_e( 'View Demo', 'photo-nook' ); ?>
                    </a>
                    <a href="<?php echo esc_url( $pro_url ); ?>" class="button button-primary button-hero" target="_blank">
                        <?php esc_html_e( 'Get Photo Nook Pro', 'photo-nook' ); ?>
                    </a>
                </p>
            </div>

            <div class="photo-nook-promo-image">
                <img src="<?php echo esc_url( $image_url ); ?>" alt="<?php esc_attr_e( 'Photo Nook Pro', 'photo-nook' ); ?>" style="max-width: 100%; height: auto; border: 1px solid #ddd; border-radius: 6px;">
            </div>
        </div>
    </div>
    <?php
}