<?php
/**
 * Custom functions that act independently of the theme templates.
 *
 * Eventually, some of the functionality here could be replaced by core features.
 *
 * @package organic_gardening
 */

/**
 * Adds custom classes to the array of body classes.
 *
 * @param array $classes Classes for the body element.
 * @return array
 */
function organic_gardening_body_classes( $classes ) {
  global $organic_gardening_post;
  
    if( !is_page_template( 'template-home.php' ) ){
        $classes[] = 'inner';
        // Adds a class of group-blog to blogs with more than 1 published author.
    }

    if ( is_multi_author() ) {
        $classes[] = 'group-blog ';
    }

    // Adds a class of custom-background-image to sites with a custom background image.
    if ( get_background_image() ) {
        $classes[] = 'custom-background-image';
    }
    
    // Adds a class of custom-background-color to sites with a custom background color.
    if ( get_background_color() != 'ffffff' ) {
        $classes[] = 'custom-background-color';
    }
    

    if( organic_gardening_woocommerce_activated() && ( is_shop() || is_product_category() || is_product_tag() || 'product' === get_post_type() ) && ! is_active_sidebar( 'shop-sidebar' ) ){
        $classes[] = 'full-width';
    }    

    // Adds a class of hfeed to non-singular pages.
    if ( ! is_page() ) {
        $classes[] = 'hfeed ';
    }
  
    if( is_404() ||  is_search() ){
        $classes[] = 'full-width';
    }
  
    if( ! is_active_sidebar( 'right-sidebar' ) ) {
        $classes[] = 'full-width'; 
    }

    return $classes;
}
add_filter( 'body_class', 'organic_gardening_body_classes' );

 /**
 * 
 * @link http://www.altafweb.com/2011/12/remove-specific-tag-from-php-string.html
 */
function organic_gardening_strip_single( $tag, $string ){
    $string=preg_replace('/<'.$tag.'[^>]*>/i', '', $string);
    $string=preg_replace('/<\/'.$tag.'>/i', '', $string);
    return $string;
}

if ( ! function_exists( 'organic_gardening_excerpt_more' ) ) :
/**
 * Replaces "[...]" (appended to automatically generated excerpts) with ... * 
 */
function organic_gardening_excerpt_more($more) {
  return is_admin() ? $more : ' &hellip; ';
}
endif;
add_filter( 'excerpt_more', 'organic_gardening_excerpt_more' );


if( ! function_exists( 'organic_gardening_footer_credit' ) ):
/**
 * Footer Credits
*/
function organic_gardening_footer_credit() {

    // Check if footer copyright is enabled
    $organic_gardening_show_footer_copyright = get_theme_mod( 'organic_gardening_footer_setting', true );

    if ( ! $organic_gardening_show_footer_copyright ) {
        return; 
    }

    $organic_gardening_copyright_text = get_theme_mod('organic_gardening_footer_copyright_text');

    $organic_gardening_text = '<div class="site-info"><div class="container"><span class="copyright">';
    if ($organic_gardening_copyright_text) {
        $organic_gardening_text .= wp_kses_post($organic_gardening_copyright_text); 
    } else {
        $organic_gardening_text .= esc_html__('&copy; ', 'organic-gardening') . date_i18n(esc_html__('Y', 'organic-gardening')); 
        $organic_gardening_text .= ' <a href="' . esc_url(home_url('/')) . '">' . esc_html(get_bloginfo('name')) . '</a>' . esc_html__('. All Rights Reserved.', 'organic-gardening');
    }
    $organic_gardening_text .= '</span>';
    $organic_gardening_text .= '<span class="by"> <a href="' . esc_url('https://www.themeignite.com/products/free-gardening-wordpress-theme') . '" rel="nofollow" target="_blank">' . ORGANIC_GARDENING_THEME_NAME . '</a>' . esc_html__(' By ', 'organic-gardening') . '<a href="' . esc_url('https://themeignite.com/') . '" rel="nofollow" target="_blank">' . esc_html__('Themeignite', 'organic-gardening') . '</a>.';
    $organic_gardening_text .= sprintf(esc_html__(' Powered By %s', 'organic-gardening'), '<a href="' . esc_url(__('https://wordpress.org/', 'organic-gardening')) . '" target="_blank">WordPress</a>.');
    if (function_exists('the_privacy_policy_link')) {
        $organic_gardening_text .= get_the_privacy_policy_link();
    }
    $organic_gardening_text .= '</span></div></div>';
    echo apply_filters('organic_gardening_footer_text', $organic_gardening_text);
}
add_action('organic_gardening_footer', 'organic_gardening_footer_credit');
endif;

/**
 * Is Woocommerce activated
*/
if ( ! function_exists( 'organic_gardening_woocommerce_activated' ) ) {
  function organic_gardening_woocommerce_activated() {
    if ( class_exists( 'woocommerce' ) ) { return true; } else { return false; }
  }
}

if( ! function_exists( 'organic_gardening_change_comment_form_default_fields' ) ) :
/**
 * Change Comment form default fields i.e. author, email & url.
 * https://blog.josemcastaneda.com/2016/08/08/copy-paste-hurting-theme/
*/
function organic_gardening_change_comment_form_default_fields( $fields ){    
    // get the current commenter if available
    $organic_gardening_commenter = wp_get_current_commenter();
 
    // core functionality
    $req      = get_option( 'require_name_email' );
    $organic_gardening_aria_req = ( $req ? " aria-required='true'" : '' );
    $organic_gardening_required = ( $req ? " required" : '' );
    $organic_gardening_author   = ( $req ? __( 'Name*', 'organic-gardening' ) : __( 'Name', 'organic-gardening' ) );
    $organic_gardening_email    = ( $req ? __( 'Email*', 'organic-gardening' ) : __( 'Email', 'organic-gardening' ) );
 
    // Change just the author field
    $fields['author'] = '<p class="comment-form-author"><label class="screen-reader-text" for="author">' . esc_html__( 'Name', 'organic-gardening' ) . '<span class="required">*</span></label><input id="author" name="author" placeholder="' . esc_attr( $organic_gardening_author ) . '" type="text" value="' . esc_attr( $organic_gardening_commenter['comment_author'] ) . '" size="30"' . $organic_gardening_aria_req . $organic_gardening_required . ' /></p>';
    
    $fields['email'] = '<p class="comment-form-email"><label class="screen-reader-text" for="email">' . esc_html__( 'Email', 'organic-gardening' ) . '<span class="required">*</span></label><input id="email" name="email" placeholder="' . esc_attr( $organic_gardening_email ) . '" type="text" value="' . esc_attr(  $organic_gardening_commenter['comment_author_email'] ) . '" size="30"' . $organic_gardening_aria_req . $organic_gardening_required. ' /></p>';
    
    $fields['url'] = '<p class="comment-form-url"><label class="screen-reader-text" for="url">' . esc_html__( 'Website', 'organic-gardening' ) . '</label><input id="url" name="url" placeholder="' . esc_attr__( 'Website', 'organic-gardening' ) . '" type="text" value="' . esc_attr( $organic_gardening_commenter['comment_author_url'] ) . '" size="30" /></p>'; 
    
    return $fields;    
}
endif;
add_filter( 'comment_form_default_fields', 'organic_gardening_change_comment_form_default_fields' );

if( ! function_exists( 'organic_gardening_change_comment_form_defaults' ) ) :
/**
 * Change Comment Form defaults
 * https://blog.josemcastaneda.com/2016/08/08/copy-paste-hurting-theme/
*/
function organic_gardening_change_comment_form_defaults( $defaults ){    
    $defaults['comment_field'] = '<p class="comment-form-comment"><label class="screen-reader-text" for="comment">' . esc_html__( 'Comment', 'organic-gardening' ) . '</label><textarea id="comment" name="comment" placeholder="' . esc_attr__( 'Comment', 'organic-gardening' ) . '" cols="45" rows="8" aria-required="true" required></textarea></p>';
    
    return $defaults;    
}
endif;
add_filter( 'comment_form_defaults', 'organic_gardening_change_comment_form_defaults' );

if( ! function_exists( 'organic_gardening_escape_text_tags' ) ) :
/**
 * Remove new line tags from string
 *
 * @param $text
 * @return string
 */
function organic_gardening_escape_text_tags( $text ) {
    return (string) str_replace( array( "\r", "\n" ), '', strip_tags( $text ) );
}
endif;

if( ! function_exists( 'wp_body_open' ) ) :
/**
 * Fire the wp_body_open action.
 * Added for backwards compatibility to support pre 5.2.0 WordPress versions.
*/
function wp_body_open() {
    /**
     * Triggered after the opening <body> tag.
    */
    do_action( 'wp_body_open' );
}
endif;

if ( ! function_exists( 'organic_gardening_get_fallback_svg' ) ) :    
/**
 * Get Fallback SVG
*/
function organic_gardening_get_fallback_svg( $organic_gardening_post_thumbnail ) {
    if( ! $organic_gardening_post_thumbnail ){
        return;
    }
    
    $organic_gardening_image_size = organic_gardening_get_image_sizes( $organic_gardening_post_thumbnail );
     
    if( $organic_gardening_image_size ){ ?>
        <div class="svg-holder">
             <svg class="fallback-svg" viewBox="0 0 <?php echo esc_attr( $organic_gardening_image_size['width'] ); ?> <?php echo esc_attr( $organic_gardening_image_size['height'] ); ?>" preserveAspectRatio="none">
                    <rect width="<?php echo esc_attr( $organic_gardening_image_size['width'] ); ?>" height="<?php echo esc_attr( $organic_gardening_image_size['height'] ); ?>" style="fill:#dedddd;"></rect>
            </svg>
        </div>
        <?php
    }
}
endif;

function organic_gardening_enqueue_google_fonts() {

    require get_template_directory() . '/inc/wptt-webfont-loader.php';

    wp_enqueue_style(
        'google-fonts-roboto',
        wptt_get_webfont_url( 'https://fonts.googleapis.com/css2?family=Poppins:ital,wght@0,100;0,200;0,300;0,400;0,500;0,600;0,700;0,800;0,900;1,100;1,200;1,300;1,400;1,500;1,600;1,700;1,800;1,900&family=Roboto:ital,wght@0,100..900;1,100..900&display=swap' ),
        array(),
        '1.0'
    );
}
add_action( 'wp_enqueue_scripts', 'organic_gardening_enqueue_google_fonts' );


if( ! function_exists( 'organic_gardening_site_branding' ) ) :
/**
 * Site Branding
*/
function organic_gardening_site_branding(){
    $organic_gardening_logo_site_title = get_theme_mod( 'header_site_title', true );
    $organic_gardening_tagline = get_theme_mod( 'header_tagline', false );
    $organic_gardening_logo_width = get_theme_mod('logo_width', 100); // Retrieve the logo width setting

    ?>
    <div class="site-branding" style="max-width: <?php echo esc_attr(get_theme_mod('logo_width', '-1'))?>px;">
        <?php 
        // Check if custom logo is set and display it
        if (function_exists('has_custom_logo') && has_custom_logo()) {
            the_custom_logo();
        }
        if ($organic_gardening_logo_site_title):
             if (is_front_page()): ?>
            <h1 class="site-title" style="font-size: <?php echo esc_attr(get_theme_mod('organic_gardening_site_title_size', '30')); ?>px;">
            <a href="<?php echo esc_url(home_url('/')); ?>" rel="home"><?php bloginfo('name'); ?></a>
          </h1>
            <?php else: ?>
                <p class="site-title" itemprop="name">
                    <a href="<?php echo esc_url(home_url('/')); ?>" rel="home"><?php bloginfo('name'); ?></a>
                </p>
            <?php endif; ?>
        <?php endif; 
    
        if ($organic_gardening_tagline) :
            $organic_gardening_description = get_bloginfo('description', 'display');
            if ($organic_gardening_description || is_customize_preview()) :
        ?>
                <p class="site-description" itemprop="description"><?php echo $organic_gardening_description; ?></p>
            <?php endif;
        endif;
        ?>
    </div> 
    <?php
}
endif;
if( ! function_exists( 'organic_gardening_navigation' ) ) :
    /**
     * Site Navigation
    */
    function organic_gardening_navigation(){
        ?>
        <nav class="main-navigation" id="site-navigation" role="navigation">
            <?php 
            wp_nav_menu( array( 
                'theme_location' => 'primary', 
                'menu_id' => 'primary-menu' 
            ) ); 
            ?>
        </nav>
        <?php
    }
endif;

if( ! function_exists( 'organic_gardening_header' ) ) :
    /**
     * Header Start
    */
    function organic_gardening_header(){
        $organic_gardening_header_image = get_header_image();
        $organic_gardening_sticky_header = get_theme_mod('organic_gardening_sticky_header');
        $organic_gardening_header_btn_text     = get_theme_mod( 'organic_gardening_header_btn_text' );
        $organic_gardening_header_btn_url     = get_theme_mod( 'organic_gardening_header_btn_url' );
        $organic_gardening_phone        = get_theme_mod( 'organic_gardening_header_phone' );
        $organic_gardening_social_icon = get_theme_mod( 'organic_gardening_social_icon_setting', false);
        $organic_gardening_email        = get_theme_mod( 'organic_gardening_header_email' );
        ?>
        <div id="page-site-header" class="main-header">
            <header id="masthead" style="background-image: url('<?php echo esc_url( $organic_gardening_header_image ); ?>');" class="site-header header-inner" role="banner">
                    <div class="mid-head">
                        <div class="container">
                            <div class="row">
                                <div class="col-xl-3 col-lg-3 col-md-3 align-self-center">
                                    <?php if ( $organic_gardening_phone ){?>
                                        <div class="location">
                                            <span class="contact-icon">
                                                <i class="fas fa-phone-alt"></i>
                                            </span>
                                            <span class="contact-box">
                                                 <span class="location-text">
                                                    <a href="tel:<?php echo esc_attr($organic_gardening_phone);?>">
                                                        <?php echo esc_html( $organic_gardening_phone);?>
                                                    </a>
                                                </span>
                                            </span>
                                        </div>
                                    <?php } ?>
                                </div>
                                <div class="col-xl-3 col-lg-3 col-md-5 align-self-center text-md-left">
                                    <?php if ( $organic_gardening_email ){?>
                                        <div class="location">
                                            <span class="contact-icon">
                                                <i class="fas fa-envelope-open-text"></i>
                                            </span>
                                            <span class="contact-box">
                                                <span class="location-text">
                                                    <a href="mailto:<?php echo esc_attr($organic_gardening_email);?>">
                                                        <?php echo esc_html($organic_gardening_email);?>
                                                    </a>
                                                </span>
                                            </span>
                                            
                                        </div>
                                    <?php } ?>
                                </div>
                                <div class="col-xl-6 col-lg-6 col-md-4 align-self-center">
                                    <?php if ( $organic_gardening_social_icon ){?>
                                        <div class="social-links">
                                            <?php 
                                            $organic_gardening_social_link1 = get_theme_mod( 'organic_gardening_social_link_1' );
                                            $organic_gardening_social_link2 = get_theme_mod( 'organic_gardening_social_link_2' );
                                            $organic_gardening_social_link3 = get_theme_mod( 'organic_gardening_social_link_3' );
                                            $organic_gardening_social_link4 = get_theme_mod( 'organic_gardening_social_link_4' );

                                            if ( ! empty( $organic_gardening_social_link1 ) ) {
                                                echo '<a class="social1" href="' . esc_url( $organic_gardening_social_link1 ) . '" target="_blank"><i class="fab fa-facebook-f"></i></a>';
                                            }
                                            if ( ! empty( $organic_gardening_social_link2 ) ) {
                                                echo '<a class="social2" href="' . esc_url( $organic_gardening_social_link2 ) . '" target="_blank"><i class="fab fa-twitter"></i></a>';
                                            } 
                                            if ( ! empty( $organic_gardening_social_link3 ) ) {
                                                echo '<a class="social3" href="' . esc_url( $organic_gardening_social_link3 ) . '" target="_blank"><i class="fab fa-instagram"></i></a>';
                                            }
                                            if ( ! empty( $organic_gardening_social_link4 ) ) {
                                                echo '<a class="social4" href="' . esc_url( $organic_gardening_social_link4 ) . '" target="_blank"><i class="fab fa-pinterest-p"></i></a>';
                                            }
                                            ?>
                                        </div>
                                    <?php } ?>
                                </div>
                            </div>
                        </div> 
                    </div>
                    <div class="theme-menu head_bg" style="background-image: url('<?php echo esc_url( $organic_gardening_header_image ); ?>');" data-sticky="<?php echo $organic_gardening_sticky_header; ?>">
                        <div class="container">
                            <div class="row">
                            <div class="col-xl-3 col-lg-3 col-md-5 align-self-center">
                               <?php organic_gardening_site_branding(); ?> 
                            </div>
                            <div class="col-xl-7 col-lg-7 col-md-1 align-self-center">
                               <?php organic_gardening_navigation(); ?> 
                            </div>
                            <div class="col-xl-2 col-lg-2 col-md-6 align-self-center text-lg-right">
                                <?php if ( $organic_gardening_header_btn_text ){?>
                                    <div class="menudiv-button">
                                        <a href="<?php echo esc_url($organic_gardening_header_btn_url);?>"><?php echo esc_html($organic_gardening_header_btn_text);?><i class="far fa-arrow-alt-circle-right"></i></a>
                                    </div>
                                <?php } ?>
                            </div>
                        </div>
                        </div>
                    </div>
                    
            </header>
        </div>
        <?php
    }
endif;
add_action( 'organic_gardening_header', 'organic_gardening_header', 20 );
