<?php
/**
 * Orbit custom settings
 *
 * @package Orbit
 * @since Orbit 1.1.0
 */

/**
 * Replaces "[...]" in excerpts with "Read more"
 *
 * @since Orbit 1.0.0
 */
function orbit_remove_ellipsis_excerpts( $more ) {

	return '<p><a class="more-link" href="' . esc_url( get_permalink() ) . '">' . __( 'Read more &#8250;', 'orbit' ) . '</a></p>';
}
add_filter( 'excerpt_more', 'orbit_remove_ellipsis_excerpts' );

/**
 * Get our wp_nav_menu() fallback, wp_page_menu(), to show a home link.
 *
 * @since Orbit 1.0.0
 */
function orbit_add_home_link_menu( $args ) {

	$args['show_home'] = true;

	return $args;
}
add_filter( 'wp_page_menu_args', 'orbit_add_home_link_menu' );

if ( ! function_exists( '_wp_render_title_tag' ) ) {
	/**
	 * Creates a nicely formatted and more specific title element text
	 * for output in head of document, based on current view.
	 *
	 * @since Orbit 1.0.0
	 */
	function orbit_wp_title( $title, $sep ) {
		if ( is_feed() )
			return $title;

		// Add the site name.
		$title .= get_bloginfo( 'name' );

		// Add the site description for the home/front page.
		$site_description = get_bloginfo( 'description', 'display' );
		if ( $site_description && ( is_home() || is_front_page() ) )
			$title = $title . ' ' . $sep . ' ' .$site_description;

		return $title;
	}
	add_filter( 'wp_title', 'orbit_wp_title', 10, 2 );
}

/**
 * Filter the separator for the document title.
 *
 * @since Orbit 1.2.0
 */
function orbit_title_separator( $sep ){
	$sep = '|';
	return $sep; 
}
add_filter( 'document_title_separator', 'orbit_title_separator', 10 );

/**
 * Template for comments and pingbacks.
 *
 * To override this walker in a child theme without modifying the comments template
 * simply create your own orbit_render_list_comments(), and that function will be used instead.
 *
 * Used as a callback by wp_list_comments() for displaying the comments.
 *
 * @since Orbit 1.0.0
 */
function orbit_render_list_comments( $comment, $args, $depth ) {
	$GLOBALS['comment'] = $comment;

	if ( $comment->comment_approved == '1' ): ?>
		<li <?php comment_class(); ?> id="li-comment-<?php comment_ID(); ?>">
			<article id="comment-<?php comment_ID() ?>">
			<?php if ( $comment->comment_type == 'pingback' || $comment->comment_type == 'trackback' ): ?>
				<header class="header-comment">
					<h3><?php comment_author_link(); ?> <?php edit_comment_link( __( 'Edit', 'orbit' ), '<span class="edit-link"> | ', '</span>' ); ?> </h3>
				</header>
			<?php else: ?>
				<header class="header-comment">
					<div class="comment-avatar">
						<?php echo get_avatar( $comment, 42 ); ?>
					</div>
					<div class="comment-text">
						<h3><?php comment_author_link() ?></h3>
						<p><?php printf( '<time datetime="%1$s">%2$s %3$s %4$s</time>',
									get_comment_time( 'c' ), get_comment_date(), __( 'at', 'orbit' ), get_comment_time() ); ?> <?php edit_comment_link( __( 'Edit', 'orbit' ), '<span class="edit-link"> | ', '</span>' ); ?><?php comment_reply_link( array_merge( $args, array( 'reply_text' => __( 'Reply <span>&darr;</span>', 'orbit' ), 'depth' => $depth, 'max_depth' => $args['max_depth'], 'before' => '<span class="edit-link"> | ', 'after' => '</span>' ) ) ); ?></p>
						<?php comment_text() ?>
					</div>
				</header>
			<?php endif ?>
			</article>
	<?php endif;
}

/**
 * Capture title from current menu item and set a global variable to
 * use in orbit_render_breadcrumb (ex. if you page for posts is different of "blog")
 *
 * @since Orbit 1.0.0
 */
function orbit_capture_title_page_for_breadcrumb( $sorted_menu_items )
{
	foreach ( $sorted_menu_items as $menu_item ) {
		if ( $menu_item->current ) {
			$GLOBALS['orbit_current_menu_title'] = $menu_item->title;
			break;
		}
	}
	return $sorted_menu_items;
}
add_filter( 'wp_nav_menu_objects', 'orbit_capture_title_page_for_breadcrumb' );

/**
 * Render Breadcrumb (Child Theme Support)
 *
 * @since Orbit 1.1.0
 */
if ( function_exists('childorbit_render_breadcrumb' ) )  {
	/**
	 * Render Breadcrumb from child Theme
	 *
	 * @since Orbit 1.1.0
	 */
	function orbit_render_breadcrumb() {
		childorbit_render_breadcrumb();
	}
} else {
	/**
	 * Render Breadcrumb.
	 *
	 * @since Orbit 1.0.0
	 */
	function orbit_render_breadcrumb() {
		global $post;
		$content = '';
		$chevron = '<span class="chevron">&#8250;</span>';
		$before = '<div class="breadcrumb"><p>';
		$options = orbit_get_theme_options();

		if ( ( $options['pagination'] == 'pages' )
			&& get_query_var( 'paged' )
			&& ( is_category() || is_day() || is_month() || is_year() || is_search() || is_tag() || is_author() ) ) {
				$after = ' ( ' . __( 'Page','orbit' ) . ' ' . get_query_var( 'paged' ) . ')</p></div>';
		} else {
				$after = '</p></div>';
		}

		if ( !is_front_page()
			 || ( 'posts' == get_option( 'show_on_front' ) && is_front_page() ) ){

			if ( is_search() ) {
				$content = sprintf( __( 'Search Results for: "%s"', 'orbit' ), get_search_query() );

			} elseif ( is_home() ) {

				// Name of page used to show post blog (Can you use News, Last news...)
				if ( isset( $GLOBALS['orbit_current_menu_title'] ) )
					$content = $GLOBALS['orbit_current_menu_title'];
				else
					$content = 'Blog';

			} elseif ( is_page() && !$post->post_parent ) {
				$content = get_the_title();

			} elseif ( is_page() && $post->post_parent ) {
				$parent_id	= $post->post_parent;
				$breadcrumbs = array();

				while ( $parent_id ) {
					$page = get_page( $parent_id );
					$breadcrumbs[] = '<a href="' . esc_url( get_permalink( $page->ID ) ) . '">' . get_the_title( $page->ID ) . '</a>';
					$parent_id	= $page->post_parent;
				}
				$breadcrumbs = array_reverse( $breadcrumbs );

				foreach ($breadcrumbs as $crumb)
					$content .= $crumb . ' ' . $chevron . ' ';
				$content .=  get_the_title();

			} elseif ( is_single() && !is_attachment() ) {

				if ( get_post_type() != 'post' ) {
					$post_type = get_post_type_object( get_post_type() );
					$slug = $post_type->rewrite;
					$content = '<a href="' . home_url() . '/' . $slug['slug'] . '/">' . $post_type->labels->singular_name . '</a> ' . $chevron . ' ' . get_the_title();

				} else {
					$cat = get_the_category(); $cat = $cat[0];
					$content = get_category_parents( $cat, TRUE, ' ' . $chevron . ' ' ) . get_the_title();
				}

			} elseif ( is_category() ) {
				global $wp_query;

				$cat_obj = $wp_query->get_queried_object();
				$thisCat = $cat_obj->term_id;
				$thisCat = get_category( $thisCat );
				$parentCat = get_category( $thisCat->parent );
				$content = __( 'Category: ', 'orbit' );

				if ($thisCat->parent != 0)
					$content .= get_category_parents( $parentCat, TRUE, ' ' . $chevron . ' ' );

				$content .= single_cat_title( '', false );

			} elseif ( is_tag() ) {
				$content = sprintf( __( 'Tag: %s', 'orbit' ), single_tag_title( '', false ) );

			} elseif ( is_author() ) {
				global $author;

				$userdata = get_userdata( $author );
				$content = sprintf( __( 'All posts by %s', 'orbit' ), $userdata->display_name );

			} elseif ( is_404() ) {
				$content = __( 'Error 404 ','orbit' );

			} elseif ( is_day() ) {
				$content = __( 'Daily Archives: ', 'orbit' );
				$content .= '<a href="' . get_year_link( get_the_time( 'Y' ) ) . '">' . get_the_time( 'Y' ) . '</a> ' . $chevron . ' ';
				$content .= '<a href="' . get_month_link( get_the_time( 'Y' ),get_the_time( 'm' ) ) . '">' . get_the_time( 'F' ) . '</a> ' . $chevron . ' ';
				$content .= get_the_time( 'd' );

			} elseif ( is_month() ) {
				$content = __( 'Monthly Archives: ', 'orbit' );
				$content .= '<a href="' . get_year_link( get_the_time( 'Y' ) ) . '">' . get_the_time( 'Y' ) . '</a> ' . $chevron . ' ';				$content .= get_the_time( 'F' );

			} elseif ( is_year() ) {
				$content = __( 'Yearly Archives: ', 'orbit' ) . get_the_time( 'Y' );
			}
		}
		$output = array( 'content' => $content , 'before' => $before, 'after' => $after );
		$output = apply_filters( 'orbit_render_breadcrumb', $output );
		$outputlast = $output['before'] . $output['content'] . $output['after'];
		echo $outputlast;

	}
}

/**
 * Render social items
 *
 * @since Orbit 1.0.0
 */
function orbit_render_social_items(){
	$options = orbit_get_theme_options();
	$before = '<div class="social-icons"><ul>';
	$after = '</ul></div>';
	$content = '';

	foreach ( orbit_get_social_theme_options() as $social => $social_text) {
		if ( !empty( $options[ $social ] ) ) {
			$content .= '<li><a href="' . esc_url( $options[ $social ] ) . '" class="social-icons-all social-icons-' . $social . '"></a></li>';
		}
	}

	if ( !empty( $content ) ) {
		$output = apply_filters( 'orbit_render_social_items', $content );
		$output = $before . $output . $after;
		echo $output;
	}
	
}

/**
 * Render sidebar
 *
 * @since Orbit 1.0.0
 */
function orbit_get_sidebar( $location ){
	$options = orbit_get_theme_options();

	if ( is_page_template( 'full-width-page.php' ) && is_front_page() ){
		return;
	}

	$sidebar = '';
	if ( is_page_template( 'second-page.php' ) ){
		$sidebar = 'two';
	}
	
	if ( ( $location == 'rigth' ) &&  ( $options['theme_layout'] == 'content-sidebar' ) ) {
		get_sidebar( $sidebar );
	}

	if ( ( $location == 'left' ) &&	 ( $options['theme_layout'] == 'sidebar-content' ) ) {
		get_sidebar( $sidebar );
	}
}

/**
 * Render pagination
 *
 * @since Orbit 1.0.0
 */
function orbit_navigatin_pages( $wp_query ){
	$big = 999999999;

	echo paginate_links(array(
		'base' => str_replace( $big, '%#%', get_pagenum_link( $big ) ),
		'format' => '/page/%#%/',
		'current' => max( 1, get_query_var( 'paged' ) ),
		'total' => $wp_query->max_num_pages,
		'prev_text' => '<span class="previous-page"> &#8249; ' . __( 'Previous Page', 'orbit' ) . ' </span>',
		'next_text' => '<span class="next-page"> ' . __( 'Next Page', 'orbit' ) . ' &#8250;</span>' ) );

}

/**
 * Check if all images for slider are empty
 *
 * @since Orbit 1.1.0
 */
function orbit_slider_image_field_empty() {
	$options = orbit_get_theme_options();
	
	for ( $i=1; $i <= orbit_get_setting( 'orbit_slide_images' ) ; $i++ ) { 
		if ( isset( $options[ 'slide_image_' . $i ] ) ) {
			if ( $options[ 'slide_image_' . $i ] != '' )
				return false;
		}
	}
	return true;
}

/**
 * Render slider in front page
 *
 * @since Orbit 1.1.0
 */
function orbit_render_front_boxes(){
	$show = TRUE;
	$show = apply_filters('orbit_show_front_boxes', $show);
	
	if ( $show ) {
		$text_boxes = array(
			__( 'Responsive Design', 'orbit' ) => __( 'Design site adjusts to any mobile device including the iOS and Android devices. <br/><strong> Widget Area "Front Box 1"</strong>', 'orbit' ),
			__( 'Feature Slider', 'orbit' ) => __( 'Orbit get a Responsive Slider that displays images on any device. <br/><strong> Widget Area "Front Box 2"</strong>', 'orbit' ),
			__( 'Free Support', 'orbit' ) => sprintf( __( 'If you need help, visit our %s or %s <br/><strong> Widget Area "Front Box 3"</strong>', 'orbit' ), '<a href="' . esc_url( orbit_get_language_setting( 'orbit_doc_support' ) ) . '">' . __( 'Documentation', 'orbit' ) . '</a>', '<a href="' . esc_url( orbit_get_language_setting( 'orbit_forum_support' ) ) . '">' . __( 'Support Forum', 'orbit' ) . '</a>'));
				
		$num_box = 1;
		foreach ($text_boxes as $title => $desc) {
			$widget_box = 'front-box-' . $num_box;
	?>
			<div class="front-box">
				<?php if ( !dynamic_sidebar( $widget_box ) ) : ?>
					<h2><?php echo $title; ?></h2>
					<p><?php echo $desc; ?></p>
				<?php endif ?>
			</div> <!-- end front-box <?php echo $num_box; ?> -->
	<?php
			$num_box++;
		}
	}
}

/**
 * Render slider in front page
 *
 * @since Orbit 1.0.0
 */
function orbit_render_slider(){
	$options = orbit_get_theme_options();
	$images = orbit_get_setting( 'orbit_slide_images' );
	$show = TRUE;
	$show = apply_filters('orbit_show_slider', $show);

	if ($show) {
	
		if ( orbit_slider_image_field_empty() ) {
?>
	<div class="flexslider">
		<ul class="slides">
		<?php for ( $i = 1; $i <= $images ; $i++ ) { ?>
			<li>
				<img src="<?php echo get_template_directory_uri() . '/images/DefaultSlides_' . sprintf( '%02d', $i ) . '.jpg'; ?>" width="500" height="325" alt="slide<?php echo $i ?>">
				<a href="#"><p class="flex-caption"><?php printf( __( 'Text Slider %s', 'orbit' ), $i); ?></p></a>
			</li>
		<?php } ?>
		</ul>
	</div>
<?php
	} else {
?>
	<div class="flexslider">
		<ul class="slides">
			<?php for ( $i=1; $i <= $images ; $i++ ) {
				if ( isset( $options[ 'slide_image_' . $i ] ) ) {
					if ( $options[ 'slide_image_' . $i ] != '' ) { ?>
						<li>
							<img src="<?php echo esc_url( $options[ 'slide_image_' . $i ] ); ?>" alt="<?php if ( $options[ 'slide_text_' . $i ] != '' ) echo esc_attr( $options[ 'slide_text_' . $i ] ); ?>">							
							<?php if ( $options[ 'slide_text_' . $i ] != '' ) { ?>
									<a href="<?php if ( $options[ 'slide_url_' . $i ] != '' ) echo esc_url( $options[ 'slide_url_' . $i ] ); else echo '#'; ?>"><p class="flex-caption"><?php echo $options[ 'slide_text_' . $i ]; ?></p></a>
							<?php } ?>
						</li>
			<?php 	}
				}
			} ?>			
		</ul>
	</div>
<?php
		}
	}
}

/**
 * Adds the Orbit and Bitado credit.
 *
 * @since Orbit 1.0.0
 */
function orbit_credit() {
	$cr = __( 'Powered by ', 'orbit' );
	$cr = $cr . '<a href="' . esc_url( orbit_get_language_setting( 'orbit_homepage' ) ) . '" title="Orbit Theme by bitado.com">Orbit</a>';
	$cr = $cr . ' &amp;  <a href="' . esc_url( __( 'http://wordpress.org/','orbit' ) ) . '" title="Semantic Personal Publishing Platform">WordPress.</a>';
	
	echo $cr;
}
add_action( 'bit_footer_credit', 'orbit_credit', 10 );


/**
 * Add responsive container to embeds
 *
 * @since Orbit 1.1.5
 */
function orbit_embed_html( $html ) {
    return '<div class="video-container">' . $html . '</div>';
}
 
add_filter( 'embed_oembed_html', 'orbit_embed_html', 10, 3 );
add_filter( 'video_embed_html', 'orbit_embed_html' ); // Jetpack

?>