<?php 
/**
 * Orbit Validate options and customize page
 *
 * @package Orbit
 * @since Orbit 1.1.0
 */

/**
 * Sanitize and validate form input. Accepts an array, return a sanitized array.
 *
 * @since Orbit 1.0.0
 */
function orbit_options_validate( $input ) {
	global $allowedposttags;

	$output = $defaults = orbit_get_default_theme_options();

	// Our checkboxes value is either 0 or 1
	if ( ! isset( $input['show_author'] ) )
		$input['show_author'] = null;
	$output['show_author'] = ( $input['show_author'] == 1 ? 1 : 0 );
	if ( ! isset( $input['show_categories'] ) )
		$input['show_categories'] = null;
	$output['show_categories'] = ( $input['show_categories'] == 1 ? 1 : 0 );
	if ( ! isset( $input['show_tags'] ) )
		$input['show_tags'] = null;
	$output['show_tags'] = ( $input['show_tags'] == 1 ? 1 : 0 );
	if ( ! isset( $input['breadcrumb'] ) )
		$input['breadcrumb'] = null;
	$output['breadcrumb'] = ( $input['breadcrumb'] == 1 ? 1 : 0 );
	if ( ! isset( $input['slide_controlNav'] ) )
		$input['slide_controlNav'] = null;
	$output['slide_controlNav'] = ( $input['slide_controlNav'] == 1 ? 1 : 0 );
	if ( ! isset( $input['slide_slideshow'] ) )
		$input['slide_slideshow'] = null;
	$output['slide_slideshow'] = ( $input['slide_slideshow'] == 1 ? 1 : 0 );

	// Title color must be 3 or 6 hexadecimal characters
	if ( isset( $input['color_title_main'] ) && preg_match( '/^#?([a-f0-9]{3}){1,2}$/i', $input['color_title_main'] ) )
		$output['color_title_main'] = '#' . strtolower( ltrim( $input['color_title_main'], '#' ) );
	// Title color must be 3 or 6 hexadecimal characters
	if ( isset( $input['color_title_seconds'] ) && preg_match( '/^#?([a-f0-9]{3}){1,2}$/i', $input['color_title_seconds'] ) )
		$output['color_title_seconds'] = '#' . strtolower( ltrim( $input['color_title_seconds'], '#' ) );


	// Theme layout must be in our array of theme layout options
	if ( isset( $input['theme_layout'] ) && array_key_exists( $input['theme_layout'], orbit_layouts() ) )
		$output['theme_layout'] = $input['theme_layout'];

	// Pagination must be in our array of pagination options
	if ( isset( $input['pagination'] ) && array_key_exists( $input['pagination'], orbit_pagination() ) )
		$output['pagination'] = $input['pagination'];

	// Color Schemes layout must be in our array of Color Schemes options
	if ( isset( $input['color_scheme'] ) && array_key_exists( $input['color_scheme'], orbit_color_scheme() ) )
		$output['color_scheme'] = $input['color_scheme'];

	// check text for input footer info
	if ( isset( $input['footer_info'] ) )
		$output['footer_info'] = wp_kses_stripslashes( $input['footer_info'] );

	// check text for css user
	if ( isset( $input['css_user'] ) )
		$output['css_user'] = wp_kses_stripslashes( $input['css_user'] );

	// Font title must be in our array of theme font options
	if ( isset( $input['orbit_font_title'] ) && in_array( $input['orbit_font_title'], orbit_font_all() ) )
		$output['orbit_font_title'] = $input['orbit_font_title'];

	// check text for input font google title
	if ( isset( $input['google_font_title'] ) )
		$output['google_font_title'] = wp_kses_stripslashes( $input['google_font_title'] );

	// Font base must be in our array of theme font options
	if ( isset( $input['orbit_font_base'] ) && in_array( $input['orbit_font_base'], orbit_font_all() ) )
		$output['orbit_font_base'] = $input['orbit_font_base'];

	// check text for input font google base
	if ( isset( $input['google_font_base'] ) )
		$output['google_font_base'] = wp_kses_stripslashes( $input['google_font_base'] );

	// check text for input text for pages without title
	if ( isset( $input['show_title_page'] ) )
		$output['show_title_page'] = wp_kses_stripslashes( $input['show_title_page'] );

	// check text for input text in front page
	if ( isset( $input['front_page_header'] ) )
		$output['front_page_header'] = wp_kses_data( $input['front_page_header'] );
	if ( isset( $input['front_page_subheader'] ) )
		$output['front_page_subheader'] = wp_kses_data( $input['front_page_subheader'] );
		
	// Validation like html editor post
	if ( isset( $input['front_page_textarea'] ) )
		$output['front_page_textarea'] =  wp_kses( $input['front_page_textarea'], $allowedposttags );

	if ( isset( $input['front_page_call_to_action'] ) )
		$output['front_page_call_to_action'] = wp_kses_data( $input['front_page_call_to_action'] );
	if ( isset( $input['url_call_to_action'] ) )
		$output['url_call_to_action'] = wp_kses_data( $input['url_call_to_action'] );


	// check text for input images slide
	for ( $i = 1; $i <= orbit_get_setting( 'orbit_slide_images' ); $i++ ) { 
		if ( isset( $input['slide_image_' . $i] ) )
			$output['slide_image_' . $i] = wp_kses_data( $input['slide_image_' . $i] );
		if ( isset( $input['slide_text_' . $i] ) )
			$output['slide_text_' . $i] = wp_kses_data( $input['slide_text_' . $i] );
		if ( isset( $input['slide_url_' . $i] ) )
			$output['slide_url_' . $i] = wp_kses_data( $input['slide_url_' . $i] );
	}	
		
	if ( isset( $input['favicon'] ) )
		$output['favicon'] = wp_kses_data( $input['favicon'] );

	// check number for input posts in front page
	if ( ! is_numeric( $input['show_posts'] ) )
		$input['show_posts'] = 0;
	$output['show_posts'] = ( $input['show_posts'] > 0 ? $input['show_posts'] : 0 );

	// check number for slide show Speed
	if ( ! is_numeric( $input['slide_slideshowSpeed'] ) )
		$input['slide_slideshowSpeed'] = 7000;
	$output['slide_slideshowSpeed'] = ( $input['slide_slideshowSpeed'] > 0 ? $input['slide_slideshowSpeed'] : 7000 );

	// check text for input social
	foreach ( orbit_get_social_theme_options() as $social => $social_text) {
		if ( isset( $input[$social] ) )
			$output[$social] = wp_kses_data( $input[$social] );
	}

	$resetDefault = ( ! empty( $input['orbit_defaults']) ? true : false );

	if ($resetDefault)
		$output=$defaults;

	return $output;
}

/**
 * Sanitize and validate color_scheme in Customize page
 *
 * @since Orbit 1.2.0
 */
function orbit_sanitize_color_scheme( $input ) {
	
    $valid = orbit_color_scheme();
 
    if ( array_key_exists( $input, $valid ) ) {
        return $input;
    } else {
        return orbit_get_default_theme_options( 'color_scheme' );
    }
}

/**
 * Sanitize and validate theme_layout in Customize page
 *
 * @since Orbit 1.2.0
 */
function orbit_sanitize_theme_layout( $input ) {
	
    $valid = orbit_layouts();
 
    if ( array_key_exists( $input, $valid ) ) {
        return $input;
    } else {
        return orbit_get_default_theme_options( 'theme_layout' );
    }
}

/**
 * Sanitize pagination values
 *
 * @since Orbit 1.2.0
 */
function orbit_sanitize_pagination( $input ) {
	
    $valid = orbit_pagination();
 
    if ( array_key_exists( $input, $valid ) ) {
        return $input;
    } else {
        return orbit_get_default_theme_options( 'pagination' );
    }
}
	
/**
 * Sanitize orbit_font_title values
 *
 * @since Orbit 1.2.0
 */
function orbit_sanitize_font_title( $input ) {

    $valid = orbit_font_all();

    if ( in_array( $input, $valid ) ) {
        return $input;
    } else {
        return orbit_get_default_theme_options( 'orbit_font_title' );
    }
}

/**
 * Sanitize orbit_font_base values
 *
 * @since Orbit 1.2.0
 */
function orbit_sanitize_font_base( $input ) {

    $valid = orbit_font_all();

    if ( in_array( $input, $valid ) ) {
        return $input;
    } else {
        return orbit_get_default_theme_options( 'orbit_font_base' );
    }
}
	
/**
 * Sanitize integers
 *
 * @since Orbit 1.2.0
 */
function orbit_sanitize_integer( $input ) {
	return absint( $input );
}

/**
 * Sanitize checkbox values
 *
 * @since Orbit 1.2.0
 */
function orbit_sanitize_checkbox( $input ) {
	if ( $input ) {
		$output = '1';
	} else {
		$output = false;
	}
	return $output;
}

/**
 * Sanitize slug pages values
 *
 * @since Orbit 1.2.0
 */
function orbit_sanitize_text( $input ) {
	return wp_kses_stripslashes( $input );
}
	
/**
 * Sanitize colors
 *
 * @since Orbit 1.2.0
 */
function orbit_sanitize_hex_color( $color ) {
    if ( '' === $color )
        return '';

    // 3 or 6 hex digits, or the empty string.
    if ( preg_match('|^#([A-Fa-f0-9]{3}){1,2}$|', $color ) )
        return $color;

    return '';
}