<?php
/**
 * Orbit Options in admin panel
 *
 * @package Orbit
 * @since Orbit 1.0.0
 */

/**
 * Init plugin options to white list our options
 *
 * @since Orbit 1.0.0
 */
function orbit_options_init(){

	register_setting( 'orbit_options', 'orbit_theme_options', 'orbit_options_validate' );

	// Sections
	add_settings_section( 'front_page_section', __( 'Front Page Options', 'orbit' ), '__return_false', 'theme_options' );
	add_settings_section( 'status_report', __( 'Status Report', 'orbit' ), 'orbit_status_report_page', 'theme_options' );

	// Fields for general section front_page_section
	add_settings_field( 'front_page_header', __( 'Header', 'orbit' ), 'orbit_settings_field_front_page_header', 'theme_options', 'front_page_section' );
	add_settings_field( 'front_page_subheader', __( 'Subheader', 'orbit' ), 'orbit_settings_field_front_page_subheader', 'theme_options', 'front_page_section' );
	add_settings_field( 'front_page_textarea', __( 'Long text', 'orbit' ), 'orbit_settings_field_front_page_textarea', 'theme_options', 'front_page_section' );
	add_settings_field( 'front_page_call_to_action', __( 'Text button "Call to Action"', 'orbit' ), 'orbit_settings_field_front_page_call_to_action', 'theme_options', 'front_page_section' );
	add_settings_field( 'front_page_url_call_to_action', __( 'URL button "Call to Action"', 'orbit' ), 'orbit_settings_field_front_page_url_call_to_action', 'theme_options', 'front_page_section' );
	add_settings_field( 'show_posts', __( 'Posts to Display on Front Page', 'orbit' ), 'orbit_settings_field_show_posts', 'theme_options', 'front_page_section' );

	// Fields for section slider
	orbit_add_slider_fields();
}
add_action( 'admin_init', 'orbit_options_init' );


/**
 * Add field for image slider: image, text and url field
 *
 * @since Orbit 1.1.0
 */
function orbit_add_slider_fields() {

	add_settings_field( 'slide_controlNav', __( 'Slider Control', 'orbit' ), 'orbit_settings_field_slide_controlNav', 'theme_options', 'front_page_section' );
	add_settings_field( 'slide_slideshow', __( 'Animate slider automatically', 'orbit' ), 'orbit_settings_field_slide_slideshow', 'theme_options', 'front_page_section' );
	add_settings_field( 'slide_slideshowSpeed', __( 'Speed of the slideshow', 'orbit' ), 'orbit_settings_field_slide_slideshowSpeed', 'theme_options', 'front_page_section' );

	for ($i=1; $i <= orbit_get_setting( 'orbit_slide_images' ); $i++) {
		add_settings_field( 'slide_image_' . $i, sprintf( __( 'Slider Image %s', 'orbit' ), $i ), 'orbit_settings_field_slide_image', 'theme_options', 'front_page_section', $i );
		add_settings_field( 'slide_text_' . $i, sprintf( __( 'Slider Text %s', 'orbit' ), $i ), 'orbit_settings_field_slide_text', 'theme_options', 'front_page_section', $i );
		add_settings_field( 'slide_url_' . $i, sprintf( __( 'Slider Url %s', 'orbit' ), $i ), 'orbit_settings_field_slide_url', 'theme_options', 'front_page_section', $i );
	}

}

/**
 * Load up the menu page
 *
 * @since Orbit 1.0.0
 */
function orbit_options_add_page() {
	add_theme_page( __( 'Orbit Options', 'orbit' ), __( 'Orbit Options', 'orbit' ), 'edit_theme_options', 'orbit_options', 'orbit_options_do_page' );
}
add_action( 'admin_menu', 'orbit_options_add_page' );

/**
 * A safe way of adding JavaScripts to a WordPress generated page.
 *
 * @since Orbit 1.0.0
 */
function orbit_admin_enqueue_scripts( $hook_suffix ) {
	wp_enqueue_style( 'orbit-theme-options', get_template_directory_uri() . '/inc/css/theme-options.css', false, '1.0' );
	wp_enqueue_script( 'media-upload' );
	wp_enqueue_script( 'thickbox' );
	wp_enqueue_style( 'thickbox' );
	wp_enqueue_style( 'farbtastic' );
	wp_enqueue_script( 'jquery-plugin-cookie', get_template_directory_uri() . '/inc/js/jquery.ck.js', array( 'jquery' ), '1.0' );
	wp_enqueue_script( 'orbit-theme-options', get_template_directory_uri() . '/inc/js/theme-options.js', array( 'jquery', 'media-upload', 'thickbox', 'farbtastic', 'jquery-plugin-cookie' ), '1.0' );

}
add_action( 'admin_print_styles-appearance_page_orbit_options', 'orbit_admin_enqueue_scripts' );

/**
 * Render sections in admin panel
 *
 * @since Orbit 1.0.0
 */
function orbit_do_settings_sections( $page ) {
	global $wp_settings_sections, $wp_settings_fields;

	if ( ! isset( $wp_settings_sections ) || !isset( $wp_settings_sections[$page] ) )
		return;

	foreach ( (array) $wp_settings_sections[$page] as $section ) {
		echo '<div class="section-options">';
		echo '	<h3 id="orbad_' . $section['id'] . '" class="rwd-toggle"><a href="#">' . $section['title'] . '</a></h3>';
		echo '	<div class="rwd-container">';

		if ( $section['callback'] )
			call_user_func( $section['callback'], $section );

		if ( $section['id'] != 'status_report' ) {
			echo '		<table class="form-table">';
			orbit_do_settings_fields( $page, $section['id'] );
			echo '		</table>';
		}
		echo '	</div>';
		echo '</div>';
	}
}

/**
 * Render custom settings fields for a particular section
 *
 * @since Orbit 1.1.0
 */
function orbit_do_settings_fields($page, $section) {
	global $wp_settings_fields;

	if ( !isset($wp_settings_fields) || !isset($wp_settings_fields[$page]) || !isset($wp_settings_fields[$page][$section]) )
		return;

	foreach ( (array) $wp_settings_fields[$page][$section] as $field ) {
		if ( strpos( $field['id'], 'slide_' ) === false)
			$classes = '';
		else
			$classes = substr_replace( $field['id'], "", -2);

		if ( (strpos( $field['id'], 'slide_text' ) === false ) && ( strpos( $field['id'], 'show_posts' ) === false) )
			$classes .= '';
		else
			$classes .= ' ' . $field['id'];
			
		if ( $classes != '' )
			$custom_class = 'class="' . $classes . '"';
		else
			$custom_class = '';
			
		echo '<tr valign="top" ' . $custom_class . '>';
		if ( !empty($field['args']['label_for']) )
			echo '<th scope="row"><label for="' . esc_attr( $field['args']['label_for'] ) . '">' . $field['title'] . '</label></th>';
		else
			echo '<th scope="row">' . $field['title'] . '</th>';
		echo '<td>';
		call_user_func( $field['callback'], $field['args'] );
		echo '</td>';
		echo '</tr>';
	}
}

/**
 * Create the options page
 *
 * @since Orbit 1.0.0
 */
function orbit_options_do_page() {
	?>
	<div class="wrap">
		<div>
			<img style="float: left;" src="<?php echo get_template_directory_uri() . '/images/logo-orbit-mini.png' ?>" />
			<h2><?php printf( __( 'Orbit Options', 'orbit' ), wp_get_theme() ); ?>		<span style="font-size: 13px; margin-left: 24px;">Orbit <?php echo orbit_get_version_theme() . ' ' . __( 'by', 'orbit' ) . ' '; ?>  <a href="<?php echo esc_url( orbit_get_setting( 'orbit_developer' ) ); ?>">bitado.com</a></span></h2>
		</div>
		<div style="border-bottom: 1px solid #ccc;padding-bottom:12px;">
			<a class="button button-primary button-orbit-support" href="<?php echo orbit_get_language_setting('orbit_forum_support'); ?>"><?php _e( 'Support Forum', 'orbit' ); ?></a>
			<a class="button button-primary button-orbit-support" href="<?php echo orbit_get_language_setting('orbit_doc_support'); ?>"><?php _e( 'Documentation', 'orbit' ); ?></a>
			<a class="button button-primary button-orbit-support" href="<?php echo orbit_get_language_setting('orbit_homepage'); ?>"><?php _e( 'Orbit Homepage', 'orbit' ); ?></a>
		</div>
		<?php settings_errors(); ?>
	</div>
	<div class="wrap optionsorbit">
		<div class="lefty">
			<form method="post" action="options.php">
				<?php
					settings_fields( 'orbit_options' );
					orbit_do_settings_sections( 'theme_options' );
				?>
				<input type="submit" name="submit" id="submit" class="button button-primary" value="<?php _e( 'Save Settings', 'orbit' ); ?>">
				<input name="orbit_theme_options[orbit_defaults]" id="orbit_theme_options[orbit_defaults]" class="button button-secondary alignright" type="submit" value="<?php _e( 'Reset to Defaults', 'orbit' ); ?>" />
			</form>
		</div>
		<div class="righty">
			<div class="postbox support">
				<?php orbit_settings_box_last_news(); ?>
			</div>
		</div>
	</div>
	<?php
}


/**
 * Renders css user setting field.
 *
 * @since Orbit 1.0.0
 */
function orbit_settings_field_css_user() {
	$options = orbit_get_theme_options();
	?>
		<label class="description">
			<textarea id="orbit_theme_options[css_user]" class="large-text" cols="50" rows="10" name="orbit_theme_options[css_user]"><?php if (!empty($options['css_user'])) echo esc_html($options['css_user']); ?></textarea>
		</label>
	<?php
}

/**
 * Renders the 'front_page_header' setting field.
 *
 * @since Orbit 1.0.0
 */
function orbit_settings_field_front_page_header() {
	orbit_settings_field_regular_text( 'front_page_header' );
}

/**
 * Renders the 'front_page_subheader' setting field.
 *
 * @since Orbit 1.0.0
 */
function orbit_settings_field_front_page_subheader() {
	orbit_settings_field_regular_text( 'front_page_subheader' );
}

/**
 * Renders the 'front_page_textarea' setting field.
 *
 * @since Orbit 1.0.0
 */
function orbit_settings_field_front_page_textarea() {
	$options = orbit_get_theme_options();
	?>
		<label class="description">
			<textarea id="orbit_theme_options[front_page_textarea]" class="large-text" cols="50" rows="10" name="orbit_theme_options[front_page_textarea]"><?php if (!empty($options['front_page_textarea'])) echo $options['front_page_textarea']; ?></textarea>
		</label>
	<?php
}

/**
 * Renders the 'front_page_call_to_action' setting field.
 *
 * @since Orbit 1.0.0
 */
function orbit_settings_field_front_page_call_to_action() {
	orbit_settings_field_regular_text( 'front_page_call_to_action' );
}

/**
 * Renders the 'url_call_to_action' setting field.
 *
 * @since Orbit 1.0.0
 */
function orbit_settings_field_front_page_url_call_to_action() {
	orbit_settings_field_regular_text( 'url_call_to_action' );
}

/**
 * Renders the 'slide_controlNav' setting field.
 *
 * @since Orbit 1.1.3
 */
function orbit_settings_field_slide_controlNav() {
	orbit_settings_field_regular_checkbox( 'slide_controlNav' );
}

/**
 * Renders the 'slide_slideshow' setting field.
 *
 * @since Orbit 1.1.3
 */
function orbit_settings_field_slide_slideshow() {
	orbit_settings_field_regular_checkbox( 'slide_slideshow' );
}

/**
 * Renders the 'url_call_to_action' setting field.
 *
 * @since Orbit 1.1.3
 */
function orbit_settings_field_slide_slideshowSpeed() {
	$options = orbit_get_theme_options();
	?>
		<label for="slide_slideshowSpeed">
			<input id="orbit_theme_options[slide_slideshowSpeed]" type="text" size="5" name="orbit_theme_options[slide_slideshowSpeed]" value="<?php echo esc_attr( $options['slide_slideshowSpeed'] ); ?>" />
			<br /><?php _e( 'In milliseconds', 'orbit' ); ?>
		</label>
	<?php
}

/**
 * Renders general images field for slide images
 *
 * @since Orbit 1.1.0
 */
function orbit_settings_field_slide_image( $number ) {
	$image = 'slide_image_' . $number;
	$options = orbit_get_theme_options();
	$id_image = 'upload_' . $image;
	?>
	<label for="<?php echo $image; ?>">
		<input id="orbit_theme_options[<?php echo $image; ?>]" type="text" size="33" name="orbit_theme_options[<?php echo $image; ?>]" value="<?php if ( !empty( $options[ $image ] ) ) echo esc_attr( $options[ $image ] ); ?>" />
		<input id="<?php echo $id_image; ?>" class="upload_image_button" type="button" value="<?php _e( 'Upload Image', 'orbit'); ?>" />
		<br /><?php _e( 'Enter an URL or upload an image for the slider.', 'orbit' ); ?>
	</label>
	<?php
}

/**
 * Renders general text field for slide images
 *
 * @since Orbit 1.1.0
 */
function orbit_settings_field_slide_text( $number ) {
	$image = 'slide_text_' . $number;
	$options = orbit_get_theme_options();
	?>
		<label for="<?php echo $image; ?>">
			<input id="orbit_theme_options[<?php echo $image; ?>]" type="text" size="33" name="orbit_theme_options[<?php echo $image; ?>]" value="<?php if ( !empty( $options[ $image ] ) ) echo esc_attr( $options[ $image ] ); ?>" />
		</label>
	<?php
}

/**
 * Renders general url field for slide images
 *
 * @since Orbit 1.1.0
 */
function orbit_settings_field_slide_url( $number ) {
	$image = 'slide_url_' . $number;
	$options = orbit_get_theme_options();
	?>
		<label for="<?php echo $image; ?>">
			<input id="orbit_theme_options[<?php echo $image; ?>]" type="text" size="33" name="orbit_theme_options[<?php echo $image; ?>]" value="<?php if ( !empty( $options[ $image ] ) ) echo esc_attr( $options[ $image ] ); ?>" />
		</label>
	<?php
}


/**
 * Renders show posts in front page.
 *
 * @since Orbit 1.1.0
 */
function orbit_settings_field_show_posts() {
	$options = orbit_get_theme_options();
	?>
		<label for="show_posts">
			<input id="orbit_theme_options[show_posts]" type="text" size="5" name="orbit_theme_options[show_posts]" value="<?php echo esc_attr( $options['show_posts'] ); ?>" />
			<br /><?php _e( 'Greater than 0 Front Sidebar is activated (Widget Area "Front Sidebar")', 'orbit' ); ?>
		</label>
	<?php
}

/**
 * Renders box for RSS feed News Orbit
 *
 * @since Orbit 1.1.0
 */
function orbit_settings_box_last_news(){
	$feed_last_news = orbit_get_language_setting( 'orbit_feed_news' );
?>
	<h3 class="hndle"><?php _e( 'Orbit Latest News', 'orbit' ); ?></h3>

	<div class="inside" style="height:238px;overflow:auto;">
		<?php
			$maxitems = 0;
			$orbit_news = fetch_feed( array( $feed_last_news ) );
			if ( ! is_wp_error( $orbit_news ) ) {
				$maxitems = $orbit_news->get_item_quantity( 10 );
				$news_items = $orbit_news->get_items( 0, $maxitems );
			}
		?>
		<ul class="news-list">
			<?php if ( $maxitems == 0 ) : echo '<li>' . __( 'No news items.', 'orbit' ) . '</li>'; else :
				foreach( $news_items as $news_item ) : ?>
				<li>
					<a class="news-header" href='<?php echo esc_url( $news_item->get_permalink() ); ?>'><?php echo esc_html( $news_item->get_title() ); ?></a><br />
					<span class="news-item-date"><?php echo __( 'Posted on ', 'orbit' ) . $news_item->get_date( get_option( 'date_format' ) ); ?></span><br />

					<?php echo orbit_truncate_words( strip_tags( $news_item->get_description() ), 20, '...' ) ; ?>
					<br><a class="news-read" href='<?php echo esc_url( $news_item->get_permalink() ); ?>'><?php _e( 'Read more &raquo;', 'orbit' ); ?></a>
				</li>
			<?php endforeach; endif; ?>
		</ul>
	</div><!-- inside -->
<?php
}

/**
 * Renders Status report for support
 *
 * @since Orbit 1.1.0
 */
function orbit_status_report_page() {
?>
	<table class="orbit-status-report" cellspacing="0">
		<thead>
			<tr>
				<th colspan="2"><p><?php _e( 'Please include this information when requesting support', 'orbit' ); ?></p></th>
			</tr>
		</thead>
		<tbody>
			<tr>
                <td><strong><?php _e( 'Home URL','orbit' ); ?>:</strong></td>
                <td><?php echo home_url(); ?></td>
            </tr>
            <tr>
                <td><strong><?php _e( 'Site URL','orbit' ); ?>:</strong></td>
                <td><?php echo site_url(); ?></td>
            </tr>
            <tr>
                <td><strong><?php _e( 'Orbit Version','orbit' ); ?>:</strong></td>
                <td><?php echo esc_html( orbit_get_version_theme() ); ?></td>
            </tr>
            <tr>
                <td><strong><?php _e( 'WP Version','orbit' ); ?>:</strong></td>
                <td><?php if ( is_multisite() ) echo 'WPMU'; else echo 'WP'; ?> <?php echo bloginfo('version'); ?></td>
            </tr>
            <tr>
                <td><strong><?php _e( 'Web Server Info','orbit' ); ?>:</strong></td>
                <td><?php echo esc_html( $_SERVER['SERVER_SOFTWARE'] );  ?></td>
            </tr>
            <tr>
                <td><strong><?php _e( 'PHP Version','orbit' ); ?>:</strong></td>
                <td><?php if ( function_exists( 'phpversion' ) ) echo esc_html( phpversion() ); ?></td>
            </tr>
            <tr>
                <td><strong><?php _e( 'MySQL Version','orbit' ); ?>:</strong></td>
                <td><?php global $wpdb; echo $wpdb->db_version(); ?></td>
            </tr>
			<tr>
         		<td><strong><?php _e( 'Installed Plugins','orbit' ); ?>:</strong></td>
         		<td><?php
         			$active_plugins = (array) get_option( 'active_plugins', array() );

         			if ( is_multisite() )
						$active_plugins = array_merge( $active_plugins, get_site_option( 'active_sitewide_plugins', array() ) );

					$wc_plugins = array();

					foreach ( $active_plugins as $plugin ) {

						$plugin_data    = @get_plugin_data( WP_PLUGIN_DIR . '/' . $plugin );
						$dirname        = dirname( $plugin );
						$version_string = '';

						if ( ! empty( $plugin_data['Name'] ) ) {

							$wc_plugins[] = $plugin_data['Name'] . ' ' . __( 'by', 'orbit' ) . ' ' . $plugin_data['Author'] . ' ' . __( 'version', 'orbit' ) . ' ' . $plugin_data['Version'] . $version_string;

						}
					}

					if ( sizeof( $wc_plugins ) == 0 )
						echo '-';
					else
						echo implode( ', <br/>', $wc_plugins );

         			?>
				</td>
         	</tr>
		</tbody>
	</table>
<?php
}
