<?php
/**
 * Orbit helper functions
 *
 * @package Orbit
 * @since Orbit 1.1.0
 */

/**
 * Returns the general variables for Orbit
 *
 * @since Orbit 1.1.0
 */
function orbit_get_setting( $option = null ) {
	$orbit_feed_news[ 'es-ES' ] = 'feed://bitado.com/foros/foro/plantilla-orbit/noticias-y-actualizaciones/feed/';
	$orbit_feed_news[ 'en-US' ] = 'feed://bitado.com/foros/foro/news-and-updates/feed/';
	
	$orbit_forum_support[ 'es-ES' ] = 'http://bitado.com/foros/foro/plantilla-orbit/';
	$orbit_forum_support[ 'en-US' ] = 'http://bitado.com/foros/foro/theme-orbit-eng/';
	
	$orbit_doc_support[ 'es-ES' ] = 'http://bitado.com/documentacion-orbit/';
	$orbit_doc_support[ 'en-US' ] = 'http://bitado.com/orbit-documentation-eng/';

	$orbit_homepage[ 'es-ES' ] = 'http://bitado.com/orbit-plantilla-wordpress-gratis/';
	$orbit_homepage[ 'en-US' ] = 'http://bitado.com/orbit-theme-wordpress-free-eng/';
	
	$orbit_gen[ 'orbit_feed_news' ] = $orbit_feed_news;
	$orbit_gen[ 'orbit_forum_support' ] = $orbit_forum_support;
	$orbit_gen[ 'orbit_doc_support' ] = $orbit_doc_support;
	$orbit_gen[ 'orbit_homepage' ] = $orbit_homepage;
		
	$orbit_gen[ 'orbit_developer' ] = 'http://bitado.com';
	$orbit_gen[ 'orbit_version' ] = '1.2.1';
	$orbit_gen[ 'orbit_slide_images' ] = 3;
	$orbit_gen[ 'orbit_slide_max' ] = 6;
	
	$orbit_gen = apply_filters( 'orbit_get_custom_setting', $orbit_gen );
	
	// Returns 'orbit_slide_max' number of images if 'orbit_slide_images' exceeds the maximum.
	// Option 'orbit_slide_images' (all options) can be changed by filter hook (apply filters above )
	if ( ( $option == 'orbit_slide_images' ) && ( $orbit_gen[ 'orbit_slide_images' ] > $orbit_gen[ 'orbit_slide_max' ] ) )
		$orbit_gen[ 'orbit_slide_images' ] = $orbit_gen[ 'orbit_slide_max' ];
	
	// Return full array or particular option
	if ( $option == null)
		return $orbit_gen;
	else
		return $orbit_gen[ $option ];
}

/**
 * Returns the general variables for Orbit in right language
 *
 * @since Orbit 1.1.0
 */
function orbit_get_language_setting( $orbit_setting ) {
	$setting_list = orbit_get_setting( $orbit_setting );
	$setting_lang = $setting_list[ 'en-US' ];
	$language = get_bloginfo( 'language' );

	if (  $language != 'en_US' && array_key_exists( $language, $setting_list )) {
		$setting_lang = $setting_list[ $language ];
	}

	return $setting_lang;
}

/**
 * Returns the default options for admin pnael
 *
 * @since Orbit 1.0.0
 */
function orbit_get_default_theme_options( $option = null ) {
	$default_theme_options = array(
		'theme_layout' => 'content-sidebar',
		'pagination' => 'pages',
		'breadcrumb' => false,
		'color_scheme' => 'blue',
		'footer_info' => '',
		'show_title_page' => '',
		'css_user' => '',
		'show_author' => true,
		'show_categories' => true,
		'show_tags' => true,
		'color_title_main' => '#333',
		'color_title_seconds' => '#333',
		'google_font_title' => '',
		'google_font_base' => '',
		'orbit_font_title' => 'Georgia, Times, serif',
		'orbit_font_base' => 'sans-serif, Verdana, Geneva',
		'front_page_header' => '',
		'front_page_subheader' => '',
		'front_page_textarea' => '',
		'front_page_call_to_action' => '',
		'url_call_to_action' => '#',
		'show_posts' => '0',
		'slide_controlNav' => '1',
		'slide_slideshow' => '1',
		'slide_slideshowSpeed' => '7000',
	);

	
	$default_theme_options = array_merge( $default_theme_options, orbit_get_default_slider_options() );
	$default_theme_options = array_merge( $default_theme_options, orbit_get_default_social_options() );
	
	if ( $option == null) {
		return $default_theme_options;
	}
	else {
		return $default_theme_options[ $option ];
	}
}

/**
 * Returns the slider fields options default value array
 *
 * @since Orbit 1.1.0
 */
function orbit_get_default_slider_options() {
	$orbit_slider_options = array();

	for ( $i = 1; $i <= orbit_get_setting( 'orbit_slide_images' ); $i++ ) {
		$orbit_slider_options[ 'slide_image_' . $i ] = '';
		$orbit_slider_options[ 'slide_text_' . $i ] = '';
		$orbit_slider_options[ 'slide_url_' . $i ] = '';
	}

	return $orbit_slider_options;
}

/**
 * Returns the social options default value array
 *
 * @since Orbit 1.1.0
 */
function orbit_get_default_social_options() {
	$orbit_social_options = array();
	
	foreach ( orbit_get_social_theme_options() as $social => $social_text ) {
		$orbit_social_options[ $social ] = '';
	}

	return $orbit_social_options;
}

/**
 * Returns the social options array for Orbit.
 *
 * @since Orbit 1.1.0
 */
function orbit_get_social_theme_options() {
	$social_theme = array (
		'rss'        => __( 'RSS url', 'orbit'),
		'twitter'    => __( 'Twitter url', 'orbit'),
		'facebook'   => __( 'Facebook url', 'orbit'),
		'googleplus' => __( 'Google+ url', 'orbit'),
		'linkedin'   => __( 'LinkedIn url', 'orbit'),
		'vimeo'      => __( 'Vimeo url', 'orbit'),
		'youtube'    => __( 'YouTube url', 'orbit'),
		'instagram'  => __( 'Instagram url', 'orbit'),
		'flickr'     => __( 'Flickr url', 'orbit'),
		'pinterest'  => __( 'Pinterest url', 'orbit'),
		);
	return $social_theme;
}

/**
 * Returns the options array for Orbit.
 *
 */
function orbit_get_theme_options( $name_option = false ) {
	$options = wp_parse_args( 
		get_option( 'orbit_theme_options', array() ), 
		orbit_get_default_theme_options()
	);
		
	if ( ! $name_option )
		return $options;
	else
		return $options[$name_option];
}

/**
 * Returns an array of layout options registered for orbit.
 *
 * @since Orbit 1.0.0
 */
function orbit_layouts() {
	$layout_options = array(
		'content-sidebar' => array(
			'value' => 'content-sidebar',
			'label' => __( 'Sidebar on right', 'orbit' ),
			'thumbnail' => get_template_directory_uri() . '/inc/images/content-sidebar.png',
		),
		'sidebar-content' => array(
			'value' => 'sidebar-content',
			'label' => __( 'Sidebar on left', 'orbit' ),
			'thumbnail' => get_template_directory_uri() . '/inc/images/sidebar-content.png',
		),
	);

	return $layout_options;
}

/**
 * Returns an array of pagination options registered for orbit.
 *
 * @since Orbit 1.0.0
 */
function orbit_pagination() {
	$pagination_options = array(
		'older-newer' => array(
			'value' => 'older-newer',
			'label' => __( 'Older-newer posts link', 'orbit' ),
		),
		'pages' => array(
			'value' => 'pages',
			'label' => __( 'Pages posts link', 'orbit' ),
		),
	);

	return $pagination_options;
}

/**
 * Returns an array of color scheme options registered for orbit.
 *
 * @since Orbit 1.0.0
 */
function orbit_color_scheme() {
	$color_scheme = array(
		'default' => array(
			'value' => 'blue',
			'label' => __( 'Blue', 'orbit' ),
			'thumbnail' => get_template_directory_uri() . '/inc/images/scheme-blue.png',
		),
		'orange' => array(
			'value' => 'orange',
			'label' => __( 'Orange', 'orbit' ),
			'thumbnail' => get_template_directory_uri() . '/inc/images/scheme-orange.png',
		),
		'green' => array(
			'value' => 'green',
			'label' => __( 'Green', 'orbit' ),
			'thumbnail' => get_template_directory_uri() . '/inc/images/scheme-green.png',
		),
	);

	return $color_scheme;
}

/**
 * Returns an array of font sans options registered for orbit.
 *
 * @since Orbit 1.0.0
 */
function orbit_font_sans() {
	$fontSans = array ( "sans-serif, Verdana, Geneva" ,
						"Geneva, sans-serif ",
						"Helvetica, sans-serif" ,
						"Futura, Century Gothic, AppleGothic, sans-serif",
						"Tahoma, Geneva, sans-serif" ,
						"Arial, Helvetica, sans-serif" ,
						"Arial Black, Gadget, sans-serif" );
						
	$fontSans = apply_filters( 'orbit_font_sans', $fontSans );
						
	return $fontSans;
}


/**
 * Returns an array of font sans options registered for orbit.
 *
 * @since Orbit 1.0.0
 */
function orbit_font_serif() {
	$fontSerif = array( "Georgia, Times, serif" ,
						"Times New Roman, Times, serif",
						"Cambria, Georgia, Times, serif",
						"Baskerville, Times, serif",
						"Bodoni MT, serif",
						"Garamond, Times, serif");
						
	$fontSerif = apply_filters( 'orbit_font_serif', $fontSerif );

	return $fontSerif;
}

/**
 * Returns an array of all fonts
 *
 * @since Orbit 1.0.0
 */
function orbit_font_all() {
	return array_merge( orbit_font_sans(), orbit_font_serif() );
}

/**
 * Returns an array of all fonts in two arrays
 *
 * @since Orbit 1.2.0
 */
function orbit_font_choise() {
	
	$sans_sans = orbit_font_sans();
	$sans_sans_final = array();
	foreach ( $sans_sans as $key => $value ) {
		$sans_sans_final[$value] = $value;
	}

	$sans_sans_serif = orbit_font_serif();
	$sans_sans_serif_final = array();
	foreach ( $sans_sans_serif as $key => $value ) {
		$sans_sans_serif_final[$value] = $value;
	}
	
	$choise['sans-serif'] = $sans_sans_serif_final;
	$choise['serif'] = $sans_sans_final;

	return $choise;
}

/*
 * Return array true if font selected by the user is different from default font or
 * google font is used for the user. (title and general fonts)
 *
 * @since Orbit 1.0.0
 */
function orbit_is_changed_custom_style(){
	$orbit_options = orbit_get_theme_options();
	$orbit_options = apply_filters( 'orbit_is_changed_custom_style', $orbit_options );
	$default_theme_options = orbit_get_default_theme_options();
	$custom_style =	 array( 'changed_style' => false,
							'changed_title' => false,
							'changed_base' => false,
							'font_title' => '',
							'font_base' => '',
							'css_user' => '',
							'color_scheme' => '',
							'color_title_main' => '',
							'color_title_seconds' => '');

	if ( $orbit_options['orbit_font_title'] != $default_theme_options['orbit_font_title'] ) {
		$custom_style['changed_style'] = true;
		$custom_style['changed_title'] = true;
		$custom_style['font_title'] = $orbit_options['orbit_font_title'];
	}

	if ( $orbit_options['google_font_title'] != $default_theme_options['google_font_title'] ) {
		$custom_style['changed_style'] = true;
		$custom_style['changed_title'] = true;
		$custom_style['font_title'] = $orbit_options['google_font_title'];
	}

	if ( $orbit_options['orbit_font_base']	!= $default_theme_options['orbit_font_base'] ) {
		$custom_style['changed_style'] = true;
		$custom_style['changed_base'] = true;
		$custom_style['font_base'] = $orbit_options['orbit_font_base'];
	}

	if ( $orbit_options['google_font_base']	 != $default_theme_options['google_font_base'] ) {
		$custom_style['changed_style'] = true;
		$custom_style['changed_base'] = true;
		$custom_style['font_base'] = $orbit_options['google_font_base'];
	}

	if ( $orbit_options['css_user']	 != $default_theme_options['css_user'] ) {
		$custom_style['changed_style'] = true;
		$custom_style['css_user'] = $orbit_options['css_user'];
	}

	if ( $orbit_options['color_scheme'] != $default_theme_options['color_scheme'] ) {
		$custom_style['changed_style'] = true;
		$custom_style['color_scheme'] = $orbit_options['color_scheme'];
	}

	if ( $orbit_options['color_title_main'] != $default_theme_options['color_title_main'] ) {
		$custom_style['changed_style'] = true;
		$custom_style['color_title_main'] = $orbit_options['color_title_main'];
	}
	
	if ( $orbit_options['color_title_seconds'] != $default_theme_options['color_title_seconds'] ) {
		$custom_style['changed_style'] = true;
		$custom_style['color_title_seconds'] = $orbit_options['color_title_seconds'];
	}

	return $custom_style;
}

/*
 * Return if h1 must be show in current page
 *
 * @since Orbit 1.0.0
 */
function orbit_show_title_page(){
	global $post;
	$show_title_page = orbit_get_theme_options( 'show_title_page' );
	$pages_no_title = explode( ' ', $show_title_page );

	// If show_title_page contains All or all the title don't show
	if ( in_array( 'All', $pages_no_title) || in_array( 'all', $pages_no_title ) )
		return false;

	// If current page is in pages_no_title, the title don't show
	if ( in_array( $post->post_name,  $pages_no_title ) )
		return false;

	return true;
}

/**
 * Truncate function
 *
 * @since Orbit 1.1.0
 */
function orbit_truncate_words( $string, $words = 20, $ellipsis = ' ...') {
	$new = preg_replace('/((\w+\W*){'.($words-1).'}(\w+))(.*)/', '${1}', $string);
	if( $new != $string ){
		return $new.$ellipsis;
	}else{
		return $string;
	}
}

/**
 * Template to render regular text setting field for options.
 *
 * @since Orbit 1.1.3
 */
function orbit_settings_field_regular_text( $field, $help = ''  ){
	$options = orbit_get_theme_options();
	?>
	<label class="<?php echo $field; ?>">
		<input id="orbit_theme_options[<?php echo $field; ?>]" class="regular-text" type="text" name="orbit_theme_options[<?php echo $field; ?>]" value="<?php if ( !empty( $options[$field] ) ) echo esc_attr( $options[$field] ); ?>" />
		<?php if ( $help != '' ) echo '<br />' .$help; ?>
	</label>
	<?php
}

/**
 * Template to render font selector field for options.
 *
 * @since Orbit 1.1.3
 */
function orbit_settings_field_font_selector( $field ) {
	$options = orbit_get_theme_options();
	?>
	<label class="<?php echo $field; ?>">
		<select id='orbit_theme_options[<?php echo $field; ?>]' name='orbit_theme_options[<?php echo $field; ?>]'>";
			<optgroup label='Sans-Serif'>
			<?php foreach ( orbit_font_sans() as $font_item ) {
				echo "<option style='font-family:$font_item;' value='$font_item'";
				selected( $options[$field], $font_item );
				echo ">$font_item</option>";
			} ?>
			</optgroup>
			<optgroup label='Serif'>
			<?php foreach ( orbit_font_serif() as $font_item ) {
				echo "<option style='font-family:$font_item;' value='$font_item'";
				selected( $options[$field], $font_item );
				echo ">$font_item</option>";
			} ?>
			</optgroup>
		</select>
	</label>
	<?php
}

/**
 * Template to checkbox setting field for options
 *
 * @since Orbit 1.1.3
 */
function orbit_settings_field_regular_checkbox( $field, $help = '' ) {
	$options = orbit_get_theme_options();
	?>
	<label class="description">
		<input id="orbit_theme_options[<?php echo $field; ?>]" type="checkbox" name="orbit_theme_options[<?php echo $field; ?>]" value="1" <?php checked( $options[$field], 1 ); ?> />
		<?php if ( $help != '') echo '<br />' .$help; ?>
	</label>
	<?php
}

/**
 * Get version theme
 *
 * @since Orbit 1.1.3
 */
function orbit_get_flexslider_setting() {
	$options = orbit_get_theme_options();
	
	$output = '';
	if ( $options['slide_controlNav'] != '1' )
		$output .= 'controlNav:false,';
	if ( $options['slide_slideshow'] != '1' )
		$output .= 'slideshow:false,';
	if ( $options['slide_slideshowSpeed'] != '7000' )
		$output .= 'slideshowSpeed:' . $options['slide_slideshowSpeed'] . ',';
		
	if ( $output != '' ) {
		$output = '{' . $output . '}';
		echo $output;
	}
}


/**
 * Get version theme
 *
 * @since Orbit 1.1.3
 */
function orbit_get_version_theme() {
	return orbit_get_setting( 'orbit_version' );
}


/**
 * Custom Walker for main menu
 *
 * @since Orbit 1.2.0
 */
class Walker_Orbit_Nav_Menu extends Walker_Nav_Menu {

	public function start_el( &$output, $item, $depth = 0, $args = array(), $id = 0 ) {
		$indent = ( $depth ) ? str_repeat( "\t", $depth ) : '';

		$classes = empty( $item->classes ) ? array() : (array) $item->classes;
		$classes[] = 'menu-item-' . $item->ID;

		$class_names = join( ' ', apply_filters( 'nav_menu_css_class', array_filter( $classes ), $item, $args, $depth ) );
		$class_names = $class_names ? ' class="' . esc_attr( $class_names ) . '"' : '';

		$id = apply_filters( 'nav_menu_item_id', 'menu-item-'. $item->ID, $item, $args, $depth );
		$id = $id ? ' id="' . esc_attr( $id ) . '"' : '';

		$output .= $indent . '<li' . $id . $class_names .'>';

		$atts = array();
		$atts['title']  = ! empty( $item->attr_title ) ? $item->attr_title : '';
		$atts['target'] = ! empty( $item->target )     ? $item->target     : '';
		$atts['rel']    = ! empty( $item->xfn )        ? $item->xfn        : '';
		$atts['href']   = ! empty( $item->url )        ? $item->url        : '';

		$atts = apply_filters( 'nav_menu_link_attributes', $atts, $item, $args, $depth );

		$attributes = '';
		foreach ( $atts as $attr => $value ) {
			if ( ! empty( $value ) ) {
				$value = ( 'href' === $attr ) ? esc_url( $value ) : esc_attr( $value );
				$attributes .= ' ' . $attr . '="' . $value . '"';
			}
		}

		$item_output = $args->before;
		$item_output .= '<a'. $attributes .'>';
		/** This filter is documented in wp-includes/post-template.php */
		$item_output .= $args->link_before . apply_filters( 'the_title', $item->title, $item->ID );
		
		if ( $args->walker->has_children )
			$item_output .= '<span class="sub-menu-expand"><span class="screen-reader-text">Expandir</span></span>';	
		
		$item_output .= $args->link_after;
		$item_output .= '</a>';
		$item_output .= $args->after;

		$output .= apply_filters( 'walker_nav_menu_start_el', $item_output, $item, $depth, $args );
	}

}

?>