<?php

/**
 * Add action and add filter
 * Class Online_Learning_Include
 */
class Online_Learning_Include {
	public function __construct() {
		$this->include();
		// action back to top
		add_action( 'after_setup_theme', array( $this, 'setup_theme' ), 10 );
		add_action( 'wp_enqueue_scripts', array( $this, 'enqueue_scripts' ), 10 );
		add_action( 'login_init', array( $this, 'redirect_wp_login_to_custom_page' ), 10 );
		add_filter( 'query_loop_block_query_vars', array( $this, 'filter_query_loop_related_posts' ), 10, 2 );
		add_filter( 'body_class', array( $this, 'body_class' ), 10 );
		add_filter( 'comment_form_submit_button', array( $this, 'custom_submit_button' ), 10, 2 );
		add_action( 'tgmpa_register', array( $this, 'register_required_plugins' ) );
		add_filter( 'render_block', array( $this, 'add_fallback_featured_image' ), 10, 2 );
	}

	public function include() {
		require ONLINE_LEARNING_DIR . 'inc/class-tgm-plugin-activation.php';
		require ONLINE_LEARNING_DIR . 'inc/admin/dashboard.php';
	}

	public function setup_theme() {
		// Adding support for core block visual styles.
		add_theme_support( 'wp-block-styles' );
		add_theme_support( 'editor-styles' );

		// Adds theme support for post formats.
		add_theme_support( 'post-formats', array( 'aside', 'audio', 'chat', 'gallery', 'image', 'link', 'quote', 'status', 'video' ) );

		// Enqueue editor styles.
		add_editor_style( 'style.css' );

		/**
		 * Load core file
		 */
		require get_theme_file_path() . '/inc/core/block-patterns.php';
		require get_theme_file_path() . '/inc/core/block-styles.php';
	}

	public function enqueue_scripts() {
		// Enqueue theme stylesheet.
		wp_enqueue_style( 'thim-style', get_stylesheet_uri(), array(), ONLINE_LEARNING_VERSION );
		// Enqueue theme script
		wp_enqueue_script( 'gsap', ONLINE_LEARNING_URI . 'assets/js/gsap.js', array( 'jquery' ), ONLINE_LEARNING_VERSION, true );
		wp_enqueue_script( 'scrolltrigger', ONLINE_LEARNING_URI . 'assets/js/scrolltrigger.js', array( 'jquery' ), ONLINE_LEARNING_VERSION, true );
		wp_enqueue_script( 'thim-custom', ONLINE_LEARNING_URI . 'assets/js/thim-custom.js', array( 'jquery' ), ONLINE_LEARNING_VERSION, true );
	}

	/**
	 * Change default WordPress login URL to a custom page with slug 'account'.
	 *
	 * @param string $login_url Default login URL.
	 * @param string $redirect Redirect destination after login.
	 *
	 * @return string
	 */
	public function redirect_wp_login_to_custom_page() {
		if ( ! is_user_logged_in() && empty( $_REQUEST['action'] ) && class_exists( '\ThimPress\ThimBlocks\Thim_Blocks' ) ) {
			$account_page = get_page_by_path( 'account' );
			if ( ! $account_page ) {
				return;
			}

			$redirect_to = ! empty( $_GET['redirect_to'] ) ? $_GET['redirect_to'] : '';
			$login_page  = get_permalink( $account_page->ID );

			if ( $redirect_to && strpos( urldecode( $redirect_to ), '/account' ) !== false ) {
				return;
			}

			if ( $redirect_to ) {
				$login_page = add_query_arg( 'redirect_to', urlencode( $redirect_to ), esc_url( $login_page ) );
			}

			wp_redirect( $login_page );
			exit;
		}
	}

	/**
	 * Modify query loop block to show related posts in the same category when viewing a single post.
	 *
	 * @param array $query_vars Query vars for the block.
	 * @param WP_Block|array $block Block instance.
	 *
	 * @return array Modified query vars.
	 */
	public function filter_query_loop_related_posts( $query_vars, $block ) {
		if ( $block instanceof WP_Block ) {
			$block_name = $block->name;
		} else {
			$block_name = $block['blockName'] ?? '';
		}

		if ( $block_name === 'core/post-template' && is_singular( 'post' ) ) {
			$current_id = get_queried_object_id();
			$cats       = wp_get_post_categories( $current_id );

			if ( ! empty( $cats ) ) {
				$query_vars['category__in'] = $cats;
				$query_vars['post__not_in'] = array( $current_id );
			}
		}

		return $query_vars;
	}

	/**
	 * Adds custom classes to the array of body classes.
	 *
	 * @param array $classes Classes for the body element.
	 *
	 * @return array
	 */
	public function body_class( $classes ) {
		$classes[] = 'animate-button-hover';

		return $classes;
	}

	/**
	 * Replace the default comment form submit input with a <button> element.
	 *
	 * @param string $submit_button The default HTML markup for the submit button.
	 * @param array $args An array of default arguments for the comment form.
	 *
	 * @return string
	 */
	public function custom_submit_button( $submit_button, $args ) {
		$button = sprintf(
			'<button name="%1$s" type="submit" id="%2$s" class="%3$s">%4$s</button>',
			esc_attr( $args['name_submit'] ),
			esc_attr( $args['id_submit'] ),
			esc_attr( $args['class_submit'] ),
			esc_html( $args['label_submit'] )
		);

		return $button;
	}


	public function register_required_plugins() {
		$plugins = array(
			array(
				'name'     => 'Thim Blocks',
				'slug'     => 'thim-blocks',
				'required' => true,
			),
			array(
				'name'     => 'Contact Form 7',
				'slug'     => 'contact-form-7',
				'required' => false,
			),
			array(
				'name'     => 'LearnPress',
				'slug'     => 'learnpress',
				'required' => false,
			),
			array(
				'name'     => 'LearnPress – Course Wishlist',
				'slug'     => 'learnpress-wishlist',
				'required' => false,
			),
			array(
				'name'     => 'LearnPress – Course Review',
				'slug'     => 'learnpress-course-review',
				'required' => false,
			),

			array(
				'name'     => 'MailChimp for WordPress',
				'slug'     => 'mailchimp-for-wp',
				'required' => false,
			),
		);

		$config = array(
			'id'           => 'online-learning',                 // Unique ID for hashing notices for multiple instances of TGMPA.
			'default_path' => '',                      // Default absolute path to bundled plugins.
			'menu'         => 'tgmpa-install-plugins', // Menu slug.
			'has_notices'  => true,                    // Show admin notices or not.
			'dismissable'  => true,                    // If false, a user cannot dismiss the nag message.
			'dismiss_msg'  => '',                      // If 'dismissable' is false, this message will be output at top of nag.
			'is_automatic' => false,                   // Automatically activate plugins after installation or not.
			'message'      => '',                      // Message to output right before the plugins table.
		);

		tgmpa( $plugins, $config );
	}

	/**
	 * Add fallback image for core/post-featured-image block
	 *
	 * @param string $block_content
	 * @param array  $block
	 *
	 * @return string
	 */
	public function add_fallback_featured_image( $block_content, $block ) {
		if ( $block['blockName'] === 'core/post-featured-image' ) {
			if ( empty( $block_content ) ) {
				$default_url = ONLINE_LEARNING_URI . '/assets/images/no-image.png';
				$alt         = esc_attr__( 'Default featured image', 'online-learning' );

				$block_content = sprintf(
					'<img src="%s" alt="%s" class="wp-block-post-featured-image" />',
					esc_url( $default_url ),
					$alt
				);
			}
		}

		return $block_content;
	}
}

new Online_Learning_Include();
