<?php
/**
 * Performance Features
 *
 * @package niche_blog
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

class Niche_Blog_Performance {

	/**
	 * Instance of this class.
	 */
	private static $instance = null;

	/**
	 * Get instance.
	 */
	public static function get_instance() {
		if ( null === self::$instance ) {
			self::$instance = new self();
		}
		return self::$instance;
	}

	/**
	 * Constructor.
	 */
	public function __construct() {
		// Initialize hooks based on customizer settings
		add_action( 'init', array( $this, 'init_features' ) );
		// These run earlier or differently
		if ( get_theme_mod( 'perf_minify_html' ) && ! is_admin() ) {
			add_action( 'template_redirect', array( $this, 'buffer_start' ), 1 );
		}
		if ( get_theme_mod( 'perf_page_cache' ) && ! is_admin() && ! is_user_logged_in() ) {
			add_action( 'template_redirect', array( $this, 'serve_cache' ), 0 );
			add_action( 'shutdown', array( $this, 'save_cache' ), 999 );
			add_action( 'save_post', array( $this, 'clear_cache' ) );
			add_action( 'comment_post', array( $this, 'clear_cache' ) );
		}
	}

	public function init_features() {
		if ( get_theme_mod( 'perf_disable_emojis' ) ) {
			$this->disable_emojis();
		}
		if ( get_theme_mod( 'perf_lazy_load', true ) ) {
			add_filter( 'wp_lazy_loading_enabled', '__return_true' );
		} else {
			add_filter( 'wp_lazy_loading_enabled', '__return_false' );
		}
		if ( get_theme_mod( 'perf_reduce_layout_shifts' ) ) {
			add_filter( 'the_content', array( $this, 'add_width_height' ) );
		}
		if ( get_theme_mod( 'perf_defer_scripts' ) ) {
			add_filter( 'script_loader_tag', array( $this, 'defer_scripts' ), 10, 2 );
		}
		if ( get_theme_mod( 'perf_delay_scripts' ) ) {
			add_filter( 'script_loader_tag', array( $this, 'delay_scripts' ), 10, 2 );
			add_action( 'wp_footer', array( $this, 'delay_scripts_trigger' ) );
		}
		if ( get_theme_mod( 'perf_preload_resources' ) ) {
			add_action( 'wp_head', array( $this, 'preload_resources' ), 1 );
		}
		if ( get_theme_mod( 'perf_instant_navigation' ) ) {
			add_action( 'wp_footer', array( $this, 'inject_prefetch_script' ) );
		}
		if ( get_theme_mod( 'perf_google_fonts_optimize' ) ) {
			add_filter( 'niche_blog_get_fonts_url', array( $this, 'optimize_fonts_url' ) );
			add_action( 'wp_head', array( $this, 'preconnect_fonts' ), 0 );
		}
		
		// Customizer JS for Clear Cache
		add_action( 'customize_controls_enqueue_scripts', array( $this, 'customizer_scripts' ) );
		if ( get_theme_mod( 'perf_page_cache' ) ) {
			// Admin Bar Clear Cache
			if ( is_user_logged_in() ) {
				add_action( 'admin_bar_menu', array( $this, 'add_admin_bar_node' ), 100 );
				add_action( 'wp_footer', array( $this, 'print_admin_bar_script' ) );
				add_action( 'admin_footer', array( $this, 'print_admin_bar_script' ) );
			}
		}
		// AJAX for Clear Cache
		add_action( 'wp_ajax_niche_blog_clear_cache', array( $this, 'ajax_clear_cache' ) );
	}
	
	public function add_admin_bar_node( $wp_admin_bar ) {
		if ( ! current_user_can( 'edit_theme_options' ) ) return;
		
		$icon = '<span class="ab-icon dashicons dashicons-trash" style="margin-top: 2px;"></span>';
		
		$wp_admin_bar->add_node( array(
			'id'    => 'niche-blog-clear-cache',
			'title' => $icon . 'Clear Cache',
			'href'  => '#',
			'meta'  => array( 
				'class' => 'niche-blog-clear-cache-btn',
				'title' => 'Clear Niche Blog Page Cache',
				'onclick' => 'return false;'
			)
		) );
	}

	public function print_admin_bar_script() {
		if ( ! current_user_can( 'edit_theme_options' ) ) return;
		?>
		<script>
		jQuery(document).ready(function($) {
			$('#wp-admin-bar-niche-blog-clear-cache').click(function(e) {
				e.preventDefault();
				var item = $(this);
				var originalText = item.find('.ab-item').text();
				item.find('.ab-item').text('Clearing...');
				
				$.ajax({
					url: '<?php echo admin_url( 'admin-ajax.php' ); ?>',
					type: 'POST',
					data: {
						action: 'niche_blog_clear_cache',
						security: '<?php echo wp_create_nonce("niche_blog_clear_cache"); ?>'
					},
					success: function(response) {
						alert('Cache Cleared!');
						item.find('.ab-item').html('<span class="ab-icon dashicons dashicons-trash" style="margin-top: 2px;"></span>Clear Cache');
					},
					error: function() {
						alert('Error clearing cache.');
						item.find('.ab-item').html('<span class="ab-icon dashicons dashicons-trash" style="margin-top: 2px;"></span>Clear Cache');
					}
				});
			});
		});
		</script>
		<?php
	}

	public function customizer_scripts() {
		wp_enqueue_script( 'niche-customizer-controls', get_template_directory_uri() . '/assets/js/niche-customizer-controls.js', array( 'jquery' ), '1.0', true );
		wp_localize_script( 'niche-customizer-controls', 'niche_blog_vars', array(
			'nonce' => wp_create_nonce( 'niche_blog_clear_cache' )
		) );
	}
	
	public function ajax_clear_cache() {
		check_ajax_referer( 'niche_blog_clear_cache', 'security' );
		if ( ! current_user_can( 'edit_theme_options' ) ) {
			wp_send_json_error( 'Permission denied' );
		}
		$this->clear_cache();
		wp_send_json_success( 'Cache cleared' );
	}

	public function disable_emojis() {
		remove_action( 'wp_head', 'print_emoji_detection_script', 7 );
		remove_action( 'admin_print_scripts', 'print_emoji_detection_script' );
		remove_action( 'wp_print_styles', 'print_emoji_styles' );
		remove_action( 'admin_print_styles', 'print_emoji_styles' );
		remove_filter( 'the_content_feed', 'wp_staticize_emoji' );
		remove_filter( 'comment_text_rss', 'wp_staticize_emoji' );
		remove_filter( 'wp_mail', 'wp_staticize_emoji_for_email' );
	}

	// Layout Shifts: Add width/height to images in content
	public function add_width_height( $content ) {
		// Basic WP implementation usually handles this, ensures it's enforced
		if ( ! function_exists( 'wp_img_tag_add_width_and_height_attributes' ) ) {
			return $content;
		}
		// We could do custom regex here if WP core fails, but we'll trust core filters are active
		return $content; 
	}

	// Minify HTML
	public function buffer_start() {
		ob_start( array( $this, 'minify_html_output' ) );
	}

	public function minify_html_output( $buffer ) {
		// Minify Inline CSS
		$buffer = preg_replace_callback( '/<style[^>]*>(.*?)<\/style>/s', function( $matches ) {
			$css = $matches[1];
			$css = preg_replace( '!/\*[^*]*\*+([^/][^*]*\*+)*/!', '', $css ); // Remove comments
			$css = str_replace( array( "\r\n", "\r", "\n", "\t" ), '', $css ); // Remove newlines/tabs
			$css = preg_replace( '/\s*([{}|:;,])\s+/', '$1', $css ); // Remove space around chars
			$css = preg_replace( '/\s\s+(?!^)?/', ' ', $css ); // Reduce multiple spaces
			return str_replace( $matches[1], $css, $matches[0] );
		}, $buffer );

		$search = array(
			'/\>[^\S ]+/s',     // strip whitespaces after tags, except space
			'/[^\S ]+\</s',     // strip whitespaces before tags, except space
			'/(\s)+/s',         // shorten multiple whitespace sequences
			'/<!--(.|\s)*?-->/' // Remove HTML comments
		);
		$replace = array(
			'>',
			'<',
			'\\1',
			''
		);
		$buffer = preg_replace( $search, $replace, $buffer );
		return $buffer;
	}

	// Defer Scripts
	public function defer_scripts( $tag, $handle ) {
		if ( is_admin() ) return $tag;
		// Skip jquery or crucial scripts if needed
		if ( strpos( $handle, 'jquery' ) !== false ) return $tag;
		return str_replace( ' src', ' defer src', $tag );
	}

	// Delay Scripts
	public function delay_scripts( $tag, $handle ) {
		if ( is_admin() ) return $tag;
		// Skip jquery for safety, or targeted delay
		if ( strpos( $handle, 'jquery' ) !== false ) return $tag;
		
		// Change type to prevent execution
		$tag = str_replace( ' type=\'text/javascript\'', '', $tag );
		$tag = str_replace( ' type="text/javascript"', '', $tag );
		
		// Move src to data-src
		// Simple replacement, might need regex for robustness
		$tag = str_replace( ' src=', ' type="niche-delayed" data-src=', $tag );
		$tag = str_replace( '<script', '<script type="niche-delayed"', $tag );
		
		return $tag;
	}

	public function delay_scripts_trigger() {
		?>
		<script>
		function niche_load_delayed() {
			var scripts = document.querySelectorAll('script[type="niche-delayed"]');
			scripts.forEach(function(script) {
				var newScript = document.createElement('script');
				if (script.getAttribute('data-src')) {
					newScript.src = script.getAttribute('data-src');
				} else {
					newScript.innerHTML = script.innerHTML;
				}
				document.body.appendChild(newScript);
			});
			// Remove events
			window.removeEventListener('mousemove', niche_load_delayed);
			window.removeEventListener('scroll', niche_load_delayed);
			window.removeEventListener('touchstart', niche_load_delayed);
		}
		window.addEventListener('mousemove', niche_load_delayed);
		window.addEventListener('scroll', niche_load_delayed);
		window.addEventListener('touchstart', niche_load_delayed);
		</script>
		<?php
	}

	// Instant Navigation (Prefetch)
	public function inject_prefetch_script() {
		?>
		<script>
		/*! Instant Page Navigation */
		(function(){
			const prefetcher = document.createElement('link');
			const support = (prefetcher.relList && prefetcher.relList.supports && prefetcher.relList.supports('prefetch'));
			if (!support) return;

			const fetcher = (url) => {
				const link = document.createElement('link');
				link.rel = 'prefetch';
				link.href = url;
				document.head.appendChild(link);
			};

			let timer;
			document.addEventListener('mouseover', (e) => {
				const link = e.target.closest('a');
				if (!link || !link.href || link.href.startsWith('#') || link.href.includes('wp-admin') || link.href.includes('wp-login')) return;
				
				// Same domain check
				if (link.origin !== location.origin) return;

				timer = setTimeout(() => {
					fetcher(link.href);
				}, 65); // 65ms delay to prevent prefetching on fast scroll
			});

			document.addEventListener('mouseout', (e) => {
				if (timer) clearTimeout(timer);
			});
		})();
		</script>
		<?php
	}

	// Resources Preload
	public function preload_resources() {
		if ( has_post_thumbnail() ) {
			$id = get_post_thumbnail_id();
			$src = wp_get_attachment_image_src( $id, 'full' );
			if ( $src ) {
				echo '<link rel="preload" as="image" href="' . esc_url( $src[0] ) . '" fetchpriority="high">';
			}
		}
	}

	// Google Fonts
	public function optimize_fonts_url( $url ) {
		if ( strpos( $url, 'display=swap' ) === false ) {
			$url .= '&display=swap';
		}
		return $url;
	}

	public function preconnect_fonts() {
		echo '<link rel="preconnect" href="https://fonts.gstatic.com" crossorigin>';
	}

	// Caching
	private function get_cache_path() {
		$upload_dir = wp_upload_dir();
		return $upload_dir['basedir'] . '/niche-blog-cache/' . md5( $_SERVER['REQUEST_URI'] ) . '.html';
	}

	public function serve_cache() {
		$path = $this->get_cache_path();
		if ( file_exists( $path ) && ( time() - filemtime( $path ) < 86400 ) ) { // 24h expiry
			readfile( $path );
			exit;
		}
		ob_start();
	}

	public function save_cache() {
		if ( is_404() || is_search() ) return;
		$path = $this->get_cache_path();
		$dir = dirname( $path );
		if ( ! file_exists( $dir ) ) {
			mkdir( $dir, 0755, true );
		}
		$content = ob_get_contents();
		if ( $content ) {
			file_put_contents( $path, $content );
		}
	}

	public function clear_cache() {
		$upload_dir = wp_upload_dir();
		$dir = $upload_dir['basedir'] . '/niche-blog-cache/';
		// Recursive delete would be better, but simple glob for now
		array_map( 'unlink', glob( "$dir/*" ) );
	}
}

// Initialize
Niche_Blog_Performance::get_instance();
