<?php
/**
 * Functions which enhance the theme by hooking into WordPress
 *
 * @package News Express
 */

/**
 * Adds custom classes to the array of body classes.
 *
 * @param array $classes Classes for the body element.
 * @return array
 */
function news_express_body_classes( $classes ) {
	// Adds a class of hfeed to non-singular pages.
	if ( ! is_singular() ) {
		$classes[] = 'hfeed';
	}

	// Adds a class of no-sidebar when there is no sidebar present.
	if ( ! is_active_sidebar( 'sidebar-1' ) && ! is_active_sidebar( 'left-sidebar' ) ) {
		$classes[] = 'no-sidebar';
	}

	$news_express_site_container_layout = news_express_get_customizer_option_value( 'news_express_site_container_layout' );
	if ( ! empty( $news_express_site_container_layout ) ) {
		$classes[] = 'news-express-site-layout--'.$news_express_site_container_layout;
	}

    if ( isset( $_COOKIE["news-express-site-mode-cookie"] ) ) {
        $classes[] = $_COOKIE["news-express-site-mode-cookie"];
    } else {
    	$classes[] = news_express_get_customizer_option_value( 'news_express_site_mode' );
    }

	// archive page style
	if ( !is_page() || !is_singular() || !is_single() ) {
		$news_express_archive_page_style = news_express_get_customizer_option_value( 'news_express_archive_page_style' );
		$classes[] = esc_attr( $news_express_archive_page_style );
	}

	$global_archive_sidebar = news_express_get_customizer_option_value( 'news_express_archive_sidebar_layout' );
	$global_posts_sidebar   = news_express_get_customizer_option_value( 'news_express_posts_sidebar_layout' );
	$global_pages_sidebar   = news_express_get_customizer_option_value( 'news_express_pages_sidebar_layout' );

	if ( is_page() ) {
	    $classes[] = esc_attr( $global_pages_sidebar );
	} elseif ( is_single() || is_singular() ) {
	    $classes[] = esc_attr( $global_posts_sidebar );
	    $news_express_single_posts_layout = news_express_get_customizer_option_value( 'news_express_single_posts_layout' );
	    $classes[] = 'single-'.esc_attr( $news_express_single_posts_layout );
	} elseif ( is_archive() || is_search() ) {
	    $classes[] = esc_attr( $global_archive_sidebar );
	} elseif ( is_home() || is_front_page() ) {
		$classes[] = esc_attr( $global_archive_sidebar );
	}

	return $classes;
}
add_filter( 'body_class', 'news_express_body_classes' );

/**
 * Add a pingback url auto-discovery header for single posts, pages, or attachments.
 */
function news_express_pingback_header() {
	if ( is_singular() && pings_open() ) {
		printf( '<link rel="pingback" href="%s">', esc_url( get_bloginfo( 'pingback_url' ) ) );
	}
}
add_action( 'wp_head', 'news_express_pingback_header' );


/*---------------------------- Enqueue style and scripts -------------------------------------------------------------*/

	if ( ! function_exists( 'news_express_admin_scripts' ) ) :

		/**
		 * Enqueue admin scripts and styles.
		 */
		function news_express_admin_scripts( $hook ) {

		    // Only needed on these admin screens
		    if ( $hook != 'edit.php' && $hook != 'post.php' && $hook != 'post-new.php' && 'widgets.php' != $hook ) {
		        return;
		    }

		    wp_enqueue_style( 'news-express-widget-style', get_template_directory_uri() . '/inc/widgets/assets/css/widget-admin-style.css', array(), NEWS_EXPRESS_VERSION );

		    wp_enqueue_script( 'news-express-widget-script', get_template_directory_uri() . '/inc/widgets/assets/js/widget-admin-script.js', array( 'jquery' ), NEWS_EXPRESS_VERSION, true );
		}

	endif;

	add_action( 'admin_enqueue_scripts', 'news_express_admin_scripts' );

	if ( ! function_exists( 'news_express_scripts' ) ) :

		/**
		 * Enqueue scripts and styles.
		 */
		function news_express_scripts() {

			wp_enqueue_style( 'news-express-fonts', news_express_google_font_callback(), array(), null );

			wp_enqueue_style( 'lightslider', get_template_directory_uri() . '/assets/library/lightslider/css/lightslider.min.css', array(), NEWS_EXPRESS_VERSION );

			wp_enqueue_style( 'news-express-preloader', get_template_directory_uri(). '/assets/css/ne-preloader.css', array(), NEWS_EXPRESS_VERSION );

			wp_enqueue_style( 'news-express-dark-mode', get_template_directory_uri(). '/assets/css/ne-dark-mode.css', array(), NEWS_EXPRESS_VERSION );

			wp_enqueue_style( 'news-express-style', get_stylesheet_uri(), array(), NEWS_EXPRESS_VERSION );

			wp_enqueue_style( 'news-express-responsive-style', get_template_directory_uri() . '/assets/css/ne-responsive.css', array(), NEWS_EXPRESS_VERSION );
			
			wp_enqueue_style( 'font-awesome', get_template_directory_uri() . '/assets/library/font-awesome/css/all.min.css', array(), '6.5.1' );

			wp_enqueue_script( 'jquery-ui-tabs' );

			wp_enqueue_script( 'jquery-cookie', get_template_directory_uri() . '/assets/library/jquery-cookie/jquery.cookie.js', array(), NEWS_EXPRESS_VERSION, true );

			wp_enqueue_script( 'lightslider', get_template_directory_uri() . '/assets/library/lightslider/js/lightslider.min.js', array(), NEWS_EXPRESS_VERSION, true );

			wp_enqueue_script( 'jquery-marquee', get_template_directory_uri() . '/assets/library/js-marquee/jquery.marquee.js', array(), NEWS_EXPRESS_VERSION, true );

			wp_enqueue_script( 'jquery-header-sticky', get_template_directory_uri() . '/assets/library/sticky/jquery.sticky.min.js', array(), NEWS_EXPRESS_VERSION, true );

			wp_enqueue_script( 'jquery-sticky-sidebar', get_template_directory_uri() . '/assets/library/sticky-sidebar/theia-sticky-sidebar.min.js', array(), NEWS_EXPRESS_VERSION, true );

			wp_enqueue_script( 'news-express-navigation', get_template_directory_uri() . '/assets/js/navigation.js', array(), NEWS_EXPRESS_VERSION, true );

			wp_enqueue_script( 'news-express-main-scripts', get_template_directory_uri() . '/assets/js/main-scripts.js', array( 'jquery' ), NEWS_EXPRESS_VERSION, true );

			wp_enqueue_script( 'news-express-keyboard-accessibility', get_template_directory_uri() . '/assets/js/keyboard-accessibility.js', array( 'jquery' ), NEWS_EXPRESS_VERSION, true );

			$news_express_header_sticky_enable 	= news_express_get_customizer_option_value( 'news_express_header_sticky_enable' );
			$news_express_sidebar_sticky_enable 	= news_express_get_customizer_option_value( 'news_express_sidebar_sticky_enable' );
			$header_sticky = $news_express_header_sticky_enable ? 'true' : 'false';
			$sidebar_sticky = $news_express_sidebar_sticky_enable ? 'true' : 'false';

			wp_localize_script( 'news-express-main-scripts', 'OG_JSObject',
				array(
		            'sidebar_sticky'    => $sidebar_sticky,
		            'header_sticky'     => $header_sticky
		        )
		    );

			if ( is_singular() && comments_open() && get_option( 'thread_comments' ) ) {
				wp_enqueue_script( 'comment-reply' );
			}
		}

	endif;

	add_action( 'wp_enqueue_scripts', 'news_express_scripts' );

if ( ! function_exists( 'news_express_social_icons_array' ) ) :

    /**
     * List of icons
     *
     * @return array();
     * @since 1.0.0
     */
    function news_express_social_icons_array() {
        return array( "fab fa-facebook","fab fa-facebook-messenger","fab fa-threads","fab fa-twitter","fab fa-x-twitter","fab fa-instagram","fab fa-linkedin","fab fa-youtube","fab fa-pinterest","fab fa-snapchat","fab fa-tiktok","fab fa-whatsapp","fab fa-telegram","fab fa-reddit","fab fa-tumblr","fab fa-vk","fab fa-weibo","fab fa-weixin","fab fa-qq","fab fa-quora","fab fa-discord","fab fa-medium","fab fa-github","fab fa-slack","fab fa-behance","fab fa-dribbble","fab fa-xing","fab fa-line","fab fa-viber","fab fa-skype","fab fa-flickr","fab fa-spotify","fab fa-soundcloud","fab fa-mixcloud","fab fa-renren");
    }

endif;


if ( ! function_exists( 'news_express_get_date_format_args' ) ) :

	/**
	 * Generate date format array for query arguments
	 * 
	 * @return array
	 * @since 1.0.0
	 */
	function news_express_get_date_format_args( $format ) {

		switch( $format ) {

			case 'today':
				$today_date = getdate();
				$get_args 	= array(
					'year'  => $today_date['year'],
					'month' => $today_date['mon'],
					'day'   => $today_date['mday'],
				);
				
				return $get_args ;
				break;

			case 'this-week':
				$get_args = array(
					'year'  => date( 'Y' ),
					'week'  => date( 'W' )
				);
				
				return $get_args;
				break;

			case 'last-week':
				$this_week = date( 'W' );

				if ( $this_week != 1 ) {
					$last_week = $this_week - 1;
				} else {
					$last_week = 52;
				}

				$this_year = date( 'Y' );

				if ( $last_week != 52 ) {
					$this_year = date( 'Y' );
				} else {
					$this_year = date( 'Y' ) -1;
				}

				$get_args = array(
					'year'  => $this_year,
					'week'  => $last_week
				);

				return $get_args;
				break;

			case 'this-month':
				$today_date = getdate();
				$get_args 	= array(
					'month' => $today_date['mon']
				);

				return $get_args;
				break;

			case 'last-month': 
				$this_date = getdate();

				if ( $this_date['mon'] != 1 ) {
					$last_month = $this_date['mon'] - 1;
				} else {
					$last_month = 12;
				}

				$this_year = date( 'Y' );
				if ( $last_month != 12 ) {
					$this_year = date('Y');
				} else {
					$this_year = date('Y') - 1;
				}

				$get_args = array(
					'year'  => $this_year,
					'month'  => $last_month
				);
				
				return $get_args;
				break;
			
			default: return [];
		}

	}

endif;

if ( ! function_exists( 'news_express_render_tab_posts' ) ) :

	/**
	 * function to output the tab posts.
	 *
	 * @since 1.0.0
	 */
	function news_express_render_tab_posts( $news_express_tab_type ) {

		$news_express_banner_tab_post_count	= news_express_get_customizer_option_value( 'news_express_banner_tab_post_count' );

		$news_express_tab_args = array(

			'posts_per_page'        => $news_express_banner_tab_post_count,
            'ignore_sticky_posts'   => true
		);

		switch ( $news_express_tab_type ) {
			case 'top-stories':
				$news_express_tab_args['orderby'] = 'comment_count';
				break;

			case 'trending':
				$news_express_trending_category = news_express_get_customizer_option_value( 'news_express_banner_tab_trending_category' );
				if ( 'all' !== $news_express_trending_category ) {
					$news_express_tab_args['category_name'] = $news_express_trending_category;
				}
				break;
			default:
				break;
		}

?>
		<div class="<?php echo esc_attr( $news_express_tab_type ); ?>-posts-wrapper tab-posts-wrapper">
			<?php
				$news_express_tab_query = new WP_Query( $news_express_tab_args );
				if ( $news_express_tab_query->have_posts() ) {
					while ( $news_express_tab_query->have_posts() ) {
						$news_express_tab_query->the_post();
						if ( has_post_thumbnail() ) {
	                        $post_img	= 'has-image';
	                    } else {
	                        $post_img   = 'no-image';
	                    }
	        ?>
	        			<div class="single-post-wrap <?php echo esc_attr( $post_img ); ?>">
	        				<?php news_express_post_thumbnail( 'news-express-block-medium' ); ?>
                            <div class="post-content-wrap">

                                <div class="post-title-wrap">
                                    <h2 class="post-title"><a href="<?php the_permalink(); ?>"><?php the_title(); ?></a></h2>
                                </div><!-- .post-title-wrap -->
								<div class="post-cats-meta-wrap news-express-flex">
                                        <?php news_express_the_post_categories_list( get_the_ID(), 1 ); ?>
                            			<?php news_express_posted_by(get_the_ID(), 3 ); ?>
									</div><!-- .post-cats-meta-wrap -->
							
                            </div><!-- .post-content-wrap -->
	        			</div><!-- .single-post-wrap -->
	        <?php
					}
				}
			?>
		</div><!-- .tab-posts-wrapper -->
<?php

	}

endif;

/*---------------------------Site Mode Switcher---------------------------------*/

if ( ! function_exists( 'news_express_site_mode_switcher' ) ) :

    /**
     * Display the site mode switcher icon in header section
     * 
     * @since 1.5.2
     */
    function news_express_site_mode_switcher() {

        $news_express_site_mode_switcher_option = news_express_get_customizer_option_value( 'news_express_site_mode_switcher_option' );
        if ( false === $news_express_site_mode_switcher_option ) {
            return;
        }

        $site_mode = isset( $_COOKIE['news-express-site-mode-cookie'] ) ? $_COOKIE['news-express-site-mode-cookie'] : 'light-mode';
        
        $site_mode = in_array( $site_mode, ['light-mode', 'dark-mode'] ) ? $site_mode : 'light-mode';
?>
        <div id="np-site-mode-wrap" class="np-icon-elements">
            <a id="mode-switcher" class="<?php echo esc_attr( $site_mode ); ?>" href="#">
                <span class="site-mode-icon"><?php esc_html_e( 'site mode button', 'news-express' ); ?></span>
            </a>
        </div><!-- #np-site-mode-wrap -->
<?php
    }

endif;

if ( ! function_exists( 'news_express_sticky_sidebar_toggle' ) ) :

	/**
	 * display sticky sidebar toggle icon
	 * 
	 * @since 1.0.0
	 */
	function news_express_sticky_sidebar_toggle() {
		$news_express_header_sticky_sidebar_toggle_enable = news_express_get_customizer_option_value( 'news_express_header_sticky_sidebar_toggle_enable' );
		if ( false === $news_express_header_sticky_sidebar_toggle_enable ) {
			return;
		}
?>
		<div class="sidebar-menu-toggle-wrap news-express-icon-elements">
            <button class="sidebar-menu-toggle news-express-modal-toggler" data-popup-content=".sticky-header-sidebar">
                <a href="javascript:void(0)">
                    <div class="sidebar-menu-toggle-nav">
                        <span class="smtn-top"></span>
                        <span class="smtn-mid"></span>
                        <span class="smtn-bot"></span>
                    </div>
                </a>
            </button>
            <div class="sticky-header-sidebar news-express-modal-popup-content">
                <div class="sticky-header-widget-wrapper">
                    <?php
                        if ( is_active_sidebar( 'header-sticky-sidebar' ) ) {
                            dynamic_sidebar( 'header-sticky-sidebar' );
                        }
                    ?>
                </div>
                <div class="sticky-header-sidebar-overlay"> </div>
                <button class="sticky-sidebar-close news-express-madal-close" data-focus=".sidebar-menu-toggle.news-express-modal-toggler"><i class="fas fa-x"></i></button>
            </div><!-- .sticky-header-sidebar -->
        </div><!-- .sidebar-menu-toggle-wrap -->
<?php
	}

endif;

/*----------------------------- Schema Markup ---------------------------------------------*/

	if ( ! function_exists( 'news_express_get_schema_markup' ) ) :

		/**
		 * Return correct schema markup
		 *
		 * @since 1.0.0
		 */
		function news_express_get_schema_markup( $location ) {

			// Default
			$schema = $itemprop = $itemtype = '';

			// HTML
			if ( 'html' == $location ) {
				if ( is_home() || is_front_page() ) {
					$schema = 'itemscope=itemscope itemtype=https://schema.org/WebPage';
				} elseif ( is_category() || is_tag() || is_singular( 'post') ) {
					$schema = 'itemscope=itemscope itemtype=https://schema.org/Blog';
				} elseif ( is_page() ) {
					$schema = 'itemscope=itemscope itemtype=https://schema.org/WebPage';
				} else {
					$schema = 'itemscope=itemscope itemtype=https://schema.org/WebPage';
				}
			}

			// Creative work
			if ( 'creative_work' == $location ) {
				if ( is_single() ) {
					$schema = 'itemscope=itemscope itemtype=https://schema.org/creative_work';
				} elseif ( is_home() || is_archive() ) {
					$schema = 'itemscope=itemscope itemtype=https://schema.org/creative_work';
				}
			}

			// Header
			if ( 'header' == $location ) {
				$schema = 'itemscope=itemscope itemtype=https://schema.org/WPHeader';
			}

			// Logo
			if ( 'logo' == $location ) {
				$schema = 'itemscope itemtype=https://schema.org/Organization';
			}

			// Navigation
			if ( 'site_navigation' == $location ) {
				$schema = 'itemscope=itemscope itemtype=https://schema.org/SiteNavigationElement';
			}

			// Main
			if ( 'main' == $location ) {
				$itemtype = 'https://schema.org/WebPageElement';
				$itemprop = 'mainContentOfPage';
			}

			// Sidebar
			if ( 'sidebar' == $location ) {
				$schema = 'itemscope=itemscope itemtype=https://schema.org/WPSideBar';
			}

			// Footer widgets
			if ( 'footer' == $location ) {
				$schema = 'itemscope=itemscope itemtype=https://schema.org/WPFooter';
			}

			// Headings
			if ( 'headline' == $location ) {
				$schema = 'itemprop=headline';
			}

			// Posts
			if ( 'entry_content' == $location ) {
				$schema = 'itemprop=text';
			}

			// Publish date
			if ( 'publish_date' == $location ) {
				$schema = 'itemprop=datePublished';
			}

			// Modified date
			if ( 'modified_date' == $location ) {
				$schema = 'itemprop=dateModified';
			}

			// Author name
			if ( 'author_name' == $location ) {
				$schema = 'itemprop=name';
			}

			// Author link
			if ( 'author_link' == $location ) {
				$schema = 'itemprop=author itemscope=itemscope itemtype=https://schema.org/Person';
			}

			// Item
			if ( 'item' == $location ) {
				$schema = 'itemprop=item';
			}

			// Url
			if ( 'url' == $location ) {
				$schema = 'itemprop=url';
			}

			// Position
			if ( 'position' == $location ) {
				$schema = 'itemprop=position';
			}

			// Image
			if ( 'image' == $location ) {
				$schema = 'itemprop=image';
			}

	        // Avatar
	        if ( 'avatar' == $location ) {
	            $schema = 'itemprop=avatar';
	        }

	        // Description
	        if ( 'description' == $location ) {
	            $schema = 'itemprop=description';
	        }

			return ' ' . apply_filters( 'news_express_schema_markup_items', $schema );

		}

	endif;

	if ( ! function_exists( 'news_express_schema_markup' ) ) :

		/**
		 * Outputs correct schema markup
		 *
		 * @since 1.0.0
		 */
		function news_express_schema_markup( $location ) {


			$news_express_site_schema_enable = news_express_get_customizer_option_value( 'news_express_site_schema_enable' );

			if ( false === $news_express_site_schema_enable ) {
				return;
			}

			echo news_express_get_schema_markup( $location );

		}

	endif;

/*------------------------ Primary Menu Settings --------------------------*/

	/**
	 *  Menu items - Add "Custom sub-menu" in menu item render output
	 *  if menu item has class "menu-item-target"
	 */
	add_filter( 'walker_nav_menu_start_el', 'news_express_nav_description', 10, 4 );

	if ( ! function_exists( 'news_express_nav_description' ) ) :

		function news_express_nav_description( $item_output, $item, $depth, $args ) {

			$news_express_primary_menu_description_enable = news_express_get_customizer_option_value( 'news_express_primary_menu_description_enable' );

		    if ( ! empty( $item->description ) && false !== $news_express_primary_menu_description_enable ) {
		        $item_output = str_replace( $args->link_after . '</a>', '<span class="menu-item-description">' . $item->description . '</span>' . $args->link_after . '</a>', $item_output );
		    }
		    return $item_output;
		}

	endif;

if ( ! function_exists( 'news_express_minify_css' ) ) {

    /**
     * Minify CSS
     *
     * @since 1.0.1
     */
    function news_express_minify_css( $css = '' ) {

        // Return if no CSS
        if ( ! $css ) return;

        // Normalize whitespace
        $css = preg_replace( '/\s+/', ' ', $css );

        // Remove ; before }
        $css = preg_replace( '/;(?=\s*})/', '', $css );

        // Remove space after , : ; { } */ >
        $css = preg_replace( '/(,|:|;|\{|}|\*\/|>) /', '$1', $css );

        // Remove space before , ; { }
        $css = preg_replace( '/ (,|;|\{|})/', '$1', $css );

        // Strips leading 0 on decimal values (converts 0.5px into .5px)
        $css = preg_replace( '/(:| )0\.([0-9]+)(%|em|ex|px|in|cm|mm|pt|pc)/i', '${1}.${2}${3}', $css );

        // Strips units if value is 0 (converts 0px to 0)
        $css = preg_replace( '/(:| )(\.?)0(%|em|ex|px|in|cm|mm|pt|pc)/i', '${1}0', $css );

        // Trim
        $css = trim( $css );

        // Return minified CSS
        return $css;

    }

}

/*----------------------------- Google fonts ---------------------------------------------*/
	if ( ! function_exists( 'news_express_get_google_font_variants')) :
		/**
		 * get google font variants
		 */

		function news_express_get_google_font_variants() {
			$news_express_font_list = get_option(' news_express_google_font');
			$font_family = $_REQUEST['font_family'];
			$variants_array = $news_express_font_list[$font_family] ['0'];
			$options_array = '<option value="inherit">'. __('Inherit', 'news-express'). '</option>';
			foreach ( $variants_array as $variant){
				$variant_html = news_express_convert_font_variants ($variant);
				 $options_array .= '<option value="'.esc_attr( $variant ).'">'. esc_html( $variant_html ).'</option>';
            }
			echo $options_array;
			die();
		}
	endif;
	add_action("wp_ajax_get_google_font_variants", "news_express_get_google_font_variants");

	if (!function_exists('news_express_convert_font_variants')):
		 /**
         * Returns the font variant name based on its value.
         *
         * @param string $value Font variant value.
         * @return string Variant name.
         * @since 1.0.0
         */
         function news_express_convert_font_variants($value){
         	switch ($value)	{
         		case '100':
         		return __('Thin 100', 'news-express');
         		break;

         		case '200':
         		return __('Extra-Light 200', 'news-express');
         		 case '300':
                    return __( 'Light 300', 'news-express' );
                    break;

                case '400':
                    return __( 'Normal 400', 'news-express' );
                    break;

                case '500':
                    return __( 'Medium 500', 'news-express' );
                    break;

                case '600':
                    return __( 'Semi-Bold 600', 'news-express' );
                    break;

                case '700':
                    return __( 'Bold 700', 'news-express' );
                    break;

                case '800':
                    return __( 'Extra-Bold 800', 'news-express' );
                    break;

                case '900':
                    return __( 'Ultra-Bold 900', 'news-express' );
                    break;

                case 'inherit':
                    return __( 'Inherit', 'news-express' );
                    break;
                
                default:
                    break;
            }
        }

    endif;
    if (!function_exists('news_express_google_font_callback')):
        /**
         * Loads the Google Fonts API link.
         *
         * @since 1.0.0
         */
        function news_express_google_font_callback(){
        	$news_express_get_font_list = get_option('news_express_google_font');
        	if( empty( $news_express_get_font_list)){
        		$query_args = array(
        		'family' => urlencode( 'Roboto' ),
                    'subset' => urlencode( 'latin,cyrillic-ext,greek-ext,greek,vietnamese,latin-ext,cyrillic,khmer,devanagari,arabic,hebrew,telugu' )
                );

                $google_fonts_url = add_query_arg( $query_args, 'https://fonts.googleapis.com/css' );
                return $google_fonts_url;
            }
            $news_express_body_font_family = news_express_get_customizer_option_value('news_express_body_font_family');
            $news_express_body_font_weight = implode( ',', $news_express_get_font_list[$news_express_body_font_family]['0'] );
            $body_typo_combo        = $news_express_body_font_family.":".$news_express_body_font_weight;

            $news_express_heading_font_family  = news_express_get_customizer_option_value( 'news_express_heading_font_family' );
            $news_express_heading_font_weight  = implode( ',', $news_express_get_font_list[$news_express_heading_font_family]['0'] );
            $heading_typo_combo     = $news_express_heading_font_family.":".$news_express_heading_font_weight;

            $get_fonts          = array( $body_typo_combo, $heading_typo_combo );

            $final_font_string = implode( '|', $get_fonts );

            $google_fonts_url = '';

            if ( $final_font_string ) {
                $query_args = array(
                    'family' => urlencode( $final_font_string ),
                    'subset' => urlencode( 'latin,cyrillic-ext,greek-ext,greek,vietnamese,latin-ext,cyrillic,khmer,devanagari,arabic,hebrew,telugu' )
                );

                $google_fonts_url = add_query_arg( $query_args, 'https://fonts.googleapis.com/css' );
            }

            return $google_fonts_url;
        }

    endif;