<?php
/**
 * Functions which enhance the theme by hooking into WordPress
 *
 * @package News Event
 */
use NewsEvent\CustomizerDefault as NEV;
/**
 * Adds custom classes to the array of body classes.
 *
 * @param array $classes Classes for the body element.
 * @return array
 */
function news_event_body_classes( $classes ) {
	global $post;

	// Adds a class of hfeed to non-singular pages.
	if ( ! is_singular() ) {
		$classes[] = 'hfeed';
	}

	$classes[] = 'news-event-variables';
	$classes[] = esc_attr( 'news-event-title-' . NEV\news_event_get_customizer_option( 'post_title_hover_effects'  ) ); // post title hover effects
	$classes[] = esc_attr( 'news-event-image-hover--effect-' . NEV\news_event_get_customizer_option( 'site_image_hover_effects' ) ); // site image hover effects
	$classes[] = esc_attr( 'site-' . NEV\news_event_get_customizer_option( 'website_layout' ) ); // site layout
	$theme_mode = array_key_exists( 'themeMode', $_COOKIE ) ? $_COOKIE['themeMode'] : 'light';
	if( $theme_mode == 'dark' ) {
		$classes[] = esc_attr( 'news_event_dark_mode' ); // site mode
		$classes[] = 'news_event_font_typography';
	} else {
		$classes[] = 'news_event_main_body news_event_font_typography';
	}
	$header_width_layout = NEV\news_event_get_customizer_option('header_width_layout');
	$classes[] = esc_attr('header-width--' . $header_width_layout);

	$classes[] = 'block-title--layout-four';

	if( NEV\news_event_get_customizer_option( 'header_search_option' ) ) :
		$classes[] = 'search-popup--style-one';
	endif;

	if( NEV\news_event_get_customizer_option( 'header_off_canvas_option' ) ) :
		$off_canvas_position = NEV\news_event_get_customizer_option('off_canvas_position');
		$classes[] = esc_attr('off-canvas-sidebar-appear--' . $off_canvas_position);
	endif;
	
	// page layout
	if( is_page() || is_404() || is_search() ) :
		if( is_front_page() ) {
			$frontpage_sidebar_layout = NEV\news_event_get_customizer_option( 'frontpage_sidebar_layout' );
			$frontpage_sidebar_sticky_option = NEV\news_event_get_customizer_option( 'frontpage_sidebar_sticky_option' );
			if( $frontpage_sidebar_sticky_option ) $classes[] = esc_attr( 'sidebar-sticky' );
			$classes[] = esc_attr( $frontpage_sidebar_layout );
		} else {
			if( is_page() ) {
				// default value set for post sidebar layout.
				if( !metadata_exists( 'post', $post->ID, 'post_sidebar_layout' ) ) {
					// add post sidebar layout value "custmomizer-setting".
					update_post_meta( $post->ID, 'post_sidebar_layout', 'customizer-setting' );
				}
				$post_sidebar_layout = get_post_meta( $post->ID, 'post_sidebar_layout', true );
				$post_sidebar_layout = ( $post_sidebar_layout ) ? $post_sidebar_layout : 'customizer-setting';
				if( $post_sidebar_layout == 'customizer-setting' ){
					$page_sidebar_layout = NEV\news_event_get_customizer_option( 'page_sidebar_layout' );
				} else {
					$page_sidebar_layout = $post_sidebar_layout;
				}
			} else {
				$page_sidebar_layout = NEV\news_event_get_customizer_option( 'page_sidebar_layout' );
			}
			$page_sidebar_sticky_option = NEV\news_event_get_customizer_option( 'page_sidebar_sticky_option' );
			if( $page_sidebar_sticky_option ) $classes[] = esc_attr( 'sidebar-sticky' );
			$classes[] = esc_attr( $page_sidebar_layout );
		}
	endif;

	// single post layout
	if( is_single() ) :
		// default value set for post sidebar layout.
		if( !metadata_exists( 'post', $post->ID, 'post_sidebar_layout' ) ) {
			// add post sidebar layout value "custmomizer-setting".
			update_post_meta( $post->ID, 'post_sidebar_layout', 'customizer-setting' );
		}
		$post_sidebar_layout = get_post_meta( $post->ID, 'post_sidebar_layout', true );
		$post_sidebar_layout = ( $post_sidebar_layout ) ? $post_sidebar_layout : 'customizer-setting';
		if( $post_sidebar_layout == 'customizer-setting' ){
			$single_sidebar_layout = NEV\news_event_get_customizer_option( 'single_sidebar_layout' );
		} else {
			$single_sidebar_layout = $post_sidebar_layout;
		}
		$single_sidebar_sticky_option = NEV\news_event_get_customizer_option( 'single_sidebar_sticky_option' );
		if( $single_sidebar_sticky_option ) $classes[] = esc_attr( 'sidebar-sticky' );
		$classes[] = 'single-layout--three';
		$classes[] = esc_attr( $single_sidebar_layout );
	endif;

	// archive layout
	if( is_archive() || is_home() || is_search() ) :
		$archive_sidebar_layout = NEV\news_event_get_customizer_option( 'archive_sidebar_layout' );
		$archive_page_layout = NEV\news_event_get_customizer_option( 'archive_page_layout' );
		$archive_layout_meta = news_event_get_archive_layout_meta();
		$archive_sidebar_meta = news_event_get_archive_sidebar_meta();
		$is_customizer_layout = ( $archive_layout_meta == 'customizer-layout' );
		$is_customizer_settings = ( $archive_sidebar_meta === 'customizer-setting' );
		$archive_sidebar_sticky_option = NEV\news_event_get_customizer_option( 'archive_sidebar_sticky_option' );
		if( $archive_sidebar_sticky_option ) $classes[] = esc_attr( 'sidebar-sticky' );
		$classes[] = esc_attr( 'post-layout--'. ( $is_customizer_layout ? $archive_page_layout : $archive_layout_meta ) );
		$classes[] = esc_attr( $is_customizer_settings ? $archive_sidebar_layout : $archive_sidebar_meta );
	endif;

	$site_background_animation = NEV\news_event_get_customizer_option( 'site_background_animation' );
	$classes[] = 'background-animation--' . $site_background_animation;

	$website_content_layout = NEV\news_event_get_customizer_option( 'website_content_layout' );
    $classes[] = 'global-content-layout--' . $website_content_layout;
	
	$top_header_date_time_option = NEV\news_event_get_customizer_option( 'top_header_date_time_option' );
    if( ! $top_header_date_time_option ) $classes[] = 'date-time--off';

	return $classes;
}
add_filter( 'body_class', 'news_event_body_classes' );


/**
 * Add a pingback url auto-discovery header for single posts, pages, or attachments.
 */
function news_event_pingback_header() {
	if ( is_singular() && pings_open() ) {
		printf( '<link rel="pingback" href="%s">', esc_url( get_bloginfo( 'pingback_url' ) ) );
	}
}
add_action( 'wp_head', 'news_event_pingback_header' );

//define constant
define( 'NEWS_EVENT_INCLUDES_PATH', get_template_directory() . '/inc/' );

/**
 * Enqueue theme scripts and styles.
 */
function news_event_scripts() {
	global $wp_query;
	require_once get_theme_file_path( 'inc/wptt-webfont-loader.php' );
	$preloader_option = NEV\news_event_get_customizer_option('preloader_option');
	$ticker_news_visible = NEV\news_event_get_customizer_option('ticker_news_visible');
	$archive_pagination_type = NEV\news_event_get_customizer_option('archive_pagination_type');
	$enqueue_scripts_args = [ 'strategy' => 'defer', 'in_footer' => true ];
	$css_dependency = [];
	$js_dependency = [ 'jquery' ];
	$template_directory_uri = get_template_directory_uri();
	wp_enqueue_style( 'fontawesome', $template_directory_uri . '/assets/lib/fontawesome/css/all.min.css', $css_dependency, '6.5.1', 'all' );
	wp_enqueue_style( 'slick', $template_directory_uri . '/assets/lib/slick/slick.css', $css_dependency, '1.8.1', 'all' );
	wp_enqueue_style( 'magnific-popup', $template_directory_uri . '/assets/lib/magnific-popup/magnific-popup.css', $css_dependency, '1.1.0', 'all' );
	wp_enqueue_style( 'news-event-typo-fonts', wptt_get_webfont_url( news_event_typo_fonts_url() ), $css_dependency, null );
	// enqueue inline style
	wp_enqueue_style( 'news-event-style', get_stylesheet_uri(), $css_dependency, NEWS_EVENT_VERSION );
	wp_add_inline_style( 'news-event-style', news_event_current_styles() );
	wp_enqueue_style( 'news-event-main-style', $template_directory_uri.'/assets/css/main.css', $css_dependency, NEWS_EVENT_VERSION );
	// additional css
	wp_enqueue_style( 'news-event-main-style-additional', $template_directory_uri.'/assets/css/add.css', $css_dependency, NEWS_EVENT_VERSION );
	if( $preloader_option ) wp_enqueue_style( 'news-event-loader-style', $template_directory_uri.'/assets/css/loader.css', $css_dependency, NEWS_EVENT_VERSION );
	wp_enqueue_style( 'news-event-responsive-style', $template_directory_uri.'/assets/css/responsive.css', $css_dependency, NEWS_EVENT_VERSION );
	wp_style_add_data( 'news-event-style', 'rtl', 'replace' );

	wp_enqueue_script( 'slick', $template_directory_uri . '/assets/lib/slick/slick.min.js', $js_dependency, '1.8.1', $enqueue_scripts_args );
	wp_enqueue_script( 'magnific-popup', $template_directory_uri . '/assets/lib/magnific-popup/magnific-popup.min.js', $js_dependency, '1.1.0', $enqueue_scripts_args );
	if( $ticker_news_visible !== 'none' ) wp_enqueue_script( 'js-marquee', $template_directory_uri . '/assets/lib/js-marquee/jquery.marquee.min.js', $js_dependency, '1.6.0', $enqueue_scripts_args );
	wp_enqueue_script( 'news-event-navigation', $template_directory_uri . '/assets/js/navigation.js', array(), NEWS_EVENT_VERSION, $enqueue_scripts_args );
	wp_enqueue_script( 'news-event-theme', $template_directory_uri . '/assets/js/theme.js', $js_dependency, NEWS_EVENT_VERSION, $enqueue_scripts_args );
	
	$scriptVars['_wpnonce'] = wp_create_nonce( 'news-event-nonce' );
	$scriptVars['ajaxUrl'] 	= esc_url(admin_url('admin-ajax.php'));
	$scriptVars['stt']	= NEV\news_event_get_customizer_option('stt_responsive_option');
	$scriptVars['is_customizer'] = is_customize_preview();
	$scriptVars['isSingle'] = is_single();
	$scriptVars['isPage'] = is_page();	

	// localize scripts
	wp_localize_script( 'news-event-theme', 'newsEventObject' , $scriptVars);

	if ( is_singular() && comments_open() && get_option( 'thread_comments' ) ) {
		wp_enqueue_script( 'comment-reply' );
	}
}
add_action( 'wp_enqueue_scripts', 'news_event_scripts' );

/**
 * Minify dynamic css
 * 
 * @since 1.0.0
 */
if( ! function_exists( 'news_event_minifyCSS' ) ) :
	function news_event_minifyCSS( $css ) {
		// Remove comments
		$css = preg_replace( '!/\*.*?\*/!s', '', $css );
		// Remove space after colons
		$css = preg_replace( '/\s*:\s*/', ':', $css );
		// Remove whitespace
		$css = preg_replace( '/\s+/', ' ', $css );
		// Remove space before/after brackets and semicolons
		$css = preg_replace( '/\s*{\s*/', '{', $css );
		$css = preg_replace( '/\s*}\s*/', '}', $css );
		$css = preg_replace( '/\s*;\s*/', ';', $css );
		// Remove final semicolon in a block
		$css = preg_replace( '/;}/', '}', $css );
		// Trim the final output
		return trim( $css );
	}
endif;

if( ! function_exists( 'news_event_current_styles' ) ) :
	/**
	 * Generates the current changes in styling of the theme.
	 * 
	 * @package News Event
	 * @since 1.0.0
	 */
	function news_event_current_styles() {
		$archive_post_element_order = json_decode( NEV\news_event_get_customizer_option('archive_post_element_order'), true );
		$site_breadcrumb_option = NEV\news_event_get_customizer_option('site_breadcrumb_option');
		$top_header_option = NEV\news_event_get_customizer_option('top_header_option');
		$header_off_canvas_option = NEV\news_event_get_customizer_option('header_off_canvas_option');
		$header_search_option = NEV\news_event_get_customizer_option('header_search_option');
		$header_random_news_option = NEV\news_event_get_customizer_option('header_random_news_option');
		$header_newsletter_option = NEV\news_event_get_customizer_option('header_newsletter_option');
		$main_banner_option = NEV\news_event_get_customizer_option('main_banner_option');
		$homepage_content_order = json_decode( NEV\news_event_get_customizer_option('homepage_content_order'), true );
		$footer_option = NEV\news_event_get_customizer_option('footer_option');
		$bottom_footer_option = NEV\news_event_get_customizer_option('bottom_footer_option');

		// enqueue inline style
		ob_start();
			news_event_preset_color_styles( 'solid_presets', '--news-event-global-preset-color-' );
			news_event_preset_color_styles( 'gradient_presets', '--news-event-global-preset-gradient-color-' );

			news_event_responsive_range_css( 'body.news-event-variables', 'header_vertical_padding', '--header-padding');

			$nBackgroundCode = function( $identifier, $id, $property = 'background' ) {
				news_event_color_css( $id, $identifier, $property );
			};

			$nBackgroundCode('.news_event_main_body .site-header.layout--default .menu-section, .news_event_main_body .site-header.layout--three .header-smh-button-wrap, body.search-popup--style-three .site-header:not(.layout--three) .search-form-wrap','header_menu_background_color_group');
			$nBackgroundCode('.news_event_main_body .main-navigation ul.menu ul, .news_event_main_body .main-navigation ul.nav-menu ul','header_sub_menu_background_color_group');

			$nTypoCode = function($identifier,$id) {
				news_event_typography_css($identifier,$id);
			};
			$nTypoCode( "--site-title", 'site_title_typo' );
			$nTypoCode( "--site-tagline", 'site_tagline_typo' );
			$nTypoCode( "--block-title", 'site_section_block_title_typo');
			$nTypoCode("--menu", 'header_menu_typo');
			$nTypoCode("--submenu", 'header_sub_menu_typo');
			$nTypoCode("--post-title",'site_archive_post_title_typo');
			$nTypoCode("--meta", 'site_archive_post_meta_typo');
			$nTypoCode("--content", 'site_archive_post_content_typo');
			if( get_theme_mod( 'custom_logo' ) ) news_event_responsive_range_css("body .site-branding img", 'news_event_site_logo_width');
			$nColorGroupCode = function( $selector, $id, $property = 'color' ) {
				news_event_color_css( $id, $selector, $property );
			};
			$nColorCode = function( $property, $id, $selector = 'body.news-event-variables' ) {
				news_event_color_css( $id, $selector, $property );
			};
			$nColorCode('--site-bk-color', 'site_background_color');
			news_event_visibility_options('.ads-banner','header_ads_banner_responsive_option');
			news_event_visibility_options('body #news-event-scroll-to-top.show','stt_responsive_option');
			news_event_border_option('.widget .post_thumb_image, .widget .widget-tabs-content .post-thumb, .widget .popular-posts-wrap article .post-thumb, .widget.widget_news_event_news_filter_tabbed_widget .tabs-content-wrap .post-thumb, .widget .news-event-widget-carousel-posts .post-thumb-wrap, .author-wrap.layout-two .post-thumb, .widget_news_event_category_collection_widget .categories-wrap .category-item, .news-event-advertisement-block img','widgets_styles_image_border', 'border');
			$nBackgroundCode('body.news_event_main_body .site-header.layout--default .site-branding-section, .search-popup--style-three .site-header.layout--three .search-form-wrap', 'header_background_color_group');
			news_event_assign_variable('--news-event-global-preset-theme-color','theme_color');
			if( NEV\news_event_get_customizer_option( 'site_background_animation' ) !== 'none' ) $nColorCode('--news-event-animation-object-color', 'animation_object_color');
			news_event_box_shadow_styles('.widget .post_thumb_image, .widget .widget-tabs-content .post-thumb, .widget .popular-posts-wrap article .post-thumb, .widget.widget_news_event_news_filter_tabbed_widget .tabs-content-wrap .post-thumb, .widget .news-event-widget-carousel-posts.layout--two .slick-list, .author-wrap.layout-two .post-thumb, .widget_news_event_category_collection_widget .categories-wrap .category-item','widgets_styles_image_box_shadow');
			news_event_value_change_responsive('.widget .post_thumb_image, .widget .widget-tabs-content .post-thumb, .widget .popular-posts-wrap article .post-thumb, .widget.widget_news_event_news_filter_tabbed_widget .tabs-content-wrap .post-thumb, .widget .news-event-widget-carousel-posts .post-thumb-wrap, .author-wrap.layout-two .post-thumb, .widget .news-event-widget-carousel-posts.layout--two .slick-list, .widget_news_event_category_collection_widget .categories-wrap .category-item, .news-event-advertisement-block img','widgets_styles_image_border_radius', 'border-radius');
			if( $main_banner_option ) news_event_value_change_responsive('body #main-banner-section article','main_banner_image_border_radius', 'border-radius');
			news_event_category_colors_styles();
			echo 'body .social-icons .social-icon {' . NEV\news_event_get_customizer_option( 'social_icons_border_radius' ) . "px;\n }";

			// Global button
			if( $archive_post_element_order[ 'button' ] ) $nTypoCode("--post-link-btn", 'global_button_typo');

			// Top Header
			if( $top_header_option ) :
				$nBackgroundCode('.news_event_main_body .site-header.layout--default .top-header','top_header_background_color_group');
			endif;

			// Newsletter / subscribe button
			if( $header_newsletter_option ) $nColorCode('--newsletter-bk-color', 'header_newsletter_background_color');

			// Left content - right sidebar
			if( $homepage_content_order[ 'leftc_rights_section' ] ) :
				$nBackgroundCode('.news_event_main_body #leftc-rights-section, body #leftc-rights-section .layout--four .news-list-post-wrap .list-item .post-title, #leftc-rights-section .news-grid.layout--four .post-element, #leftc-rights-section .widget .posts-grid-wrap .post-content-wrap, #leftc-rights-section .news-carousel.layout--four .post-element', 'leftc_rights_blocks_background_color_group');
			endif;

			// Left sidebar - right content
			if( $homepage_content_order[ 'lefts_rightc_section' ] ) :
				$nBackgroundCode('.news_event_main_body #lefts-rightc-section, body #lefts-rightc-section .layout--four .news-list-post-wrap .list-item .post-title, #lefts-rightc-section .news-grid.layout--four .post-element, #lefts-rightc-section .widget .posts-grid-wrap.layout-two .post-content-wrap, #lefts-rightc-section .news-carousel.layout--four .post-element', 'lefts_rightc_blocks_background_color_group');
			endif;

			// Latest Posts
			if( $homepage_content_order[ 'latest_posts' ] ) :
			endif;

			// Bottom full width
			if( $homepage_content_order[ 'bottom_full_width_section' ] ) :
				$nBackgroundCode( ".news_event_main_body #bottom-full-width-section, body #bottom-full-width-section .layout--four .news-list-post-wrap .list-item .post-title, #bottom-full-width-section .news-grid.layout--four .post-element, #bottom-full-width-section .news-carousel.layout--four .post-element", "bottom_full_width_blocks_background_color_group" );
			endif;

			// Two column section
			if( $homepage_content_order[ 'two_column_section' ] ) :
				$nBackgroundCode( ".news_event_main_body .two-column-section, body .two-column-section .layout--four .news-list-post-wrap .list-item .post-title, .two-column-section .news-grid.layout--four .post-element, .two-column-section .news-carousel.layout--four .post-element", "two_column_background_color_group" );
			endif;

			// Archive
			if( is_home() || is_front_page() || is_archive() || is_search() ) :
				news_event_image_ratio_variable('--news-event-archive-image-ratio','archive_image_ratio');
				news_event_responsive_range_css( 'body.news-event-variables', 'archive_vertical_spacing_top', '--archive-padding-top');
				news_event_responsive_range_css( 'body.news-event-variables', 'archive_vertical_spacing_bottom', '--archive-padding-bottom');
			endif;

			// Single
			if( is_single() ) :
				$nTypoCode("--single-title",'single_post_title_typo');
				$nTypoCode("--single-meta", 'single_post_meta_typo');
				$nTypoCode("--single-content", 'single_post_content_typo');
				$nTypoCode("--single-content-h1", 'single_post_content_h1_typo');
				$nTypoCode("--single-content-h2", 'single_post_content_h2_typo');
				$nTypoCode("--single-content-h3", 'single_post_content_h3_typo');
				$nTypoCode("--single-content-h4", 'single_post_content_h4_typo');
				$nTypoCode("--single-content-h5", 'single_post_content_h5_typo');
				$nTypoCode("--single-content-h6", 'single_post_content_h6_typo');
				news_event_image_ratio_variable('--news-event-single-image-ratio','single_post_image_ratio');
			endif;

			// Page
			if( is_page() ) news_event_image_ratio_variable('--news-event-page-image-ratio','single_page_image_ratio');

			// Theme footer
			news_event_border_option('body .site-footer.dark_bk','footer_top_border', 'border-top');
			if( $footer_option ) $nBackgroundCode('body.news_event_main_body .site-footer .main-footer, body .dark_bk .posts-grid-wrap.layout-two .post-content-wrap', 'footer_background_color_group');

			// Bottom footer
			if( $bottom_footer_option ) $nBackgroundCode('body.news_event_main_body .site-footer .bottom-footer', 'bottom_footer_background_color_group');
			// front sections image settings styles
		$current_styles = ob_get_clean();
		return apply_filters( 'news_event_current_styles', wp_strip_all_tags( news_event_minifyCSS( $current_styles ) ) );
	}
endif;

if( ! function_exists( 'news_event_customizer_social_icons' ) ) :
	/**
	 * Functions get social icons
	 * 
	 */
	function news_event_customizer_social_icons() {
		$social_icons = NEV\news_event_get_customizer_option( 'social_icons' );
		$social_icons_target = NEV\news_event_get_customizer_option( 'social_icons_target' );
		$social_icons_enable_title = NEV\news_event_get_customizer_option( 'social_icons_enable_title' );
		$social_icons_columns = NEV\news_event_get_customizer_option( 'social_icons_columns' );
		$decoded_social_icons = json_decode( $social_icons );
		$socialIconClasses[] = 'social-icons';
		$socialIconClasses[] = 'column--' . news_event_convert_number_to_numeric_string( $social_icons_columns );
		if( $social_icons_enable_title ) $socialIconClasses[] = 'is-title';
		$socialIconClasses[] = 'is-background';

		echo '<div class="', implode( ' ', $socialIconClasses ), '">';
			foreach( $decoded_social_icons as $icon ) :
				if( $icon->item_option === 'show' ) :
					$icon_args = news_event_get_all_social_share( $icon->icon_class );
					$label = $icon_args[ 'label' ];
					$iconClass = 'social-icon ' . str_replace( ' ', '-', strtolower( $label ) );
					?>
						<a class="<?php echo esc_attr( $iconClass ); ?>" href="<?php echo esc_url( $icon->icon_url ); ?>" target="<?php echo esc_attr( $social_icons_target ); ?>">
							<i class="<?php echo esc_attr( $icon->icon_class ); ?>"></i>
							<?php if( $social_icons_enable_title ) echo '<span class="icon-label">', esc_html( $label ), '</span>'; ?>
						</a>
					<?php
				endif;
			endforeach;
		echo '</div>';
	}
endif;

if( ! function_exists( 'news_event_get_date_filter_choices_array' ) ) :
	/**
	 * Return array of date filter choices.
	 * 
	 */
	function news_event_get_date_filter_choices_array() {
		return apply_filters( 'news_event_get_date_filter_choices_array_filter', [
			'all'	=> esc_html__('All', 'news-event' ),
			'last-seven-days'	=> esc_html__('Last 7 days', 'news-event' ),
			'today'	=> esc_html__('Today', 'news-event' ),
			'this-week'	=> esc_html__('This Week', 'news-event' ),
			'last-week'	=> esc_html__('Last Week', 'news-event' ),
			'this-month'	=> esc_html__('This Month', 'news-event' ),
			'last-month'	=> esc_html__('Last Month', 'news-event' ),
			'this-year'	=> esc_html__('This Year', 'news-event' )
		]);
	}
endif;

if( ! function_exists( 'news_event_get_array_key_string_to_int' ) ) :
	/**
	 * Return array of int values.
	 * 
	 */
	function news_event_get_array_key_string_to_int( $array ) {
		if( ! isset( $array ) || empty( $array ) ) return;
		$filtered_array = array_map( function($arr) {
			if( is_numeric( $arr ) ) return (int) $arr;
		}, $array);
		return apply_filters( 'news_event_array_with_int_values', $filtered_array );
	}
endif;

/**
 * Return string with "implode" execution in param
 * 
 */
 function news_event_get_categories_for_args($array) {
	if( ! $array ) return apply_filters( 'news_event_categories_for_argument', '' );
	foreach( $array as $value ) {
		if( is_array( $value ) ) $string_array[] = $value['value'];
		if( is_object( $value ) ) $string_array[] = $value->value;
	}
	return apply_filters( 'news_event_categories_for_argument', implode( ',', $string_array ) );
}

/**
 * Return array with execution in param
 * 
 */
function news_event_get_post_id_for_args($array) {
	if( ! $array ) return apply_filters( 'news_event_posts_slugs_for_argument', '' );
	foreach( $array as $value ) {
		if( is_array( $value ) ) $string_array[] = $value['value'];
		if( is_object( $value ) ) $string_array[] = $value->value;
	}
	return apply_filters( 'news_event_posts_slugs_for_argument', $string_array );
}

if( !function_exists( 'news_event_typo_fonts_url' ) ) :
	/**
	 * Filter and Enqueue typography fonts
	 * 
	 * @package News Event
	 * @since 1.0.0
	 */
	function news_event_typo_fonts_url() {
		$filter = NEWS_EVENT_PREFIX . 'typo_combine_filter';
		$fonts[] = apply_filters( $filter, 'site_title_typo' );
		$fonts[] = apply_filters( $filter, 'site_tagline_typo' );
		$fonts[] = apply_filters( $filter, 'header_menu_typo' );
		$fonts[] = apply_filters( $filter, 'header_sub_menu_typo' );
		$fonts[] = apply_filters( $filter, 'site_section_block_title_typo' );
		$fonts[] = apply_filters( $filter, 'site_archive_post_title_typo' );
		$fonts[] = apply_filters( $filter, 'site_archive_post_meta_typo' );
		$fonts[] = apply_filters( $filter, 'site_archive_post_content_typo' );
		$fonts[] = apply_filters( $filter, 'single_post_title_typo' );
		$fonts[] = apply_filters( $filter, 'single_post_meta_typo' );
		$fonts[] = apply_filters( $filter, 'single_post_content_typo' );
		$fonts[] = apply_filters( $filter, 'single_post_content_h1_typo' );
		$fonts[] = apply_filters( $filter, 'single_post_content_h2_typo' );
		$fonts[] = apply_filters( $filter, 'single_post_content_h3_typo' );
		$fonts[] = apply_filters( $filter, 'single_post_content_h4_typo' );
		$fonts[] = apply_filters( $filter, 'single_post_content_h5_typo' );
		$fonts[] = apply_filters( $filter, 'single_post_content_h6_typo' );
		$fonts[] = apply_filters( $filter, 'global_button_typo' );
		$fonts[] = apply_filters( $filter, 'default_typo_one' );
		$fonts[] = apply_filters( $filter, 'default_typo_two' );

		$get_filtered_fonts = apply_filters( 'news_event_get_fonts_toparse', $fonts );
		$_new_fonts_array = [];
		foreach( $get_filtered_fonts as $fonts ) {
			if( ! isset( $_new_fonts_array[$fonts['font_family']] ) ) {
				$_new_fonts_array[$fonts['font_family']] = [
					$fonts['variant']	=> [$fonts['font_weight']]
				];
			} else {
				if( ! isset( $_new_fonts_array[$fonts['font_family']][$fonts['variant']] ) ) {
					$_new_fonts_array[$fonts['font_family']][$fonts['variant']] = [$fonts['font_weight']];
				} else {
					if( ! in_array( $fonts['font_weight'], $_new_fonts_array[$fonts['font_family']][$fonts['variant']] ) ) $_new_fonts_array[$fonts['font_family']][$fonts['variant']][] = $fonts['font_weight'];
				}
			}
		}
		$_new_fonts_string = '';
		$_new_fonts_strings = [];
		foreach( $_new_fonts_array as $_new_font_key => $_new_font_value ) {
			$prefix_weight = false;
			$normal_weight = false;
			$_new_fonts_string = $_new_font_key . ':';
			if( isset( $_new_font_value['italic'] ) ) {
				$prefix_weight = true;
				$_new_fonts_string .= 'ital,';
			}
			$_new_fonts_string .= 'wght@';
			if( isset( $_new_font_value['normal'] ) && is_array( $_new_font_value['normal'] ) ) {
				$sorted_new_font_value = $_new_font_value['normal'];
				sort( $sorted_new_font_value, SORT_NUMERIC );
				foreach( $sorted_new_font_value as $font_weight_key => $font_weight_value ) {
					if( $font_weight_key > 0 ) $_new_fonts_string .= ';';
					if( $prefix_weight ) $_new_fonts_string .= '0,'. $font_weight_value;
					if( ! $prefix_weight ) $_new_fonts_string .= $font_weight_value;
				}
				$normal_weight = true;
			}

			if( isset( $_new_font_value['italic'] ) && is_array( $_new_font_value['italic'] ) ) {
				$sorted_new_font_value = $_new_font_value['italic'];
				sort( $sorted_new_font_value, SORT_NUMERIC );
				foreach( $sorted_new_font_value as $font_weight_key => $font_weight_value ) {
					if( $normal_weight ) $_new_fonts_string .= ';';
					if( ! $normal_weight && $font_weight_key > 0 ) $_new_fonts_string .= ';';
					if( $prefix_weight ) $_new_fonts_string .= '1,'. $font_weight_value;
					if( ! $prefix_weight ) $_new_fonts_string .= $font_weight_value;
				}
			}
			$_new_fonts_strings[] = urlencode($_new_fonts_string);
		}
		$google_fonts_url = add_query_arg( [
			'family'	=> implode( '&family=', $_new_fonts_strings ),
			'display'	=> 'swap'
		], 'https://fonts.googleapis.com/css2' );
		return $google_fonts_url;
	}
endif;

if(! function_exists('news_event_get_color_format')):
    function news_event_get_color_format( $color ) {
		if( is_array( $color ) ) return $color;
		if( str_contains( $color, '--news-event-global-preset' ) ) {
			return( 'var( ' .esc_html( $color ). ' )' );
		} else {
			return $color;
		}
    }
endif;

if( ! function_exists( 'news_event_get_rcolor_code' ) ) :
	/**
	 * Returns randon color code
	 * 
	 * @package News Event
	 * @since 1.0.0
	 */
	function news_event_get_rcolor_code() {
		$color_array["color"] = "#333333";
		$color_array["hover"] = "#448bef";
		return apply_filters( 'news_event_apply_random_color_shuffle_value', $color_array );
	}
endif;

require get_template_directory() . '/inc/theme-starter.php'; // theme starter functions.
require get_template_directory() . '/inc/customizer/customizer.php'; // Customizer additions.
require get_template_directory() . '/inc/extras/helpers.php'; // helpers files.
require get_template_directory() . '/inc/extras/extras.php'; // extras files.
require get_template_directory() . '/inc/extras/extend-api.php'; // extras files.
require get_template_directory() . '/inc/widgets/widgets.php'; // widget handlers
require get_template_directory() . '/inc/metaboxes/metabox.php'; // metabox handlers
include get_template_directory() . '/inc/styles.php';
include get_template_directory() . '/inc/admin/admin.php';
new News_Event_Admin\Admin_Page();
new News_Event_Extend_Api();

/**
 * Filter posts layout 6 ajax function
 *
 * @package News Event Pro
 * @since 1.0.0
 */
function news_event_filter_layout_five_posts_load_tab_content() {
	check_ajax_referer( 'news-event-nonce', 'security' );
	$options = isset( $_GET['options'] ) ? json_decode( stripslashes( $_GET['options'] ) ): '';
	if( empty( $options ) ) return;
	$query = json_decode( $options->query );
	$layout = $options->layout;
	$orderArray = explode( '-', $query->order );
	$posts_args = array(
		'posts_per_page'   => absint( $query->count ),
		'order' => esc_html( $orderArray[1] ),
		'orderby' => esc_html( $orderArray[0] ),
		'cat' => esc_html( $options->category_id ),
		'ignore_sticky_posts'    => true
	);
	if( $query->ids ) $posts_args['post__not_in'] = news_event_get_array_key_string_to_int( $query->ids );
	$post_query = new \WP_Query( $posts_args );
	$total_posts = $post_query->post_count;
	if( $post_query->have_posts() ):
		ob_start();
		?>
			<div class="tab-content content-<?php echo esc_attr( $options->category_id )?>">
                <?php
                    unset( $posts_args['category_name'] );
					$delay = 0;
					$row_count = 0;
					while( $post_query->have_posts() ) : $post_query->the_post();
						$res['loaded'] = true;
						if( $layout === 'five' ) {
							$current_post = $post_query->current_post;
							$options->featuredPosts = false;
							$last_item = ( $total_posts === ( $current_post + 1 ) );
							if( ( $current_post % 5 ) === 0 && $row_count < 2  ) {
								echo '<div class="row-wrap">';
								$row_count++;
							}
								if( $current_post === 0 ) {
									echo '<div class="featured-post">';
									$options->featuredPosts = true;
								}
									if( $current_post === 1 || $current_post === 5 ) {
										?>
										<div class="trailing-post <?php if($current_post === 5) echo esc_attr('bottom-trailing-post'); ?>">
										<?php
									}
										if( $current_post === 1 ) echo '<div class="grid-posts">';
										if( $current_post === 3 ) echo '<div class="list-posts">';
						}
									$options->delay = $delay;
									// get template file w.r.t par
										get_template_part( 'template-parts/news-filter/content', 'one', $options );
						if( $layout === 'five' ) {
												if( $current_post === 2 || ( $last_item && $total_posts <= 3 && $total_posts >= 2 ) ) echo '</div><!-- .grid-posts -->';
										if( $current_post === 4 || ( $last_item && $total_posts <= 5 && $total_posts >= 4 ) ) echo '</div><!-- .list-posts -->';
									if( $current_post === 4 || $last_item ) echo '</div><!-- .trailing-post -->';
								if( $current_post === 0 ) echo '</div><!-- .featured-post-->';
								if( ! in_array( $current_post, [ 0, 4 ] ) && $last_item ) echo '</div><!-- .total-posts-close -->';
							if( $current_post === 4 && $row_count <= 2 ) echo '</div><!-- .row-wrap -->';
						}
						$delay += 50;
					endwhile;
                ?>
            </div>
		<?php
		$res['posts'] = ob_get_clean();
	else :
		$res['loaded'] = false;
		$res['posts'] = esc_html__( 'No posts found', 'news-event' );
	endif;
	wp_send_json_success( $res );
	wp_die();
}
add_action( 'wp_ajax_news_event_filter_layout_five_posts_load_tab_content', 'news_event_filter_layout_five_posts_load_tab_content');
add_action( 'wp_ajax_nopriv_news_event_filter_layout_five_posts_load_tab_content', 'news_event_filter_layout_five_posts_load_tab_content' );

if( ! function_exists( 'news_event_lazy_load_value' ) ) :
	/**
	 * Echos lazy load attribute value.
	 * 
	 * @package News Event
	 * @since 1.0.0
	 */
	function news_event_lazy_load_value() {
		echo esc_attr( apply_filters( 'news_event_lazy_load_value', 'lazy' ) );
	}
endif;

if( ! function_exists( 'news_event_add_menu_description' ) ) :
	// merge menu description element to the menu 
	function news_event_add_menu_description( $item_output, $item, $depth, $args ) {
		if($args->theme_location != 'menu-2') return $item_output;
		
		if ( !empty( $item->description ) ) {
			$item_output = str_replace( $args->link_after . '</a>', '<span class="menu-item-description">' . $item->description . '</span>' . $args->link_after . '</a>', $item_output );
		}
		return $item_output;
	}
	add_filter( 'walker_nav_menu_start_el', 'news_event_add_menu_description', 10, 4 );
endif;

if( ! function_exists( 'news_event_bool_to_string' ) ) :
	// boolean value to string 
	function news_event_bool_to_string( $bool ) {
		$string = ( $bool ) ? '1' : '0';
		return $string;
	}
endif;

 if( ! function_exists( 'news_event_get_image_sizes_option_array' ) ) :
	/**
	 * Get list of image sizes
	 * 
	 * @since 1.0.0
	 * @package News Event
	 */
	function news_event_get_image_sizes_option_array() {
		$image_sizes = get_intermediate_image_sizes();
		foreach( $image_sizes as $image_size ) :
			$sizes[] = [
				'label'	=> esc_html( $image_size ),
				'value'	=> esc_html( $image_size )
			];
		endforeach;
		return $sizes;
	}
endif;

if( ! function_exists( 'news_event_get_style_tag' ) ) :
	/**
	 * Generate Style tag for image ratio and image radius for news grid, list, carousel
	 * 
	 * @since 1.0.0
	 * @package News Event
	 */
	function news_event_get_style_tag( $variables, $selectors = '' ) {
		echo '<style id="'. esc_attr( $variables['unique_id'] ) .'-style">';
			if( isset( $variables['image_ratio'] ) ) {
				$image_ratio = json_decode( $variables['image_ratio'] );

				if( $image_ratio->desktop > 0 ) echo "#" . $variables['unique_id']. " article figure.post-thumb-wrap { padding-bottom: calc( " . $image_ratio->desktop . " * 100% ) }\n";

				if( $image_ratio->tablet > 0 ) echo " @media (max-width: 769px){ #" . $variables['unique_id']. " article figure.post-thumb-wrap { padding-bottom: calc( " . $image_ratio->tablet . " * 100% ) } }\n";

				if( $image_ratio->smartphone > 0 ) echo " @media (max-width: 548px){ #" . $variables['unique_id']. " article figure.post-thumb-wrap { padding-bottom: calc( " . $image_ratio->smartphone . " * 100% ) }}\n";

			}
		echo "</style>";
	}
endif;

if( ! function_exists( 'news_event_get_style_tag_fb' ) ) :
	/**
	 * Generates style tag for image ratio and image radius for news filter and new block
	 * 
	 * @since 1.0.0
	 * @package News Event
	 */
	function news_event_get_style_tag_fb( $variables, $selectors = '' ) {
		echo '<style id="'. esc_attr( $variables['unique_id'] ) .'-style">';
			if( isset( $variables['image_ratio'] ) ) {
				$image_ratio = json_decode( $variables['image_ratio'] );
				// for featured post
				if( $variables['layout'] == 'three' ) {

					if( $image_ratio->desktop > 0 ) echo "#" . $variables['unique_id']. ".news-event-block.layout--three .featured-post figure { padding-bottom: calc( " . ( $image_ratio->desktop * 0.4 ) . " * 100% ) }\n";

					if( $image_ratio->tablet > 0 ) echo "@media (max-width: 769px) {#" . $variables['unique_id']. ".news-event-block.layout--three .featured-post figure { padding-bottom: calc( " . ( $image_ratio->tablet * 0.4 ) . " * 100% ) } }\n";

					if( $image_ratio->smartphone > 0 ) echo "@media (max-width: 769px) {#" . $variables['unique_id']. ".news-event-block.layout--three .featured-post figure { padding-bottom: calc( " . ( $image_ratio->smartphone * 0.4 ) . " * 100% ) } }\n";	

				} else {

					if( $image_ratio->desktop > 0 ) echo "#" . $variables['unique_id']. ".news-event-block .featured-post figure { padding-bottom: calc( " . $image_ratio->desktop . " * 100% ) }\n";

					if( $image_ratio->tablet > 0 ) echo "@media (max-width: 769px) {#" . $variables['unique_id']. ".news-event-block .featured-post figure { padding-bottom: calc( " . $image_ratio->tablet . " * 100% ) } }\n";

					if( $image_ratio->smartphone > 0 ) echo "@media (max-width: 769px) {#" . $variables['unique_id']. ".news-event-block .featured-post figure { padding-bottom: calc( " . $image_ratio->smartphone . " * 100% ) } }\n";

				}

				// for trailing post
				if( $variables['layout'] == 'two' ) {
					
					if( $image_ratio->desktop > 0 ) echo "#" . $variables['unique_id']. ".news-event-block.layout--two .trailing-post figure { padding-bottom: calc( " . ( $image_ratio->desktop * 0.78 ) . " * 100% ) }\n";

					if( $image_ratio->tablet > 0 ) echo "@media (max-width: 769px) {#" . $variables['unique_id']. ".news-event-block .trailing-post figure { padding-bottom: calc( " . ( $image_ratio->tablet * 0.78 ) . " * 100% ) } }\n";

					if( $image_ratio->smartphone > 0 ) echo "@media (max-width: 769px) {#" . $variables['unique_id']. ".news-event-block.layout--two .trailing-post figure { padding-bottom: calc( " . ( $image_ratio->smartphone * 0.78 ) . " * 100% ) } }\n";

				} else {

					if( $image_ratio->desktop > 0 ) echo "#" . $variables['unique_id']. ".news-event-block .trailing-post figure { padding-bottom: calc( " . ( $image_ratio->desktop * 0.3 ) . " * 100% ) }\n";

					if( $image_ratio->tablet > 0 ) echo "@media (max-width: 769px) {#" . $variables['unique_id']. ".news-event-block .trailing-post figure { padding-bottom: calc( " . ( $image_ratio->tablet * 0.3 ) . " * 100% ) } }\n";

					if( $image_ratio->smartphone > 0 ) echo "@media (max-width: 769px) {#" . $variables['unique_id']. ".news-event-block .trailing-post figure { padding-bottom: calc( " . ( $image_ratio->smartphone * 0.3 ) . " * 100% ) } }\n";

				}

			}
		echo "</style>";
	}
endif;

if( ! function_exists( 'news_event_get_image_sizes' ) ) :
	/**
	 * get a list of all images sizes
	 * 
	 * @since 1.0.0
	 * @package News Event
	 */
	function news_event_get_image_sizes() {
		$image_sizes_args = get_intermediate_image_sizes();
		$image_size_elements_args = [];
		if( ! empty( $image_sizes_args ) ) :
			foreach( $image_sizes_args as $image_size ) :
				$image_size_elements_args[$image_size] = esc_html( $image_size );
			endforeach;
		endif;
		return $image_size_elements_args;
	}
endif;

if( ! function_exists( 'news_event_parse_icon_picker_value' ) ) :
	/**
	 * Function to return image url for icon picker
	 */
	function news_event_parse_icon_picker_value ( $control ) {
		if( $control['type'] == 'svg' ) :
			$control['url'] = wp_get_attachment_image_url( $control['value'], 'full' );
		endif;
		return $control;
	}
endif;

if( ! function_exists( 'news_event_get_all_social_share' ) ) :
	/**
	 * All social share icons and urls
	 * 
	 * @since 1.0.0
	 */
	function news_event_get_all_social_share( $icon = '' ) {
		$postUrl = 'http' . ( isset( $_SERVER['HTTPS'] ) ? 's' : '' ) . '://' . "{$_SERVER['HTTP_HOST']}{$_SERVER['REQUEST_URI']}";
        $allIcons = [
			'fa-brands fa-facebook'	=>	[
				'label'	=>	esc_html__( 'Facebook', 'news-event' ),
				'value'	=>	'https://www.facebook.com/sharer/sharer.php?u=',
			],
			'fa-brands fa-facebook-f'	=>	[
				'label'	=>	esc_html__( 'Facebook', 'news-event' ),
				'value'	=>	'https://www.facebook.com/sharer/sharer.php?u=',
			],
			'fa-brands fa-square-facebook'	=>	[
				'label'	=>	esc_html__( 'Facebook', 'news-event' ),
				'value'	=>	'https://www.facebook.com/sharer/sharer.php?u=',
			],
			'fa-brands fa-square-x-twitter'	=>	[
				'label'	=>	esc_html__( 'Twitter', 'news-event' ),
				'value'	=>	'https://twitter.com/intent/tweet?url=',
			],
			'fa-brands fa-x-twitter'	=>	[
				'label'	=>	esc_html__( 'Twitter', 'news-event' ),
				'value'	=>	'https://twitter.com/intent/tweet?url=',
			],
			'fa-brands fa-twitter'	=>	[
				'label'	=>	esc_html__( 'Twitter', 'news-event' ),
				'value'	=>	'https://twitter.com/intent/tweet?url=',
			],
			'fa-brands fa-linkedin'	=>	[
				'label'	=>	esc_html__( 'Linkedin', 'news-event' ),
				'value'	=>	'https://www.linkedin.com/sharing/share-offsite/?url=',
			],
			'fa-brands fa-linkedin-in'	=>	[
				'label'	=>	esc_html__( 'Linkedin', 'news-event' ),
				'value'	=>	'https://www.linkedin.com/sharing/share-offsite/?url=',
			],
			'fa-solid fa-envelope'	=>	[
				'label'	=>	esc_html__( 'Envelope', 'news-event' ),
				'value'	=>	'mailto:?subject={title}&body=',
			],
			'fa-regular fa-envelope'	=>	[
				'label'	=>	esc_html__( 'Envelope', 'news-event' ),
				'value'	=>	'https://mail.google.com/mail/?view=cm&to={email_address}&su={title}&body=',
			],
			'fa-brands fa-instagram'	=>	[
				'label'	=>	esc_html__( 'Instagram', 'news-event' ),
				'value'	=>	'http://www.instagram.com',
			],
			'fa-brands fa-square-instagram'	=>	[
				'label'	=>	esc_html__( 'Instagram', 'news-event' ),
				'value'	=>	'http://www.instagram.com',
			],
			'fa-brands fa-whatsapp'	=>	[
				'label'	=>	esc_html__( 'Whatsapp', 'news-event' ),
				'value'	=>	'https://api.whatsapp.com/send?phone={phone_number}&text=',
			],
			'fa-brands fa-square-whatsapp'	=>	[
				'label'	=>	esc_html__( 'Whatsapp', 'news-event' ),
				'value'	=>	'https://api.whatsapp.com/send?phone={phone_number}&text=',
			],
			'fa-brands fa-reddit'	=>	[
				'label'	=>	esc_html__( 'Reddit', 'news-event' ),
				'value'	=>	'https://reddit.com/submit?url=',
			],
			'fa-brands fa-reddit-alien'	=>	[
				'label'	=>	esc_html__( 'Reddit', 'news-event' ),
				'value'	=>	'https://reddit.com/submit?url=',
			],
			'fa-brands fa-square-reddit'	=>	[
				'label'	=>	esc_html__( 'Reddit', 'news-event' ),
				'value'	=>	'https://reddit.com/submit?url=',
			],
			'fa-brands fa-weixin'	=>	[
				'label'	=>	esc_html__( 'Weixin', 'news-event' ),
				'value'	=>	'https://widget.wechat.com/o/qrcode/',
			],
			'fa-brands fa-tumblr'	=>	[
				'label'	=>	esc_html__( 'Tumblr', 'news-event' ),
				'value'	=>	'https://www.tumblr.com/widgets/share/tool?canonicalUrl=',
			],
			'fa-brands fa-square-tumblr'	=>	[
				'label'	=>	esc_html__( 'Tumblr', 'news-event' ),
				'value'	=>	'https://www.tumblr.com/widgets/share/tool?canonicalUrl=',
			],
			'fa-brands fa-weibo'	=>	[
				'label'	=>	esc_html__( 'Weibo', 'news-event' ),
				'value'	=>	'http://service.weibo.com/share/share.php?url=',
			],
			'fa-brands fa-google-plus'	=>	[
				'label'	=>	esc_html__( 'Google Plus', 'news-event' ),
				'value'	=>	'https://plus.google.com/share?url=',
			],
			'fa-brands fa-google-plus-g'	=>	[
				'label'	=>	esc_html__( 'Google Plus', 'news-event' ),
				'value'	=>	'https://plus.google.com/share?url=',
			],
			'fa-brands fa-square-google-plus'	=>	[
				'label'	=>	esc_html__( 'Google Plus', 'news-event' ),
				'value'	=>	'https://plus.google.com/share?url=',
			],
			'fa-brands fa-skype'	=>	[
				'label'	=>	esc_html__( 'Skype', 'news-event' ),
				'value'	=>	'https://web.skype.com/share?url=',
			],
			'fa-brands fa-telegram'	=>	[
				'label'	=>	esc_html__( 'Telegram', 'news-event' ),
				'value'	=>	'https://telegram.me/share/url?url=',
			],
			'fa-brands fa-pinterest'	=>	[
				'label'	=>	esc_html__( 'Pinterest', 'news-event' ),
				'value'	=>	'http://pinterest.com/pin/create/link/?url=',
			],
			'fa-brands fa-pinterest-p'	=>	[
				'label'	=>	esc_html__( 'Pinterest', 'news-event' ),
				'value'	=>	'http://pinterest.com/pin/create/link/?url=',
			],
			'fa-brands fa-square-pinterest'	=>	[
				'label'	=>	esc_html__( 'Pinterest', 'news-event' ),
				'value'	=>	'http://pinterest.com/pin/create/link/?url=',
			],
			'fa-brands fa-vk'	=>	[
				'label'	=>	esc_html__( 'VK', 'news-event' ),
				'value'	=>	'http://vk.com/share.php?url=',
			],
			'fa-brands fa-line'	=>	[
				'label'	=>	esc_html__( 'Line', 'news-event' ),
				'value'	=>	'https://social-plugins.line.me/lineit/share?url=',
			],
			'fa-brands fa-blogger'	=>	[
				'label'	=>	esc_html__( 'Blogger', 'news-event' ),
				'value'	=>	'https://www.blogger.com/blog-this.g?u=',
			],
			'fa-brands fa-blogger-b'	=>	[
				'label'	=>	esc_html__( 'Blogger', 'news-event' ),
				'value'	=>	'https://www.blogger.com/blog-this.g?u=',
			],
			'fa-brands fa-evernote'	=>	[
				'label'	=>	esc_html__( 'Evernote', 'news-event' ),
				'value'	=>	'https://www.evernote.com/clip.action?url=',
			],
			'fa-brands fa-yahoo'	=>	[
				'label'	=>	esc_html__( 'Yahoo', 'news-event' ),
				'value'	=>	'http://compose.mail.yahoo.com/?to={email_address}&subject={title}&body=',
			],
			'fa-brands fa-odnoklassniki'	=>	[
				'label'	=>	esc_html__( 'Odnoklassniki', 'news-event' ),
				'value'	=>	'https://connect.ok.ru/dk?st.cmd=WidgetSharePreview&st.shareUrl=',
			],
			'fa-brands fa-square-odnoklassniki'	=>	[
				'label'	=>	esc_html__( 'Odnoklassniki', 'news-event' ),
				'value'	=>	'https://connect.ok.ru/dk?st.cmd=WidgetSharePreview&st.shareUrl=',
			],
			'fa-brands fa-viber'	=>	[
				'label'	=>	esc_html__( 'Viber', 'news-event' ),
				'value'	=>	'viber://forward?text=',
			],
			'fa-brands fa-get-pocket'	=>	[
				'label'	=>	esc_html__( 'Get Pocket', 'news-event' ),
				'value'	=>	'https://getpocket.com/edit?url=',
			],
			'fa-brands fa-mix'	=>	[
				'label'	=>	esc_html__( 'Mix', 'news-event' ),
				'value'	=>	'https://mix.com/add?url=',
			],
			'fa-brands fa-flipboard'	=>	[
				'label'	=>	esc_html__( 'Flipboard', 'news-event' ),
				'value'	=>	'https://share.flipboard.com/bookmarklet/popout?v=2&title=[TITLE]&url=',
			],
			'fa-brands fa-square-xing'	=>	[
				'label'	=>	esc_html__( 'Xing', 'news-event' ),
				'value'	=>	'https://www.xing.com/app/user?op=share;url=',
			],
			'fa-brands fa-xing'	=>	[
				'label'	=>	esc_html__( 'Xing', 'news-event' ),
				'value'	=>	'https://www.xing.com/app/user?op=share;url=',
			],
			'fa-brands fa-digg'	=>	[
				'label'	=>	esc_html__( 'Digg', 'news-event' ),
				'value'	=>	'http://digg.com/submit?url=',
			],
			'fa-brands fa-stumbleupon-circle'	=>	[
				'label'	=>	esc_html__( 'Stumbleupon', 'news-event' ),
				'value'	=>	'http://www.stumbleupon.com/submit?url=',
			],
			'fa-brands fa-stumbleupon'	=>	[
				'label'	=>	esc_html__( 'Stumbleupon', 'news-event' ),
				'value'	=>	'http://www.stumbleupon.com/submit?url=',
			],
			'fa-brands fa-delicious'	=>	[
				'label'	=>	esc_html__( 'Delicious', 'news-event' ),
				'value'	=>	'https://delicious.com/save?v=5&provider=[PROVIDER]&noui&jump=close&url=',
			],
			'fa-brands fa-buffer'	=>	[
				'label'	=>	esc_html__( 'Buffer', 'news-event' ),
				'value'	=>	'http://bufferapp.com/add?text=[post-title]&url=',
			],
			'fa-brands fa-diaspora'	=>	[
				'label'	=>	esc_html__( 'Diaspora', 'news-event' ),
				'value'	=>	'https://share.diasporafoundation.org/?title={title}&url=',
			],
			'fa-brands fa-hacker-news'	=>	[
				'label'	=>	esc_html__( 'Hacker News', 'news-event' ),
				'value'	=>	'https://news.ycombinator.com/submitlink?u=',
			],
			'fa-solid fa-comment-sms'	=>	[
				'label'	=>	esc_html__( 'SMS', 'news-event' ),
				'value'	=>	'sms:{phone_number}?body=',
			],
			'fa-brands fa-wordpress'	=>	[
				'label'	=>	esc_html__( 'Wordpress', 'news-event' ),
				'value'	=>	'https://wordpress.com/press-this.php?u=',
			],
			'fa-brands fa-wordpress-simple'	=>	[
				'label'	=>	esc_html__( 'Wordpress', 'news-event' ),
				'value'	=>	'https://wordpress.com/press-this.php?u=',
			],
			'fa-solid fa-copy'	=>	[
				'label'	=>	esc_html__( 'Copy', 'news-event' ),
				'value'	=>	$postUrl,
			],
			'fa-solid fa-print'	=>	[
				'label'	=>	esc_html__( 'Print', 'news-event' ),
				'value'	=>	'print',
			],
			'fa-brands fa-amazon'	=>	[
				'label'	=>	esc_html__( 'Amazon', 'news-event' ),
				'value'	=>	'http://www.amazon.com/wishlist/add?u=',
			],
			'fa-brands fa-renren'	=>	[
				'label'	=>	esc_html__( 'Renren', 'news-event' ),
				'value'	=>	'https://www.connect.renren.com/share/sharer?url=',
			],
			'fa-brands fa-trello'	=>	[
				'label'	=>	esc_html__( 'Trello', 'news-event' ),
				'value'	=>	'https://trello.com/add-card?mode=popup&url=',
			],
			'fa-brands fa-viadeo'	=>	[
				'label'	=>	esc_html__( 'Viadeo', 'news-event' ),
				'value'	=>	'http://www.viadeo.com/shareit/share/?url=',
			],
			'fa-brands fa-square-viadeo'	=>	[
				'label'	=>	esc_html__( 'Viadeo', 'news-event' ),
				'value'	=>	'http://www.viadeo.com/shareit/share/?url='
			]
		];
		if( $icon === '' ) :
			return $allIcons;
		else:
			return $allIcons[ $icon ];
		endif;
	}
endif;

if ( ! function_exists( 'news_event_create_elementor_kit' ) ) {
    /**
     * Create Elementor default kit
     *
     * @since 1.0.0
     */
    function news_event_create_elementor_kit() {
        if ( ! did_action( 'elementor/loaded' ) ) return;
        $kit = Elementor\Plugin::$instance->kits_manager->get_active_kit();
        if ( ! $kit->get_id() ) :
            $created_default_kit = Elementor\Plugin::$instance->kits_manager->create_default();
            update_option( 'elementor_active_kit', $created_default_kit );
		endif;
    }
    add_action( 'init', 'news_event_create_elementor_kit' );
}

 if( ! function_exists( 'news_event_get_archive_layout_meta' ) ) :
	/**
	 * Get archive layout
	 * 
	 * @since 1.0.0
	 */
 	function news_event_get_archive_layout_meta() {
		$current_id = get_queried_object_id();
		$archive_layout_meta = 'customizer-layout';
		if( is_category() ) :
			$archive_meta_key = '_news_event_category_archive_custom_meta_field';
			$archive_layout_meta = metadata_exists( 'term', $current_id, $archive_meta_key ) ? get_term_meta( $current_id, $archive_meta_key, true ) : 'customizer-layout';
		elseif ( is_tag() ) :
			$archive_meta_key = '_news_event_post_tag_archive_custom_meta_field';
			$archive_layout_meta = metadata_exists( 'term', $current_id, $archive_meta_key ) ? get_term_meta( $current_id, $archive_meta_key, true ) : 'customizer-layout';
		endif;

		return $archive_layout_meta;
	}
 endif;

 if( ! function_exists( 'news_event_get_archive_sidebar_meta' ) ) :
	/**
	 * Get archive sidebar layout
	 * 
	 * @since 1.0.0
	 */
 	function news_event_get_archive_sidebar_meta() {
		$current_id = get_queried_object_id();
		$archive_sidebar_meta = 'customizer-setting';
		if( is_category() ) :
			$archive_meta_key = '_news_event_category_sidebar_custom_meta_field';
			$archive_sidebar_meta = metadata_exists( 'term', $current_id, $archive_meta_key ) ? get_term_meta( $current_id, $archive_meta_key, true ) : 'customizer-setting';
		elseif ( is_tag() ) :
			$archive_meta_key = '_news_event_post_tag_sidebar_custom_meta_field';
			$archive_sidebar_meta = metadata_exists( 'term', $current_id, $archive_meta_key ) ? get_term_meta( $current_id, $archive_meta_key, true ) : 'customizer-setting';
		endif;

		return $archive_sidebar_meta;
	}
 endif;