<?php

namespace Elementor;

if (!defined('ABSPATH')) {
    exit;
}

/**
 * Define our MusicVibe Elementor Album Card Widget.
 */
class musicvibe_elementor_album_card extends Widget_Base {
    
    /**
     * Define our get_name settings.
     */
    public function get_name() {
        return 'musicvibe-album-card';
    }
    
    /**
     * Define our get_title settings.
     */
    public function get_title() {
        return __('Album Card', 'musicvibe');
    }
    
    /**
     * Define our get_icon settings.
     */
    public function get_icon() {
        return 'eicon-album';
    }
    
    /**
     * Define our get_categories settings.
     */
    public function get_categories() {
        return ['musicvibe-elementor-widget'];
    }
    
    /**
     * Define our _register_controls settings.
     */
    protected function register_controls() {
        // Content Tab
        $this->start_controls_section(
            'musicvibe_album_content_section',
            [
                'label' => esc_html__('Album Content', 'musicvibe'),
                'tab' => Controls_Manager::TAB_CONTENT,
            ]
        );
        
        $this->add_control(
            'album_image',
            [
                'label' => __('Album Image', 'musicvibe'),
                'type' => Controls_Manager::MEDIA,
                'default' => [
                    'url' => Utils::get_placeholder_image_src(),
                ],
            ]
        );
        
        $this->add_control(
            'album_title',
            [
                'label' => __('Album Title', 'musicvibe'),
                'type' => Controls_Manager::TEXT,
                'default' => __('Lost On The Road', 'musicvibe'),
                'placeholder' => __('Enter album title', 'musicvibe'),
                'label_block' => true,
            ]
        );
        
        $this->add_control(
            'stream_text',
            [
                'label' => __('Stream Text', 'musicvibe'),
                'type' => Controls_Manager::TEXT,
                'default' => __('Stream On', 'musicvibe'),
                'placeholder' => __('Enter stream text', 'musicvibe'),
            ]
        );
        
        $this->add_control(
            'show_spotify',
            [
                'label' => __('Show Spotify', 'musicvibe'),
                'type' => Controls_Manager::SWITCHER,
                'label_on' => __('Show', 'musicvibe'),
                'label_off' => __('Hide', 'musicvibe'),
                'return_value' => 'yes',
                'default' => 'yes',
            ]
        );
        
        $this->add_control(
            'spotify_link',
            [
                'label' => __('Spotify Link', 'musicvibe'),
                'type' => Controls_Manager::URL,
                'placeholder' => __('https://your-link.com', 'musicvibe'),
                'condition' => [
                    'show_spotify' => 'yes',
                ],
            ]
        );
        
        $this->add_control(
            'show_soundcloud',
            [
                'label' => __('Show SoundCloud', 'musicvibe'),
                'type' => Controls_Manager::SWITCHER,
                'label_on' => __('Show', 'musicvibe'),
                'label_off' => __('Hide', 'musicvibe'),
                'return_value' => 'yes',
                'default' => 'yes',
            ]
        );
        
        $this->add_control(
            'soundcloud_link',
            [
                'label' => __('SoundCloud Link', 'musicvibe'),
                'type' => Controls_Manager::URL,
                'placeholder' => __('https://your-link.com', 'musicvibe'),
                'condition' => [
                    'show_soundcloud' => 'yes',
                ],
            ]
        );
        
        $this->add_control(
            'show_itunes',
            [
                'label' => __('Show Itunes', 'musicvibe'),
                'type' => Controls_Manager::SWITCHER,
                'label_on' => __('Show', 'musicvibe'),
                'label_off' => __('Hide', 'musicvibe'),
                'return_value' => 'yes',
                'default' => 'yes',
            ]
        );
        
        $this->add_control(
            'itunes_link',
            [
                'label' => __('Itunes Link', 'musicvibe'),
                'type' => Controls_Manager::URL,
                'placeholder' => __('https://your-link.com', 'musicvibe'),
                'condition' => [
                    'show_itunes' => 'yes',
                ],
            ]
        );
        
        $this->end_controls_section();
        
        // Style Tab: Card
        $this->start_controls_section(
            'musicvibe_album_card_style_section',
            [
                'label' => esc_html__('Card', 'musicvibe'),
                'tab' => Controls_Manager::TAB_STYLE,
            ]
        );
        
        $this->add_control(
            'card_max_width',
            [
                'label' => __('Max Width', 'musicvibe'),
                'type' => Controls_Manager::SLIDER,
                'size_units' => ['px', '%'],
                'range' => [
                    'px' => [
                        'min' => 200,
                        'max' => 1000,
                        'step' => 1,
                    ],
                    '%' => [
                        'min' => 0,
                        'max' => 100,
                    ],
                ],
                'default' => [
                    'unit' => 'px',
                    'size' => 400,
                ],
                'selectors' => [
                    '{{WRAPPER}} .album-card' => 'max-width: {{SIZE}}{{UNIT}};',
                ],
            ]
        );
        
        $this->add_control(
            'card_background_color',
            [
                'label' => __('Background Color', 'musicvibe'),
                'type' => Controls_Manager::COLOR,
                'default' => '#ffffff',
                'selectors' => [
                    '{{WRAPPER}} .album-card' => 'background-color: {{VALUE}};',
                ],
            ]
        );
        
        $this->add_control(
            'card_border_radius',
            [
                'label' => __('Border Radius', 'musicvibe'),
                'type' => Controls_Manager::DIMENSIONS,
                'size_units' => ['px', '%'],
                'selectors' => [
                    '{{WRAPPER}} .album-card' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );
        
        $this->add_group_control(
            Group_Control_Box_Shadow::get_type(),
            [
                'name' => 'card_box_shadow',
                'label' => __('Box Shadow', 'musicvibe'),
                'selector' => '{{WRAPPER}} .album-card',
            ]
        );
        
        $this->add_responsive_control(
            'card_padding',
            [
                'label' => __('Padding', 'musicvibe'),
                'type' => Controls_Manager::DIMENSIONS,
                'size_units' => ['px', 'em', '%'],
                'selectors' => [
                    '{{WRAPPER}} .album-card' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );
        
        $this->add_responsive_control(
            'card_margin',
            [
                'label' => __('Margin', 'musicvibe'),
                'type' => Controls_Manager::DIMENSIONS,
                'size_units' => ['px', 'em', '%'],
                'selectors' => [
                    '{{WRAPPER}} .album-card' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );
        
        $this->end_controls_section();
        
        // Style Tab: Image
        $this->start_controls_section(
            'musicvibe_album_image_style_section',
            [
                'label' => esc_html__('Image', 'musicvibe'),
                'tab' => Controls_Manager::TAB_STYLE,
            ]
        );
        
        $this->add_control(
            'image_border_radius',
            [
                'label' => __('Border Radius', 'musicvibe'),
                'type' => Controls_Manager::DIMENSIONS,
                'size_units' => ['px', '%'],
                'selectors' => [
                    '{{WRAPPER}} .album-card img' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );
        
        $this->add_responsive_control(
            'image_margin',
            [
                'label' => __('Margin', 'musicvibe'),
                'type' => Controls_Manager::DIMENSIONS,
                'size_units' => ['px', 'em', '%'],
                'selectors' => [
                    '{{WRAPPER}} .album-card img' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );
        
        $this->end_controls_section();
        
        // Style Tab: Album Info
        $this->start_controls_section(
            'musicvibe_album_info_style_section',
            [
                'label' => esc_html__('Album Info', 'musicvibe'),
                'tab' => Controls_Manager::TAB_STYLE,
            ]
        );
        
        $this->add_control(
            'info_padding',
            [
                'label' => __('Padding', 'musicvibe'),
                'type' => Controls_Manager::DIMENSIONS,
                'size_units' => ['px', 'em', '%'],
                'selectors' => [
                    '{{WRAPPER}} .album-info' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );
        
        $this->end_controls_section();
        
        // Style Tab: Title
        $this->start_controls_section(
            'musicvibe_album_title_style_section',
            [
                'label' => esc_html__('Title', 'musicvibe'),
                'tab' => Controls_Manager::TAB_STYLE,
            ]
        );
        
        $this->add_control(
            'title_color',
            [
                'label' => __('Color', 'musicvibe'),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .album-title h2' => 'color: {{VALUE}};',
                ],
            ]
        );
        
        $this->add_group_control(
            Group_Control_Typography::get_type(),
            [
                'name' => 'title_typography',
                'label' => __('Typography', 'musicvibe'),
                'selector' => '{{WRAPPER}} .album-title h2',
            ]
        );
        
        $this->add_responsive_control(
            'title_margin',
            [
                'label' => __('Margin', 'musicvibe'),
                'type' => Controls_Manager::DIMENSIONS,
                'size_units' => ['px', 'em', '%'],
                'selectors' => [
                    '{{WRAPPER}} .album-title h2' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );
        
        $this->add_control(
            'title_text_transform',
            [
                'label' => __('Text Transform', 'musicvibe'),
                'type' => Controls_Manager::SELECT,
                'default' => 'uppercase',
                'options' => [
                    '' => __('Default', 'musicvibe'),
                    'uppercase' => __('Uppercase', 'musicvibe'),
                    'lowercase' => __('Lowercase', 'musicvibe'),
                    'capitalize' => __('Capitalize', 'musicvibe'),
                ],
                'selectors' => [
                    '{{WRAPPER}} .album-title h2' => 'text-transform: {{VALUE}};',
                ],
            ]
        );
        
        $this->add_control(
            'title_alignment',
            [
                'label' => __('Alignment', 'musicvibe'),
                'type' => Controls_Manager::CHOOSE,
                'options' => [
                    'left' => [
                        'title' => __('Left', 'musicvibe'),
                        'icon' => 'eicon-text-align-left',
                    ],
                    'center' => [
                        'title' => __('Center', 'musicvibe'),
                        'icon' => 'eicon-text-align-center',
                    ],
                    'right' => [
                        'title' => __('Right', 'musicvibe'),
                        'icon' => 'eicon-text-align-right',
                    ],
                ],
                'default' => 'left',
                'selectors' => [
                    '{{WRAPPER}} .album-title h2' => 'text-align: {{VALUE}};',
                ],
            ]
        );
        
        $this->end_controls_section();
        
        // Style Tab: Divider
        $this->start_controls_section(
            'musicvibe_album_divider_style_section',
            [
                'label' => esc_html__('Divider', 'musicvibe'),
                'tab' => Controls_Manager::TAB_STYLE,
            ]
        );
        
        $this->add_control(
            'divider_color',
            [
                'label' => __('Color', 'musicvibe'),
                'type' => Controls_Manager::COLOR,
                'default' => '#ccc',
                'selectors' => [
                    '{{WRAPPER}} .divider' => 'border-bottom-color: {{VALUE}};',
                ],
            ]
        );
        
        $this->add_control(
            'divider_width',
            [
                'label' => __('Width', 'musicvibe'),
                'type' => Controls_Manager::SLIDER,
                'size_units' => ['px', '%'],
                'range' => [
                    'px' => [
                        'min' => 1,
                        'max' => 10,
                        'step' => 1,
                    ],
                ],
                'default' => [
                    'unit' => 'px',
                    'size' => 1,
                ],
                'selectors' => [
                    '{{WRAPPER}} .divider' => 'border-bottom-width: {{SIZE}}{{UNIT}};',
                ],
            ]
        );
        
        $this->add_control(
            'divider_margin',
            [
                'label' => __('Margin', 'musicvibe'),
                'type' => Controls_Manager::DIMENSIONS,
                'size_units' => ['px', 'em', '%'],
                'selectors' => [
                    '{{WRAPPER}} .divider' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );
        
        $this->end_controls_section();
        
        // Style Tab: Stream Text
        $this->start_controls_section(
            'musicvibe_album_stream_style_section',
            [
                'label' => esc_html__('Stream Text', 'musicvibe'),
                'tab' => Controls_Manager::TAB_STYLE,
            ]
        );
        
        $this->add_control(
            'stream_text_color',
            [
                'label' => __('Color', 'musicvibe'),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .stream span' => 'color: {{VALUE}};',
                ],
            ]
        );
        
        $this->add_group_control(
            Group_Control_Typography::get_type(),
            [
                'name' => 'stream_text_typography',
                'label' => __('Typography', 'musicvibe'),
                'selector' => '{{WRAPPER}} .stream span',
            ]
        );
        
        $this->add_control(
            'stream_text_alignment',
            [
                'label' => __('Alignment', 'musicvibe'),
                'type' => Controls_Manager::CHOOSE,
                'options' => [
                    'left' => [
                        'title' => __('Left', 'musicvibe'),
                        'icon' => 'eicon-text-align-left',
                    ],
                    'center' => [
                        'title' => __('Center', 'musicvibe'),
                        'icon' => 'eicon-text-align-center',
                    ],
                    'right' => [
                        'title' => __('Right', 'musicvibe'),
                        'icon' => 'eicon-text-align-right',
                    ],
                ],
                'default' => 'left',
                'selectors' => [
                    '{{WRAPPER}} .stream' => 'justify-content: {{VALUE}};',
                ],
            ]
        );
        
        $this->end_controls_section();
        
        // Style Tab: Icons
        $this->start_controls_section(
            'musicvibe_album_icons_style_section',
            [
                'label' => esc_html__('Icons', 'musicvibe'),
                'tab' => Controls_Manager::TAB_STYLE,
            ]
        );
        
        $this->add_control(
            'icons_size',
            [
                'label' => __('Size', 'musicvibe'),
                'type' => Controls_Manager::SLIDER,
                'default' => [
                    'size' => 22,
                ],
                'range' => [
                    'px' => [
                        'min' => 10,
                        'max' => 50,
                    ],
                ],
                'selectors' => [
                    '{{WRAPPER}} .icons svg' => 'width: {{SIZE}}{{UNIT}}; height: {{SIZE}}{{UNIT}};',
                ],
            ]
        );
        
        $this->add_control(
            'icons_color',
            [
                'label' => __('Color', 'musicvibe'),
                'type' => Controls_Manager::COLOR,
                'default' => '#000000',
                'selectors' => [
                    '{{WRAPPER}} .icons svg' => 'fill: {{VALUE}};',
                ],
            ]
        );
        
        $this->add_control(
            'icons_hover_color',
            [
                'label' => __('Hover Color', 'musicvibe'),
                'type' => Controls_Manager::COLOR,
                'default' => '#D14343',
                'selectors' => [
                    '{{WRAPPER}} .icons svg:hover' => 'fill: {{VALUE}};',
                ],
            ]
        );
        
        $this->add_control(
            'icons_spacing',
            [
                'label' => __('Spacing', 'musicvibe'),
                'type' => Controls_Manager::SLIDER,
                'default' => [
                    'size' => 12,
                ],
                'range' => [
                    'px' => [
                        'min' => 0,
                        'max' => 30,
                    ],
                ],
                'selectors' => [
                    '{{WRAPPER}} .icons' => 'gap: {{SIZE}}{{UNIT}};',
                ],
            ]
        );
        
        $this->end_controls_section();
    }
    
    /**
     * Define our Content Display settings.
     */
    protected function render() {
        $settings = $this->get_settings_for_display();
        
        // Get image URL
        $image_url = $settings['album_image']['url'];
        $image_alt = get_post_meta($settings['album_image']['id'], '_wp_attachment_image_alt', true);
        if (empty($image_alt)) {
            $image_alt = $settings['album_title'];
        }
        
        // Get link attributes
        $spotify_link = $settings['spotify_link'];
        $spotify_target = $spotify_link['is_external'] ? ' target="_blank"' : '';
        $spotify_nofollow = $spotify_link['nofollow'] ? ' rel="nofollow"' : '';
        
        $soundcloud_link = $settings['soundcloud_link'];
        $soundcloud_target = $soundcloud_link['is_external'] ? ' target="_blank"' : '';
        $soundcloud_nofollow = $soundcloud_link['nofollow'] ? ' rel="nofollow"' : '';
        
        $itunes_link = $settings['itunes_link'];
        $itunes_target = $itunes_link['is_external'] ? ' target="_blank"' : '';
        $itunes_nofollow = $itunes_link['nofollow'] ? ' rel="nofollow"' : '';
        ?>
        <div class="album-card">
            <?php if ($image_url) : ?>
                <img src="<?php echo esc_url($image_url); ?>" alt="<?php echo esc_attr($image_alt); ?>">
            <?php endif; ?>
            
            <div class="album-info">
                <?php if ($settings['album_title']) : ?>
                    <div class="album-title"><h2><?php echo esc_html($settings['album_title']); ?></h2></div>
                <?php endif; ?>
                
                <div class="divider"></div>
                
                <div class="stream">
                    <?php if ($settings['stream_text']) : ?>
                        <span><?php echo esc_html($settings['stream_text']); ?></span>
                    <?php endif; ?>
                    
                    <div class="icons">
                        <?php if ($settings['show_spotify'] === 'yes') : ?>
                            <a href="<?php echo esc_url($spotify_link['url']); ?>" <?php echo $spotify_target . $spotify_nofollow; ?>>
                                <svg viewBox="0 0 496 512" xmlns="http://www.w3.org/2000/svg">
                                    <path d="M248 8C111 8 0 119 0 256s111 248 248 248 
                                    248-111 248-248S385 8 248 8zm113.5 364.8c-4.2 
                                    0-6.8-1.3-10.8-3.6-62.4-37.6-136.5-39.2-207.7-24.5
                                    -3.9.8-9 .8-11.9-2.7-3.6-3.9-2.7-10.2 2.7-13 
                                    76.2-16.1 156.6-14.2 225.6 26.5 4.6 2.7 6.1 
                                    8.9 3.3 13.5-1.9 2.9-5 4.6-8.9 
                                    4.6zm14.3-63.6c-5.2 0-8.6-1.9-12.7-4.2-62.4-37-156.8-52.1
                                    -238.4-28.7-4.6 1.3-10.2-1.3-11.5-5.9-1.3-5 
                                    1.3-10.2 5.9-11.5 88.7-25.7 
                                    189.4-9.9 257.8 31.2 4.2 2.7 
                                    5.9 8.2 3.6 12.7-2.4 3.4-6 5.4-10.2 
                                    5.4zm20.1-65.7c-5.2 0-8.6-1.3-13.4-3.9
                                    -71.1-42.2-189.9-46.5-265.3-25.4-5.2 
                                    1.3-10.8-1.9-12.1-7.2-1.3-5.2 
                                    1.9-10.8 7.2-12.1C208.6 184 335.1 
                                    188.8 412 234c4.6 2.7 6.5 8.9 
                                    3.9 13.8-1.9 3.6-5.5 5.7-9.5 
                                    5.7z"/>
                                </svg>
                            </a>
                        <?php endif; ?>
                        
                        <?php if ($settings['show_soundcloud'] === 'yes') : ?>
                            <a href="<?php echo esc_url($soundcloud_link['url']); ?>" <?php echo $soundcloud_target . $soundcloud_nofollow; ?>>
                                <svg viewBox="0 0 640 512" xmlns="http://www.w3.org/2000/svg">
                                    <path d="M537.6 226.6c-4.3 0-8.6.3-12.9.8-13.7-59.5-66.7-103.9-130.7-103.9-15.4 
                                    0-30.1 2.6-43.9 7.3-19.9-64.5-79.7-111.3-151.1-111.3-87.1 
                                    0-157.7 70.7-157.7 157.7 0 3.6.1 7.2.3 10.7C17.7 
                                    206.3 0 233.9 0 265.6c0 45.3 36.7 82 82 82h455.6c47.3 
                                    0 86.4-38.4 86.4-86.4s-39.1-86.6-86.4-86.6z"/>
                                </svg>
                            </a>
                        <?php endif; ?>
                        
                        <?php if ($settings['show_itunes'] === 'yes') : ?>
                            <a href="<?php echo esc_url($itunes_link['url']); ?>" <?php echo $itunes_target . $itunes_nofollow; ?>>
                                <svg class="e-font-icon-svg e-fab-apple" viewBox="0 0 384 512" xmlns="http://www.w3.org/2000/svg"><path d="M318.7 268.7c-.2-36.7 16.4-64.4 50-84.8-18.8-26.9-47.2-41.7-84.7-44.6-35.5-2.8-74.3 20.7-88.5 20.7-15 0-49.4-19.7-76.4-19.7C63.3 141.2 4 184.8 4 273.5q0 39.3 14.4 81.2c12.8 36.7 59 126.7 107.2 125.2 25.2-.6 43-17.9 75.8-17.9 31.8 0 48.3 17.9 76.4 17.9 48.6-.7 90.4-82.5 102.6-119.3-65.2-30.7-61.7-90-61.7-91.9zm-56.6-164.2c27.3-32.4 24.8-61.9 24-72.5-24.1 1.4-52 16.4-67.9 34.9-17.5 19.8-27.8 44.3-25.6 71.9 26.1 2 49.9-11.4 69.5-34.3z"></path></svg>
                            </a>
                        <?php endif; ?>
                    </div>
                </div>
            </div>
        </div>
        <?php
    }
    
    /**
     * Define our Content Display in editor settings.
     */
    protected function content_template() {
        ?>
        <#
        var image = {
            id: settings.album_image.id,
            url: settings.album_image.url,
            size: settings.image_size,
            dimension: settings.image_custom_dimension,
            model: view.getEditModel()
        };
        var image_url = elementor.imagesManager.getImageUrl( image );
        var image_alt = settings.album_title;
        
        var spotify_link = settings.spotify_link;
        var spotify_target = spotify_link.is_external ? ' target="_blank"' : '';
        var spotify_nofollow = spotify_link.nofollow ? ' rel="nofollow"' : '';
        
        var soundcloud_link = settings.soundcloud_link;
        var soundcloud_target = soundcloud_link.is_external ? ' target="_blank"' : '';
        var soundcloud_nofollow = soundcloud_link.nofollow ? ' rel="nofollow"' : '';
        
        var itunes_link = settings.itunes_link;
        var itunes_target = itunes_link.is_external ? ' target="_blank"' : '';
        var itunes_nofollow = itunes_link.nofollow ? ' rel="nofollow"' : '';
        #>
        <div class="album-card">
            <# if ( image_url ) { #>
                <img src="{{ image_url }}" alt="{{ image_alt }}">
            <# } #>
            
            <div class="album-info">
                <# if ( settings.album_title ) { #>
                    <div class="album-title">{{{ settings.album_title }}}</div>
                <# } #>
                
                <div class="divider"></div>
                
                <div class="stream">
                    <# if ( settings.stream_text ) { #>
                        <span>{{{ settings.stream_text }}}</span>
                    <# } #>
                    
                    <div class="icons">
                        <# if ( settings.show_spotify === 'yes' ) { #>
                            <a href="{{ spotify_link.url }}"{{ spotify_target }}{{ spotify_nofollow }}>
                                <svg viewBox="0 0 496 512" xmlns="http://www.w3.org/2000/svg">
                                    <path d="M248 8C111 8 0 119 0 256s111 248 248 248 
                                    248-111 248-248S385 8 248 8zm113.5 364.8c-4.2 
                                    0-6.8-1.3-10.8-3.6-62.4-37.6-136.5-39.2-207.7-24.5
                                    -3.9.8-9 .8-11.9-2.7-3.6-3.9-2.7-10.2 2.7-13 
                                    76.2-16.1 156.6-14.2 225.6 26.5 4.6 2.7 6.1 
                                    8.9 3.3 13.5-1.9 2.9-5 4.6-8.9 
                                    4.6zm14.3-63.6c-5.2 0-8.6-1.9-12.7-4.2-62.4-37-156.8-52.1
                                    -238.4-28.7-4.6 1.3-10.2-1.3-11.5-5.9-1.3-5 
                                    1.3-10.2 5.9-11.5 88.7-25.7 
                                    189.4-9.9 257.8 31.2 4.2 2.7 
                                    5.9 8.2 3.6 12.7-2.4 3.4-6 5.4-10.2 
                                    5.4zm20.1-65.7c-5.2 0-8.6-1.3-13.4-3.9
                                    -71.1-42.2-189.9-46.5-265.3-25.4-5.2 
                                    1.3-10.8-1.9-12.1-7.2-1.3-5.2 
                                    1.9-10.8 7.2-12.1C208.6 184 335.1 
                                    188.8 412 234c4.6 2.7 6.5 8.9 
                                    3.9 13.8-1.9 3.6-5.5 5.7-9.5 
                                    5.7z"/>
                                </svg>
                            </a>
                        <# } #>
                        
                        <# if ( settings.show_soundcloud === 'yes' ) { #>
                            <a href="{{ soundcloud_link.url }}"{{ soundcloud_target }}{{ soundcloud_nofollow }}>
                                <svg viewBox="0 0 640 512" xmlns="http://www.w3.org/2000/svg">
                                    <path d="M537.6 226.6c-4.3 0-8.6.3-12.9.8-13.7-59.5-66.7-103.9-130.7-103.9-15.4 
                                    0-30.1 2.6-43.9 7.3-19.9-64.5-79.7-111.3-151.1-111.3-87.1 
                                    0-157.7 70.7-157.7 157.7 0 3.6.1 7.2.3 10.7C17.7 
                                    206.3 0 233.9 0 265.6c0 45.3 36.7 82 82 82h455.6c47.3 
                                    0 86.4-38.4 86.4-86.4s-39.1-86.6-86.4-86.6z"/>
                                </svg>
                            </a>
                        <# } #>
                        
                        <# if ( settings.show_itunes === 'yes' ) { #>
                            <a href="{{ itunes_link.url }}"{{ itunes_target }}{{ itunes_nofollow }}>
                                <svg class="e-font-icon-svg e-fab-apple" viewBox="0 0 384 512" xmlns="http://www.w3.org/2000/svg"><path d="M318.7 268.7c-.2-36.7 16.4-64.4 50-84.8-18.8-26.9-47.2-41.7-84.7-44.6-35.5-2.8-74.3 20.7-88.5 20.7-15 0-49.4-19.7-76.4-19.7C63.3 141.2 4 184.8 4 273.5q0 39.3 14.4 81.2c12.8 36.7 59 126.7 107.2 125.2 25.2-.6 43-17.9 75.8-17.9 31.8 0 48.3 17.9 76.4 17.9 48.6-.7 90.4-82.5 102.6-119.3-65.2-30.7-61.7-90-61.7-91.9zm-56.6-164.2c27.3-32.4 24.8-61.9 24-72.5-24.1 1.4-52 16.4-67.9 34.9-17.5 19.8-27.8 44.3-25.6 71.9 26.1 2 49.9-11.4 69.5-34.3z"></path></svg>
                            </a>
                        <# } #>
                    </div>
                </div>
            </div>
        </div>
        <?php
    }
}

// Register the widget
Plugin::instance()->widgets_manager->register_widget_type(new musicvibe_elementor_album_card());