<?php

/**
 * Theme functions and definitions.
 *
 * @link https://codex.wordpress.org/Functions_File_Explained
 *
 * @package Musical_Vibe
 */

if (! function_exists('musical_vibe_setup')) :

	/**
	 * Sets up theme defaults and registers support for various WordPress features.
	 *
	 * Note that this function is hooked into the after_setup_theme hook, which
	 * runs before the init hook. The init hook is too late for some features, such
	 * as indicating support for post thumbnails.
	 */
	function musical_vibe_setup()
	{
		/*
		 * Make theme available for translation.
		 */
		load_theme_textdomain('musical-vibe', get_template_directory() . '/languages');

		// Add default posts and comments RSS feed links to head.
		add_theme_support('automatic-feed-links');

		/*
		 * Let WordPress manage the document title.
		 */
		add_theme_support('title-tag');

		/*
		 * Enable support for Post Thumbnails on posts and pages.
		 */
		add_theme_support('post-thumbnails');
		add_image_size('musical-vibe-thumb', 380, 360, true);


		// This theme uses wp_nav_menu() in four location.
		register_nav_menus(array(
			'primary'  => esc_html__('Primary Menu', 'musical-vibe'),
			'footer'   => esc_html__('Footer Menu', 'musical-vibe'),
			'social'   => esc_html__('Social Menu', 'musical-vibe'),
			'notfound' => esc_html__('404 Menu', 'musical-vibe'),
		));

		/*
		 * Switch default core markup for search form, comment form, and comments
		 * to output valid HTML5.
		 */
		add_theme_support('html5', array(
			'search-form',
			'comment-form',
			'comment-list',
			'gallery',
			'caption',
		));

		// Set up the WordPress core custom background feature.
		add_theme_support('custom-background', apply_filters('musical_vibe_custom_background_args', array(
			'default-color' => 'FFFFFF',
			'default-image' => '',
		)));

		/*
		 * Enable support for selective refresh of widgets in Customizer.
		 */
		add_theme_support('customize-selective-refresh-widgets');

		/*
		 * Enable support for custom logo.
		 */
		add_theme_support('custom-logo');

		// Load default block styles.
		add_theme_support('wp-block-styles');

		// Add support for responsive embeds.
		add_theme_support('responsive-embeds');

		// Enable support for footer widgets.
		add_theme_support('footer-widgets', 4);

		// Add WooCommerce Support.
		add_theme_support('woocommerce');
		add_theme_support('wc-product-gallery-lightbox');

		// Load Supports.
		require get_template_directory() . '/inc/support.php';
	}
endif;

add_action('after_setup_theme', 'musical_vibe_setup');

/**
 * Set the content width in pixels, based on the theme's design and stylesheet.
 *
 * Priority 0 to make it available to lower priority callbacks.
 *
 * @global int $content_width
 */
function musical_vibe_content_width()
{
	$GLOBALS['content_width'] = apply_filters('musical_vibe_content_width', 640);
}
add_action('after_setup_theme', 'musical_vibe_content_width', 0);

/**
 * Register widget area.
 *
 * @link https://developer.wordpress.org/themes/functionality/sidebars/#registering-a-sidebar
 */
function musical_vibe_widgets_init()
{
	register_sidebar(array(
		'name'          => esc_html__('Primary Sidebar', 'musical-vibe'),
		'id'            => 'sidebar-1',
		'description'   => esc_html__('Add widgets here to appear in your Primary Sidebar.', 'musical-vibe'),
		'before_widget' => '<aside id="%1$s" class="widget %2$s">',
		'after_widget'  => '</aside>',
		'before_title'  => '<h2 class="widget-title">',
		'after_title'   => '</h2>',
	));
	register_sidebar(array(
		'name'          => esc_html__('Secondary Sidebar', 'musical-vibe'),
		'id'            => 'sidebar-2',
		'description'   => esc_html__('Add widgets here to appear in your Secondary Sidebar.', 'musical-vibe'),
		'before_widget' => '<aside id="%1$s" class="widget %2$s">',
		'after_widget'  => '</aside>',
		'before_title'  => '<h2 class="widget-title">',
		'after_title'   => '</h2>',
	));
	register_sidebar(array(
		'name'          => esc_html__('Front Page Widget Area', 'musical-vibe'),
		'id'            => 'sidebar-front-page-widget-area',
		'description'   => esc_html__('Add widgets here to appear in your Front Page.', 'musical-vibe'),
		'before_widget' => '<aside id="%1$s" class="widget %2$s"><div class="container">',
		'after_widget'  => '</div></aside>',
		'before_title'  => '<h2 class="widget-title">',
		'after_title'   => '</h2><div class="title-divider">
			</div>',
	));
}
add_action('widgets_init', 'musical_vibe_widgets_init');

/**
 * Enqueue scripts and styles.
 */
function musical_vibe_scripts()
{

	$theme_version = wp_get_theme()->get('Version');

	$min = defined('SCRIPT_DEBUG') && SCRIPT_DEBUG ? '' : '.min';

	wp_enqueue_style('font-awesome', get_template_directory_uri() . '/third-party/font-awesome/css/all' . $min . '.css', '', '6.7.2');

	$fonts_url = musical_vibe_fonts_url();

	if (! empty($fonts_url)) {
		wp_enqueue_style('musical-vibe-google-fonts', $fonts_url, array(), null);
	}

	wp_enqueue_style('jquery-sidr', get_template_directory_uri() . '/third-party/sidr/css/jquery.sidr.dark' . $min . '.css', '', '2.2.1');

	wp_enqueue_style('musical-vibe-style', get_stylesheet_uri(), array(), $theme_version);

	wp_enqueue_script('musical-vibe-skip-link-focus-fix', get_template_directory_uri() . '/js/skip-link-focus-fix' . $min . '.js', array(), '20130115', true);

	wp_enqueue_script('jquery-cycle2', get_template_directory_uri() . '/third-party/cycle2/js/jquery.cycle2' . $min . '.js', array('jquery'), '2.1.6', true);

	wp_enqueue_script('jquery-sidr', get_template_directory_uri() . '/third-party/sidr/js/jquery.sidr' . $min . '.js', array('jquery'), '2.2.1', true);

	wp_enqueue_script('musical-vibe-custom', get_template_directory_uri() . '/js/custom' . $min . '.js', array('jquery'), '1.0.0', true);

	$custom_args = array(
		'go_to_top_status' => (true === musical_vibe_get_option('go_to_top')) ? 1 : 0,
	);
	wp_localize_script('musical-vibe-custom', 'Musical_Vibe_Custom_Options', $custom_args);

	if (is_singular() && comments_open() && get_option('thread_comments')) {
		wp_enqueue_script('comment-reply');
	}
}
add_action('wp_enqueue_scripts', 'musical_vibe_scripts');

/**
 * Enqueue admin scripts and styles.
 */
function musical_vibe_admin_scripts($hook)
{

	$min = defined('SCRIPT_DEBUG') && SCRIPT_DEBUG ? '' : '.min';

	if (in_array($hook, array('post.php', 'post-new.php'))) {
		wp_enqueue_style('musical-vibe-metabox', get_template_directory_uri() . '/css/metabox' . $min . '.css', '', '1.0.0');
		wp_enqueue_script('musical-vibe-custom-admin', get_template_directory_uri() . '/js/admin' . $min . '.js', array('jquery', 'jquery-ui-core', 'jquery-ui-tabs'), '1.0.0', true);
	}

	if ('widgets.php' === $hook) {
		wp_enqueue_style('wp-color-picker');
		wp_enqueue_script('wp-color-picker');
		wp_enqueue_media();
		wp_enqueue_style('musical-vibe-custom-widgets-style', get_template_directory_uri() . '/css/widgets' . $min . '.css', array(), '1.0.0');
		wp_enqueue_script('musical-vibe-custom-widgets', get_template_directory_uri() . '/js/widgets' . $min . '.js', array('jquery'), '1.0.0', true);
	}
}
add_action('admin_enqueue_scripts', 'musical_vibe_admin_scripts');

/**
 * Load init.
 */
require_once get_template_directory() . '/inc/init.php';


if (! function_exists('musical_vibe_blocks_support')) :
	/**
	 * Create add default blocks support
	 */
	function musical_vibe_blocks_support()
	{
		// Add support for Block Styles.
		add_theme_support('wp-block-styles');

		// Add support for full and wide align images.
		add_theme_support('align-wide');

		// Add support for editor styles.
		add_theme_support('editor-styles');

		// Add support for responsive embeds.
		add_theme_support('responsive-embeds');

		// Add custom editor font sizes.
		add_theme_support(
			'editor-font-sizes',
			array(
				array(
					'name'      => esc_html__('Small', 'musical-vibe'),
					'shortName' => esc_html__('S', 'musical-vibe'),
					'size'      => 14,
					'slug'      => 'small',
				),
				array(
					'name'      => esc_html__('Normal', 'musical-vibe'),
					'shortName' => esc_html__('M', 'musical-vibe'),
					'size'      => 18,
					'slug'      => 'normal',
				),
				array(
					'name'      => esc_html__('Large', 'musical-vibe'),
					'shortName' => esc_html__('L', 'musical-vibe'),
					'size'      => 42,
					'slug'      => 'large',
				),
				array(
					'name'      => esc_html__('Huge', 'musical-vibe'),
					'shortName' => esc_html__('XL', 'musical-vibe'),
					'size'      => 54,
					'slug'      => 'huge',
				),
			)
		);

		// Add support for custom color scheme.
		add_theme_support('editor-color-palette', array(
			array(
				'name'  => esc_html__('White', 'musical-vibe'),
				'slug'  => 'white',
				'color' => '#ffffff',
			),
			array(
				'name'  => esc_html__('Black', 'musical-vibe'),
				'slug'  => 'black',
				'color' => '#111111',
			),
			array(
				'name'  => esc_html__('Gray', 'musical-vibe'),
				'slug'  => 'gray',
				'color' => '#f4f4f4',
			),
			array(
				'name'  => esc_html__('Blue', 'musical-vibe'),
				'slug'  => 'blue',
				'color' => '00aeef',
			),
			array(
				'name'  => esc_html__('Orange', 'musical-vibe'),
				'slug'  => 'orange',
				'color' => '#ffc300',
			),
		));
	}
	add_action('after_setup_theme', 'musical_vibe_blocks_support', 20);
endif; //musical_vibe_blocks_support

if (! function_exists('musical_vibe_add_blocks_style')) :
	/**
	 * Add Blocks Style
	 */
	function musical_vibe_add_blocks_style()
	{
		// Theme block stylesheet.
		wp_enqueue_style('musical-vibe-block-style', get_theme_file_uri('/css/blocks.css'), array('musical-vibe-style'), date('Ymd-Gis', filemtime(get_template_directory() . '/css/blocks.css')));
	}
	add_action('wp_enqueue_scripts', 'musical_vibe_add_blocks_style');
endif; //musical_vibe_add_blocks_style

if (! function_exists('musical_vibe_block_editor_styles')) :
	/**
	 * Enqueue editor styles for Blocks
	 */
	function musical_vibe_block_editor_styles()
	{
		// Block styles.
		wp_enqueue_style('musical-vibe-block-editor-style', get_theme_file_uri('/css/editor-blocks.css'), null, date('Ymd-Gis', filemtime(get_template_directory() . '/css/editor-blocks.css')));

		// Add custom fonts.
		wp_enqueue_style('musical-vibe-fonts', musical_vibe_fonts_url(), array(), null);
	}
	add_action('enqueue_block_editor_assets', 'musical_vibe_block_editor_styles');
endif; //musical_vibe_block_editor_styles
