<?php

/**
 * multivas functions and definitions
 *
 * @link https://developer.wordpress.org/themes/basics/theme-functions/
 *
 * @package multivas
 */

if (!defined('MULTIVAS__VERSION')) {
	// Replace the version number of the theme on each release.
	define('MULTIVAS__VERSION', '1.0.0');
}

/**
 * Sets up theme defaults and registers support for various WordPress features.
 *
 * Note that this function is hooked into the after_setup_theme hook, which
 * runs before the init hook. The init hook is too late for some features, such
 * as indicating support for post thumbnails.
 */
function multivas_setup()
{
	/*
		* Make theme available for translation.
		* Translations can be filed in the /languages/ directory.
		* If you're building a theme based on multivas, use a find and replace
		* to change 'multivas' to the name of your theme in all the template files.
		*/
	load_theme_textdomain('multivas', get_template_directory() . '/languages');

	// Add default posts and comments RSS feed links to head.
	add_theme_support('automatic-feed-links');

	/*
		* Let WordPress manage the document title.
		* By adding theme support, we declare that this theme does not use a
		* hard-coded <title> tag in the document head, and expect WordPress to
		* provide it for us.
		*/
	add_theme_support('title-tag');

	/*
		* Enable support for Post Thumbnails on posts and pages.
		*
		* @link https://developer.wordpress.org/themes/functionality/featured-images-post-thumbnails/
		*/
	add_theme_support('post-thumbnails');
	add_image_size('multivas-slider-size', 1400, 800, true);

	// This theme uses wp_nav_menu() in one location.
	register_nav_menus(
		array(
			'main-menu' => esc_html__('Primary', 'multivas'),
		)
	);

	/*
		* Switch default core markup for search form, comment form, and comments
		* to output valid HTML5.
		*/
	add_theme_support(
		'html5',
		array(
			'search-form',
			'comment-form',
			'comment-list',
			'gallery',
			'caption',
			'style',
			'script',
		)
	);

	// Set up the WordPress core custom background feature.
	add_theme_support(
		'custom-background',
		apply_filters(
			'multivas_custom_background_args',
			array(
				'default-color' => 'ffffff',
				'default-image' => '',
			)
		)
	);

	// Add theme support for selective refresh for widgets.
	add_theme_support('customize-selective-refresh-widgets');

	/**
	 * Add support for core custom logo.
	 *
	 * @link https://codex.wordpress.org/Theme_Logo
	 */
	add_theme_support(
		'custom-logo',
		array(
			'height'      => 250,
			'width'       => 150,
			'flex-width'  => true,
			'flex-height' => true,
		)
	);
	// Add support for full and wide align images.
	add_theme_support('align-wide');

	// Add support for responsive embeds.
	add_theme_support('responsive-embeds');

	// Add support for editor style
	add_editor_style('assets/css/editor-style.css');

	// Add support for block style
	add_theme_support("wp-block-styles");

	// Restore Classic Widgets
	remove_theme_support('widgets-block-editor');
}
add_action('after_setup_theme', 'multivas_setup');

/**
 * Set the content width in pixels, based on the theme's design and stylesheet.
 *
 * Priority 0 to make it available to lower priority callbacks.
 *
 * @global int $content_width
 */
function multivas_content_width()
{
	$GLOBALS['content_width'] = apply_filters('multivas_content_width', 640);
}
add_action('after_setup_theme', 'multivas_content_width', 0);

/**
 * Register widget area.
 *
 * @link https://developer.wordpress.org/themes/functionality/sidebars/#registering-a-sidebar
 */
function multivas_widgets_init()
{
	register_sidebar(
		array(
			'name'          => esc_html__('Blog Sidebar', 'multivas'),
			'id'            => 'blog-sidebar',
			'description'   => esc_html__('Add widgets here.', 'multivas'),
			'before_widget' => '<section id="%1$s" class="widget %2$s bg-light">',
			'after_widget'  => '</section>',
			'before_title'  => '<h2 class="widget-title px-4 py-2">',
			'after_title'   => '</h2>',
		)
	);
	register_sidebar(
		array(
			'name'          => esc_html__('Footer Widget 1', 'multivas'),
			'id'            => 'footer-widget-1',
			'description'   => esc_html__('Add widgets here.', 'multivas'),
			'before_widget' => '<aside id="%1$s" class="widget %2$s bg-light">',
			'after_widget'  => '</aside>',
			'before_title'  => '<h3 class="widget-title px-4 py-2">',
			'after_title'   => '</h3>',
		)
	);
	register_sidebar(
		array(
			'name'          => esc_html__('Footer Widget 2', 'multivas'),
			'id'            => 'footer-widget-2',
			'description'   => esc_html__('Add widgets here.', 'multivas'),
			'before_widget' => '<aside id="%1$s" class="widget %2$s bg-light">',
			'after_widget'  => '</aside>',
			'before_title'  => '<h3 class="widget-title px-4 py-2">',
			'after_title'   => '</h3>',
		)
	);
	register_sidebar(
		array(
			'name'          => esc_html__('Footer Widget 3', 'multivas'),
			'id'            => 'footer-widget-3',
			'description'   => esc_html__('Add widgets here.', 'multivas'),
			'before_widget' => '<aside id="%1$s" class="widget %2$s bg-light">',
			'after_widget'  => '</aside>',
			'before_title'  => '<h3 class="widget-title px-4 py-2">',
			'after_title'   => '</h3>',
		)
	);
	register_sidebar(
		array(
			'name'          => esc_html__('Footer Widget 4', 'multivas'),
			'id'            => 'footer-widget-4',
			'description'   => esc_html__('Add widgets here.', 'multivas'),
			'before_widget' => '<aside id="%1$s" class="widget %2$s bg-light">',
			'after_widget'  => '</aside>',
			'before_title'  => '<h3 class="widget-title px-4 py-2">',
			'after_title'   => '</h3>',
		)
	);
}
add_action('widgets_init', 'multivas_widgets_init');

//estimated reading time
function multivas_reading_time()
{
	global $post;
	$content = get_post_field('post_content', $post->ID);
	$word_count = str_word_count(strip_tags($content));
	$readingtime = ceil($word_count / 200);

	if ($readingtime == 1) {
		$timer = __(' minute read', 'multivas');
	} else {
		$timer = __(' minutes read', 'multivas');
	}
	$totalreadingtime = $readingtime . $timer;

	return $totalreadingtime;
}

//Default function for theme options 
function multivas_get_defaults()
{
	$defaults = array();
	//Logo default settings

	$defaults['multivas_feature_slider_category'] = '';
	$defaults['multivas_total_feature_slides'] = 3;
	$defaults['multivas_caption_checkbox'] = true;
	$defaults['multivas_blog_grid_settings'] = 'right';
	$defaults['multivas_posts_grid_settings'] = '1';
	$defaults['multivas_footer_widgets_checkbox'] = false;
	$defaults['multivas_fb_social_link'] = 'https://www.facebook.com/';
	$defaults['multivas_twitter_social_link'] = 'https://twitter.com/';
	$defaults['multivas_instagram_social_link'] = 'https://www.instagram.com/';
	$defaults['multivas_youtube_social_link'] = 'https://youtube.com/';

	return apply_filters('multivas_get_defaults', $defaults);
}

//Blank footer widget area 
function multivas_blog_blank_widget_area()
{
	if (is_user_logged_in() && current_user_can('edit_theme_options')) {
		echo '<a href="' .
			esc_url(admin_url('widgets.php')) .
			'" target="_blank"><i class="fa fa-plus-circle"></i> ' .
			esc_html__('Add Widget', 'multivas') .
			'</a>';
	}
}

//Blank slider section 
function multivas_slider_section()
{
	if (is_user_logged_in() && current_user_can('edit_theme_options')) {
		echo '<a href="' .
			esc_url(admin_url('customize.php?autofocus[section]=multivas_feature_images_slider')) .
			'"><i class="fa fa-plus-circle"></i> ' .
			esc_html__('Add Slider', 'multivas') .
			'</a>';
	}
}

add_filter('nav_menu_link_attributes', 'multivas_prefix_bs5_dropdown_data_attribute', 20, 3);
/**
 * Use namespaced data attribute for Bootstrap's dropdown toggles.
 *
 * @param array    $atts HTML attributes applied to the item's `<a>` element.
 * @param WP_Post  $item The current menu item.
 * @param stdClass $args An object of wp_nav_menu() arguments.
 * @return array
 */
function multivas_prefix_bs5_dropdown_data_attribute($atts, $item, $args)
{
	if (is_a($args->walker, 'WP_Bootstrap_Navwalker')) {
		if (array_key_exists('data-toggle', $atts)) {
			unset($atts['data-toggle']);
			$atts['data-bs-toggle'] = 'dropdown';
		}
	}
	return $atts;
}

/**
 * Enqueue scripts and styles.
 */
function multivas_scripts()
{
	// Register Styles
	wp_enqueue_style('google-fonts', wptt_get_webfont_url('https://fonts.googleapis.com/css2?family=DM+Sans:wght@400;500;700&family=Montserrat:wght@400;500;600;700;800&family=Open+Sans:wght@300;400;500;600&display=swap'), array(), MULTIVAS__VERSION, 'all');
	wp_enqueue_style('font-awesome-css', get_template_directory_uri() . '/assets/css/font-awesome.min.css', array(), MULTIVAS__VERSION, 'all');
	wp_enqueue_style('slick-css', get_template_directory_uri() . '/assets/css/slick.css', array(), MULTIVAS__VERSION, 'all');
	wp_enqueue_style('bootstrap-css', get_template_directory_uri() . '/assets/css/bootstrap.min.css', array(), MULTIVAS__VERSION, 'all');
	wp_enqueue_style('multivas-style', get_stylesheet_uri(), array(), MULTIVAS__VERSION);
	wp_enqueue_style('responsive-css', get_template_directory_uri() . '/assets/css/responsive.css', array(), MULTIVAS__VERSION, 'all');

	// Register Scripts
	wp_enqueue_script('bootstrap-bundle-js', get_template_directory_uri() . '/assets/js/bootstrap.bundle.min.js', array('jquery'), MULTIVAS__VERSION, true);
	wp_enqueue_script('slick-js', get_template_directory_uri() . '/assets/js/slick.min.js', array('jquery'), MULTIVAS__VERSION, true);
	wp_enqueue_script('slick-animation-js', get_template_directory_uri() . '/assets/js/slick-animation.min.js', array('jquery'), MULTIVAS__VERSION, true);
	wp_enqueue_script('settings-js', get_template_directory_uri() . '/js/settings.js', array('jquery'), MULTIVAS__VERSION, true);
	wp_enqueue_script('multivas-navigation', get_template_directory_uri() . '/js/navigation.js', array(), MULTIVAS__VERSION, true);
	
	if (is_singular() && comments_open() && get_option('thread_comments')) {
		wp_enqueue_script('comment-reply');
	}
}
add_action('wp_enqueue_scripts', 'multivas_scripts');

/**
 * Register Bootstrap Nav Walker
 */
require get_template_directory() . '/inc/class-wp-bootstrap-navwalker.php';

/**
 * Implement the Custom Header feature.
 */
require get_template_directory() . '/inc/custom-header.php';

/**
 * Custom template tags for this theme.
 */
require get_template_directory() . '/inc/template-tags.php';

/**
 * Functions which enhance the theme by hooking into WordPress.
 */
require get_template_directory() . '/inc/template-functions.php';

/**
 * Customizer additions.
 */
require get_template_directory() . '/inc/customizer.php';

/**
 * Include webfont-loader
 */
require_once get_theme_file_path('inc/wptt-webfont-loader.php');

/**
 * Load Jetpack compatibility file.
 */
if (defined('JETPACK__VERSION')) {
	require get_template_directory() . '/inc/jetpack.php';
}
