<?php

/**
 * Functions which enhance the theme by hooking into WordPress
 *
 * @package mirrors_interior
 */

function mirrors_interior_body_classes( $mirrors_interior_classes ) {
	// Adds a class of hfeed to non-singular pages.
	if ( ! is_singular() ) {
		$mirrors_interior_classes[] = 'hfeed';
	}

	// Adds a class of no-sidebar when there is no sidebar present.
	if ( ! is_active_sidebar( 'sidebar-1' ) ) {
		$mirrors_interior_classes[] = 'no-sidebar';
	}

	$mirrors_interior_classes[] = mirrors_interior_sidebar_layout();

	return $mirrors_interior_classes;
}
add_filter( 'body_class', 'mirrors_interior_body_classes' );

/**
 * Add a pingback url auto-discovery header for single posts, pages, or attachments.
 */
function mirrors_interior_pingback_header() {
	if ( is_singular() && pings_open() ) {
		printf( '<link rel="pingback" href="%s">', esc_url( get_bloginfo( 'pingback_url' ) ) );
	}
}
add_action( 'wp_head', 'mirrors_interior_pingback_header' );


/**
 * Get all posts for customizer Post content type.
 */
function mirrors_interior_get_post_choices() {
	$mirrors_interior_choices = array( '' => esc_html__( '--Select--', 'mirrors-interior' ) );
	$mirrors_interior_args    = array( 'numberposts' => -1 );
	$mirrors_interior_posts   = get_posts( $mirrors_interior_args );

	foreach ( $mirrors_interior_posts as $mirrors_interior_post ) {
		$mirrors_interior_id             = $mirrors_interior_post->ID;
		$mirrors_interior_title          = $mirrors_interior_post->post_title;
		$mirrors_interior_choices[ $mirrors_interior_id ] = $mirrors_interior_title;
	}

	return $mirrors_interior_choices;
}

/**
 * Get all pages for customizer Page content type.
 */
function mirrors_interior_get_page_choices() {
	$mirrors_interior_choices = array( '' => esc_html__( '--Select--', 'mirrors-interior' ) );
	$mirrors_interior_pages   = get_pages();

	foreach ( $mirrors_interior_pages as $mirrors_interior_page ) {
		$mirrors_interior_choices[ $mirrors_interior_page->ID ] = $mirrors_interior_page->post_title;
	}

	return $mirrors_interior_choices;
}

if ( ! function_exists( 'mirrors_interior_excerpt_length' ) ) :
	/**
	 * Excerpt length.
	 */
	function mirrors_interior_excerpt_length( $mirrors_interior_length ) {
		if ( is_admin() ) {
			return $mirrors_interior_length;
		}

		return get_theme_mod( 'mirrors_interior_excerpt_length', 20 );
	}
endif;
add_filter( 'excerpt_length', 'mirrors_interior_excerpt_length', 999 );

if ( ! function_exists( 'mirrors_interior_excerpt_more' ) ) :
	/**
	 * Excerpt more.
	 */
	function mirrors_interior_excerpt_more( $mirrors_interior_more ) {
		if ( is_admin() ) {
			return $mirrors_interior_more;
		}

		return '&hellip;';
	}
endif;
add_filter( 'excerpt_more', 'mirrors_interior_excerpt_more' );

if ( ! function_exists( 'mirrors_interior_sidebar_layout' ) ) {
	/**
	 * Get sidebar layout.
	 */
	function mirrors_interior_sidebar_layout() {
		$mirrors_interior_sidebar_position      = get_theme_mod( 'mirrors_interior_sidebar_position', 'right-sidebar' );
		$mirrors_interior_sidebar_position_post = get_theme_mod( 'mirrors_interior_post_sidebar_position', 'right-sidebar' );
		$mirrors_interior_sidebar_position_page = get_theme_mod( 'mirrors_interior_page_sidebar_position', 'right-sidebar' );

		if ( is_single() ) {
			$mirrors_interior_sidebar_position = $mirrors_interior_sidebar_position_post;
		} elseif ( is_page() ) {
			$mirrors_interior_sidebar_position = $mirrors_interior_sidebar_position_page;
		}

		return $mirrors_interior_sidebar_position;
	}
}

if ( ! function_exists( 'mirrors_interior_is_sidebar_enabled' ) ) {
	/**
	 * Check if sidebar is enabled.
	 */
	function mirrors_interior_is_sidebar_enabled() {
		$mirrors_interior_sidebar_position      = get_theme_mod( 'mirrors_interior_sidebar_position', 'right-sidebar' );
		$mirrors_interior_sidebar_position_post = get_theme_mod( 'mirrors_interior_post_sidebar_position', 'right-sidebar' );
		$mirrors_interior_sidebar_position_page = get_theme_mod( 'mirrors_interior_page_sidebar_position', 'right-sidebar' );

		$mirrors_interior_sidebar_enabled = true;
		if ( is_home() || is_archive() || is_search() ) {
			if ( 'no-sidebar' === $mirrors_interior_sidebar_position ) {
				$mirrors_interior_sidebar_enabled = false;
			}
		} elseif ( is_single() ) {
			if ( 'no-sidebar' === $mirrors_interior_sidebar_position || 'no-sidebar' === $mirrors_interior_sidebar_position_post ) {
				$mirrors_interior_sidebar_enabled = false;
			}
		} elseif ( is_page() ) {
			if ( 'no-sidebar' === $mirrors_interior_sidebar_position || 'no-sidebar' === $mirrors_interior_sidebar_position_page ) {
				$mirrors_interior_sidebar_enabled = false;
			}
		}
		return $mirrors_interior_sidebar_enabled;
	}
}

if ( ! function_exists( 'mirrors_interior_get_homepage_sections ' ) ) {
	/**
	 * Returns homepage sections.
	 */
	function mirrors_interior_get_homepage_sections() {
		$mirrors_interior_sections = array(
			'banner'  => esc_html__( 'Banner Section', 'mirrors-interior' ),
			'trending-product' => esc_html__( 'Trending Product Section', 'mirrors-interior' ),
		);
		return $mirrors_interior_sections;
	}
}

/**
 * Renders customizer section link
 */
function mirrors_interior_section_link( $section_id ) {
	$mirrors_interior_section_name      = str_replace( 'mirrors_interior_', ' ', $section_id );
	$mirrors_interior_section_name      = str_replace( '_', ' ', $mirrors_interior_section_name );
	$mirrors_interior_starting_notation = '#';
	?>
	<span class="section-link">
		<span class="section-link-title"><?php echo esc_html( $mirrors_interior_section_name ); ?></span>
	</span>
	<style type="text/css">
		<?php echo $mirrors_interior_starting_notation . $section_id; // phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped ?>:hover .section-link {
			visibility: visible;
		}
	</style>
	<?php
}

/**
 * Adds customizer section link css
 */
function mirrors_interior_section_link_css() {
	if ( is_customize_preview() ) {
		?>
		<style type="text/css">
			.section-link {
				visibility: hidden;
				background-color: black;
				position: relative;
				top: 80px;
				z-index: 99;
				left: 40px;
				color: #fff;
				text-align: center;
				font-size: 20px;
				border-radius: 10px;
				padding: 20px 10px;
				text-transform: capitalize;
			}

			.section-link-title {
				padding: 0 10px;
			}

			.banner-section {
				position: relative;
			}

			.banner-section .section-link {
				position: absolute;
				top: 100px;
			}
		</style>
		<?php
	}
}
add_action( 'wp_head', 'mirrors_interior_section_link_css' );

/**
 * Breadcrumb.
 */
function mirrors_interior_breadcrumb( $mirrors_interior_args = array() ) {
	if ( ! get_theme_mod( 'mirrors_interior_enable_breadcrumb', true ) ) {
		return;
	}

	$mirrors_interior_args = array(
		'show_on_front' => false,
		'show_title'    => true,
		'show_browse'   => false,
	);
	breadcrumb_trail( $mirrors_interior_args );
}
add_action( 'mirrors_interior_breadcrumb', 'mirrors_interior_breadcrumb', 10 );

/**
 * Add separator for breadcrumb trail.
 */
function mirrors_interior_breadcrumb_trail_print_styles() {
	$mirrors_interior_breadcrumb_separator = get_theme_mod( 'mirrors_interior_breadcrumb_separator', '/' );

	$mirrors_interior_style = '
		.trail-items li::after {
			content: "' . $mirrors_interior_breadcrumb_separator . '";
		}'; // phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped

	$mirrors_interior_style = apply_filters( 'mirrors_interior_breadcrumb_trail_inline_style', trim( str_replace( array( "\r", "\n", "\t", '  ' ), '', $mirrors_interior_style ) ) );

	if ( $mirrors_interior_style ) {
		echo "\n" . '<style type="text/css" id="breadcrumb-trail-css">' . $mirrors_interior_style . '</style>' . "\n"; // phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped
	}
}
add_action( 'wp_head', 'mirrors_interior_breadcrumb_trail_print_styles' );

/**
 * Pagination for archive.
 */
function mirrors_interior_render_posts_pagination() {
	$mirrors_interior_is_pagination_enabled = get_theme_mod( 'mirrors_interior_enable_pagination', true );
	if ( $mirrors_interior_is_pagination_enabled ) {
		$mirrors_interior_pagination_type = get_theme_mod( 'mirrors_interior_pagination_type', 'default' );
		if ( 'default' === $mirrors_interior_pagination_type ) :
			the_posts_navigation();
		else :
			the_posts_pagination();
		endif;
	}
}
add_action( 'mirrors_interior_posts_pagination', 'mirrors_interior_render_posts_pagination', 10 );

/**
 * Pagination for single post.
 */
function mirrors_interior_render_post_navigation() {
	the_post_navigation(
		array(
			'prev_text' => '<span>&#10229;</span> <span class="nav-title">%title</span>',
			'next_text' => '<span class="nav-title">%title</span> <span>&#10230;</span>',
		)
	);
}
add_action( 'mirrors_interior_post_navigation', 'mirrors_interior_render_post_navigation' );

/**
 * Adds footer copyright text.
 */
function mirrors_interior_output_footer_copyright_content() {
    $mirrors_interior_theme_data = wp_get_theme();
    $mirrors_interior_copyright_text = get_theme_mod('mirrors_interior_footer_copyright_text');

    if (!empty($mirrors_interior_copyright_text)) {
        $mirrors_interior_text = $mirrors_interior_copyright_text;
    } else {
        $mirrors_interior_default_text = '<a href="'. esc_url(__('https://asterthemes.com/products/free-interior-wordpress-theme','mirrors-interior')) . '" target="_blank"> ' . esc_html($mirrors_interior_theme_data->get('Name')) . '</a>' . '&nbsp;' . esc_html__('by', 'mirrors-interior') . '&nbsp;<a target="_blank" href="' . esc_url($mirrors_interior_theme_data->get('AuthorURI')) . '">' . esc_html(ucwords($mirrors_interior_theme_data->get('Author'))) . '</a>';
        /* translators: %s: WordPress.org URL */
		$mirrors_interior_default_text .= sprintf(esc_html__(' | Powered by %s', 'mirrors-interior'), '<a href="' . esc_url(__('https://wordpress.org/', 'mirrors-interior')) . '" target="_blank">WordPress</a>. ');

        $mirrors_interior_text = $mirrors_interior_default_text;
    }
    ?>
    <span><?php echo wp_kses_post($mirrors_interior_text); ?></span>
    <?php
}
add_action('mirrors_interior_footer_copyright', 'mirrors_interior_output_footer_copyright_content');

/* Footer Social Icons */ 
function mirrors_interior_footer_social_links() {

    if ( get_theme_mod('mirrors_interior_enable_footer_icon_section', true) ) {

            ?>
            <div class="socialicons">
				<div class="asterthemes-wrapper">
					<?php if ( get_theme_mod('mirrors_interior_footer_facebook_link', 'https://www.facebook.com/') != '' ) { ?>
						<a target="_blank" href="<?php echo esc_url(get_theme_mod('mirrors_interior_footer_facebook_link', 'https://www.facebook.com/')); ?>">
							<i class="<?php echo esc_attr(get_theme_mod('mirrors_interior_facebook_icon', 'fab fa-facebook-f')); ?>"></i>
							<span class="screen-reader-text"><?php esc_html_e('Facebook', 'mirrors-interior'); ?></span>
						</a>
					<?php } ?>
					<?php if ( get_theme_mod('mirrors_interior_footer_twitter_link', 'https://x.com/') != '' ) { ?>
						<a target="_blank" href="<?php echo esc_url(get_theme_mod('mirrors_interior_footer_twitter_link', 'https://x.com/')); ?>">
							<i class="<?php echo esc_attr(get_theme_mod('mirrors_interior_twitter_icon', 'fab fa-twitter')); ?>"></i>
							<span class="screen-reader-text"><?php esc_html_e('Twitter', 'mirrors-interior'); ?></span>
						</a>
					<?php } ?>
					<?php if ( get_theme_mod('mirrors_interior_footer_instagram_link', 'https://www.instagram.com/') != '' ) { ?>
						<a target="_blank" href="<?php echo esc_url(get_theme_mod('mirrors_interior_footer_instagram_link', 'https://www.instagram.com/')); ?>">
							<i class="<?php echo esc_attr(get_theme_mod('mirrors_interior_instagram_icon', 'fab fa-instagram')); ?>"></i>
							<span class="screen-reader-text"><?php esc_html_e('Instagram', 'mirrors-interior'); ?></span>
						</a>
					<?php } ?>
					<?php if ( get_theme_mod('mirrors_interior_footer_linkedin_link', 'https://in.linkedin.com/') != '' ) { ?>
						<a target="_blank" href="<?php echo esc_url(get_theme_mod('mirrors_interior_footer_linkedin_link', 'https://in.linkedin.com/')); ?>">
							<i class="<?php echo esc_attr(get_theme_mod('mirrors_interior_linkedin_icon', 'fab fa-linkedin')); ?>"></i>
							<span class="screen-reader-text"><?php esc_html_e('Linkedin', 'mirrors-interior'); ?></span>
						</a>
					<?php } ?>
					<?php if ( get_theme_mod('mirrors_interior_footer_youtube_link', 'https://www.youtube.com/') != '' ) { ?>
						<a target="_blank" href="<?php echo esc_url(get_theme_mod('mirrors_interior_footer_youtube_link', 'https://www.youtube.com/')); ?>">
							<i class="<?php echo esc_attr(get_theme_mod('mirrors_interior_youtube_icon', 'fab fa-youtube')); ?>"></i>
							<span class="screen-reader-text"><?php esc_html_e('Youtube', 'mirrors-interior'); ?></span>
						</a>
					<?php } ?>
				</div>
            </div>
            <?php
    }
}
add_action('wp_footer', 'mirrors_interior_footer_social_links');

/**
 * GET START FUNCTION
 */

function mirrors_interior_getpage_css($hook) {
	wp_enqueue_script( 'mirrors-interior-admin-script', get_template_directory_uri() . '/resource/js/mirrors-interior-admin-notice-script.js', array( 'jquery' ) );
    wp_localize_script( 'mirrors-interior-admin-script', 'mirrors_interior_ajax_object',
        array( 'ajax_url' => admin_url( 'admin-ajax.php' ) )
    );
    wp_enqueue_style( 'mirrors-interior-notice-style', get_template_directory_uri() . '/resource/css/notice.css' );
}

add_action( 'admin_enqueue_scripts', 'mirrors_interior_getpage_css' );


add_action('wp_ajax_mirrors_interior_dismissable_notice', 'mirrors_interior_dismissable_notice');
function mirrors_interior_switch_theme() {
    delete_user_meta(get_current_user_id(), 'mirrors_interior_dismissable_notice');
}
add_action('after_switch_theme', 'mirrors_interior_switch_theme');
function mirrors_interior_dismissable_notice() {
    update_user_meta(get_current_user_id(), 'mirrors_interior_dismissable_notice', true);
    die();
}

function mirrors_interior_deprecated_hook_admin_notice() {
    global $mirrors_interior_pagenow;
    
    // Check if the current page is the one where you don't want the notice to appear
    if ( $mirrors_interior_pagenow === 'themes.php' && isset( $_GET['page'] ) && $_GET['page'] === 'mirrors-interior-getting-started' ) {
        return;
    }

    $mirrors_interior_dismissed = get_user_meta( get_current_user_id(), 'mirrors_interior_dismissable_notice', true );
    if ( !$mirrors_interior_dismissed) { ?>
        <div class="getstrat updated notice notice-success is-dismissible notice-get-started-class">
            <div class="at-admin-content" >
                <h2><?php esc_html_e('Welcome to Mirror Interior', 'mirrors-interior'); ?></h2>
                <p><?php _e('Explore the features of our Pro Theme and take your Mirror to the next level.', 'mirrors-interior'); ?></p>
                <p ><?php _e('Get Started With Theme By Clicking On Getting Started.', 'mirrors-interior'); ?><p>
                <div style="display: flex; justify-content: center; align-items:center; flex-wrap: wrap; gap: 5px">
                    <a class="admin-notice-btn button button-primary button-hero" href="<?php echo esc_url( admin_url( 'themes.php?page=mirrors-interior-getting-started' )); ?>"><?php esc_html_e( 'Get started', 'mirrors-interior' ) ?></a>
                    <a  class="admin-notice-btn button button-primary button-hero" target="_blank" href="https://demo.asterthemes.com/mirrors-interior"><?php esc_html_e('View Demo', 'mirrors-interior') ?></a>
                    <a  class="admin-notice-btn button button-primary button-hero" target="_blank" href="https://asterthemes.com/products/mirrors-interior-wordpress-theme"><?php esc_html_e('Buy Now', 'mirrors-interior') ?></a>
                    <a  class="admin-notice-btn button button-primary button-hero" target="_blank" href="<?php echo esc_url( MIRRORS_INTERIOR_BUNDLE_PAGE ); ?>"><?php esc_html_e('Get Bundle', 'mirrors-interior') ?></a>
                </div>
            </div>
            <div class="at-admin-image">
                <img style="width: 100%;max-width: 320px;line-height: 40px;display: inline-block;vertical-align: top;border: 2px solid #ddd;border-radius: 4px;" src="<?php echo esc_url(get_stylesheet_directory_uri()) .'/screenshot.png'; ?>" />
            </div>
        </div>
    <?php }
}

add_action( 'admin_notices', 'mirrors_interior_deprecated_hook_admin_notice' );


//Admin Notice For Getstart
function mirrors_interior_ajax_notice_handler() {
    if ( isset( $_POST['type'] ) ) {
        $type = sanitize_text_field( wp_unslash( $_POST['type'] ) );
        update_option( 'dismissed-' . $type, TRUE );
    }
}

if ( ! function_exists( 'mirrors_interior_footer_widget' ) ) :
	function mirrors_interior_footer_widget() {
		$mirrors_interior_footer_widget_column = get_theme_mod('mirrors_interior_footer_widget_column','4');

		$mirrors_interior_column_class = '';
		if ($mirrors_interior_footer_widget_column == '1') {
			$mirrors_interior_column_class = 'one-column';
		} elseif ($mirrors_interior_footer_widget_column == '2') {
			$mirrors_interior_column_class = 'two-columns';
		} elseif ($mirrors_interior_footer_widget_column == '3') {
			$mirrors_interior_column_class = 'three-columns';
		} else {
			$mirrors_interior_column_class = 'four-columns';
		}
	
		if($mirrors_interior_footer_widget_column !== ''): 
		?>
		<div class="dt_footer-widgets <?php echo esc_attr($mirrors_interior_column_class); ?>">
			<div class="footer-widgets-column">
				<?php
				$footer_widgets_active = false;

				// Loop to check if any footer widget is active
				for ($i = 1; $i <= $mirrors_interior_footer_widget_column; $i++) {
					if (is_active_sidebar('mirrors-interior-footer-widget-' . $i)) {
						$footer_widgets_active = true;
						break;
					}
				}

				if ($footer_widgets_active) {
					// Display active footer widgets
					for ($i = 1; $i <= $mirrors_interior_footer_widget_column; $i++) {
						if (is_active_sidebar('mirrors-interior-footer-widget-' . $i)) : ?>
							<div class="footer-one-column">
								<?php dynamic_sidebar('mirrors-interior-footer-widget-' . $i); ?>
							</div>
						<?php endif;
					}
				} else {
				?>
				<div class="footer-one-column default-widgets">
					<aside id="search-2" class="widget widget_search default_footer_search">
						<div class="widget-header">
							<h4 class="widget-title"><?php esc_html_e('Search Here', 'mirrors-interior'); ?></h4>
						</div>
						<?php get_search_form(); ?>
					</aside>
				</div>
				<div class="footer-one-column default-widgets">
					<aside id="recent-posts-2" class="widget widget_recent_entries">
						<h2 class="widget-title"><?php esc_html_e('Recent Posts', 'mirrors-interior'); ?></h2>
						<ul>
							<?php
							$recent_posts = wp_get_recent_posts(array(
								'numberposts' => 5,
								'post_status' => 'publish',
							));
							foreach ($recent_posts as $post) {
								echo '<li><a href="' . esc_url(get_permalink($post['ID'])) . '">' . esc_html($post['post_title']) . '</a></li>';
							}
							wp_reset_query();
							?>
						</ul>
					</aside>
				</div>
				<div class="footer-one-column default-widgets">
					<aside id="recent-comments-2" class="widget widget_recent_comments">
						<h2 class="widget-title"><?php esc_html_e('Recent Comments', 'mirrors-interior'); ?></h2>
						<ul>
							<?php
							$recent_comments = get_comments(array(
								'number' => 5,
								'status' => 'approve',
							));
							foreach ($recent_comments as $comment) {
								echo '<li><a href="' . esc_url(get_comment_link($comment)) . '">' .
									/* translators: %s: details. */
									sprintf(esc_html__('Comment on %s', 'mirrors-interior'), get_the_title($comment->comment_post_ID)) .
									'</a></li>';
							}
							?>
						</ul>
					</aside>
				</div>
				<div class="footer-one-column default-widgets">
					<aside id="calendar-2" class="widget widget_calendar">
						<h2 class="widget-title"><?php esc_html_e('Calendar', 'mirrors-interior'); ?></h2>
						<?php get_calendar(); ?>
					</aside>
				</div>
			</div>
			<?php } ?>
		</div>
		<?php
		endif;
	}
	endif;
add_action( 'mirrors_interior_footer_widget', 'mirrors_interior_footer_widget' );

function mirrors_interior_footer_text_transform_css() {
    $mirrors_interior_footer_text_transform = get_theme_mod('mirrors_interior_footer_text_transform', 'none');
    ?>
    <style type="text/css">
        .site-footer h4,footer#colophon h2.wp-block-heading,footer#colophon .widgettitle,footer#colophon .widget-title {
            text-transform: <?php echo esc_html($mirrors_interior_footer_text_transform); ?>;
        }
    </style>
    <?php
}
add_action('wp_head', 'mirrors_interior_footer_text_transform_css');