<?php
/**
 * Minimalblog Theme Customizer
 *
 * @package minimalblog
 */

if ( ! defined( 'ABSPATH' ) ) {
    exit; // Exit if accessed directly
}

/**
 * Theme Customizer setup
 *
 * @param WP_Customize_Manager $wp_customize Theme Customizer object.
 */
function minimalblog_customize_register( WP_Customize_Manager $wp_customize ) {
    // Enable live preview for default settings
    $settings = [
        'blogname',
        'blogdescription',
        'header_textcolor'
    ];

    foreach ( $settings as $setting ) {
        $wp_customize->get_setting( $setting )->transport = 'postMessage';
    }

    // Add selective refresh partials
    if ( isset( $wp_customize->selective_refresh ) ) {
        $partials = [
            'blogname' => [
                'selector'        => '.site-title a',
                'render_callback' => 'minimalblog_customize_partial_blogname',
            ],
            'blogdescription' => [
                'selector'        => '.site-description',
                'render_callback' => 'minimalblog_customize_partial_blogdescription',
            ],
        ];

        foreach ( $partials as $id => $args ) {
            $wp_customize->selective_refresh->add_partial( $id, $args );
        }
    }
}
add_action( 'customize_register', 'minimalblog_customize_register' );

/**
 * Render the site title for selective refresh
 *
 * @return void
 */
function minimalblog_customize_partial_blogname() {
    bloginfo( 'name' );
}

/**
 * Render the site description for selective refresh
 *
 * @return void
 */
function minimalblog_customize_partial_blogdescription() {
    bloginfo( 'description' );
}

/**
 * Enqueue customizer preview scripts
 *
 * @return void
 */
function minimalblog_customize_preview_js() {
    wp_enqueue_script(
        'minimalblog-customizer',
        get_template_directory_uri() . '/js/customizer.js',
        ['customize-preview'],
        wp_get_theme()->get( 'Version' ), // Use theme version for cache busting
        true
    );
}
add_action( 'customize_preview_init', 'minimalblog_customize_preview_js' );

/**
 * Custom Customizer Controls
 */
function minimalblog_register_custom_controls( WP_Customize_Manager $wp_customize ) {
    /**
     * Multiple Select Control
     */
    class Minimalblog_Customize_Control_Multiple_Select extends WP_Customize_Control {
        public $type = 'multiple-select';

        public function render_content() {
            if ( empty( $this->choices ) ) {
                return;
            }
            ?>
            <label>
                <span class="customize-control-title"><?php echo esc_html( $this->label ); ?></span>
                <select <?php $this->link(); ?> multiple="multiple" style="height: 100%;">
                    <?php
                    foreach ( $this->choices as $value => $label ) {
                        $selected = in_array( $value, (array) $this->value(), true ) ? 'selected="selected"' : '';
                        printf(
                            '<option value="%s" %s>%s</option>',
                            esc_attr( $value ),
                            $selected,
                            esc_html( $label )
                        );
                    }
                    ?>
                </select>
            </label>
            <?php
        }
    }

    /**
     * Note Control
     */
    class Minimalblog_Note_Control extends WP_Customize_Control {
        public $type = 'note';

        public function render_content() {
            ?>
            <span class="customize-control-title"><?php echo wp_kses_post( $this->label ); ?></span>
            <?php if ( $this->description ) : ?>
                <span class="description customize-control-description"><?php echo wp_kses_post( $this->description ); ?></span>
            <?php endif; ?>
            <?php
        }
    }

    /**
     * Separator Control
     */
    class Minimalblog_Customize_Control_Separator extends WP_Customize_Control {
        public $type = 'separator';

        public function render_content() {
            ?>
            <hr>
            <h4 style="text-align: center; text-transform: uppercase;"><?php echo esc_html( $this->label ); ?></h4>
            <hr>
            <?php
        }
    }
}
add_action( 'customize_register', 'minimalblog_register_custom_controls', 0 ); // Higher priority

/**
 * Kirki Configuration and Panel Setup
 */
function minimalblog_setup_kirki() {
    if ( ! class_exists( 'Kirki' ) ) {
        return;
    }

    Kirki::add_config( 'minimalblog_config', [
        'capability'  => 'edit_theme_options',
        'option_type' => 'theme_mod',
    ]);

    Kirki::add_panel( 'minimalblog', [
        'title'    => esc_html__( 'Minimal Blog Options', 'minimalblog' ),
        'priority' => 10,
    ]);

    // List of customizer settings files
    $settings_files = [
        'header-options',
        'social-links',
        'featured-options',
        'category-slider',
        'banner-options',
        'blog-options',
        'typography',
        'preloader',
        'contact',
        'colors',
        'footer-options',
    ];

    // Include settings files
    foreach ( $settings_files as $file ) {
        $path = get_theme_file_path( "inc/themeoptions/{$file}.php" );
        if ( file_exists( $path ) ) {
            require_once $path;
        }
    }
}
add_action( 'init', 'minimalblog_setup_kirki' );