<?php
/**
 * Dashboard Page
 *
 * @package Minimal Blog
 */
if (!class_exists('Wpinterface_Dashboard')) {
    /**
     * Main class.
     *
     * @since 1.1.7
     */
    class Wpinterface_Dashboard
    {
        /**
         * Theme name.
         *
         * @var string $theme_name Theme name.
         *
         * @since 1.1.7
         */
        private $theme_name;
        /**
         * Theme slug.
         *
         * @var string $theme_slug Theme slug.
         *
         * @since 1.1.7
         */
        private $theme_slug;
        /**
         * Current theme object.
         *
         * @var WP_Theme $theme Current theme.
         */
        private $theme;
        /**
         * Some globals url
         */
        private $shop_url = 'https://wpinterface.com';
        private $doc_url = 'https://docs.wpinterface.com/minimal-blog';
        private $support_url = 'https://wpinterface.com/support/';
        private $demo_url = 'https://preview.wpinterface.com/minimal-blog/';
        private $theme_version = '';
        private $menu_name = '';
        private $page_name = '';
        private $page_slug = '';
        /**
         * Constructor.
         *
         * @since 1.1.7
         */
        public function __construct()
        {
            global $pagenow;
            $theme = wp_get_theme();
            if (is_child_theme()) {
                $this->theme_name = $theme->parent()->get('Name');
                $this->theme = $theme->parent();
            } else {
                $this->theme_name = $theme->get('Name');
                $this->theme = $theme->parent();
            }
            $this->theme_version = $theme->get('Version');
            $this->theme_slug = $theme->get_template();
            $this->menu_name = sprintf(esc_html__('%s', 'minimal-blog'), $this->theme_name);
            $this->page_name = sprintf(esc_html__('%s', 'minimal-blog'), $this->theme_name);
            $this->page_slug = $this->theme_slug . '-welcome';
            if (current_user_can('manage_options')) {
                add_action('admin_menu', array($this, 'add_menu'));
                add_action('admin_bar_menu', array($this, 'add_bar_menu'), 999);
                if (($pagenow == 'index.php') || ($pagenow == 'plugins.php') || ($pagenow == 'themes.php')) {
                    add_action('admin_notices', array($this, 'display_admin_notice'), 999);
                }
            }
            add_action('admin_enqueue_scripts', array($this, 'enqueue_scripts'));
            add_action('wp_ajax_minimal_blog_dismiss', array($this, 'dismiss_callback'));
            add_action('wp_ajax_nopriv_minimal_blog_dismiss', array($this, 'dismiss_callback'));
        }
        /**
         * Add theme dashboard page.
         *
         * @since 1.1.7
         */
        public function add_menu()
        {
            add_theme_page(
                $this->menu_name,
                $this->page_name,
                'manage_options',
                $this->page_slug,
                array($this, 'render_main_page'),
                $position = 1
            );
        }
        /**
         * Add link in menu bar
         *
         * @since 1.1.7
         */
        public function add_bar_menu()
        {
            global $wp_admin_bar;
            if (!is_super_admin() || !is_admin_bar_showing()) {
                return;
            }
            $wp_admin_bar->add_menu(array(
                'parent' => 'site-name',
                'id' => $this->page_slug,
                'title' => $this->theme_name,
                'href' => admin_url("themes.php?page=$this->page_slug")
            ));
        }
        /**
         * Display notice in admin area
         *
         * @since 1.0.1
         */
        public function display_admin_notice()
        {
            $screen_id = null;
            $current_screen = get_current_screen();
            if ($current_screen) {
                $screen_id = $current_screen->id;
            }
            $user_id = get_current_user_id();
            $dismiss_status = get_user_meta($user_id, 'minimal_blog_dismiss_status', true);
            $theme_info = wp_get_theme();
            $theme_name = $theme_info->__get('Name');
            ?>
            <?php if (current_user_can('edit_theme_options') && 1 !== absint($dismiss_status)) : ?>
            <div class="minimal-blog-about-notice notice notice-about">
                <p class="notice-text">
                    <?php
                    $current_user = wp_get_current_user();
                    printf(
                    /* Translators: %1$s current user display name., %2$s this theme name. */
                        esc_html__(
                            'Dear %1$s, We hope you are enjoying using our %2$s WordPress theme. We are constantly working to improve and enhance the user experience. If you need any assistance, feel free to reach out.',
                            'minimal-blog'
                        ),
                        '<strong>' . esc_html($current_user->display_name) . '</strong>',
                        '<strong>' . esc_html($theme_name) . '</strong>'
                    );
                    ?>
                </p>
                <p class="notice-text">
                    <?php esc_html_e('Thank you for choosing our theme. Happy blogging!', 'minimal-blog'); ?>
                </p>
                <div class="dashboard-btn-group">
                    <a href="<?php echo $this->shop_url; ?>/themes/minimal-blog/#choose-pricing-plan" target="_blank" rel="noopener noreferrer"
                       class="button button-primary button-upgrade">
                        <span class="dashicons dashicons-thumbs-up"></span>
                        <span><?php esc_html_e('Upgrade to Pro', 'minimal-blog'); ?></span>
                    </a>
                    <a href="<?php echo esc_url(admin_url("themes.php?page=$this->page_slug")) ?>"
                       class="button button-primary">
                        <span class="dashicons dashicons-superhero"></span>
                        <span><?php echo sprintf(esc_html__('Get started with %1$s', 'minimal-blog'), $this->theme_name) ?></span>
                    </a>
                    <a href="<?php echo $this->demo_url; ?>" target="_blank" rel="noopener noreferrer" class="button button-primary">
                        <span class="dashicons dashicons-welcome-view-site"></span>
                        <span><?php esc_html_e('Live Preview', 'minimal-blog'); ?></span>
                    </a>
                    <a href="#" class="btn-dismiss" data-userid="<?php echo esc_attr(get_current_user_id()); ?>"
                       data-nonce="<?php echo esc_attr(wp_create_nonce('minimal_blog_dismiss_nonce')); ?>">
                        <?php esc_html_e('Dismiss this notice.', 'minimal-blog'); ?>
                    </a>
                </div>
            </div>
        <?php endif; ?>
            <?php
        }
        /**
         * Dashboard Page
         *
         * @since 1.1.7
         */
        public function render_main_page()
        {
            ?>
            <div class="wrap about-wrap minimal-blog-about-wrap">
                <header class="theme-dashboard-header">
                    <h1 class="dashboard-header-title">
                        <span class="dashboard-theme-name"><?php echo esc_html($this->theme_name); ?></span>
                        <span class="dashboard-theme-version"><?php echo esc_html($this->theme_version); ?></span>
                    </h1>
                    <p class="about-text">
                        <?php _e('Great job getting our theme up and running! At WPInterface, we go above and beyond to ensure our customers have the best possible experience. As a token of our appreciation for choosing our theme, we\'ve gathered all the important information you need in one easy-to-access spot. Thank you for entrusting us with your website design!', 'minimal-blog'); ?>
                    </p>
                </header>
                <div class="dashboard-btn-group">
                    <a href="<?php echo $this->shop_url; ?>/themes/minimal-blog/#choose-pricing-plan" target="_blank" rel="noopener noreferrer"
                       class="button button-primary button-upgrade">
                        <span class="dashicons dashicons-thumbs-up"></span>
                        <span><?php esc_html_e('Upgrade to Pro', 'minimal-blog'); ?></span>
                    </a>
                    <a href="<?php echo $this->shop_url; ?>/themes/minimal-blog/" class="button button-primary"
                       target="_blank" rel="noopener noreferrer">
                        <span class="dashicons dashicons-info"></span>
                        <span><?php _e('Theme Details', 'minimal-blog'); ?></span>
                    </a>
                    <a href="<?php echo $this->demo_url; ?>" class="button button-primary" target="_blank" rel="noopener noreferrer">
                        <span class="dashicons dashicons-welcome-view-site"></span>
                        <span><?php _e('Live Preview', 'minimal-blog'); ?></span>
                    </a>
                    <a href="<?php echo $this->doc_url; ?>" class="button button-primary" target="_blank" rel="noopener noreferrer">
                        <span class="dashicons dashicons-book"></span>
                        <span><?php _e('View Documentation', 'minimal-blog'); ?></span>
                    </a>
                </div>
                <?php $current_tab = isset($_GET['tab']) ? wp_unslash($_GET['tab']) : 'getting-started'; ?>
                <div class="nav-tab-wrapper">
                    <a href="<?php echo esc_url(admin_url('themes.php?page=' . $this->page_slug)) . '&tab=getting-started' ?>"
                       class="nav-tab tab-getting-started <?php echo ($current_tab == 'getting-started') ? ' nav-tab-active' : ''; ?>">
                        <?php _e('Getting Started', 'minimal-blog'); ?>
                    </a>
                    <a href="<?php echo esc_url(admin_url('themes.php?page=' . $this->page_slug)) . '&tab=comparison-table' ?>"
                       class="nav-tab tab-comparison-table <?php echo ($current_tab == 'comparison-table') ? ' nav-tab-active' : ''; ?>">
                        <?php _e('Free Vs Pro', 'minimal-blog'); ?>
                    </a>
                </div>
                <?php
                $current_tab = isset($_GET['tab']) ? wp_unslash($_GET['tab']) : 'getting-started';
                $method = str_replace('-', '_', esc_attr($current_tab));
                if (method_exists($this, $method)) {
                    $this->{$method}();
                } else {
                    printf(esc_html__('%s() method does not exist.', 'minimal-blog'), $method);
                }
                ?>
            </div><!-- minimal-blog-about-wrap -->
            <?php
        }
        /**
         * Render getting started.
         *
         * @since 1.1.7
         */
        public function getting_started()
        {
            ?>
            <div id="getting-started" class="minimal-blog-section minimal-blog-getting-started">
                <div class="col">
                    <h3>
                        <span class="dashicons dashicons-format-aside"></span>
                        <?php _e('Theme Documentation', 'minimal-blog'); ?>
                    </h3>
                    <p><?php _e('Unlock the full potential of your website with our step-by-step guide on setting up and personalizing the theme to fit your brand\'s unique style.', 'minimal-blog'); ?></p>
                    <a href="<?php echo $this->doc_url; ?>" class="button button-primary" target="_blank" rel="noopener noreferrer">
                        <?php _e('View Documentation', 'minimal-blog'); ?>
                    </a>
                </div>
                <div class="col">
                    <h3>
                        <span class="dashicons dashicons-admin-customizer"></span>
                        <?php _e('Widget Options', 'minimal-blog'); ?>
                    </h3>
                    <p><?php _e('Stand out from the crowd by using our customizable widgets and widget areas to create a truly unique and personalized website experience.', 'minimal-blog'); ?></p>
                    <a href="<?php echo admin_url('widgets.php') ?>" class="button button-primary"
                       target="_self"><?php _e('Widgets', 'minimal-blog'); ?></a>
                </div>
                <div class="col">
                    <h3>
                        <span class="dashicons dashicons-admin-customizer"></span>
                        <?php _e('Theme Options', 'minimal-blog'); ?>
                    </h3>
                    <p><?php _e('Elevate your website design with our user-friendly and extensive customizer settings, giving you complete control to make your site truly one-of-a-kind.', 'minimal-blog'); ?></p>
                    <a href="<?php echo wp_customize_url(); ?>" class="button button-primary"
                       target="_self"><?php _e('Customize', 'minimal-blog'); ?></a>
                </div>
                <div class="col">
                    <h3>
                        <span class="dashicons dashicons-welcome-view-site"></span>
                        <?php _e('Theme Preview', 'minimal-blog'); ?>
                    </h3>
                    <p><?php _e('Discover how to make your website stand out and see how our theme can be fully customized to fit your unique style and brand.', 'minimal-blog'); ?></p>
                    <a href="<?php echo $this->demo_url; ?>" class="button button-primary"
                       target="_blank" rel="noopener noreferrer"><?php _e('View Demos', 'minimal-blog'); ?></a>
                </div>
                <div class="col">
                    <h3>
                        <span class="dashicons dashicons-sos"></span>
                        <?php _e('Contact Support', 'minimal-blog'); ?>
                    </h3>
                    <p><?php _e('Get expert help and guidance. Whether you have a question, have found a bug, or have some feedback, we\'re here to assist you every step of the way.', 'minimal-blog'); ?></p>
                    <a href="<?php echo $this->support_url; ?>" class="button button-primary"
                       target="_blank" rel="noopener noreferrer"><?php _e('Contact Support', 'minimal-blog'); ?></a>
                </div>
            </div>
            <?php
        }
        /**
         *
         * @since 1.1.7
         */
        public function comparison_table()
        {
            //Free vs Pro
            $minimal_blog_panel_compare = array(
                array(
                    'title' => __('Help and Support', 'minimal-blog'),
                    'free' => __('Standard Support', 'minimal-blog'),
                    'pro' => __('Premium Dedicated Support', 'minimal-blog'),
                ),
                array(
                    'title' => __('Responsive Design', 'minimal-blog'),
                    'free' => __('yes', 'minimal-blog'),
                    'pro' => __('yes', 'minimal-blog'),
                ),
                array(
                    'title' => __('Translation Ready', 'minimal-blog'),
                    'free' => __('yes', 'minimal-blog'),
                    'pro' => __('yes', 'minimal-blog'),
                ),
                array(
                    'title' => __('Seo Optimized', 'minimal-blog'),
                    'free' => __('yes', 'minimal-blog'),
                    'pro' => __('yes', 'minimal-blog'),
                ),
                array(
                    'title' => __('Supports Post Formats', 'minimal-blog'),
                    'free' => __('yes', 'minimal-blog'),
                    'pro' => __('yes', 'minimal-blog'),
                ),
                array(
                    'title' => __('Upload your own logo', 'minimal-blog'),
                    'free' => __('yes', 'minimal-blog'),
                    'pro' => __('yes', 'minimal-blog'),
                ),
                array(
                    'title' => __('Sticky Primary Menu', 'minimal-blog'),
                    'free' => __('yes', 'minimal-blog'),
                    'pro' => __('yes', 'minimal-blog'),
                ),
                array(
                    'title' => __('Preloader Option', 'minimal-blog'),
                    'free' => __('yes', 'minimal-blog'),
                    'pro' => __('yes', 'minimal-blog'),
                ),
                array(
                    'title' => __('Color Options', 'minimal-blog'),
                    'free' => __('no', 'minimal-blog'),
                    'pro' => __('yes', 'minimal-blog'),
                ),
                array(
                    'title' => __('Dark Mode Options', 'minimal-blog'),
                    'free' => __('no', 'minimal-blog'),
                    'pro' => __('yes', 'minimal-blog'),
                ),
                array(
                    'title' => __('Font Options', 'minimal-blog'),
                    'free' => __('no', 'minimal-blog'),
                    'pro' => __('yes', 'minimal-blog'),
                ),
                array(
                    'title' => __('Single Post Options', 'minimal-blog'),
                    'free' => __('no', 'minimal-blog'),
                    'pro' => __('yes', 'minimal-blog'),
                ),
                array(
                    'title' => __('Archive Options', 'minimal-blog'),
                    'free' => __('no', 'minimal-blog'),
                    'pro' => __('yes', 'minimal-blog'),
                ),
                array(
                    'title' => __('Mailchimp Option', 'minimal-blog'),
                    'free' => __('no', 'minimal-blog'),
                    'pro' => __('yes', 'minimal-blog'),
                ),
                array(
                    'title' => __('Footer Credit Customization', 'minimal-blog'),
                    'free' => __('no', 'minimal-blog'),
                    'pro' => __('yes', 'minimal-blog'),
                ),
            );
            ?>
            <div id="comparison-table" class="minimal-blog-section minimal-blog-comparison-table">
                <?php
                $free_pro = $minimal_blog_panel_compare;
                if (!empty($free_pro)) {
                    $defaults = array(
                        'title' => '',
                        'desc' => '',
                        'free' => '',
                        'pro' => '',
                    );
                    if (!empty($free_pro) && is_array($free_pro)) {
                        $theme_info = wp_get_theme();
                        $theme_name = $theme_info->__get('Name');
                        $pro_theme_name = $theme_name . ' Pro';
                        echo '<div id="theme-comparision-table">';
                        echo '<table class="free-pro-table">';
                        echo '<thead>';
                        echo '<tr>';
                        echo '<th></th>';
                        echo '<th>' . $theme_name . '</th>';
                        echo '<th>' . $pro_theme_name . '</th>';
                        echo '</tr>';
                        echo '</thead>';
                        echo '<tbody>';
                        foreach ($free_pro as $feature) {
                            $instance = wp_parse_args((array)$feature, $defaults);
                            /*allowed 7 value in array */
                            $title = $instance['title'];
                            $desc = $instance['desc'];
                            $free = $instance['free'];
                            $pro = $instance['pro'];
                            echo '<tr>';
                            if (!empty($title) || !empty($desc)) {
                                echo '<td>';
                                if (!empty($title)) {
                                    echo '<h3 class="compare-tabel-title">' . wp_kses_post($title) . '</h3>';
                                }
                                if (!empty($desc)) {
                                    echo '<p>' . wp_kses_post($desc) . '</p>';
                                }
                                echo '</td>';
                            }
                            if (!empty($free)) {
                                if ('yes' === $free) {
                                    echo '<td class="theme-feature-check"><span class="dashicons-before dashicons-yes"></span></td>';
                                } elseif ('no' === $free) {
                                    echo '<td class="theme-feature-cross"><span class="dashicons-before dashicons-no-alt"></span></td>';
                                } else {
                                    echo '<td class="theme-feature-check">' . esc_html($free) . '</td>';
                                }
                            }
                            if (!empty($pro)) {
                                if ('yes' === $pro) {
                                    echo '<td class="theme-feature-check"><span class="dashicons-before dashicons-yes"></span></td>';
                                } elseif ('no' === $pro) {
                                    echo '<td class="theme-feature-cross"><span class="dashicons-before dashicons-no-alt"></span></td>';
                                } else {
                                    echo '<td class="theme-feature-check">' . esc_html($pro) . '</td>';
                                }
                            }
                            echo '</tr>';
                        }
                        echo '</tbody>';
                        echo '</table>';
                        echo '</div>';
                    }
                } ?>
            </div><!-- .minimal-blog-section -->
            <?php
        }
        /**
         * Enqueue scripts for dashboard page.
         *
         * @param string $hook Page hook.
         * @since 1.1.7
         *
         */
        public function enqueue_scripts($hook)
        {
            $valid_pages = array(
                "themes.php",
                "plugins.php",
                "index.php",
                "toplevel_page_$this->page_slug",
                "appearance_page_$this->page_slug",
            );
            if (in_array($hook, $valid_pages, true)) {
                wp_enqueue_media();
                wp_enqueue_style("minimal-blog-dashboard-style", get_template_directory_uri() . '/inc/dashboard/css/style.css', array(), $this->theme_version);
                wp_enqueue_script("minimal-blog-dashboard-script", get_template_directory_uri() . '/inc/dashboard/js/script.js', array('jquery'), $this->theme_version, true);
                $localize_script = array(
                    'ajaxUrl' => admin_url('admin-ajax.php'),
                    'nonce' => wp_create_nonce('ajax-nonce'),
                    'domain' => get_home_url('/')
                );
                wp_localize_script("minimal-blog-dashboard-script", 'WPINTERFACEDASHBOARD', $localize_script);
            }
        }
        /**
         * Callback for AJAX dismiss.
         *
         * @since 1.0.1
         */
        public function dismiss_callback()
        {
            $output = array();
            $output['status'] = false;
            $userid = (isset($_GET['userid'])) ? esc_attr(wp_unslash($_GET['userid'])) : '';
            $wpnonce = (isset($_GET['_wpnonce'])) ? esc_attr(wp_unslash($_GET['_wpnonce'])) : '';
            if (false === wp_verify_nonce($wpnonce, 'minimal_blog_dismiss_nonce')) {
                wp_send_json($output);
            }
            update_user_meta($userid, 'minimal_blog_dismiss_status', 1);
            $output['status'] = true;
            wp_send_json($output);
        }
    }
}