/**
 * MediHealth Deactivation Feedback Scripts
 *
 * Handles the deactivation feedback modal on themes page.
 *
 * @package MediHealth WordPress theme
 */

(function($) {
    'use strict';

    var modal = null;
    var pendingActivationUrl = null;

    $(document).ready(function() {
        modal = $('#medihealth-deactivation-modal');
        
        if (!modal.length) {
            return;
        }

        // Intercept theme activation clicks
        interceptThemeActivation();

        // Handle modal close
        modal.on('click', '.medihealth-modal-close', closeModal);
        
        // Close on overlay click
        modal.on('click', function(e) {
            if ($(e.target).hasClass('medihealth-modal-overlay')) {
                closeModal();
            }
        });

        // Close on Escape key
        $(document).on('keyup', function(e) {
            if (e.key === 'Escape' && modal.is(':visible')) {
                closeModal();
            }
        });

        // Show additional feedback textarea when reason is selected
        $('input[name="deactivation_reason"]').on('change', function() {
            var detailsBox = $('.medihealth-feedback-details');
            var selectedValue = $(this).val();
            
            // Show textarea for all options except 'temporary'
            if (selectedValue && selectedValue !== 'temporary') {
                detailsBox.slideDown(200);
            } else {
                detailsBox.slideUp(200);
            }
        });

        // Handle skip button
        modal.on('click', '.medihealth-skip-btn', function() {
            proceedWithDeactivation();
        });

        // Handle submit button
        modal.on('click', '.medihealth-submit-btn', function() {
            submitFeedback();
        });
    });

    /**
     * Intercept clicks on theme activation buttons
     */
    function interceptThemeActivation() {
        // For theme cards - intercept the Activate button
        $(document).on('click', '.theme-actions .activate', function(e) {
            var $this = $(this);
            var activationUrl = $this.attr('href');
            
            // Store the activation URL and show modal
            if (activationUrl && activationUrl.indexOf('action=activate') !== -1) {
                e.preventDefault();
                pendingActivationUrl = activationUrl;
                $('#medihealth-new-theme-url').val(activationUrl);
                showModal();
            }
        });

        // For theme overlay - intercept the Activate button
        $(document).on('click', '.theme-overlay .theme-actions .activate', function(e) {
            var $this = $(this);
            var activationUrl = $this.attr('href');
            
            if (activationUrl && activationUrl.indexOf('action=activate') !== -1) {
                e.preventDefault();
                pendingActivationUrl = activationUrl;
                $('#medihealth-new-theme-url').val(activationUrl);
                showModal();
            }
        });
    }

    /**
     * Show the modal
     */
    function showModal() {
        modal.fadeIn(200);
        $('body').addClass('medihealth-modal-open');
        
        // Reset form
        $('#medihealth-deactivation-form')[0].reset();
        $('.medihealth-feedback-details').hide();
    }

    /**
     * Close the modal
     */
    function closeModal() {
        modal.fadeOut(200);
        $('body').removeClass('medihealth-modal-open');
        pendingActivationUrl = null;
    }

    /**
     * Submit feedback and proceed with deactivation
     */
    function submitFeedback() {
        var reason = $('input[name="deactivation_reason"]:checked').val();
        var feedback = $('#medihealth-feedback-text').val();

        // Require a reason to be selected
        if (!reason) {
            alert(wp.i18n ? wp.i18n.__('Please select a reason for deactivating.', 'medihealth') : 'Please select a reason for deactivating.');
            return;
        }

        var $submitBtn = modal.find('.medihealth-submit-btn');
        $submitBtn.prop('disabled', true).text('Submitting...');

        $.ajax({
            url: medihealthDeactivation.ajaxurl,
            type: 'POST',
            data: {
                action: 'medihealth_deactivation_feedback',
                nonce: medihealthDeactivation.nonce,
                reason: reason,
                feedback: feedback
            },
            success: function(response) {
                proceedWithDeactivation();
            },
            error: function() {
                // Still proceed with deactivation even if AJAX fails
                proceedWithDeactivation();
            }
        });
    }

    /**
     * Proceed with theme deactivation
     */
    function proceedWithDeactivation() {
        if (pendingActivationUrl) {
            window.location.href = pendingActivationUrl;
        } else {
            closeModal();
        }
    }

})(jQuery);
