<?php
/**
 * Admin Getting Started Notice
 *
 * @package MediHealth WordPress theme
 */

// Exit if accessed directly
if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

/**
 * Enqueue admin notice styles and scripts
 */
function medihealth_admin_notice_scripts( $hook ) {
    // Only load on relevant admin pages
    if ( ! in_array( $hook, array( 'index.php', 'themes.php' ), true ) ) {
        return;
    }

    // Only load if notice is not dismissed
    if ( get_option( 'medihealth_notice_dismissed', false ) ) {
        return;
    }

    // Only load for users who can edit theme options
    if ( ! current_user_can( 'edit_theme_options' ) ) {
        return;
    }

    // Enqueue admin notice CSS
    wp_enqueue_style( 
        'medihealth-admin-notice', 
        get_template_directory_uri() . '/css/admin-notice.css',
        array(),
        wp_get_theme()->get( 'Version' )
    );

    // Enqueue admin notice JS
    wp_enqueue_script( 
        'medihealth-admin-notice', 
        get_template_directory_uri() . '/js/admin-notice.js',
        array( 'jquery' ),
        wp_get_theme()->get( 'Version' ),
        true
    );

    // Localize script with nonce
    wp_localize_script( 
        'medihealth-admin-notice', 
        'medihealthAdminNotice', 
        array(
            'nonce' => wp_create_nonce( 'medihealth_dismiss_notice' ),
            'ajaxurl' => admin_url( 'admin-ajax.php' ),
        )
    );
}
add_action( 'admin_enqueue_scripts', 'medihealth_admin_notice_scripts' );

/**
 * Display admin notice for theme setup
 */
function medihealth_admin_notice() {
    // Only show to users who can edit theme options
    if ( ! current_user_can( 'edit_theme_options' ) ) {
        return;
    }

    // Check if notice was dismissed
    if ( get_option( 'medihealth_notice_dismissed', false ) ) {
        return;
    }

    // Only show on relevant admin pages
    $screen = get_current_screen();
    if ( ! in_array( $screen->id, array( 'dashboard', 'themes', 'appearance_page_starter-sites' ), true ) ) {
        return;
    }

    $customizer_url = admin_url( 'customize.php' );
    $new_page_url = admin_url( 'post-new.php?post_type=page' );
    $new_post_url = admin_url( 'post-new.php' );
    ?>
    <div class="notice notice-info medihealth-welcome-notice is-dismissible" id="medihealth-welcome-notice">
        <div class="medihealth-notice-content">
            <div class="medihealth-notice-header">
                <h2><?php esc_html_e( 'Welcome to MediHealth Theme!', 'medihealth' ); ?></h2>
                <p><?php esc_html_e( 'Follow this two-step process to set up your homepage:', 'medihealth' ); ?></p>
            </div>
            
            <div class="medihealth-workflow">
                <div class="medihealth-workflow-step">
                    <div class="workflow-badge"><?php esc_html_e( 'Step 1', 'medihealth' ); ?></div>
                    <h3><?php esc_html_e( 'Create Your Content First', 'medihealth' ); ?></h3>
                    <p><?php esc_html_e( 'Create pages or posts with titles, content, and featured images. These will be used in your homepage sections.', 'medihealth' ); ?></p>
                    <div class="workflow-actions">
                        <a href="<?php echo esc_url( $new_page_url ); ?>" class="button button-primary"><?php esc_html_e( 'Create New Page', 'medihealth' ); ?></a>
                        <a href="<?php echo esc_url( $new_post_url ); ?>" class="button"><?php esc_html_e( 'Create New Post', 'medihealth' ); ?></a>
                    </div>
                </div>
                
                <div class="medihealth-workflow-arrow">→</div>
                
                <div class="medihealth-workflow-step">
                    <div class="workflow-badge"><?php esc_html_e( 'Step 2', 'medihealth' ); ?></div>
                    <h3><?php esc_html_e( 'Configure in Customizer', 'medihealth' ); ?></h3>
                    <p><?php esc_html_e( 'After creating content, go to Customizer to enable sections and select which pages/posts to display.', 'medihealth' ); ?></p>
                    <div class="workflow-actions">
                        <a href="<?php echo esc_url( $customizer_url ); ?>" class="button button-primary"><?php esc_html_e( 'Open Customizer', 'medihealth' ); ?></a>
                    </div>
                </div>
            </div>
            
            <div class="medihealth-notice-sections">
                <h4><?php esc_html_e( 'Quick Links to Section Settings:', 'medihealth' ); ?></h4>
                <div class="medihealth-section-links">
                    <a href="<?php echo esc_url( $customizer_url . '?autofocus[section]=medihealth_contact_panel' ); ?>" class="button button-small"><?php esc_html_e( 'Contact Info', 'medihealth' ); ?></a>
                    <a href="<?php echo esc_url( $customizer_url . '?autofocus[section]=medihealth_slider_section' ); ?>" class="button button-small"><?php esc_html_e( 'Slider', 'medihealth' ); ?></a>
                    <a href="<?php echo esc_url( $customizer_url . '?autofocus[section]=medihealth_service_section' ); ?>" class="button button-small"><?php esc_html_e( 'Services', 'medihealth' ); ?></a>
                    <a href="<?php echo esc_url( $customizer_url . '?autofocus[section]=medihealth_testimonial_section' ); ?>" class="button button-small"><?php esc_html_e( 'Testimonials', 'medihealth' ); ?></a>
                    <a href="<?php echo esc_url( $customizer_url . '?autofocus[section]=medihealth_portfolio_section' ); ?>" class="button button-small"><?php esc_html_e( 'Portfolio', 'medihealth' ); ?></a>
                    <a href="<?php echo esc_url( $customizer_url . '?autofocus[section]=medihealth_blog_section' ); ?>" class="button button-small"><?php esc_html_e( 'Blog', 'medihealth' ); ?></a>
                </div>
            </div>
            
            <div class="medihealth-notice-footer">
                <span class="medihealth-tip"><strong><?php esc_html_e( 'Tip:', 'medihealth' ); ?></strong> <?php esc_html_e( 'Use the "Icon" meta box for Services (Font Awesome class) and "Testimonial Designation" meta box for Testimonials.', 'medihealth' ); ?></span>
                <a href="#" class="medihealth-dismiss-notice"><?php esc_html_e( 'Dismiss this notice', 'medihealth' ); ?></a>
            </div>
        </div>
    </div>
    <?php
}
add_action( 'admin_notices', 'medihealth_admin_notice' );

/**
 * AJAX handler to dismiss the notice
 */
function medihealth_dismiss_notice_ajax() {
    // Verify nonce
    if ( ! isset( $_POST['nonce'] ) || ! wp_verify_nonce( sanitize_text_field( wp_unslash( $_POST['nonce'] ) ), 'medihealth_dismiss_notice' ) ) {
        wp_send_json_error( 'Security check failed' );
    }

    // Check capability
    if ( ! current_user_can( 'edit_theme_options' ) ) {
        wp_send_json_error( 'Unauthorized' );
    }

    // Save dismissed state
    update_option( 'medihealth_notice_dismissed', true );
    wp_send_json_success();
}
add_action( 'wp_ajax_medihealth_dismiss_notice', 'medihealth_dismiss_notice_ajax' );

/**
 * Reset notice on theme switch (for development/testing)
 */
function medihealth_reset_notice_on_switch() {
    delete_option( 'medihealth_notice_dismissed' );
}
add_action( 'after_switch_theme', 'medihealth_reset_notice_on_switch' );
