<?php
/**
 * Custom template tags for this theme
 *
 * Eventually, some of the functionality here could be replaced by core features.
 *
 * @package mateo
 */

use Mateo\ThemeOptionsMetabox;
use Mateo\Breadcrumb_Trail;

if ( ! function_exists( 'mateo_posted_on' ) ) :
	/**
	 * Prints HTML with meta information for the current post-date/time.
	 */
	function mateo_posted_on() {
		$time_string = '<time class="entry-date published" datetime="%1$s">%2$s</time>';

		$time_string = sprintf(
			$time_string,
			esc_attr( get_the_date( DATE_W3C ) ),
			esc_html( get_the_date() )
		);
		// phpcs:ignore
		printf( '<i class="material-icons" aria-hidden="true">event</i><span class="posted-on">%s</span>', $time_string );
	}
endif;


if ( ! function_exists( 'mateo_posted_by' ) ) :
	/**
	 * Prints HTML with meta information for the current author.
	 */
	function mateo_posted_by() {
		echo '<span class="byline"> ' .
		esc_html_x( 'By', 'post author', 'mateo' ) .
		'<span class="author vcard"> <a class="url fn n" href="' .
		esc_url( get_author_posts_url( get_the_author_meta( 'ID' ) ) ) . '">' .
		esc_html( get_the_author() ) .
		'</a></span>'
		. '</span>';
	}
endif;

if ( ! function_exists( 'mateo_entry_cats' ) ) :
	/**
	 * Prints categories list.
	 */
	function mateo_entry_cats() {
		$categories_list = get_the_category_list( ' ' );
		if ( $categories_list ) {
			/* translators: used before categories list */
			$label = esc_html__( 'Categories: ', 'mateo' );
			// phpcs:disable
			echo '<div class="chips-links cats-chips-group"><i class="material-icons" aria-hidden="true">explore</i><span>'
			 . '<span class="chips-links-label">'
			 . $label
			 . '</span>'
			 . $categories_list
			 . '</span></div>';
			// phpcs:enable
		}

	}
endif;

if ( ! function_exists( 'mateo_comments_info' ) ) {
	/**
	 * Displays comments meta.
	 */
	function mateo_comments_info() {
		if ( ! is_page() && ( comments_open() || get_comments_number() ) ) : ?>
		<div>
			<i class="material-icons" aria-hidden="true">comment</i>
			<span>
					<?php comments_popup_link(); ?>
			</span>
		</div>
			<?php
		endif;
	}
}
if ( ! function_exists( 'mateo_entry_tags' ) ) :
	/**
	 * Prints HTML with meta information regarding the post.
	 */
	function mateo_entry_tags() {
		// Hide tag text for pages.
		if ( 'page' !== get_post_type() ) {
			$tags_list = get_the_tag_list();

			if ( ! $tags_list instanceof WP_Error && ! empty( $tags_list ) ) {
				/* translators: used before tags list */
				$label = esc_html__( 'Tags: ', 'mateo' );
				// phpcs:disable
				echo '<div class="chips-links tags-chips-group"><i class="material-icons" aria-hidden="true">local_offer</i><span>'
				 . '<span class="chips-links-label">'
				 . $label
				 . '</span>'
				 . $tags_list
				 . '</span></div>';
				// phpcs:enable
			}
		}
	}
endif;

if ( ! function_exists( 'mateo_post_thumbnail' ) ) :
	/**
	 * Displays an optional post thumbnail.
	 *
	 * Wraps the post thumbnail in an anchor element on index views, or a div
	 * element when on single views.
	 */
	function mateo_post_thumbnail() {
		if ( post_password_required() || is_attachment() || ! has_post_thumbnail() ) {
			return;
		}

		if ( is_singular() ) :
			?>
		<figure class="wp-block-image mateo-featured-img">
			<?php the_post_thumbnail( array( mateo_get_display_width(), 0 ) ); ?>
		</figure>
			<?php
		else :
			?>
			<?php $url = get_the_post_thumbnail_url( null, array( mateo_get_display_width(), 0 ) ); ?>
		<a class="post-thumbnail" href="<?php the_permalink(); ?>">
			<div class="size-post-thumbnail" style="background-image: url(<?php echo esc_url( $url ); ?>)"></div>
		</a>
			<?php
		endif; // End is_singular().
	}
endif;

if ( ! function_exists( 'mateo_comment_form' ) ) :
	/**
	 * Displays a comment form.
	 *
	 * @param array    $args comment_form() function args.
	 * @param null|int $post_id Post id.
	 *
	 * @see comment_form()
	 */
	function mateo_comment_form( $args = array(), $post_id = null ) {

		$commenter = wp_get_current_commenter();
		$req       = get_option( 'require_name_email' );
		$html_req  = ( $req ? 'required' : '' );

		$defaults = array(
			'format'              => 'xhtml', // Set xhtml for browser validation.
			'cancel_reply_before' => '',
			'cancel_reply_after'  => '',
			'cancel_reply_link'   => '<i class="material-icons" aria-hidden="true">close</i><span class="screen-reader-text">' . __( 'Cancel', 'mateo' ) . '</span>',
			'class_submit'        => '',
			'submit_button'       => '<button name="%1$s" type="submit" id="%2$s" class="%3$s">%4$s</button>',
			'submit_field'        => '
	<div class="row">
		<div class="col s12">%1$s %2$s</div>
	</div>',

			'fields'              => array(
				'author' => '
	<div class="form-inline comment-block-author">
		<div class="form-group bmd-form-group">
		  <input id="author" name="author" class="form-control" type="text" value="' . esc_attr( $commenter['comment_author'] ) . '" maxlength="245" ' . $html_req . '/>
		  <label for="author" class="bmd-label-floating">' . __( 'Name', 'mateo' ) . ( $req ? ' <span class="required">*</span>' : '' ) . '</label>
		</div>
		<div class="form-group bmd-form-group">
		  <input id="email" name="email"  class="form-control" type="email" value="' . esc_attr( $commenter['comment_author_email'] ) . '" maxlength="100" ' . $html_req . '/>
		  <label for="email" class="bmd-label-floating">' . __( 'Email', 'mateo' ) . ( $req ? ' <span class="required">*</span>' : '' ) . '</label>
		</div>
		 <div class="form-group bmd-form-group">
		  <input id="url" name="url"  class="form-control" type="url" value="' . esc_attr( $commenter['comment_author_url'] ) . '" maxlength="200"/>
		  <label for="url" class="bmd-label-floating">' . __( 'Website', 'mateo' ) . '</label>
		</div>
	</div>',
			),
			'comment_field'       => '
	<div>
		<div class="form-group bmd-form-group">
			<label for="comment" class="bmd-label-floating">' . _x( 'Comment', 'noun', 'mateo' ) . ' <span class="required">*</span></label>
			<textarea class="form-control" id="comment" name="comment" cols="45" rows="4" maxlength="65525" ' . $html_req . '></textarea>
		</div>
	</div>',
		);
		if ( get_option( 'show_comments_cookies_opt_in' ) ) {
			$consent                       = empty( $commenter['comment_author_email'] ) ? '' : ' checked="checked"';
			$defaults['fields']['cookies'] = '<div class="form-group bmd-form-group comment-form-cookies-consent">
		<input id="wp-comment-cookies-consent" name="wp-comment-cookies-consent" type="checkbox" value="yes"' . $consent . ' />' .
			'<label for="wp-comment-cookies-consent">' . __( 'Save my name, email, and website in this browser for the next time I comment.', 'mateo' ) . '</label>' .
			'</div>';
		}

		comment_form( $args + $defaults, $post_id );
	}
endif;

if ( ! function_exists( 'mateo_html5_comment' ) ) {
	/**
	 * Print comment details.
	 * Used in comments.php by wp_list_comments() function.
	 *
	 * @param int|WP_Comment $comment Comment or its id.
	 * @param array          $args wp_list_comments() args.
	 * @param int            $depth Current thread level.
	 */
	function mateo_html5_comment( $comment, $args, $depth ) {
		$tag        = ( 'div' === $args['style'] ) ? 'div' : 'li';
		$ping_short = ( 'pingback' === $comment->comment_type || 'trackback' === $comment->comment_type ) && $args['short_ping'];

		$commenter = wp_get_current_commenter();
		if ( $commenter['comment_author_email'] ) {
			$moderation_note = __( 'Your comment is awaiting moderation.', 'mateo' );
		} else {
			$moderation_note = __( 'Your comment is awaiting moderation. This is a preview, your comment will be visible after it has been approved.', 'mateo' );
		}
		$author_name = esc_html( get_comment_author( $comment ) );
		?>
	<<?php echo $tag; // phpcs:ignore ?> id="comment-<?php comment_ID(); ?>">
	<article id="div-comment-<?php comment_ID(); ?>" class="comment-body section">
		<header class="comment-meta">
			<div class="comment-author vcard">
				<?php
				if ( 0 !== (int) $args['avatar_size'] ) {
					echo get_avatar( $comment, $args['avatar_size'], '', $author_name ); // phpcs:ignore
				}
				?>

			</div><!-- .comment-author -->
			<div class="comment-metadata">
				<?php
				if ( $ping_short ) {
					echo '<strong>' . esc_html__( 'Pingback:', 'mateo' ) . '</strong> ';
				} else {
					printf( '<strong>%s</strong>', $author_name ); // phpcs:ignore
				}
				?>
				<time datetime="<?php comment_time( 'c' ); ?>">
					<?php
					/* translators: 1: comment date, 2: comment time */
					printf( esc_html__( '%1$s at %2$s', 'mateo' ), get_comment_date( '', $comment ), get_comment_time() ); // phpcs:ignore
					?>
				</time>
			</div><!-- .comment-metadata -->
			<div class="comment-action">
				<a href="<?php echo esc_url( get_comment_link( $comment, $args ) ); ?>">
					<i class="material-icons" aria-hidden="true">link</i><span
						class="screen-reader-text"><?php esc_html_e( 'Link to comment', 'mateo' ); ?>
				</a>
					<?php
					comment_reply_link(
						array_merge(
							$args,
							array(
								'add_below'  => 'div-comment',
								'depth'      => $depth,
								'max_depth'  => $args['max_depth'],
								'reply_text' => '<i class="material-icons" aria-hidden="true">reply</i><span class="screen-reader-text">' . esc_html__( 'Reply', 'mateo' ) . '</span>',
							)
						)
					);
					?>
					<?php
					edit_comment_link(
						'<i class="material-icons" aria-hidden="true">edit</i><span class="screen-reader-text">' . esc_html__( 'Edit', 'mateo' ) . '</span>'
					);
					?>
			</div>
		</header><!-- .comment-meta -->
			<?php if ( 0 === (int) $comment->comment_approved ) : ?>
			<div class="alert alert-info">
				<em class="comment-awaiting-moderation"><?php echo esc_html( $moderation_note ); ?></em>
			</div>
		<?php endif; ?>
		<div class="comment-content">
				<?php
				if ( $ping_short ) {
					echo '<p>' . get_comment_author_link( $comment ) . '</p>';
				} else {
					comment_text();
				}
				?>
		</div><!-- .comment-content -->
	</article><!-- .comment-body -->

		<?php
	}
}
if ( ! function_exists( 'mateo_the_posts_pagination' ) ) {
	/**
	 * Print pagination links
	 */
	function mateo_the_posts_pagination() {
		$args = array(
			'type'      => 'list',
			'prev_text' => '<i class="material-icons" aria-hidden="true" aria-label="' . esc_attr__( 'Previous', 'mateo' ) . '">chevron_left</i>',
			'next_text' => '<i class="material-icons" aria-hidden="true" aria-label="' . esc_attr__( 'Next', 'mateo' ) . '">chevron_right</i>',
			'mid_size'  => 2,
		);

		the_posts_pagination( $args );
	}
}

if ( ! function_exists( 'mateo_the_post_inline_edit_button' ) ) {
	/**
	 * Displays a post/page edition link on the list.
	 *
	 * @param string $classes CSS classes to apply to the link.
	 */
	function mateo_the_post_inline_edit_button( $classes = '' ) {
		if ( ! empty( $classes ) ) {
			$classes = ' ' . $classes;
		}
		?>
	<div class="float-right">
		<?php mateo_edit_button( 'btn-post-header-edit ' . $classes ); ?>
	</div>
		<?php
	}
}

if ( ! function_exists( 'mateo_the_post_edit_button' ) ) {
	/**
	 * Displays an edition link on "singular" page/post.
	 *
	 * @param string $classes CSS classes to apply to the link.
	 */
	function mateo_the_post_edit_button( $classes = '' ) {
		if ( ! is_singular() ) {
			return;
		}
		if ( ! empty( $classes ) ) {
			$classes = ' ' . $classes;
		}
		mateo_edit_button( 'btn-float-edit' . $classes );
	}
}

if ( ! function_exists( 'mateo_edit_button' ) ) {
	/**
	 * Displays a link to post edition.
	 *
	 * @param string $classes CSS classes to apply to the link.
	 * @param string $target Link target attribute.
	 */
	function mateo_edit_button( $classes, $target = '' ) {
		$url = get_edit_post_link();
		if ( ! $url ) {
			return;
		}
		$title = sprintf( '%s %s', ( 'Edit' ), get_the_title() );
		?>
	<a class="<?php echo esc_attr( $classes ); ?>" href="<?php echo esc_url( $url ); ?>"
		title="<?php echo esc_attr( $title ); ?>"<?php echo $target ? ' target="' . esc_attr( $target ) . '"' : ''; ?>>
		<i class="material-icons" aria-hidden="true">edit</i>
		<span
			class="screen-reader-text"><?php esc_html_e( 'Edit', 'mateo' ); ?><?php echo esc_html( get_the_title() ); ?></span>
	</a>
		<?php
	}
}

if ( ! function_exists( 'mateo_has_featured_img' ) ) {
	/**
	 * Check weather featured image is available for the current post.
	 *
	 * @return string
	 */
	function mateo_has_featured_img() {
		if ( is_singular() && mateo_is_active_featured_in_header() && has_post_thumbnail() ) {
			// 1920 x 1080
			$thumbnail_data = wp_get_attachment_image_src(
				get_post_thumbnail_id( get_post() ),
				array(
					1920,
					1080,
				),
				false
			);
			if ( $thumbnail_data ) {
				return true;
			}
		}

		$url = get_header_image();

		return $url ? true : false;
	}
}

if ( ! function_exists( 'mateo_none_fallback' ) ) {
	/**
	 * Does exactly what it is supposed to do (i.e. nothing).
	 * Used in mateo_nav_menu() function.
	 */
	function mateo_none_fallback() {
		return;
	}
}


if ( ! function_exists( 'mateo_nav_menu' ) ) {
// phpcs:ignore --WPThemeReview.CoreFunctionality.FileInclude.FileIncludeFound
require_once MATEO_THEME_ABSPATH . '/inc/classes/mateo-walker-nav-menu.php';

	/**
	 * Return navigation menu.
	 *
	 * @param array $args wp_nav_menu args.
	 *
	 * @return false|string|void
	 * @see wp_nav_menu() function for arguments list.
	 */
	function mateo_nav_menu( $args = array() ) {
		return wp_nav_menu(
			array_merge(
				array(
					'echo'           => false,
					'walker'         => Mateo_Walker_Nav_Menu::get_instance(),
					// Do nothing if there are no menus assigned to required location.
					'fallback_cb'    => 'mateo_none_fallback',
					'theme_location' => 'menu-1',
				),
				$args
			)
		);
	}
}

if ( ! function_exists( 'mateo_list_post_title' ) ) {
	/**
	 * Display  the current post title with optional markup.
	 * If title is empty it displays 'Untitled'.
	 *
	 * @param string $before Optional. Markup to prepend to the title. Default empty.
	 * @param string $after Optional. Markup to append to the title. Default empty.
	 */
	function mateo_list_post_title( $before = '', $after = '' ) {

		$title = the_title( '', '', false );

		if ( strlen( trim( $title ) ) === 0 ) {
			/* translators: Used for untitled post  */
			$title = esc_html__( 'Untitled', "mateo" ); // phpcs:ignore
		}

		echo $title = $before . $title . $after; // phpcs:ignore
	}
}
if ( ! function_exists( 'mateo_header_title' ) ) {
	/**
	 * Display  the current post title with optional markup .
	 * If title is empty it displays a blog title.
	 *
	 * @param string $before Optional. Markup to prepend to the title. Default empty.
	 * @param string $after Optional. Markup to append to the title. Default empty.
	 */
	function mateo_header_title( $before = '', $after = '' ) {
		if ( ! is_singular() ) {
			// phpcs:ignore
			echo $before . ( single_post_title( '', false ) ?? bloginfo() ) . $after;
			return;
		}
		mateo_list_post_title( $before, $after );
	}
}

if ( ! function_exists( 'mateo_excerpt' ) ) {
	/**
	 * Generates an excerpt from the content.
	 * If there is no content it attempts to return a media tag.
	 *
	 * @return string   Some content (can be empty).
	 */
	function mateo_excerpt() {
		$text = wp_trim_excerpt();
		if ( strlen( $text ) === 0 ) {
			// Featured image is displayed no need to go further.
			if ( has_post_thumbnail() ) {
				return '';
			}
			// Password protected content. Exit here.
			if ( post_password_required() ) {
				return '';
			}
			// Grab media content if there is some.
			$text = get_the_content( '' );
			$text = strip_shortcodes( $text );
			if ( function_exists( 'excerpt_remove_blocks' ) ) {
				$text = excerpt_remove_blocks( $text );
			}

			/** This filter is documented in wp-includes/post-template.php */
			$text = apply_filters( 'the_content', $text ); // phpcs:ignore

			$media = get_media_embedded_in_content( $text );
			if ( count( $media ) > 0 ) {
				$text = $media[0];
			}
		}

		return $text;
	}
}

if ( ! function_exists( 'mateo_link_pages' ) ) {

	/**
	 * The formatted output of a list of pages.
	 *
	 * @param string|array $args {
	 *     Optional. Array or string of default arguments.
	 *
	 * @type string $before HTML or text to prepend to each link. Default is `<p> Pages:`.
	 * @type string $after HTML or text to append to each link. Default is `</p>`.
	 * @type string $link_before HTML or text to prepend to each link, inside the `<a>` tag.
	 *                                          Also prepended to the current item, which is not linked. Default empty.
	 * @type string $link_after HTML or text to append to each Pages link inside the `<a>` tag.
	 *                                          Also appended to the current item, which is not linked. Default empty.
	 * @type string $aria_current The value for the aria-current attribute. Possible values are 'page',
	 *                                          'step', 'location', 'date', 'time', 'true', 'false'. Default is 'page'.
	 * @type string $next_or_number Indicates whether page numbers should be used. Valid values are number
	 *                                          and next. Default is 'number'.
	 * @type string $separator Text between pagination links. Default is ' '.
	 * @type string $nextpagelink Link text for the next page link, if available. Default is 'Next Page'.
	 * @type string $previouspagelink Link text for the previous page link, if available. Default is 'Previous Page'.
	 * @type string $pagelink Format string for page numbers. The % in the parameter string will be
	 *                                          replaced with the page number, so 'Page %' generates "Page 1", "Page 2", etc.
	 *                                          Defaults to '%', just the page number.
	 * @type int|bool $echo Whether to echo or not. Accepts 1|true or 0|false. Default 1|true.
	 * }
	 * @return string Formatted output in HTML.
	 * @global int $numpages
	 * @global int $multipage
	 * @global int $more
	 *
	 * @global int $page
	 *
	 * @see mateo_filter_wp_link_pages_link() for <a> and <li> generation.
	 *
	 * Displays page links for paginated posts (i.e. including the `<!--nextpage-->`
	 * Quicktag one or more times). This tag must be within The Loop.
	 */
	function mateo_link_pages( $args = '' ) {
		$params = array_merge(
			array(
				'before' => '<nav class="pagination"><div class="nav-links"><ul class="page-numbers">',
				'after'  => '</ul></div></nav>',
				'echo'   => 1,
			),
			(array) $args
		);

		return wp_link_pages( $params );
	}
}
if ( ! function_exists( 'mateo_entry_sticky' ) ) :
	/**
	 * Prints categories list.
	 */
	function mateo_entry_sticky() {
		if ( is_sticky() ) {
			echo '<div class="sticky-post"><i class="material-icons" aria-hidden="true">star</i><span>'
			. '<span>'
			/* translators: sticky post */
			. esc_html__( 'Featured', 'mateo' )
			. '</span></div>';
		}

	}
endif;

if ( ! function_exists( 'mateo_get_sidebar_position' ) ) {
	/**
	 * Gets a sidebar position.
	 *
	 * @return string  Css class name.
	 */
	function mateo_get_sidebar_position() {
		return ( get_theme_mod( 'sidebar-position', 'default' ) === 'reverse' ) ? 'sidebar-reverse' : 'sidebar-default';
	}
}

if ( ! function_exists( 'mateo_is_show_title_area' ) ) {
	/**
	 * Says if a header image (site header or featured image) with a title must be displayed for some post.
	 * Available for a single post.
	 *
	 * @return bool TRUE to display FALSE to hide
	 */
	function mateo_is_show_title_area() {
		if ( is_singular() ) {
			global $post;
			if ( $post ) {
				$value = (int) get_post_meta( $post->ID, ThemeOptionsMetabox::META_DISABLE_HEADER, true );

				return 1 !== $value;
			}

			return true;
		}

		return true;
	}
}

if ( ! function_exists( 'mateo_breadcrumb_trail' ) ) {
	/**
	 * Prints breadcrumb.
	 *
	 * @param array $args Breadcrumbs arguments.
	 *
	 * @return string|void Generated html or void if option 'echo' is true or the display is disabled by the settings.
	 */
	function mateo_breadcrumb_trail( $args = array() ) {

		if ( ! mateo_is_active_breadcrumb() ) {
			return;
		}

		if ( ! mateo_is_show_title_area() ) {
			return;
		}

		if ( get_theme_mod( 'mateo-breadcrumb-home-disable', false ) ) {
			// Home page is a list of latest posts : do not display for the 1st page.
			if ( is_front_page() && is_home() && ! is_paged() ) {
				return;
			}
			// Home page is a static page.
			if ( is_front_page() && ! is_home() ) {
				return;
			}
		}
		if ( get_theme_mod( 'mateo-breadcrumb-page-disable', false ) && is_page() ) {
			return;
		}

		static $breadcrumb;

		if ( ! is_object( $breadcrumb ) ) {
			$args['show_browse'] = false;
			$args                = apply_filters( 'mateo_breadcrumb_trail_args', $args );
			$breadcrumb          = new Breadcrumb_Trail( $args );
		}
		return $breadcrumb->trail();
	}
}

