<?php
/**
 * Functions which enhance the theme by hooking into WordPress
 *
 * @package mateo
 */

use Mateo\ThemeOptionsMetabox;

if ( is_admin() ) {
	$mateo_options = array();
	if ( 1 === mateo_get_theme_layout() ) {
		// Disable 1 column option if we have it in site settings.
		$mateo_options['disable'] = array( ThemeOptionsMetabox::META_LAYOUT );
	}
	ThemeOptionsMetabox::init( $mateo_options );
}

// phpcs:disable -- WordPress.NamingConventions.PrefixAllGlobals.NonPrefixedHooknameFound
if ( ! function_exists( 'wp_body_open' ) ) :
	/**
	 * Fire the wp_body_open action.
	 *
	 * Added for backwards compatibility to support pre 5.2.0 WordPress versions.
	 */
	function wp_body_open() {
		do_action( 'wp_body_open' );
	}
endif;
// phpcs:enable

/**
 * Adds custom classes to the array of body classes.
 *
 * @param array $classes Classes for the body element.
 * @return array
 */
function mateo_body_classes( $classes ) {
	// Adds a class of hfeed to non-singular pages.
	if ( ! is_singular() ) {
		$classes[] = 'hfeed';
	}

	// Adds a class of no-sidebar when there is no sidebar present.
	if ( ! is_active_sidebar( 'sidebar-1' ) ) {
		$classes[] = 'no-sidebar';
	}
	if ( is_rtl() ) {
		$classes[] = 'rtl';
	}

	if ( mateo_is_layout_one_column() ) {
		$classes[] = 'layout-1';
	}
	// hide icons until fonts are loaded.
	$classes[] = 'preload';

	return $classes;
}

add_filter( 'body_class', 'mateo_body_classes' );

/**
 * Add a pingback url auto-discovery header for single posts, pages, or attachments.
 */
function mateo_pingback_header() {
	if ( is_singular() && pings_open() ) {
		printf( '<link rel="pingback" href="%s">', esc_url( get_bloginfo( 'pingback_url' ) ) );
	}
}

add_action( 'wp_head', 'mateo_pingback_header' );

/**
 * Displays post's featured image or site's header image (if there is any) in featured area below the menu.
 */
function mateo_display_header_image() {
	$url = null;
	if ( is_singular() && mateo_is_active_featured_in_header() && has_post_thumbnail() ) {
		// 1920 x 1080
		$thumbnail_data = wp_get_attachment_image_src( get_post_thumbnail_id( get_post() ), array( 1920, 9999 ), false );
		if ( $thumbnail_data ) {
			$url = $thumbnail_data[0];
		}
	}
	if ( ! $url ) {
		$url = get_header_image();
	}
	if ( $url ) {
		printf(
			'<style type="text/css">.header-featured.has-image { background-image: url( %s ); background-size: cover; min-height: 500px; height: 50vh; background-position: center;}</style>',
			esc_url( $url )
		);
	}
}

add_action( 'wp_head', 'mateo_display_header_image' );

if ( ! function_exists( 'mateo_is_show_recent_comments_widget_style' ) ) {
	/**
	 * Disable default styles for recent comments widget.
	 */
	function mateo_is_show_recent_comments_widget_style() {
		return false;
	}
}

add_filter( 'show_recent_comments_widget_style', 'mateo_is_show_recent_comments_widget_style' );

/**
 * Customize link for theme display.
 *
 * @param string $link  Generated by WP link.
 * @param int    $number   Page number.
 * @return string
 */
function mateo_filter_wp_link_pages_link( $link, $number = 0 ) {
	if ( strpos( $link, '<a' ) !== false ) {
		preg_match( '/href="([^\"]+)"/', $link, $matches );
		$out = sprintf( '<a href="%s" class="page-numbers">%s</a>', $matches[1], $number );
	} else {
		$out = sprintf( '<span class="page-numbers current">%s</span>', $number );
	}
	return '<li>' . $out . '</li>';
};
add_filter( 'wp_link_pages_link', 'mateo_filter_wp_link_pages_link', 10, 2 );

if ( ! function_exists( 'mateo_get_except_length' ) ) {
	/**
	 * Filter function for the except length (in words).
	 *
	 * @param int $length The maximum number of words.
	 * @return int
	 */
	function mateo_get_except_length( $length ) {
		if ( is_admin() ) {
			return $length;
		}
		return 100;
	}
}

add_filter( 'excerpt_length', 'mateo_get_except_length' );


if ( ! function_exists( 'mateo_page_title_classes' ) ) {
	/**
	 * Prints CSS classes for the page title.
	 *
	 * @param array $classes List of CSS class names without starting dot.
	 */
	function mateo_page_title_classes( $classes = array() ) {
		if ( ! mateo_is_active_title_bgr() ) {
			// Make background transparent.
			$classes[] = 'bgr-alpha-0';
		}
		$classes = apply_filters( 'mateo_page_title_classes', $classes );
		echo esc_attr( implode( ' ', $classes ) );
	}
}

if ( ! function_exists( 'mateo_site_content_classes' ) ) {
	/**
	 * Prints CSS classes for the #site-content.
	 *
	 * @param array $classes List of CSS class names without starting dot.
	 */
	function mateo_site_content_classes( $classes = array() ) {

		if ( ! mateo_is_show_title_area() ) {
			$classes[] = 'editor-managed';
		}
		$classes[] = mateo_get_sidebar_position();
		$classes   = apply_filters( 'mateo_site_content_classes', $classes );

		echo esc_attr( implode( ' ', $classes ) );
	}
}
