<?php
/**
 * Theme Customizer functions.
 *
 * @package mateo
 */

use Mateo\ThemeColor;
use Mateo\Width_Customize_Control;
use Mateo\Width_Customize_Settings;
use Mateo\Logo_Customize_Control;
use Mateo\Logo_Customize_Settings;
use Mateo\Customizer\Font_Size_Setting;
use Mateo\Customizer\Font_Size_Control;
use Mateo\Customizer\Font_Line_Setting;
use Mateo\Customizer\Font_Line_Control;
use Mateo\Fonts;
use Mateo\Customizer\Title_Pseudo_Control;
use Mateo\Customizer\Group_Control;
use Mateo\Customizer\Units_Control;
use Mateo\Customizer\Space_Control;

require_once __DIR__ . '/customize/features.php';

if ( ! function_exists( 'mateo_get_units' ) ) {
	/**
	 * Returns css units of measurement.
	 *
	 * @return array
	 */
	function mateo_get_units() {
		static $units = null;
		if ( null === $units ) {
			$units = apply_filters(
				'mateo_units',
				array(
					'rem' => 'rem',
					'px'  => 'px',
					'em'  => 'em',
				)
			);
		}

		return $units;
	}
}
if ( ! function_exists( 'mateo_custom_css' ) ) {
	/**
	 * Generate custom CSS.
	 *
	 * @param string $id If present the function will return css rules wrapped into <style> tag, if not only css rules will be returned.
	 *
	 * @return string
	 */
	function mateo_custom_css( $id = '' ) {
		// Custom colors.
		$theme_color = new ThemeColor( mateo_get_color_scheme() );
		$theme_color->init_user_settings();
		// are we in preview mode ?
		if ( is_customize_preview() ) {
			$theme_color->init_change_set();
		}
		$css = $theme_color->get_css();
		// Custom logo.
		$logo_round = get_theme_mod( 'logo-round' );
		if ( $logo_round ) {
			$css .= <<<DOC
			.custom-logo {
				border-radius: ${logo_round}px;
			}
DOC;
		}
		// Main areas' width.
		$css .= mateo_layout_css();
		// Post settings.
		$css .= mateo_post_custom_css();
		// Custom fonts.
		$custom_fonts = mateo_custom_fonts();
		if ( ! empty( $custom_fonts ) ) {
			$css .= $custom_fonts['css'];
		}
		// Custom sizing.
		$css .= mateo_custom_css_vars();

		if ( $id ) {
			// @deprecated.
			return $theme_color->get_style_tag( $id, $css );
		}

		return mateo_get_css( $css );
	}
}

if ( ! function_exists( 'mateo_display_blog_name' ) ) {
	/**
	 * Displays the blog name.
	 */
	function mateo_display_blog_name() {
		bloginfo( 'name' );
	}
}

if ( ! function_exists( 'mateo_display_blog_description' ) ) {
	/**
	 * Displays the blog description.
	 */
	function mateo_display_blog_description() {
		bloginfo( 'description' );
	}
}

if ( ! function_exists( 'mateo_sidebar_sanitizer' ) ) {
	/**
	 * Sanitizes sidebar position.
	 * Default value is 'right'.
	 *
	 * @param string $value Value to sanitize.
	 *
	 * @return string   Sanitized value.
	 */
	function mateo_sidebar_sanitizer( $value ) {
		return 'reverse' === $value ? 'reverse' : 'default';
	}
}

if ( ! function_exists( 'mateo_get_fonts' ) ) {
	/**
	 * Load fonts
	 *
	 * @return array
	 */
	function mateo_get_fonts() {
		static $font_data;
		if ( ! $font_data ) {
			// phpcs:ignore WPThemeReview.CoreFunctionality.FileInclude.FileIncludeFound
			$font_data = require 'google-fonts.php';
			$font_data = apply_filters( 'mateo_fonts', $font_data );
			uasort(
				$font_data,
				function ( $font1, $font2 ) {
					return strnatcmp( $font1['name'], $font2['name'] );
				}
			);
		}

		return $font_data;
	}
}
/**
 * Add postMessage support for site title and description for the Theme Customizer.
 *
 * @param WP_Customize_Manager $wp_customize Theme Customizer object.
 */
function mateo_customize_register( $wp_customize ) {

	$theme_color = new ThemeColor( mateo_get_color_scheme() );

	// Configure WP default settings.
	$wp_customize->get_setting( 'blogname' )->transport        = 'postMessage';
	$wp_customize->get_setting( 'blogdescription' )->transport = 'postMessage';

	// Logo appearance.
	$wp_customize->add_setting(
		new Logo_Customize_Settings(
			$wp_customize,
			'logo-round',
			array(
				'sanitize_callback' => array( Logo_Customize_Settings::class, 'sanitize_value' ),
			)
		)
	);

	$wp_customize->add_control(
		new Logo_Customize_Control(
			$wp_customize,
			'logo-round',
			array( 'label' => esc_html__( 'Round logo', 'mateo' ) )
		)
	);
	// Add color scheme setting and control.
	$wp_customize->add_setting(
		'color_scheme',
		array(
			'default'           => 'default',
			'sanitize_callback' => 'mateo_sanitize_color_scheme',
			'transport'         => 'postMessage',
		)
	);

	$wp_customize->add_control(
		'color_scheme',
		array(
			'label'    => _x( 'Base Color Scheme', 'Color palette', 'mateo' ),
			'section'  => 'colors',
			'type'     => 'select',
			'choices'  => mateo_get_color_scheme_choices(),
			'priority' => 1,
		)
	);
	// Primary color.
	$wp_customize->add_setting(
		'primary',
		array(
			'default'           => $theme_color->get_color( 'primary' ),
			'sanitize_callback' => 'sanitize_hex_color',
			'transport'         => 'postMessage',
		)
	);
	$wp_customize->add_control(
		new WP_Customize_Color_Control(
			$wp_customize,
			'primary',
			array(
				'label'       => _x( 'Primary color', 'Color palette', 'mateo' ),
				'description' => _x( 'Primary color is used by top menu, footer widget area, buttons and links. Whenever you change it, do not forget to update its light and dark versions.', 'Color palette', 'mateo' ),
				'section'     => 'colors',
			)
		)
	);

	// Light version of primary color.
	$wp_customize->add_setting(
		'primaryLight',
		array(
			'default'           => $theme_color->get_color( 'primaryLight' ),
			'sanitize_callback' => 'sanitize_hex_color',
			'transport'         => 'postMessage',
		)
	);
	$wp_customize->add_control(
		new WP_Customize_Color_Control(
			$wp_customize,
			'primaryLight',
			array(
				'label'       => _x( 'Primary light color', 'Color palette', 'mateo' ),
				'description' => _x( 'Used for button states.', 'Color palette', 'mateo' ),
				'section'     => 'colors',
			)
		)
	);

	// Dark version of primary color.
	$wp_customize->add_setting(
		'primaryDark',
		array(
			'default'           => $theme_color->get_color( 'primaryDark' ),
			'sanitize_callback' => 'sanitize_hex_color',
			'transport'         => 'postMessage',
		)
	);
	$wp_customize->add_control(
		new WP_Customize_Color_Control(
			$wp_customize,
			'primaryDark',
			array(
				'label'       => _x( 'Primary dark color', 'Color palette', 'mateo' ),
				'description' => _x( 'Primary dark color is applied to the copyright footer and to some button states.', 'Color palette', 'mateo' ),
				'section'     => 'colors',
			)
		)
	);
	/**
	 * WP core feature required by review team.
	 * Turn it into a shortcut for the text on primary settings.
	 *
	 * @var WP_Customize_Setting $header_textcolor_setting .
	 */
	$header_textcolor_setting = $wp_customize->get_setting( 'header_textcolor' );
	if ( $header_textcolor_setting ) {
		$header_textcolor_setting->transport = 'postMessage';
		$header_textcolor_setting->default   = $theme_color->get_color( 'textOnPrimary' );
		/**
		 * Get 'header_textcolor' control from manager.
		 *
		 * @var WP_Customize_Control $header_textcolor_ctl .
		 */
		$header_textcolor_ctl = $wp_customize->get_control( 'header_textcolor' );
		if ( $header_textcolor_ctl ) {
			$header_textcolor_ctl->description = _x( 'Shortcut for the "Text on primary" color.', 'Color palette', 'mateo' );
			$header_textcolor_ctl->priority    = 59;
		}
	}

	if ( current_theme_supports( 'custom-background' ) ) {
		/**
		 * Get 'background_color' control from manager.
		 *
		 * @var WP_Customize_Control $native_background_color .
		 */
		$native_background_color = $wp_customize->get_control( 'background_color' );
		if ( $native_background_color ) {
			$native_background_color->active_callback = 'mateo_is_active_bgr_color';
		}
	}

	// Text color on primary.
	$wp_customize->add_setting(
		'textOnPrimary',
		array(
			'default'           => $theme_color->get_color( 'textOnPrimary' ),
			'sanitize_callback' => 'sanitize_hex_color',
			'transport'         => 'postMessage',
		)
	);
	$wp_customize->add_control(
		new WP_Customize_Color_Control(
			$wp_customize,
			'textOnPrimary',
			array(
				'label'       => _x( 'Text on primary', 'Color palette', 'mateo' ),
				'description' => _x( 'The color of the text displayed over elements having primary color as a background.', 'Color palette', 'mateo' ),
				'section'     => 'colors',
			)
		)
	);
	// Text color on primary.
	$wp_customize->add_setting(
		'link',
		array(
			'default'           => $theme_color->get_color( 'link' ),
			'sanitize_callback' => 'sanitize_hex_color',
			'transport'         => 'postMessage',
		)
	);
	$wp_customize->add_control(
		new WP_Customize_Color_Control(
			$wp_customize,
			'link',
			array(
				'label'       => _x( 'Link color', 'Color palette', 'mateo' ),
				'description' => _x( 'Link color setting overrides primary color used for links.', 'Color palette', 'mateo' ),
				'section'     => 'colors',
			)
		)
	);

	// Content background.
	$wp_customize->add_setting(
		'contentBgr',
		array(
			'default'           => $theme_color->get_color( 'contentBgr' ),
			'sanitize_callback' => 'sanitize_hex_color',
			'transport'         => 'postMessage',
		)
	);
	$wp_customize->add_control(
		new WP_Customize_Color_Control(
			$wp_customize,
			'contentBgr',
			array(
				'label'   => _x( 'Content background', 'Color palette', 'mateo' ),
				'section' => 'colors',
			)
		)
	);

	// Page background color.
	$wp_customize->add_setting(
		'pageBgr',
		array(
			'default'           => $theme_color->get_color( 'pageBgr' ),
			'sanitize_callback' => 'sanitize_hex_color',
			'transport'         => 'postMessage',
		)
	);
	$wp_customize->add_control(
		new WP_Customize_Color_Control(
			$wp_customize,
			'pageBgr',
			array(
				'label'   => _x( 'Page background', 'Color palette', 'mateo' ),
				'section' => 'colors',
			)
		)
	);

	// Add main text color.
	$wp_customize->add_setting(
		'textBody',
		array(
			'default'           => $theme_color->get_color( 'textBody' ),
			'sanitize_callback' => 'sanitize_hex_color',
			'transport'         => 'postMessage',
		)
	);
	$wp_customize->add_control(
		new WP_Customize_Color_Control(
			$wp_customize,
			'textBody',
			array(
				'label'   => _x( 'Text color', 'Color palette', 'mateo' ),
				'section' => 'colors',
			)
		)
	);

	// Site title color.
	$wp_customize->add_setting(
		'title',
		array(
			'default'           => $theme_color->get_color( 'title' ),
			'sanitize_callback' => 'sanitize_hex_color',
			'transport'         => 'postMessage',
		)
	);
	$wp_customize->add_control(
		new WP_Customize_Color_Control(
			$wp_customize,
			'title',
			array(
				'label'       => _x( 'Site Title Color / Header Image Displayed', 'Color palette', 'mateo' ),
				'description' => _x( 'Applied to the site title if header image or featured image is displayed in header.', 'Color palette', 'mateo' ),
				'section'     => 'colors',
			)
		)
	);
	// Site title background.
	$wp_customize->add_setting(
		'titleBgr',
		array(
			'default'           => $theme_color->get_color( 'titleBgr' ),
			'sanitize_callback' => 'sanitize_hex_color',
			'transport'         => 'postMessage',
		)
	);
	$wp_customize->add_control(
		new WP_Customize_Color_Control(
			$wp_customize,
			'titleBgr',
			array(
				'label'           => _x( 'Site Title Background / Header Image Displayed', 'Color palette', 'mateo' ),
				'description'     => _x( 'Applied to the site title background if header image or featured image is displayed in header.', 'Color palette', 'mateo' ),
				'section'         => 'colors',
				'active_callback' => 'mateo_is_active_title_bgr',
			)
		)
	);

	$wp_customize->add_setting(
		'titleBgrDisable',
		array(
			'default'           => 0,
			'sanitize_callback' => 'mateo_sanitize_checkbox',
			'transport'         => 'postMessage',
		)
	);
	$wp_customize->add_control(
		new WP_Customize_Control(
			$wp_customize,
			'titleBgrDisable',
			array(
				'label'   => esc_html__( 'Disable Site Title Background', 'mateo' ),
				'section' => 'colors',
				'type'    => 'checkbox',
			)
		)
	);
	// Site title color if header image is not displayed.
	$wp_customize->add_setting(
		'titleNoHeader',
		array(
			'default'           => $theme_color->get_color( 'titleNoHeader' ),
			'sanitize_callback' => 'sanitize_hex_color',
			'transport'         => 'postMessage',
		)
	);
	$wp_customize->add_control(
		new WP_Customize_Color_Control(
			$wp_customize,
			'titleNoHeader',
			array(
				'label'       => _x( 'Site Title Color / Header Image Disabled', 'Color palette', 'mateo' ),
				'description' => _x( 'Applied to the site title if header image or featured image is NOT displayed in header.', 'Color palette', 'mateo' ),
				'section'     => 'colors',
			)
		)
	);
	mateo_customize_section_features( $wp_customize );

		/*
	 * Layout section.
	 */
	$wp_customize->add_section(
		'mateo-layout-section',
		array(
			'title'    => esc_html__( 'Base Layout', 'mateo' ),
			'priority' => 121,
		)
	);

	$wp_customize->add_setting(
		'sidebar-position',
		array(
			'default'           => 'default',
			'transport'         => 'postMessage',
			'sanitize_callback' => 'mateo_sidebar_sanitizer',
		)
	);
	// Global site layout.
	$wp_customize->add_setting(
		MATEO_SETTING_LAYOUT_ID,
		array(
			'default'           => MATEO_LAYOUT_1,
			'transport'         => 'refresh',
			'sanitize_callback' => 'mateo_layout_sanitizer',
		)
	);
	$wp_customize->add_control(
		'layout',
		array(
			'type'        => 'radio',
			'section'     => 'mateo-layout-section',
			'label'       => __( 'Site Layout', 'mateo' ),
			'description' => __( 'Choose between 1 or 2 column layout. Note that 2 column layout can be reset in post editor.', 'mateo' ),
			'settings'    => MATEO_SETTING_LAYOUT_ID,
			'choices'     => array(
				MATEO_LAYOUT_2 => __( '2 columns with a sidebar', 'mateo' ),
				MATEO_LAYOUT_1 => __( '1 column', 'mateo' ),
			),
		)
	);
	$wp_customize->add_control(
		'sidebar-position',
		array(
			'type'        => 'radio',
			'section'     => 'mateo-layout-section',
			'label'       => __( 'Sidebar position', 'mateo' ),
			'description' => _x( 'Available for two column layout', 'Sidebar position', 'mateo' ),
			'settings'    => 'sidebar-position',
			'choices'     => array(
				// For RTL default is on the left.
				'default' => is_rtl() ? _x( 'Left', 'Sidebar position', 'mateo' ) : _x( 'Right', 'Sidebar position', 'mateo' ),
				'reverse' => is_rtl() ? _x( 'Right', 'Sidebar position', 'mateo' ) : _x( 'Left', 'Sidebar position', 'mateo' ),
			),
		)
	);
	// Header width.
	$wp_customize->add_setting(
		new Width_Customize_Settings(
			$wp_customize,
			'header-width',
			array(
				'sanitize_callback' => array( Width_Customize_Settings::class, 'sanitize_callback' ),
			)
		)
	);
	$wp_customize->add_control(
		new Width_Customize_Control(
			$wp_customize,
			'header-width',
			array( 'label' => esc_html__( 'Header width', 'mateo' ) )
		)
	);

	// Site title width.
	$wp_customize->add_setting(
		new Width_Customize_Settings(
			$wp_customize,
			'title-width',
			array(
				'sanitize_callback' => array( Width_Customize_Settings::class, 'sanitize_callback' ),
			)
		)
	);
	$wp_customize->add_control(
		new Width_Customize_Control(
			$wp_customize,
			'title-width',
			array(
				'label' => esc_html__( 'Site title width', 'mateo' ),
			)
		)
	);

	// Content width.
	$wp_customize->add_setting(
		new Width_Customize_Settings(
			$wp_customize,
			'content-width',
			array(
				'sanitize_callback' => array( Width_Customize_Settings::class, 'sanitize_callback' ),
			)
		)
	);
	$wp_customize->add_control(
		new Width_Customize_Control(
			$wp_customize,
			'content-width',
			array(
				'label' => esc_html__( 'Content width', 'mateo' ),
			)
		)
	);
	// Footer width.
	$wp_customize->add_setting(
		new Width_Customize_Settings(
			$wp_customize,
			'footer-width',
			array(
				'sanitize_callback' => array( Width_Customize_Settings::class, 'sanitize_callback' ),
			)
		)
	);
	$wp_customize->add_control(
		new Width_Customize_Control(
			$wp_customize,
			'footer-width',
			array( 'label' => esc_html__( 'Footer width', 'mateo' ) )
		)
	);

	// Typography.
	$wp_customize->add_section(
		'mateo-typography-base',
		array(
			'title'    => esc_html__( 'Typography', 'mateo' ),
			'priority' => 122,
		)
	);
	// Font section title.
	$font_title_control = new Title_Pseudo_Control(
		$wp_customize,
		'typo-section-global',
		array(
			'label'    => esc_html__( 'Global', 'mateo' ),
			'section'  => 'mateo-typography-base',
			'settings' => array(),
		)
	);

	$wp_customize->add_control( $font_title_control );

	// Prepare fonts.
	$google_fonts = Fonts::get_instance();
	if ( ! $google_fonts->has_fonts() ) {
		$google_fonts->set_fonts( mateo_get_fonts() );
	}
	if ( ! $google_fonts->is_option_ready() ) {
		$google_fonts->init_options();
	}
	// Font family.
	$wp_customize->add_setting(
		'font-family',
		array(
			'default'           => 'roboto', // Must be a font key, not a font name.
			'sanitize_callback' => 'mateo_sanitize_font_family',
			'transport'         => 'postMessage',
		)
	);
	$wp_customize->add_control(
		'font-family',
		array(
			'label'       => esc_html__( 'Font Family', 'mateo' ),
			'description' => esc_html__( 'Font family is applied to all elements of the page', 'mateo' ),
			'section'     => 'mateo-typography-base',
			'type'        => 'select',
			'choices'     => $google_fonts->get_fonts_options( Fonts::INHERIT_OFF ),
		)
	);

	// Font Variant.
	$wp_customize->add_setting(
		'font-variant',
		array(
			'default'           => '',
			'sanitize_callback' => 'mateo_sanitize_font_variant',
			'transport'         => 'postMessage',
		)
	);

	$wp_customize->add_control(
		'font-variant',
		array(
			'label'       => esc_html__( 'Font Variant', 'mateo' ),
			'description' => esc_html__( 'By default, the theme loads 3 font variants: light(300), normal(regular) and bold(700). You can reduce this choice to one variant.', 'mateo' ),
			'section'     => 'mateo-typography-base',
			'type'        => 'select',
			'choices'     => $google_fonts->get_variants_options(),
		)
	);

	// Font Size.
	$wp_customize->add_setting(
		new Font_Size_Setting(
			$wp_customize,
			'font-size',
			array(
				'default'           => $google_fonts->get_css_rule( 'fontSize', 16 ),
				'sanitize_callback' => array( Font_Size_Setting::class, 'sanitize_callback' ),
			)
		)
	);

	$wp_customize->add_control(
		new Font_Size_Control(
			$wp_customize,
			'font-size',
			array(
				'label'       => esc_html__( 'Font Size', 'mateo' ),
				'description' => esc_html__( 'Font size setting allows to configure the base size of all elements in pixels. Default value: 16.', 'mateo' ),
				'section'     => 'mateo-typography-base',
			)
		)
	);
	// Line Height.
	$wp_customize->add_setting(
		new Font_Line_Setting(
			$wp_customize,
			'font-line-height',
			array(
				'default'           => $google_fonts->get_css_rule( 'lineHeight', 1.5 ),
				'sanitize_callback' => array( Font_Line_Setting::class, 'sanitize_callback' ),
			)
		)
	);
	$wp_customize->add_control(
		new Font_Line_Control(
			$wp_customize,
			'font-line-height',
			array(
				'label'       => esc_html__( 'Line Height', 'mateo' ),
				'description' => esc_html__( "Line height setting allows to configure the paragraph's height. Default value: 1.5", 'mateo' ),
				'section'     => 'mateo-typography-base',
				'min'         => 1,
			)
		)
	);

	// Headings font section.
	$wp_customize->add_control(
		new Title_Pseudo_Control(
			$wp_customize,
			'typo-section-headers',
			array(
				'label'    => esc_html__( 'Headers', 'mateo' ),
				'section'  => 'mateo-typography-base',
				'settings' => array(),
			)
		)
	);

	// Font family.
	$wp_customize->add_setting(
		'font-family-headers',
		array(
			'default'           => '',
			'sanitize_callback' => 'mateo_sanitize_font_family',
			'transport'         => 'postMessage',
		)
	);
	$wp_customize->add_control(
		'font-family-headers',
		array(
			'label'       => esc_html__( 'Headers Font Family', 'mateo' ),
			'description' => esc_html__( 'Headers font family is applied to site title and H1, H2, H3, H4, H5, H6 elements.', 'mateo' ),
			'section'     => 'mateo-typography-base',
			'type'        => 'select',
			'choices'     => $google_fonts->get_fonts_options(),
		)
	);

	// Font Variant.
	$wp_customize->add_setting(
		'font-variant-headers',
		array(
			'default'           => '',
			'sanitize_callback' => 'mateo_sanitize_font_variant',
			'transport'         => 'postMessage',
		)
	);

	$wp_customize->add_control(
		'font-variant-headers',
		array(
			'label'   => esc_html__( 'Headers Font Variant', 'mateo' ),
			'section' => 'mateo-typography-base',
			'type'    => 'select',
			'choices' => $google_fonts->get_variants_options(),
		)
	);

	// Headers Font Size.
	$wp_customize->add_setting(
		new Font_Size_Setting(
			$wp_customize,
			'font-size-headers',
			array(
				'default'           => $google_fonts->get_css_rule( 'fontSizeHeaders', 1 ),
				'min'               => 1,
				'is_float'          => true,
				'sanitize_callback' => array( Font_Size_Setting::class, 'sanitize_callback' ),
			)
		)
	);

	$wp_customize->add_control(
		new Font_Size_Control(
			$wp_customize,
			'font-size-headers',
			array(
				'label'       => esc_html__( 'Headers Font Size', 'mateo' ),
				'description' => esc_html__( 'Font size setting is relative to the page font size and expressed in REM. Default value: 1. ', 'mateo' ),
				'section'     => 'mateo-typography-base',
				'min'         => 1,
				'step'        => 0.1,
			)
		)
	);

	// Headers Line Height.
	$wp_customize->add_setting(
		new Font_Line_Setting(
			$wp_customize,
			'font-line-height-headers',
			array(
				'default'           => $google_fonts->get_css_rule( 'lineHeightHeaders', 1.2 ),
				'sanitize_callback' => array( Font_Line_Setting::class, 'sanitize_callback' ),
			)
		)
	);

	$wp_customize->add_control(
		new Font_Line_Control(
			$wp_customize,
			'font-line-height-headers',
			array(
				'label'       => esc_html__( 'Headers Line Height', 'mateo' ),
				'description' => esc_html__( 'Default value: 1.2 . ', 'mateo' ),
				'section'     => 'mateo-typography-base',
			)
		)
	);

	// Site title font section.
	$wp_customize->add_control(
		new Title_Pseudo_Control(
			$wp_customize,
			'typo-section-site-title',
			array(
				'label'    => esc_html__( 'Site title', 'mateo' ),
				'section'  => 'mateo-typography-base',
				'settings' => array(),
			)
		)
	);

	// Site title font family.
	$wp_customize->add_setting(
		'font-family-title',
		array(
			'default'           => '',
			'sanitize_callback' => 'mateo_sanitize_font_family',
			'transport'         => 'postMessage',
		)
	);
	$wp_customize->add_control(
		'font-family-title',
		array(
			'label'       => esc_html__( 'Title Font Family', 'mateo' ),
			'description' => esc_html__( 'Title font family inherits from Headers font family by default.', 'mateo' ),
			'section'     => 'mateo-typography-base',
			'type'        => 'select',
			'choices'     => $google_fonts->get_fonts_options(),
		)
	);

	// Site title font variant.
	$wp_customize->add_setting(
		'font-variant-title',
		array(
			'default'           => '',
			'sanitize_callback' => 'mateo_sanitize_font_variant',
			'transport'         => 'postMessage',
		)
	);

	$wp_customize->add_control(
		'font-variant-title',
		array(
			'label'   => esc_html__( 'Title Font Variant', 'mateo' ),
			'section' => 'mateo-typography-base',
			'type'    => 'select',
			'choices' => $google_fonts->get_variants_options(),
		)
	);

	// Font Size of the site title.
	$wp_customize->add_setting(
		new Font_Size_Setting(
			$wp_customize,
			'font-size-title',
			array(
				'default'           => $google_fonts->get_css_rule( 'fontSizeTitle', 4 ),
				'min'               => 1,
				'is_float'          => true,
				'sanitize_callback' => array( Font_Size_Setting::class, 'sanitize_callback' ),
			)
		)
	);

	$wp_customize->add_control(
		new Font_Size_Control(
			$wp_customize,
			'font-size-title',
			array(
				'label'       => esc_html__( 'Title Font Size', 'mateo' ),
				'description' => esc_html__( 'Font size setting is relative to the page font size and expressed in REM. Default value: 4. ', 'mateo' ),
				'section'     => 'mateo-typography-base',
				'min'         => 1,
				'step'        => 0.1,
			)
		)
	);
	// Site Title Line Height.
	$wp_customize->add_setting(
		new Font_Line_Setting(
			$wp_customize,
			'font-line-height-title',
			array(
				'default'           => 0, // 0 -> inherits from headers line height.
				'min'               => 0,
				'sanitize_callback' => array( Font_Line_Setting::class, 'sanitize_callback' ),
			)
		)
	);

	$wp_customize->add_control(
		new Font_Line_Control(
			$wp_customize,
			'font-line-height-title',
			array(
				'label'       => esc_html__( 'Title Line Height', 'mateo' ),
				'description' => esc_html__( 'Default is 0 (zero) to inherit from Headers line height.', 'mateo' ),
				'section'     => 'mateo-typography-base',
				'min'         => 0,
				'step'        => 0.1,
			)
		)
	);

	// Monospace fonts settings.
	$wp_customize->add_control(
		new Title_Pseudo_Control(
			$wp_customize,
			'typo-section-monospace-font',
			array(
				'label'    => esc_html__( 'Preformatted text (monospace)', 'mateo' ),
				'section'  => 'mateo-typography-base',
				'settings' => array(),
			)
		)
	);

	// Monospace Font family.
	$wp_customize->add_setting(
		'font-family-monospace',
		array(
			'default'           => '',
			'sanitize_callback' => 'mateo_sanitize_font_family',
			'transport'         => 'postMessage',
		)
	);

	$wp_customize->add_control(
		'font-family-monospace',
		array(
			'label'       => esc_html__( 'Monospace Font Family', 'mateo' ),
			'description' => esc_html__( 'Monospace font family is applied to tags like <pre>, <code>, etc.', 'mateo' ),
			'section'     => 'mateo-typography-base',
			'type'        => 'select',
			'choices'     => $google_fonts->get_fonts_options( Fonts::FAMILY_MONOSPACE ),
		)
	);

	// Monospace Font Variant.
	$wp_customize->add_setting(
		'font-variant-monospace',
		array(
			'default'           => '',
			'sanitize_callback' => 'mateo_sanitize_font_variant',
			'transport'         => 'postMessage',
		)
	);
	$wp_customize->add_control(
		'font-variant-monospace',
		array(
			'label'   => esc_html__( 'Monospace Font Variant', 'mateo' ),
			'section' => 'mateo-typography-base',
			'type'    => 'select',
			'choices' => $google_fonts->get_variants_options(),
		)
	);

	// Monospace Font Size.
	$wp_customize->add_setting(
		new Font_Size_Setting(
			$wp_customize,
			'font-size-monospace',
			array(
				'default'           => $google_fonts->get_css_rule( 'fontSizeMonospace', 0.87 ),
				'is_float'          => true,
				'min'               => 0.5,
				'sanitize_callback' => array( Font_Size_Setting::class, 'sanitize_callback' ),

			)
		)
	);

	$wp_customize->add_control(
		new Font_Size_Control(
			$wp_customize,
			'font-size-monospace',
			array(
				'label'       => esc_html__( 'Monospace Font Size', 'mateo' ),
				'description' => esc_html__( 'Font size setting is relative to the page font size and expressed in REM. Default value: 0.87 .', 'mateo' ),
				'section'     => 'mateo-typography-base',
				'min'         => 0.5,
				'step'        => 0.01,
			)
		)
	);
	// Monospace Line Height.
	$wp_customize->add_setting(
		new Font_Line_Setting(
			$wp_customize,
			'font-line-height-monospace',
			array(
				'default'           => $google_fonts->get_css_rule( 'lineHeightMonospace', 1.5 ),
				'sanitize_callback' => array( Font_Line_Setting::class, 'sanitize_callback' ),
			)
		)
	);

	$wp_customize->add_control(
		new Font_Line_Control(
			$wp_customize,
			'font-line-height-monospace',
			array(
				'label'       => esc_html__( 'Monospace Line Height', 'mateo' ),
				'description' => esc_html__( 'Default value: 1.5 .', 'mateo' ),
				'section'     => 'mateo-typography-base',
			)
		)
	);

	/*
	 * Header Section.
	 */
	$wp_customize->add_section(
		'mateo-header-section',
		array(
			'title'    => esc_html__( 'Header', 'mateo' ),
			'priority' => 122,
		)
	);

	$wp_customize->add_setting(
		'mateo-title-alignment',
		array(
			'default'           => 'center',
			'transport'         => 'postMessage',
			'sanitize_callback' => 'mateo_sanitize_title_alignment',
		)
	);

	$wp_customize->add_control(
		'mateo-title-alignment',
		array(
			'type'        => 'select',
			'section'     => 'mateo-header-section',
			'label'       => esc_html__( 'Title Alignment', 'mateo' ),
			'description' => esc_html__( 'You can customize title color and background in Colors section.', 'mateo' ),
			'choices'     => mateo_get_title_alignment(),
		)
	);

	$wp_customize->add_setting(
		'mateo-title-transform',
		array(
			'default'           => 'inherit',
			'transport'         => 'postMessage',
			'sanitize_callback' => 'mateo_sanitize_text_transform',
		)
	);

	$wp_customize->add_control(
		'mateo-title-transform',
		array(
			'type'    => 'select',
			'section' => 'mateo-header-section',
			'label'   => __( 'Title Transform', 'mateo' ),
			'choices' => mateo_get_text_transform(),
		)
	);

	/*
	* Theme Footer panel.
	*/
	$wp_customize->add_panel(
		'mateo-footer-panel',
		array(
			'title' => esc_html__( 'Footer', 'mateo' ),
		)
	);

	// Primary Footer section.
	$wp_customize->add_section(
		'mateo-primary-footer',
		array(
			'title' => esc_html__( 'Primary Footer', 'mateo' ),
			'panel' => 'mateo-footer-panel',
		)
	);
	// Primary footer widgets number.
	$wp_customize->add_setting(
		'footer-widgets',
		array(
			'default'           => 4,
			'transport'         => 'postMessage',
			'sanitize_callback' => 'mateo_sanitize_widget_number',
		)
	);

	$wp_customize->add_control(
		'footer-widgets',
		array(
			'type'        => 'number',
			'section'     => 'mateo-primary-footer',
			'label'       => __( 'Number Of Widgets Per Line', 'mateo' ),
			'input_attrs' => array(
				'min' => 1,
				'step => 1',
			),
		)
	);

	// Primary footer widgets alignment.
	$wp_customize->add_setting(
		'footer-alignment',
		array(
			'default'           => 'space-between',
			'transport'         => 'postMessage',
			'sanitize_callback' => 'mateo_sanitize_flex_alignment',
		)
	);

	$wp_customize->add_control(
		'footer-alignment',
		array(
			'type'    => 'select',
			'section' => 'mateo-primary-footer',
			'label'   => __( 'Alignment', 'mateo' ),
			'choices' => mateo_get_flex_alignment(),
		)
	);

	$primary_footer_font_size = $wp_customize->add_setting(
		new Font_Size_Setting(
			$wp_customize,
			'footer-font-size',
			array(
				'default'           => 0.9,
				'min'               => 0.5,
				'is_float'          => true,
				'sanitize_callback' => array( Font_Size_Setting::class, 'sanitize_callback' ),
			)
		)
	);

	$primary_footer_font_unit = $wp_customize->add_setting(
		'footer-font-unit',
		array(
			'default'           => 'rem',
			'transport'         => 'postMessage',
			'sanitize_callback' => 'mateo_sanitize_unit',
		)
	);

	$primary_footer_font_control = new Group_Control(
		$wp_customize,
		'footer-font-group',
		array(
			'label'       => esc_html__( 'Font Size', 'mateo' ),
			'description' => esc_html__( 'By default font size is relative to the page font size and expressed in REM. Default value: 0.9 .', 'mateo' ),
			'section'     => 'mateo-primary-footer',
			'settings'    => array(
				'default'                     => $primary_footer_font_size->id,
				$primary_footer_font_unit->id => $primary_footer_font_unit->id,
			),
			'controls'    => array(
				'default'                     => new Font_Size_Control(
					$wp_customize,
					'footer-font-size',
					array(
						'min'  => 0.5,
						'step' => 0.01,
					)
				),
				$primary_footer_font_unit->id => new WP_Customize_Control(
					$wp_customize,
					'footer-font-unit',
					array(
						'type'    => 'select',
						'choices' => mateo_get_units(),
					)
				),
			),
		)
	);
	$wp_customize->add_control( $primary_footer_font_control );
	// Primary Footer Widget Padding.
	$wp_customize->add_setting(
		'footer-widget-pad-top',
		array(
			'default'           => '1',
			'transport'         => 'postMessage',
			'sanitize_callback' => 'mateo_sanitize_padding',
		)
	);
	$wp_customize->add_setting(
		'footer-widget-pad-right',
		array(
			'default'           => '1',
			'transport'         => 'postMessage',
			'sanitize_callback' => 'mateo_sanitize_padding',
		)
	);
	$wp_customize->add_setting(
		'footer-widget-pad-bottom',
		array(
			'default'           => '1',
			'transport'         => 'postMessage',
			'sanitize_callback' => 'mateo_sanitize_padding',
		)
	);
	$wp_customize->add_setting(
		'footer-widget-pad-left',
		array(
			'default'           => '1',
			'transport'         => 'postMessage',
			'sanitize_callback' => 'mateo_sanitize_padding',
		)
	);
	$wp_customize->add_setting(
		'footer-widget-pad-unit',
		array(
			'default'           => 'rem',
			'transport'         => 'postMessage',
			'sanitize_callback' => 'mateo_sanitize_unit',
		)
	);
	$primary_footer_widget_padding = new Group_Control(
		$wp_customize,
		'footer-widget-padding-group',
		array(
			'label'    => esc_html__( 'Widget Inner Padding', 'mateo' ),
			'section'  => 'mateo-primary-footer',
			'css'      => 'mateo-width-50',
			'settings' => array(
				'default'                  => 'footer-widget-pad-top',
				'footer-widget-pad-right'  => 'footer-widget-pad-right',
				'footer-widget-pad-bottom' => 'footer-widget-pad-bottom',
				'footer-widget-pad-left'   => 'footer-widget-pad-left',
				'footer-widget-pad-unit'   => 'footer-widget-pad-unit',
			),
			'controls' => array(
				'default'                  => new WP_Customize_Control(
					$wp_customize,
					'footer-widget-pad-top',
					array(
						'label'       => esc_html__( 'Top', 'mateo' ),
						'type'        => 'number',
						'input_attrs' => array(
							'min'  => 0,
							'step' => 0.01,
						),
					)
				),
				'footer-widget-pad-bottom' => new WP_Customize_Control(
					$wp_customize,
					'footer-widget-pad-bottom',
					array(
						'label'       => esc_html__( 'Bottom', 'mateo' ),
						'type'        => 'number',
						'input_attrs' => array(
							'min'  => 0,
							'step' => 0.01,
						),
					)
				),
				'footer-widget-pad-left'   => new WP_Customize_Control(
					$wp_customize,
					'footer-widget-pad-left',
					array(
						'label'       => esc_html__( 'Left', 'mateo' ),
						'type'        => 'number',
						'input_attrs' => array(
							'min'  => 0,
							'step' => 0.01,
						),
					)
				),
				'footer-widget-pad-right'  => new WP_Customize_Control(
					$wp_customize,
					'footer-widget-pad-right',
					array(
						'label'       => esc_html__( 'Right', 'mateo' ),
						'type'        => 'number',
						'input_attrs' => array(
							'min'  => 0,
							'step' => 0.01,
						),
					)
				),
				'footer-widget-pad-unit'   => new WP_Customize_Control(
					$wp_customize,
					'footer-widget-pad-unit',
					array(
						'label'   => esc_html__( 'Units', 'mateo' ),
						'type'    => 'select',
						'choices' => mateo_get_units(),
					)
				),
			),
		)
	);
	$wp_customize->add_control( $primary_footer_widget_padding );

	// Footer widget inner text alignment.
	$wp_customize->add_setting(
		'footer-widget-alignment',
		array(
			'default'           => 'inherit',
			'transport'         => 'postMessage',
			'sanitize_callback' => 'mateo_sanitize_text_alignment',
		)
	);

	$wp_customize->add_control(
		'footer-widget-alignment',
		array(
			'type'    => 'select',
			'section' => 'mateo-primary-footer',
			'label'   => __( 'Widget Text Alignment', 'mateo' ),
			'choices' => mateo_get_text_alignment(),
		)
	);

	$wp_customize->add_setting(
		'footer-widget-title-transform',
		array(
			'default'           => 'uppercase',
			'transport'         => 'postMessage',
			'sanitize_callback' => 'mateo_sanitize_text_transform',
		)
	);

	$wp_customize->add_control(
		'footer-widget-title-transform',
		array(
			'type'    => 'select',
			'section' => 'mateo-primary-footer',
			'label'   => __( 'Widget Title Transform', 'mateo' ),
			'choices' => mateo_get_text_transform(),
		)
	);

	// Secondary Footer section.
	$wp_customize->add_section(
		'mateo-secondary-footer',
		array(
			'title' => esc_html__( 'Secondary Footer', 'mateo' ),
			'panel' => 'mateo-footer-panel',
		)
	);
	// Secondary footer widgets number.
	$wp_customize->add_setting(
		'footer2-widgets',
		array(
			'default'           => 1,
			'transport'         => 'postMessage',
			'sanitize_callback' => 'mateo_sanitize_widget_number',
		)
	);

	$wp_customize->add_control(
		'footer2-widgets',
		array(
			'type'        => 'number',
			'section'     => 'mateo-secondary-footer',
			'label'       => __( 'Number Of Widgets Per Line', 'mateo' ),
			'input_attrs' => array(
				'min' => 1,
				'step => 1',
			),
		)
	);

	// Secondary footer widgets alignment.
	$wp_customize->add_setting(
		'footer2-alignment',
		array(
			'default'           => 'center',
			'transport'         => 'postMessage',
			'sanitize_callback' => 'mateo_sanitize_flex_alignment',
		)
	);

	$wp_customize->add_control(
		'footer2-alignment',
		array(
			'type'    => 'select',
			'section' => 'mateo-secondary-footer',
			'label'   => __( 'Alignment', 'mateo' ),
			'choices' => mateo_get_flex_alignment(),
		)
	);
	// Secondary footer font size group.
	$wp_customize->add_setting(
		new Font_Size_Setting(
			$wp_customize,
			'footer2-font-size',
			array(
				'default'           => 0.9,
				'min'               => 0.5,
				'is_float'          => true,
				'sanitize_callback' => array( Font_Size_Setting::class, 'sanitize_callback' ),
			)
		)
	);

	$wp_customize->add_setting(
		'footer2-font-unit',
		array(
			'default'           => 'rem',
			'transport'         => 'postMessage',
			'sanitize_callback' => 'mateo_sanitize_unit',
		)
	);

	$secondary_footer_font_control = new Group_Control(
		$wp_customize,
		'footer2-font-group',
		array(
			'label'       => esc_html__( 'Font Size', 'mateo' ),
			'description' => esc_html__( 'By default font size is relative to the page font size and expressed in REM. Default value: 0.9 .', 'mateo' ),
			'section'     => 'mateo-secondary-footer',
			'settings'    => array(
				'default'           => 'footer2-font-size',
				'footer2-font-unit' => 'footer2-font-unit',
			),
			'controls'    => array(
				'default'           => new Font_Size_Control(
					$wp_customize,
					'footer2-font-size',
					array(
						'min'  => 0.5,
						'step' => 0.01,
					)
				),
				'footer2-font-unit' => new WP_Customize_Control(
					$wp_customize,
					'footer2-font-unit',
					array(
						'type'    => 'select',
						'choices' => mateo_get_units(),
					)
				),
			),
		)
	);
	$wp_customize->add_control( $secondary_footer_font_control );

	// Secondary Footer Widget Padding Group.
	$wp_customize->add_setting(
		'footer2-widget-pad-top',
		array(
			'default'           => '1',
			'transport'         => 'postMessage',
			'sanitize_callback' => 'mateo_sanitize_padding',
		)
	);
	$wp_customize->add_setting(
		'footer2-widget-pad-right',
		array(
			'default'           => '1',
			'transport'         => 'postMessage',
			'sanitize_callback' => 'mateo_sanitize_padding',
		)
	);
	$wp_customize->add_setting(
		'footer2-widget-pad-bottom',
		array(
			'default'           => '1',
			'transport'         => 'postMessage',
			'sanitize_callback' => 'mateo_sanitize_padding',
		)
	);
	$wp_customize->add_setting(
		'footer2-widget-pad-left',
		array(
			'default'           => '1',
			'transport'         => 'postMessage',
			'sanitize_callback' => 'mateo_sanitize_padding',
		)
	);
	$wp_customize->add_setting(
		'footer2-widget-pad-unit',
		array(
			'default'           => 'rem',
			'transport'         => 'postMessage',
			'sanitize_callback' => 'mateo_sanitize_unit',
		)
	);
	$footer2_padding_group = new Group_Control(
		$wp_customize,
		'footer2-widget-padding-group',
		array(
			'label'    => esc_html__( 'Widget Inner Padding', 'mateo' ),
			'section'  => 'mateo-secondary-footer',
			'css'      => 'mateo-width-50',
			'settings' => array(
				'default'                   => 'footer2-widget-pad-top',
				'footer2-widget-pad-right'  => 'footer2-widget-pad-right',
				'footer2-widget-pad-bottom' => 'footer2-widget-pad-bottom',
				'footer2-widget-pad-left'   => 'footer2-widget-pad-left',
				'footer2-widget-pad-unit'   => 'footer2-widget-pad-unit',
			),
			'controls' => array(
				'default'                   => new WP_Customize_Control(
					$wp_customize,
					'footer2-widget-pad-top',
					array(
						'label'       => esc_html__( 'Top', 'mateo' ),
						'type'        => 'number',
						'input_attrs' => array(
							'min'  => 0,
							'step' => 0.01,
						),
					)
				),
				'footer2-widget-pad-bottom' => new WP_Customize_Control(
					$wp_customize,
					'footer2-widget-pad-bottom',
					array(
						'label'       => esc_html__( 'Bottom', 'mateo' ),
						'type'        => 'number',
						'input_attrs' => array(
							'min'  => 0,
							'step' => 0.01,
						),
					)
				),
				'footer2-widget-pad-left'   => new WP_Customize_Control(
					$wp_customize,
					'footer2-widget-pad-left',
					array(
						'label'       => esc_html__( 'Left', 'mateo' ),
						'type'        => 'number',
						'input_attrs' => array(
							'min'  => 0,
							'step' => 0.01,
						),
					)
				),
				'footer2-widget-pad-right'  => new WP_Customize_Control(
					$wp_customize,
					'footer2-widget-pad-right',
					array(
						'label'       => esc_html__( 'Right', 'mateo' ),
						'type'        => 'number',
						'input_attrs' => array(
							'min'  => 0,
							'step' => 0.01,
						),
					)
				),
				'footer2-widget-pad-unit'   => new WP_Customize_Control(
					$wp_customize,
					'footer2-widget-pad-unit',
					array(
						'label'   => esc_html__( 'Units', 'mateo' ),
						'type'    => 'select',
						'choices' => mateo_get_units(),
					)
				),
			),
		)
	);
	$wp_customize->add_control( $footer2_padding_group );

	$wp_customize->add_setting(
		'footer2-widget-alignment',
		array(
			'default'           => 'center',
			'transport'         => 'postMessage',
			'sanitize_callback' => 'mateo_sanitize_text_alignment',
		)
	);

	$wp_customize->add_control(
		'footer2-widget-alignment',
		array(
			'type'    => 'select',
			'section' => 'mateo-secondary-footer',
			'label'   => __( 'Widget Text Alignment', 'mateo' ),
			'choices' => mateo_get_text_alignment(),
		)
	);
	$wp_customize->add_setting(
		'footer2-widget-title-transform',
		array(
			'default'           => 'uppercase',
			'transport'         => 'postMessage',
			'sanitize_callback' => 'mateo_sanitize_text_transform',
		)
	);

	$wp_customize->add_control(
		'footer2-widget-title-transform',
		array(
			'type'    => 'select',
			'section' => 'mateo-secondary-footer',
			'label'   => __( 'Widget Title Transform', 'mateo' ),
			'choices' => mateo_get_text_transform(),
		)
	);

	/**
	 * Site Content section.
	 */
	$wp_customize->add_section(
		'mateo-content-section',
		array(
			'title'    => esc_html__( 'Content', 'mateo' ),
			'priority' => 123,
		)
	);
	// Site Content Padding Group.
	$wp_customize->add_setting(
		'mateo-content-pad-top',
		array(
			'default'           => 3,
			'transport'         => 'postMessage',
			'sanitize_callback' => 'mateo_sanitize_padding',
		)
	);
	$wp_customize->add_setting(
		'mateo-content-pad-right',
		array(
			'default'           => 3,
			'transport'         => 'postMessage',
			'sanitize_callback' => 'mateo_sanitize_padding',
		)
	);
	$wp_customize->add_setting(
		'mateo-content-pad-bottom',
		array(
			'default'           => 3,
			'transport'         => 'postMessage',
			'sanitize_callback' => 'mateo_sanitize_padding',
		)
	);
	$wp_customize->add_setting(
		'mateo-content-pad-left',
		array(
			'default'           => 3,
			'transport'         => 'postMessage',
			'sanitize_callback' => 'mateo_sanitize_padding',
		)
	);
	$wp_customize->add_setting(
		'mateo-content-pad-unit',
		array(
			'default'           => 'rem',
			'transport'         => 'postMessage',
			'sanitize_callback' => 'mateo_sanitize_unit',
		)
	);
	$site_padding = new Group_Control(
		$wp_customize,
		'mateo-content-padding-group',
		array(
			'label'    => esc_html__( 'Padding', 'mateo' ),
			'section'  => 'mateo-content-section',
			'css'      => 'mateo-width-50',
			'settings' => array(
				'default'                  => 'mateo-content-pad-top',
				'mateo-content-pad-right'  => 'mateo-content-pad-right',
				'mateo-content-pad-bottom' => 'mateo-content-pad-bottom',
				'mateo-content-pad-left'   => 'mateo-content-pad-left',
				'mateo-content-pad-unit'   => 'mateo-content-pad-unit',
			),
			'controls' => array(
				'default'                  => new WP_Customize_Control(
					$wp_customize,
					'mateo-content-pad-top',
					array(
						'label'       => esc_html__( 'Top', 'mateo' ),
						'type'        => 'number',
						'input_attrs' => array(
							'min'  => 0,
							'step' => 0.01,
						),
					)
				),
				'mateo-content-pad-bottom' => new WP_Customize_Control(
					$wp_customize,
					'mateo-content-pad-bottom',
					array(
						'label'       => esc_html__( 'Bottom', 'mateo' ),
						'type'        => 'number',
						'input_attrs' => array(
							'min'  => 0,
							'step' => 0.01,
						),
					)
				),
				'mateo-content-pad-left'   => new WP_Customize_Control(
					$wp_customize,
					'mateo-content-pad-left',
					array(
						'label'       => esc_html__( 'Left', 'mateo' ),
						'type'        => 'number',
						'input_attrs' => array(
							'min'  => 0,
							'step' => 0.01,
						),
					)
				),
				'mateo-content-pad-right'  => new WP_Customize_Control(
					$wp_customize,
					'mateo-content-pad-right',
					array(
						'label'       => esc_html__( 'Right', 'mateo' ),
						'type'        => 'number',
						'input_attrs' => array(
							'min'  => 0,
							'step' => 0.01,
						),
					)
				),
				'mateo-content-pad-unit'   => new WP_Customize_Control(
					$wp_customize,
					'mateo-content-pad-unit',
					array(
						'label'   => esc_html__( 'Units', 'mateo' ),
						'type'    => 'select',
						'choices' => mateo_get_units(),
					)
				),
			),
		)
	);
	$wp_customize->add_control( $site_padding );

	// Site content border radius.
	$wp_customize->add_setting(
		'mateo-content-radius-top-left',
		array(
			'default'           => 0,
			'transport'         => 'postMessage',
			'sanitize_callback' => 'mateo_sanitize_border_radius',
		)
	);
	$wp_customize->add_setting(
		'mateo-content-radius-top-right',
		array(
			'default'           => 0,
			'transport'         => 'postMessage',
			'sanitize_callback' => 'mateo_sanitize_border_radius',
		)
	);
	$wp_customize->add_setting(
		'mateo-content-radius-bottom-left',
		array(
			'default'           => 0,
			'transport'         => 'postMessage',
			'sanitize_callback' => 'mateo_sanitize_border_radius',
		)
	);
	$wp_customize->add_setting(
		'mateo-content-radius-bottom-right',
		array(
			'default'           => 0,
			'transport'         => 'postMessage',
			'sanitize_callback' => 'mateo_sanitize_border_radius',
		)
	);
	$wp_customize->add_setting(
		'mateo-content-radius-unit',
		array(
			'default'           => 'px',
			'transport'         => 'postMessage',
			'sanitize_callback' => 'mateo_sanitize_unit',
		)
	);
	$site_border_radius = new Group_Control(
		$wp_customize,
		'mateo-content-radius-group',
		array(
			'label'    => esc_html__( 'Border Radius', 'mateo' ),
			'section'  => 'mateo-content-section',
			'css'      => 'mateo-width-50',
			'settings' => array(
				'default'                           => 'mateo-content-radius-top-left',
				'mateo-content-radius-top-right'    => 'mateo-content-radius-top-right',
				'mateo-content-radius-bottom-left'  => 'mateo-content-radius-bottom-left',
				'mateo-content-radius-bottom-right' => 'mateo-content-radius-bottom-right',
				'mateo-content-radius-unit'         => 'mateo-content-radius-unit',
			),
			'controls' => array(
				'default'                           => new WP_Customize_Control(
					$wp_customize,
					'mateo-content-radius-top-left',
					array(
						'label'       => esc_html__( 'Top Left', 'mateo' ),
						'type'        => 'number',
						'input_attrs' => array(
							'min'  => 0,
							'step' => 0.01,
						),
					)
				),
				'mateo-content-radius-top-right'    => new WP_Customize_Control(
					$wp_customize,
					'mateo-content-radius-top-right',
					array(
						'label'       => esc_html__( 'Top Right', 'mateo' ),
						'type'        => 'number',
						'input_attrs' => array(
							'min'  => 0,
							'step' => 0.01,
						),
					)
				),
				'mateo-content-radius-bottom-left'  => new WP_Customize_Control(
					$wp_customize,
					'mateo-content-radius-bottom-left',
					array(
						'label'       => esc_html__( 'Bottom Left', 'mateo' ),
						'type'        => 'number',
						'input_attrs' => array(
							'min'  => 0,
							'step' => 0.01,
						),
					)
				),
				'mateo-content-radius-bottom-right' => new WP_Customize_Control(
					$wp_customize,
					'mateo-content-radius-bottom-right',
					array(
						'label'       => esc_html__( 'Bottom Right', 'mateo' ),
						'type'        => 'number',
						'input_attrs' => array(
							'min'  => 0,
							'step' => 0.01,
						),
					)
				),
				'mateo-content-radius-unit'         => new Units_Control( $wp_customize, 'mateo-content-radius-unit' ),
			),
		)
	);
	$wp_customize->add_control( $site_border_radius );

	// Site content border radius.
	$wp_customize->add_setting(
		'mateo-content-margin-top',
		array(
			'default'           => 0,
			'transport'         => 'postMessage',
			'sanitize_callback' => 'mateo_sanitize_number',
		)
	);
	$wp_customize->add_setting(
		'mateo-content-margin-bottom',
		array(
			'default'           => 0,
			'transport'         => 'postMessage',
			'sanitize_callback' => 'mateo_sanitize_number',
		)
	);
	$wp_customize->add_setting(
		'mateo-content-margin-unit',
		array(
			'default'           => 'px',
			'transport'         => 'postMessage',
			'sanitize_callback' => 'mateo_sanitize_unit',
		)
	);
	$site_margin = new Group_Control(
		$wp_customize,
		'mateo-content-margin-group',
		array(
			'label'    => esc_html__( 'Margins', 'mateo' ),
			'section'  => 'mateo-content-section',
			'css'      => 'mateo-width-50',
			'settings' => array(
				'default'                     => 'mateo-content-margin-top',
				'mateo-content-margin-bottom' => 'mateo-content-margin-bottom',
				'mateo-content-margin-unit'   => 'mateo-content-margin-unit',
			),
			'controls' => array(
				'default'                     => new Space_Control(
					$wp_customize,
					'mateo-content-margin-top',
					array(
						'label' => esc_html__( 'Top Margin', 'mateo' ),
						'min'   => null,
					)
				),
				'mateo-content-margin-bottom' => new Space_Control(
					$wp_customize,
					'mateo-content-margin-bottom',
					array(
						'label' => esc_html__( 'Bottom Margin', 'mateo' ),
						'min'   => null,
					)
				),
				'mateo-content-margin-unit'   => new Units_Control( $wp_customize, 'mateo-content-margin-unit' ),
			),
		)
	);
	$wp_customize->add_control( $site_margin );

	// Sidebar section.
	$wp_customize->add_section(
		'mateo-sidebar-section',
		array(
			'title'    => esc_html__( 'Sidebar', 'mateo' ),
			'priority' => 124,
		)
	);
	// Sidebar font size group.
	$wp_customize->add_setting(
		new Font_Size_Setting(
			$wp_customize,
			'mateo-sidebar-font-size',
			array(
				'default'           => 0.9,
				'min'               => 0.5,
				'is_float'          => true,
				'sanitize_callback' => array( Font_Size_Setting::class, 'sanitize_callback' ),
			)
		)
	);

	$wp_customize->add_setting(
		'mateo-sidebar-font-unit',
		array(
			'default'           => 'rem',
			'transport'         => 'postMessage',
			'sanitize_callback' => 'mateo_sanitize_unit',
		)
	);

	$secondary_footer_font_control = new Group_Control(
		$wp_customize,
		'mateo-sidebar-font-group',
		array(
			'label'       => esc_html__( 'Font Size', 'mateo' ),
			'description' => esc_html__( 'By default font size is relative to the page font size and expressed in REM. Default value: 0.9 .', 'mateo' ),
			'section'     => 'mateo-sidebar-section',
			'settings'    => array(
				'default'                 => 'mateo-sidebar-font-size',
				'mateo-sidebar-font-unit' => 'mateo-sidebar-font-unit',
			),
			'controls'    => array(
				'default'                 => new Font_Size_Control(
					$wp_customize,
					'mateo-sidebar-font-size',
					array(
						'min'  => 0.5,
						'step' => 0.01,
					)
				),
				'mateo-sidebar-font-unit' => new WP_Customize_Control(
					$wp_customize,
					'mateo-sidebar-font-unit',
					array(
						'type'    => 'select',
						'choices' => mateo_get_units(),
					)
				),
			),
		)
	);
	$wp_customize->add_control( $secondary_footer_font_control );

	// Sidebar Widget Padding Group.
	$wp_customize->add_setting(
		'mateo-sidebar-widget-pad-top',
		array(
			'default'           => 0,
			'transport'         => 'postMessage',
			'sanitize_callback' => 'mateo_sanitize_padding',
		)
	);
	$wp_customize->add_setting(
		'mateo-sidebar-widget-pad-right',
		array(
			'default'           => 0,
			'transport'         => 'postMessage',
			'sanitize_callback' => 'mateo_sanitize_padding',
		)
	);
	$wp_customize->add_setting(
		'mateo-sidebar-widget-pad-bottom',
		array(
			'default'           => 1,
			'transport'         => 'postMessage',
			'sanitize_callback' => 'mateo_sanitize_padding',
		)
	);
	$wp_customize->add_setting(
		'mateo-sidebar-widget-pad-left',
		array(
			'default'           => 0,
			'transport'         => 'postMessage',
			'sanitize_callback' => 'mateo_sanitize_padding',
		)
	);
	$wp_customize->add_setting(
		'mateo-sidebar-widget-pad-unit',
		array(
			'default'           => 'rem',
			'transport'         => 'postMessage',
			'sanitize_callback' => 'mateo_sanitize_unit',
		)
	);
	$sidebar_padding_group = new Group_Control(
		$wp_customize,
		'mateo-sidebar-widget-padding-group',
		array(
			'label'    => esc_html__( 'Widget Inner Padding', 'mateo' ),
			'section'  => 'mateo-sidebar-section',
			'css'      => 'mateo-width-50',
			'settings' => array(
				'default'                         => 'mateo-sidebar-widget-pad-top',
				'mateo-sidebar-widget-pad-right'  => 'mateo-sidebar-widget-pad-right',
				'mateo-sidebar-widget-pad-bottom' => 'mateo-sidebar-widget-pad-bottom',
				'mateo-sidebar-widget-pad-left'   => 'mateo-sidebar-widget-pad-left',
				'mateo-sidebar-widget-pad-unit'   => 'mateo-sidebar-widget-pad-unit',
			),
			'controls' => array(
				'default'                         => new Space_Control(
					$wp_customize,
					'mateo-sidebar-widget-pad-top',
					array(
						'label' => esc_html__( 'Top', 'mateo' ),
					)
				),
				'mateo-sidebar-widget-pad-bottom' => new Space_Control(
					$wp_customize,
					'mateo-sidebar-widget-pad-bottom',
					array(
						'label' => esc_html__( 'Bottom', 'mateo' ),
					)
				),
				'mateo-sidebar-widget-pad-left'   => new Space_Control(
					$wp_customize,
					'mateo-sidebar-widget-pad-left',
					array(
						'label' => esc_html__( 'Left', 'mateo' ),
					)
				),
				'mateo-sidebar-widget-pad-right'  => new Space_Control(
					$wp_customize,
					'mateo-sidebar-widget-pad-right',
					array(
						'label' => esc_html__( 'Right', 'mateo' ),
					)
				),
				'mateo-sidebar-widget-pad-unit'   => new Units_Control( $wp_customize, 'mateo-sidebar-widget-pad-unit' ),
			),
		)
	);
	$wp_customize->add_control( $sidebar_padding_group );

	$wp_customize->add_setting(
		'mateo-sidebar-widget-alignment',
		array(
			'default'           => 'left',
			'transport'         => 'postMessage',
			'sanitize_callback' => 'mateo_sanitize_text_alignment',
		)
	);

	$wp_customize->add_control(
		'mateo-sidebar-widget-alignment',
		array(
			'type'    => 'select',
			'section' => 'mateo-sidebar-section',
			'label'   => __( 'Widget Text Alignment', 'mateo' ),
			'choices' => mateo_get_text_alignment(),
		)
	);

	$wp_customize->add_setting(
		'mateo-sidebar-title-transform',
		array(
			'default'           => 'uppercase',
			'transport'         => 'postMessage',
			'sanitize_callback' => 'mateo_sanitize_text_transform',
		)
	);

	$wp_customize->add_control(
		'mateo-sidebar-title-transform',
		array(
			'type'    => 'select',
			'section' => 'mateo-sidebar-section',
			'label'   => __( 'Widget Title Transform', 'mateo' ),
			'choices' => mateo_get_text_transform(),
		)
	);

}

add_action( 'customize_register', 'mateo_customize_register', 11 );

if ( ! function_exists( 'mateo_get_text_alignment' ) ) {
	/**
	 * Options for text alignment.
	 *
	 * @param array $exclude Options keys to exclude from the resulting array.
	 * @return array
	 */
	function mateo_get_text_alignment( $exclude = array() ) {
		static $options;
		if ( null === $options ) {
			$options = apply_filters(
				'mateo_text_alignment',
				array(
					'inherit' => __( 'Inherit', 'mateo' ),
					'left'    => __( 'Left', 'mateo' ),
					'center'  => __( 'Center', 'mateo' ),
					'right'   => __( 'Right', 'mateo' ),
					'justify' => __( 'Justify', 'mateo' ),
				)
			);
		}

		if ( $options && ! empty( $exclude ) ) {
			return array_diff_key( $options, array_combine( $exclude, $exclude ) );
		}
		return $options;
	}
}
if ( ! function_exists( 'mateo_get_title_alignment' ) ) {
	/**
	 * Options for text alignment.
	 *
	 * @return array
	 */
	function mateo_get_title_alignment() {
		$exclude = array(
			'inherit',
			'justify',
		);
		return mateo_get_text_alignment( $exclude );
	}
}
if ( ! function_exists( 'mateo_get_text_transform' ) ) {
	/**
	 * Options for text alignment.
	 *
	 * @return array
	 */
	function mateo_get_text_transform() {
		static $options;
		if ( null === $options ) {
			$options = apply_filters(
				'mateo_text_transform',
				array(
					'inherit'    => __( 'Inherit', 'mateo' ),
					'capitalize' => __( 'Capitalize', 'mateo' ),
					'uppercase'  => __( 'Uppercase', 'mateo' ),
					'lowercase'  => __( 'Lowercase', 'mateo' ),
				)
			);
		}
		return $options;
	}
}

if ( ! function_exists( 'mateo_get_flex_alignment' ) ) {
	/**
	 * Options for items alignment in a flex row.
	 *
	 * @return array
	 */
	function mateo_get_flex_alignment() {
		static $options;
		if ( null === $options ) {
			$options = apply_filters(
				'mateo_flex_alignment',
				array(
					'flex-start'    => __( 'Left', 'mateo' ),
					'center'        => __( 'Center', 'mateo' ),
					'flex-end'      => __( 'Right', 'mateo' ),
					'space-between' => __( 'Space between', 'mateo' ),
					'space-around'  => __( 'Space around', 'mateo' ),
					'space-evenly'  => __( 'Space evenly', 'mateo' ),
				)
			);
		}
		return $options;
	}
}

if ( ! function_exists( 'mateo_sanitize_text_alignment' ) ) {
	/**
	 * Sanitizes text alignment value.
	 *
	 * @param string $value Value to sanitize.
	 *
	 * @return string|WP_Error
	 */
	function mateo_sanitize_text_alignment( $value ) {
		$options = mateo_get_text_alignment();
		if ( ! $options || ! isset( $options[ $value ] ) ) {
			return new WP_Error( 'text_alignment_not_valid', esc_html__( 'Text alignment is not valid', 'mateo' ) );
		}

		return $value;
	}
}
if ( ! function_exists( 'mateo_sanitize_title_alignment' ) ) {
	/**
	 * Sanitizes text alignment value.
	 *
	 * @param string $value Value to sanitize.
	 *
	 * @return string|WP_Error
	 */
	function mateo_sanitize_title_alignment( $value ) {
		$options = mateo_get_title_alignment();
		if ( ! $options || ! isset( $options[ $value ] ) ) {
			return new WP_Error( 'title_alignment_not_valid', esc_html__( 'Title alignment is not valid', 'mateo' ) );
		}

		return $value;
	}
}

if ( ! function_exists( 'mateo_sanitize_text_transform' ) ) {
	/**
	 * Sanitizes text alignment value.
	 *
	 * @param string $value Value to sanitize.
	 *
	 * @return string|WP_Error
	 */
	function mateo_sanitize_text_transform( $value ) {
		$options = mateo_get_text_transform();
		if ( ! $options || ! isset( $options[ $value ] ) ) {
			return new WP_Error( 'text_transform_not_valid', esc_html__( 'Text transform is not valid', 'mateo' ) );
		}

		return $value;
	}
}
if ( ! function_exists( 'mateo_sanitize_flex_alignment' ) ) {
	/**
	 * Sanitizes text alignment value.
	 *
	 * @param string $value Value to sanitize.
	 *
	 * @return string|WP_Error
	 */
	function mateo_sanitize_flex_alignment( $value ) {
		$options = mateo_get_flex_alignment();
		if ( ! $options || ! isset( $options[ $value ] ) ) {
			return new WP_Error( 'flex_alignment_not_valid', esc_html__( 'Alignment is not valid', 'mateo' ) );
		}

		return $value;
	}
}

if ( ! function_exists( 'mateo_is_active_bgr_color' ) ) {
	/**
	 * Mateo theme offers its own page background.
	 * Native WP background color control is inactive cause it conflicts with a theme page background.
	 * Note: not sure if this is OK with theme review validation rules.
	 */
	function mateo_is_active_bgr_color() {
		return false;
	}
}

if ( ! function_exists( 'mateo_is_active_title_bgr' ) ) {
	/**
	 * Checks if title background is deactivated.
	 *
	 * @return bool  TRUE if deactivated FALSE if not not.
	 */
	function mateo_is_active_title_bgr() {
		return ! get_theme_mod( 'titleBgrDisable', false );
	}
}
if ( ! function_exists( 'mateo_is_active_featured_in_header' ) ) {
	/**
	 * Checks if featured image should replace header image.
	 *
	 * @return bool  TRUE to display in header FALSE to display in post content.
	 */
	function mateo_is_active_featured_in_header() {
		return true === get_theme_mod( 'mateo-featured-in-header', true );
	}
}

if ( ! function_exists( 'mateo_sanitize_checkbox' ) ) {
	/**
	 * Sanitizes header setting value.
	 *
	 * @param mixed $value Value to sanitize.
	 *
	 * @return bool
	 */
	function mateo_sanitize_checkbox( $value ) {
		return 1 === (int) $value;
	}
}

if ( ! function_exists( 'mateo_header_render_callback' ) ) {
	/**
	 * Loads and displays search header template.
	 */
	function mateo_header_render_callback() {
		get_template_part( 'template-parts/search/header' );
	}
}

if ( ! function_exists( 'mateo_layout_sanitizer' ) ) {
	/**
	 * Sanitizes layout value.
	 *
	 * @param string $value Layout value to sanitize.
	 *
	 * @return string
	 */
	function mateo_layout_sanitizer( $value ) {
		if ( in_array( $value, array( MATEO_LAYOUT_1, MATEO_LAYOUT_2 ), true ) ) {
			return $value;
		}

		return MATEO_LAYOUT_1;
	}
}
/**
 * Outputs an Underscore template for generating CSS for the color scheme.
 *
 * The template generates the css dynamically for instant display in the
 * Customizer preview.
 */
function mateo_color_scheme_css_template() {
	$theme_color = new ThemeColor( mateo_get_color_scheme() );
	?>
	<script type="text/html" id="tmpl-mateo-color-scheme">
		<?php
		// phpcs:ignore
		echo $theme_color->get_css_template( MATEO_CSS_STRIP_EOL );
		?>
	</script>
	<?php
}

add_action( 'customize_controls_print_footer_scripts', 'mateo_color_scheme_css_template' );

if ( ! function_exists( 'mateo_fonts_css_template' ) ) {
	/**
	 * Outputs an Underscore template for generating CSS with customized fonts settings.
	 */
	function mateo_fonts_css_template() {
		$google_fonts = Fonts::get_instance();
		?>
		<script type="text/html" id="tmpl-mateo-fonts-css">
			<?php
			// phpcs:ignore
			echo MATEO_CSS_STRIP_EOL ? ThemeColor::strip_eol( $google_fonts->get_css_template() ) : $google_fonts->get_css_template();
			?>
		</script>
		<?php
	}
}
add_action( 'customize_controls_print_footer_scripts', 'mateo_fonts_css_template' );

if ( ! function_exists( 'mateo_custom_css_vars' ) ) {
	/**
	 * Outputs CSS for the primary footer with customized settings.
	 *
	 * @return string CSS
	 */
	function mateo_custom_css_vars() {
		$rules = array();
		// Mapping setting ids with CSS vars.
		$simple_map = array(
			'footer-widgets'                 => '--mateo-primary-footer-widgets-lg',
			'footer-alignment'               => '--mateo-primary-footer-align',
			'footer-widget-alignment'        => '--mateo-primary-footer-widget-align',
			'footer-widget-title-transform'  => '--mateo-primary-footer-widget-title-transform',
			'footer2-widgets'                => '--mateo-second-footer-widgets-lg',
			'footer2-alignment'              => '--mateo-second-footer-align',
			'footer2-widget-alignment'       => '--mateo-second-footer-widget-align',
			'footer2-widget-title-transform' => '--mateo-second-footer-widget-title-transform',
			'mateo-sidebar-widget-alignment' => '--mateo-sidebar-widget-align',
			'mateo-sidebar-title-transform'  => '--mateo-sidebar-widget-title-transform',
			'mateo-title-alignment'          => '--mateo-page-title-alignment',
			'mateo-title-transform'          => '--mateo-page-title-transform',
		);

		$complicated_map = array(
			// Primary footer.
			array(
				'values' => array(
					'footer-widget-pad-top'    => '--mateo-primary-footer-widget-pad-top',
					'footer-widget-pad-right'  => '--mateo-primary-footer-widget-pad-right',
					'footer-widget-pad-bottom' => '--mateo-primary-footer-widget-pad-bottom',
					'footer-widget-pad-left'   => '--mateo-primary-footer-widget-pad-left',

				),
				'concat' => array(
					'setting_id' => 'footer-widget-pad-unit',
					'default'    => 'rem',
				),
			),
			array(
				'values' => array(
					'footer-font-size' => '--mateo-primary-footer-font-size',
				),
				'concat' => array(
					'setting_id' => 'footer-font-unit',
					'default'    => 'rem',
				),
			),
			// Secondary footer.
			array(
				'values' => array(
					'footer2-widget-pad-top'    => '--mateo-second-footer-widget-pad-top',
					'footer2-widget-pad-right'  => '--mateo-second-footer-widget-pad-right',
					'footer2-widget-pad-bottom' => '--mateo-second-footer-widget-pad-bottom',
					'footer2-widget-pad-left'   => '--mateo-second-footer-widget-pad-left',

				),
				'concat' => array(
					'setting_id' => 'footer2-widget-pad-unit',
					'default'    => 'rem',
				),
			),
			array(
				'values' => array(
					'footer2-font-size' => '--mateo-second-footer-font-size',
				),
				'concat' => array(
					'setting_id' => 'footer2-font-unit',
					'default'    => 'rem',
				),
			),
			// Site content.
			array(
				'values' => array(
					'mateo-content-pad-top'    => '--mateo-site-pad-top',
					'mateo-content-pad-right'  => '--mateo-site-pad-right',
					'mateo-content-pad-bottom' => '--mateo-site-pad-bottom',
					'mateo-content-pad-left'   => '--mateo-site-pad-left',

				),
				'concat' => array(
					'setting_id' => 'mateo-content-pad-unit',
					'default'    => 'rem',
				),
			),
			array(
				'values' => array(
					'mateo-content-radius-top-left'     => '--mateo-site-radius-top-left',
					'mateo-content-radius-top-right'    => '--mateo-site-radius-top-right',
					'mateo-content-radius-bottom-left'  => '--mateo-site-radius-bottom-left',
					'mateo-content-radius-bottom-right' => '--mateo-site-radius-bottom-right',

				),
				'concat' => array(
					'setting_id' => 'mateo-content-radius-unit',
					'default'    => 'px',
				),
			),
			array(
				'values' => array(
					'mateo-content-margin-top'    => '--mateo-site-margin-top',
					'mateo-content-margin-bottom' => '--mateo-site-margin-bottom',

				),
				'concat' => array(
					'setting_id' => 'mateo-content-margin-unit',
					'default'    => 'px',
				),
			),
			// Sidebar.
			array(
				'values' => array(
					'mateo-sidebar-widget-pad-top'    => '--mateo-sidebar-widget-pad-top',
					'mateo-sidebar-widget-pad-right'  => '--mateo-sidebar-widget-pad-right',
					'mateo-sidebar-widget-pad-bottom' => '--mateo-sidebar-widget-pad-bottom',
					'mateo-sidebar-widget-pad-left'   => '--mateo-sidebar-widget-pad-left',
				),
				'concat' => array(
					'setting_id' => 'mateo-sidebar-widget-pad-unit',
					'default'    => 'rem',
				),
			),
			array(
				'values' => array(
					'mateo-sidebar-font-size' => '--mateo-sidebar-font-size',
				),
				'concat' => array(
					'setting_id' => 'mateo-sidebar-font-unit',
					'default'    => 'rem',
				),
			),

		);

		foreach ( $simple_map as $id => $css_var ) {
			$value = get_theme_mod( $id, false );
			if ( false !== $value ) {
				$rules[] = "${css_var}: ${value};";
			}
		}
		foreach ( $complicated_map as $item ) {
			if ( ! empty( $item['concat']['setting_id'] ) ) {
				$concat = get_theme_mod( $item['concat']['setting_id'], $item['concat']['default'] );
			} else {
				$concat = $item['concat']['default'];
			}
			foreach ( $item['values'] as $id => $css_var ) {
				$value = get_theme_mod( $id, false );
				if ( false !== $value ) {
					$rules[] = "${css_var}: ${value}${concat};";
				}
			}
		}

		if ( empty( $rules ) ) {
			return '';

		}

		return ':root {' . implode( '', $rules ) . ' }';
	}
}

/**
 * Append css for custom controls.
 */
function mateo_customize_control_css() {
	wp_enqueue_style(
		MATEO_CSS_CUSTOMIZE_HANDLER,
		get_template_directory_uri() . '/assets/css/customize.css',
		array(),
		MATEO_VERSION
	);
	wp_style_add_data( MATEO_CSS_CUSTOMIZE_HANDLER, 'rtl', 'replace' );
}

add_action( 'customize_controls_print_styles', 'mateo_customize_control_css' );

/**
 * Enqueues JS scripts for Customizer.
 *
 * Passes color scheme data as mateoColorScheme global.
 */
function mateo_customize_control_js() {
	wp_enqueue_script(
		MATEO_JS_CUSTOMIZE_HANDLER,
		get_template_directory_uri() . '/assets/js/' . mateo_get_js_filename( 'customize-controls' ),
		array(
			'customize-controls',
			'iris',
			'underscore',
			'wp-util',
		),
		MATEO_VERSION,
		true
	);
	wp_localize_script( MATEO_JS_CUSTOMIZE_HANDLER, 'mateoColorScheme', mateo_get_color_schemes() );
	wp_localize_script( MATEO_JS_CUSTOMIZE_HANDLER, 'mateoFonts', mateo_get_fonts() );
	wp_localize_script( MATEO_JS_CUSTOMIZE_HANDLER, 'mateoFontsDefaults', ( Fonts::get_instance() )->get_css_rules() );
}

add_action( 'customize_controls_enqueue_scripts', 'mateo_customize_control_js' );

/**
 * Binds JS handlers to make the Customizer preview reload changes asynchronously.
 */
function mateo_customize_preview_js() {
	wp_enqueue_script(
		MATEO_JS_CUSTOMIZE_HANDLER,
		get_template_directory_uri() . '/assets/js/' . mateo_get_js_filename( 'customize-preview' ),
		array( 'customize-preview' ),
		MATEO_VERSION,
		true
	);
}

add_action( 'customize_preview_init', 'mateo_customize_preview_js' );


if ( ! function_exists( 'mateo_sanitize_color_scheme' ) ) {
	/**
	 * Handles sanitization for the theme color schemes.
	 *
	 * Create your own mateo_sanitize_color_scheme() function to override
	 * in a child theme.
	 *
	 * @param string $value Color scheme name value.
	 *
	 * @return string Color scheme name.
	 */
	function mateo_sanitize_color_scheme( $value ) {
		$color_schemes = mateo_get_color_scheme_choices();

		if ( ! array_key_exists( $value, $color_schemes ) ) {
			return 'default';
		}

		return $value;
	}
}

if ( ! function_exists( 'mateo_sanitize_unit' ) ) {
	/**
	 * Sanitizes size unit.
	 *
	 * @param string $value    Some value to sanitize.
	 *
	 * @return string|WP_Error
	 */
	function mateo_sanitize_unit( $value ) {
		$options = mateo_get_units();
		if ( ! $options || ! isset( $options[ $value ] ) ) {
			return new WP_Error( 'unit_not_valid', esc_html__( 'Unit is not valid.', 'mateo' ) );
		}

		return $value;
	}
}

if ( ! function_exists( 'mateo_sanitize_number' ) ) {
	/**
	 * Value to sanitize.
	 *
	 * @param string $value    Number to sanitize.
	 *
	 * @return float|int|WP_Error Number or en error.
	 */
	function mateo_sanitize_number( $value ) {
		return is_numeric( $value )
			? ( $value * 1 )
			: new WP_Error( 'number_invalid', esc_html__( 'The value is not a valid number.', 'mateo' ) );
	}
}
if ( ! function_exists( 'mateo_sanitize_widget_number' ) ) {
	/**
	 * Value to sanitize.
	 *
	 * @param string $value    Number to sanitize.
	 *
	 * @return float|int|WP_Error Number or en error.
	 */
	function mateo_sanitize_widget_number( $value ) {
		$value = mateo_sanitize_number( $value );
		if ( is_wp_error( $value ) || $value <= 0 ) {
			$msg = esc_html__( 'Number of widgets must be bigger then 0.', 'mateo' );
			return new WP_Error( 'widget_number_not_valid', $msg );
		}
		return $value;
	}
}

if ( ! function_exists( 'mateo_sanitize_padding' ) ) {
	/**
	 * Sanitizes padding value.
	 *
	 * @param string $value Value to sanitize.
	 *
	 * @return float|int|WP_Error Value or error if not valid.
	 */
	function mateo_sanitize_padding( $value ) {
		$value = mateo_sanitize_number( $value );
		if ( is_wp_error( $value ) || $value < 0 ) {
			$msg = esc_html__( 'Padding must be equal or bigger then 0.', 'mateo' );
			return new WP_Error( 'padding_not_valid', $msg );
		}
		return $value;
	}
}
if ( ! function_exists( 'mateo_sanitize_border_radius' ) ) {
	/**
	 * Sanitizes border radius value.
	 *
	 * @param string $value Value to sanitize.
	 *
	 * @return float|int|WP_Error Value or error if not valid.
	 */
	function mateo_sanitize_border_radius( $value ) {
		$value = mateo_sanitize_number( $value );
		if ( is_wp_error( $value ) || $value < 0 ) {
			$msg = esc_html__( 'Border radius must be equal or bigger then 0.', 'mateo' );
			return new WP_Error( 'border_radius_not_valid', $msg );
		}
		return $value;
	}
}
if ( ! function_exists( 'mateo_sanitize_font_family' ) ) {
	/**
	 * Handles sanitization of a font family.
	 *
	 * @param string $value Value to sanitize.
	 *
	 * @return string|WP_Error Font family key or error if not is not found.
	 */
	function mateo_sanitize_font_family( $value ) {
		// Empty font means theme default css settings and is allowed.
		if ( '' === $value ) {
			return $value;
		}
		// Check if font exists.
		$google_font = Fonts::get_instance();
		if ( ! $google_font->has_fonts() ) {
			$google_font->set_fonts( mateo_get_fonts() );
		}

		return $google_font->exists( $value ) ? $value : new WP_Error( 'font_not_valid', esc_html__( 'Font family not valid', 'mateo' ) );
	}
}
if ( ! function_exists( 'mateo_sanitize_font_variant' ) ) {
	/**
	 * Handles sanitization of a font variant.
	 *
	 * @param string $value Value to sanitize.
	 *
	 * @return string|WP_Error Font variant or error if not is not found.
	 */
	function mateo_sanitize_font_variant( $value ) {
		// Empty font variant means theme default css settings and is allowed.
		if ( '' === $value ) {
			return $value;
		}
		// Check if font variant exists.
		$google_font = Fonts::get_instance();
		if ( ! $google_font->has_fonts() ) {
			$google_font->set_fonts( mateo_get_fonts() );
		}
		if ( ! $google_font->is_option_ready() ) {
			$google_font->init_options();
		}

		return in_array( $value, $google_font->get_variants_options(), true )
			? $value
			: new WP_Error( 'font_variant_not_valid', esc_html__( 'Font variant not valid', 'mateo' ) );
	}
}

if ( ! function_exists( 'mateo_layout_css' ) ) :
	/**
	 * Generates css for customized widths.
	 * This is used in frontend and in Customizer.
	 *
	 * @param bool $as_template TRUE to return css as a template FALSE for the frontend css.
	 *
	 * @return string|null  CSS rules or null if settings not found.
	 */
	function mateo_layout_css( $as_template = false ) {
		if ( $as_template ) {
			// Use placeholder.
			$content_width = '{{data.contentWidth}}';
			$header_width  = '{{data.headerWidth}}';
			$footer_width  = '{{data.footerWidth}}';
			$title_width   = '{{data.titleWidth}}';
		} else {
			// Fetch data.
			$content_width = get_theme_mod( 'content-width', MATEO_MAX_CONTENT_WIDTH );
			$header_width  = get_theme_mod( 'header-width', MATEO_MAX_CONTENT_WIDTH );
			$footer_width  = get_theme_mod( 'footer-width', MATEO_MAX_CONTENT_WIDTH );
			$title_width   = get_theme_mod( 'title-width', MATEO_MAX_CONTENT_WIDTH );
		}
		$css = '';
		if ( mateo_is_layout_one_column() ) {
			// Declare CSS variables used in styles to calculate full and wide width.
			$css .= ":root{ --mateo-content-width: ${content_width}px;}";
		}
		$css     .= <<<EOL
	@media (min-width: 768px) {
		.site-content {
			max-width: ${content_width}px;
		}
		.page-header {
			max-width: ${title_width}px;
		}
		.header-width {
			width: ${header_width}px;
		}
		.footer-width {
			width: ${footer_width}px;
		}
	}
EOL;

		return $css;
	}
endif;
if ( ! function_exists( 'mateo_template_layout_css' ) ) :
	/**
	 * Prints JS template with layout css
	 */
	function mateo_template_layout_css() {
		?>
		<script type="text/html" id="tmpl-mateo-layout-css">
			<?php
			// phpcs:ignore
			echo ThemeColor::strip_eol( mateo_layout_css( true ) );
			?>
		</script>
		<?php
	}
endif;
add_action( 'customize_controls_print_footer_scripts', 'mateo_template_layout_css' );

if ( ! function_exists( 'mateo_custom_fonts' ) ) {
	/**
	 * Generates an URL for the Google Fonts API to load all fonts chosen by user at once.
	 * Generates CSS rules for selected elements.
	 *
	 * @return array    Array with a link to load google fonts and a set of css rules.
	 */
	function mateo_custom_fonts() {
		$settings_keys = array(
			'font-family',
			'font-variant',
			'font-size',
			'font-line-height',
			'font-family-headers',
			'font-variant-headers',
			'font-size-headers',
			'font-line-height-headers',
			'font-family-title',
			'font-variant-title',
			'font-size-title',
			'font-line-height-title',
			'font-family-monospace',
			'font-variant-monospace',
			'font-size-monospace',
			'font-line-height-monospace',
		);
		foreach ( $settings_keys as $key ) {
			$new = get_theme_mod( $key, false );
			if ( false !== $new ) {
				$fonts_settings[ $key ] = $new;
			}
		}
		if ( empty( $fonts_settings ) ) {
			return array();
		}

		$google_fonts = Fonts::get_instance();
		if ( ! $google_fonts->has_fonts() ) {
			$google_fonts->set_fonts( mateo_get_fonts() );
		}

		// Default font family.
		$default_body_font = 'roboto';
		if ( ! empty( $fonts_settings['font-family'] ) ) {
			$default_body_font = $fonts_settings['font-family'];
		}
		$google_fonts->add_font( $default_body_font );
		empty( $fonts_settings['font-variant'] )
			? $google_fonts->add_font_variants()
			: $google_fonts->add_font_variants( array( $fonts_settings['font-variant'] ) );

		if ( ! empty( $fonts_settings['font-family'] ) ) {
			$google_fonts->add_css( 'fontFamily', $google_fonts->get_family_name( $fonts_settings['font-family'] ) );
			$google_fonts->add_css( 'fontCategory', $google_fonts->get_category( $fonts_settings['font-family'] ) );
		}
		if ( ! empty( $fonts_settings['font-variant'] ) ) {
			// Analyse a font variant and adjust font display.
			$variant = $fonts_settings['font-variant'];

			if ( $google_fonts->is_italic( $variant ) ) {
				$google_fonts->add_css( 'fontStyle', 'italic' );
			}
			$font_weight = $google_fonts->get_font_weight( $variant );
			if ( $font_weight ) {
				$google_fonts->add_css( 'fontWeight', $font_weight );
			}
		}
		if ( ! empty( $fonts_settings['font-size'] ) ) {
			$google_fonts->add_css( 'fontSize', $fonts_settings['font-size'] );
		}
		if ( ! empty( $fonts_settings['font-line-height'] ) ) {
			$google_fonts->add_css( 'lineHeight', $fonts_settings['font-line-height'] );
		}

		// Headers.
		if ( ! empty( $fonts_settings['font-family-headers'] ) ) {
			$font_family_headers = $fonts_settings['font-family-headers'];
			$google_fonts->add_font( $font_family_headers );
			$google_fonts->add_css( 'fontFamilyHeaders', $google_fonts->get_family_name( $font_family_headers, Fonts::FAMILY_APPEND_CATEGORY ) );
		}

		if ( ! empty( $fonts_settings['font-variant-headers'] ) ) {
			// Analyse a font variant and adjust font display.
			$variant = $fonts_settings['font-variant-headers'];
			if ( $google_fonts->is_italic( $variant ) ) {
				$google_fonts->add_css( 'fontStyleHeaders', 'italic' );
			}
			$font_weight = $google_fonts->get_font_weight( $variant );
			if ( $font_weight ) {
				$google_fonts->add_css( 'fontWeightHeaders', $font_weight );
			}
			$google_fonts->add_font_variants( array( $variant ) );
		}
		if ( ! empty( $fonts_settings['font-size-headers'] ) ) {
			$google_fonts->add_css( 'fontSizeHeaders', $fonts_settings['font-size-headers'] );
		}
		if ( ! empty( $fonts_settings['font-line-height-headers'] ) ) {
			$google_fonts->add_css( 'lineHeightHeaders', $fonts_settings['font-line-height-headers'] );
		}

		// Site title.
		if ( ! empty( $fonts_settings['font-family-title'] ) ) {
			$font_family_headers = $fonts_settings['font-family-title'];
			$google_fonts->add_font( $font_family_headers );
			$google_fonts->add_css( 'fontFamilyTitle', $google_fonts->get_family_name( $font_family_headers, Fonts::FAMILY_APPEND_CATEGORY ) );
		}
		if ( ! empty( $fonts_settings['font-size-title'] ) ) {
			$google_fonts->add_css( 'fontSizeTitle', $fonts_settings['font-size-title'] );
		}

		if ( ! empty( $fonts_settings['font-variant-title'] ) ) {
			// Analyse a font variant and adjust font display.
			$variant = $fonts_settings['font-variant-title'];

			if ( $google_fonts->is_italic( $variant ) ) {
				$google_fonts->add_css( 'fontStyleTitle', 'italic' );
			}
			$font_weight = $google_fonts->get_font_weight( $variant );
			if ( $font_weight ) {
				$google_fonts->add_css( 'fontWeightTitle', $font_weight );
			}
			$google_fonts->add_font_variants( array( $variant ) );
		}

		if ( ! empty( $fonts_settings['font-line-height-title'] ) ) {
			$google_fonts->add_css( 'lineHeightTitle', $fonts_settings['font-line-height-title'] );
		}
		// Monospace font.
		if ( ! empty( $fonts_settings['font-family-monospace'] ) ) {
			$font_family_headers = $fonts_settings['font-family-monospace'];
			$google_fonts->add_font( $font_family_headers );
			$google_fonts->add_css( 'fontFamilyMonospace', $google_fonts->get_family_name( $font_family_headers, Fonts::FAMILY_APPEND_CATEGORY ) );
		}
		if ( ! empty( $fonts_settings['font-size-monospace'] ) ) {
			$google_fonts->add_css( 'fontSizeMonospace', $fonts_settings['font-size-monospace'] );
		}

		if ( ! empty( $fonts_settings['font-variant-monospace'] ) ) {
			// Analyse a font variant and adjust font display.
			$variant = $fonts_settings['font-variant-monospace'];

			if ( $google_fonts->is_italic( $variant ) ) {
				$google_fonts->add_css( 'fontStyleMonospace', 'italic' );
			}
			$font_weight = $google_fonts->get_font_weight( $variant );
			if ( $font_weight ) {
				$google_fonts->add_css( 'fontWeightMonospace', $font_weight );
			}
			$google_fonts->add_font_variants( array( $variant ) );
		}

		if ( ! empty( $fonts_settings['font-line-height-monospace'] ) ) {
			$google_fonts->add_css( 'lineHeightMonospace', $fonts_settings['font-line-height-monospace'] );
		}

		return array(
			'link' => $google_fonts->get_google_api_link(),
			'css'  => $google_fonts->get_parsed_template(),
		);
	}
}


