<?php
/**
 * Common functions for customizer, frontend and backend.
 *
 * @package mateo
 */

use Mateo\ThemeColor;
use Mateo\ThemeOptionsMetabox;


if ( ! function_exists( 'mateo_get_css' ) ) {
	/**
	 * Removes end lines from the content.
	 * Helper function to output CSS according to MATEO_CSS_STRIP_EOL setting.
	 *
	 * @param string $css CSS to filter.
	 *
	 * @return string  Filtered CSS.
	 */
	function mateo_get_css( $css ) {
		return MATEO_CSS_STRIP_EOL ? ThemeColor::strip_eol( $css ) : $css;
	}
}
if ( ! function_exists( 'mateo_get_js_filename' ) ) {
	/**
	 * Returns javascript filename with .min.js suffix if debugging is active (WP_DEBUG constant is true).
	 *
	 * @param string $filename The name of JS file without without path and '.js' extension.
	 * @return string File name.
	 */
	function mateo_get_js_filename( $filename ) {
		if ( ! $filename ) {
			return '';
		}
		$js_min = '.min';
		if ( defined( 'WP_DEBUG' ) && WP_DEBUG === true ) {
			$js_min = '';
		}

		return $filename . $js_min . '.js';
	}
}

if ( ! function_exists( 'mateo_get_color_schemes' ) ) {
	/**
	 *
	 * Theme color schemes.
	 * Can be filtered with 'mateo_color_schemes' filter.
	 *
	 * @return array An associative array of color scheme options.
	 */
	function mateo_get_color_schemes() {
		$schemes = array(
			'default'   => array(
				/* translators: default color scheme */
				'label'  => _x( 'Default', 'Color palette', 'mateo' ), // Pink.
				'colors' => array(
					'primary'       => '#e91e63',
					'primaryLight'  => '#ff6090',
					'primaryDark'   => '#b0003a',
					'textOnPrimary' => '#ffffff',
					'link'          => '#e91e63',
					'textBody'      => '#212121',
					'title'         => '#ffffff',
					'titleBgr'      => '#e91e63',
					'titleNoHeader' => '#212121',
					'contentBgr'    => '#ffffff',
					'pageBgr'       => '#ededed',
				),
			),
			'red'       => array(
				'label'  => _x( 'Red', 'Color palette', 'mateo' ),
				'colors' => array(
					'primary'       => '#d32f2f',
					'primaryLight'  => '#ff6659',
					'primaryDark'   => '#9a0007',
					'textOnPrimary' => '#ffffff',
					'link'          => '#d32f2f',
					'textBody'      => '#333333',
					'title'         => '#ffffff',
					'titleBgr'      => '#d32f2f',
					'titleNoHeader' => '#333333',
					'contentBgr'    => '#ffffff',
					'pageBgr'       => '#ededed',
				),
			),
			'purple'    => array(
				'label'  => _x( 'Purple', 'Color palette', 'mateo' ),
				'colors' => array(
					'primary'       => '#8e24aa',
					'primaryLight'  => '#c158dc',
					'primaryDark'   => '#5c007a',
					'textOnPrimary' => '#ffffff',
					'link'          => '#8e24aa',
					'textBody'      => '#333333',
					'title'         => '#ffffff',
					'titleBgr'      => '#8e24aa',
					'titleNoHeader' => '#333333',
					'contentBgr'    => '#ffffff',
					'pageBgr'       => '#ededed',
				),
			),
			'blue'      => array(
				'label'  => _x( 'Blue', 'Color palette', 'mateo' ),
				'colors' => array(
					'primary'       => '#2196f3',
					'primaryLight'  => '#6ec6ff',
					'primaryDark'   => '#0069c0',
					'textOnPrimary' => '#ffffff',
					'link'          => '#2196f3',
					'textBody'      => '#333333',
					'title'         => '#ffffff',
					'titleBgr'      => '#2196f3',
					'titleNoHeader' => '#333333',
					'contentBgr'    => '#ffffff',
					'pageBgr'       => '#ededed',
				),
			),
			'orange'    => array(
				'label'  => _x( 'Orange', 'Color palette', 'mateo' ),
				'colors' => array(
					'primary'       => '#ff5722',
					'primaryLight'  => '#ff8a50',
					'primaryDark'   => '#c41c00',
					'textOnPrimary' => '#ffffff',
					'link'          => '#ff5722',
					'textBody'      => '#333333',
					'title'         => '#ffffff',
					'titleBgr'      => '#ff5722',
					'titleNoHeader' => '#333333',
					'contentBgr'    => '#ffffff',
					'pageBgr'       => '#ededed',
				),
			),
			'green'     => array(
				'label'  => _x( 'Green', 'Color palette', 'mateo' ),
				'colors' => array(
					'primary'       => '#43a047',
					'primaryLight'  => '#76d275',
					'primaryDark'   => '#00701a',
					'textOnPrimary' => '#ffffff',
					'link'          => '#43a047',
					'textBody'      => '#333333',
					'title'         => '#ffffff',
					'titleBgr'      => '#43a047',
					'titleNoHeader' => '#333333',
					'contentBgr'    => '#ffffff',
					'pageBgr'       => '#ededed',
				),
			),
			'blue_grey' => array(
				'label'  => _x( 'Blue Grey', 'Color palette', 'mateo' ),
				'colors' => array(
					'primary'       => '#546e7a',
					'primaryLight'  => '#819ca9',
					'primaryDark'   => '#29434e',
					'textOnPrimary' => '#ffffff',
					'link'          => '#546e7a',
					'textBody'      => '#333333',
					'title'         => '#ffffff',
					'titleBgr'      => '#546e7a',
					'titleNoHeader' => '#333333',
					'contentBgr'    => '#ffffff',
					'pageBgr'       => '#ededed',
				),
			),
			'dark'      => array(
				'label'  => _x( 'Dark', 'Color palette', 'mateo' ),
				'colors' => array(
					'primary'       => '#333333',
					'primaryLight'  => '#5c5c5c',
					'primaryDark'   => '#0c0c0c',
					'textOnPrimary' => '#ffffff',
					'link'          => '#333333',
					'textBody'      => '#333333',
					'title'         => '#ffffff',
					'titleBgr'      => '#333333',
					'titleNoHeader' => '#333333',
					'contentBgr'    => '#ffffff',
					'pageBgr'       => '#ededed',
				),
			),
		);

		return apply_filters( 'mateo_color_schemes', $schemes );
	}
}

if ( ! function_exists( 'mateo_get_color_scheme' ) ) {
	/**
	 * Retrieves the current color scheme.
	 *
	 * Create your own mateo_get_color_scheme() function to override in a child theme.
	 *
	 * @return array An associative array of either the current or default color scheme HEX values.
	 */
	function mateo_get_color_scheme() {
		$color_scheme_option = get_theme_mod( 'color_scheme', 'default' );
		$color_schemes       = mateo_get_color_schemes();

		if ( array_key_exists( $color_scheme_option, $color_schemes ) ) {
			return $color_schemes[ $color_scheme_option ]['colors'];
		}

		return $color_schemes['default']['colors'];
	}
}

if ( ! function_exists( 'mateo_get_color_scheme_choices' ) ) {
	/**
	 * Retrieves an array of color scheme choices registered for Twenty Sixteen.
	 *
	 * Create your own mateo_get_color_scheme_choices() function to override
	 * in a child theme.
	 *
	 * @return array Array of color schemes.
	 */
	function mateo_get_color_scheme_choices() {
		$color_schemes                = mateo_get_color_schemes();
		$color_scheme_control_options = array();

		foreach ( $color_schemes as $color_scheme => $value ) {
			$color_scheme_control_options[ $color_scheme ] = $value['label'];
		}

		return $color_scheme_control_options;
	}
}

if ( ! function_exists( 'mateo_get_editor_colors' ) ) {
	/**
	 * Initializes editor color palette
	 *
	 * @return array
	 */
	function mateo_get_editor_colors() {
		$theme_color = new ThemeColor( mateo_get_color_scheme() );
		$theme_color->init_user_settings();
		$colors = array(
			// Default WP editor colors.
			array(
				'name'  => __( 'Pale pink', 'mateo' ),
				'slug'  => 'pale-pink',
				'color' => '#f78da7',
			),
			array(
				'name'  => __( 'Vivid red', 'mateo' ),
				'slug'  => 'vivid-red',
				'color' => '#cf2e2e',
			),
			array(
				'name'  => __( 'Luminous vivid orange', 'mateo' ),
				'slug'  => 'luminous-vivid-orange',
				'color' => '#ff6900',
			),
			array(
				'name'  => __( 'Luminous vivid amber', 'mateo' ),
				'slug'  => 'luminous-vivid-amber',
				'color' => '#fcb900',
			),
			array(
				'name'  => __( 'Light green cyan', 'mateo' ),
				'slug'  => 'light-green-cyan',
				'color' => '#7bdcb5',
			),
			array(
				'name'  => __( 'Vivid green cyan', 'mateo' ),
				'slug'  => 'vivid-green-cyan',
				'color' => '#00d084',
			),
			array(
				'name'  => __( 'Pale cyan blue', 'mateo' ),
				'slug'  => 'pale-cyan-blue',
				'color' => '#8ed1fc',
			),
			array(
				'name'  => __( 'Vivid cyan blue', 'mateo' ),
				'slug'  => 'vivid-cyan-blue',
				'color' => '#0693e3',
			),
			array(
				'name'  => __( 'Vivid purple', 'mateo' ),
				'slug'  => 'vivid-purple',
				'color' => '#9b51e0',
			),
			array(
				'name'  => __( 'Very light gray', 'mateo' ),
				'slug'  => 'very-light-gray',
				'color' => '#eeeeee',
			),
			array(
				'name'  => __( 'Cyan bluish gray', 'mateo' ),
				'slug'  => 'cyan-bluish-gray',
				'color' => '#abb8c3',
			),
			array(
				'name'  => __( 'Very dark gray', 'mateo' ),
				'slug'  => 'very-dark-gray',
				'color' => '#313131',
			),
			// Theme or user colors.
			array(
				'name'  => _x( 'Primary color', 'Color palette', 'mateo' ),
				'slug'  => 'primary',
				'color' => $theme_color->get_color( 'primary' ),
			),
			array(
				'name'  => _x( 'Primary light color', 'Color palette', 'mateo' ),
				'slug'  => 'primary-light',
				'color' => $theme_color->get_color( 'primaryLight' ),
			),
			array(
				'name'  => _x( 'Primary dark color', 'Color palette', 'mateo' ),
				'slug'  => 'primary-dark',
				'color' => $theme_color->get_color( 'primaryDark' ),
			),
			array(
				'name'  => _x( 'Text on primary', 'Color palette', 'mateo' ),
				'slug'  => 'text-on-primary',
				'color' => $theme_color->get_color( 'textOnPrimary' ),
			),
			array(
				'name'  => _x( 'Text color', 'Color palette', 'mateo' ),
				'slug'  => 'text-body',
				'color' => $theme_color->get_color( 'textBody' ),
			),
		);
		$colors = apply_filters( 'mateo_editor_colors', $colors );
		return $colors;
	}
}

if ( ! function_exists( 'mateo_get_content_width' ) ) {
	/**
	 * Gets container width: user selected content width or default one.
	 *
	 * @return int
	 */
	function mateo_get_content_width() {
		$width = get_theme_mod( 'content-width', MATEO_MAX_CONTENT_WIDTH );

		return apply_filters( 'mateo_content_width', $width );
	}
}
if ( ! function_exists( 'mateo_is_active_sidebar' ) ) {
	/**
	 * Checks if a sidebar is active.
	 *
	 * @return boolean  True if active false otherwise
	 */
	function mateo_is_active_sidebar() {
		// The function is called twice during normal execution so use static keyword to avoid extra processing.
		static $is_active;
		if ( null !== $is_active ) {
			return $is_active;
		}
		$is_active = is_active_sidebar( 'sidebar-1' );
		return $is_active;
	}
}

if ( ! function_exists( 'mateo_get_theme_layout' ) ) {
	/**
	 * Gets number of columns for the layout.
	 * 2 column layout can be overridden in post editor.
	 * Default: 1 column layout.
	 *
	 * @return int number of columns.
	 */
	function mateo_get_theme_layout() {
		return get_theme_mod( MATEO_SETTING_LAYOUT_ID ) === MATEO_LAYOUT_2 ? 2 : 1;
	}
}

if ( ! function_exists( 'mateo_get_post_layout' ) ) {
	/**
	 * Checks if one column layout should be applied to the post.
	 *
	 * @return int number of columns.
	 */
	function mateo_get_post_layout() {
		global $post;
		if ( $post ) {
			$value = (int) get_post_meta( $post->ID, ThemeOptionsMetabox::META_LAYOUT, true );

			if ( 1 === $value ) {
				return 1;
			}
		}
		return 2;
	}
}


if ( ! function_exists( 'mateo_is_layout_one_column' ) ) {
	/**
	 * Checks if one column layout should be applied.
	 * The function must be used for the frontend only because it checks if a single page is displayed.
	 *
	 * @return bool TRUE for 1 column layout FALSE otherwise.
	 */
	function mateo_is_layout_one_column() {
		if ( mateo_get_theme_layout() === 1 ) {
			return true;
		}

		if ( is_singular() && mateo_get_post_layout() === 1 ) {
			return true;
		}
		return false;
	}
}
if ( ! function_exists( 'mateo_get_sidebar_width' ) ) {
	/**
	 * Get sidebar width.
	 *
	 * @return int
	 */
	function mateo_get_sidebar_width() {
		return (int) get_theme_mod( 'sidebar-width', 216 );
	}
}
if ( ! function_exists( 'mateo_get_display_width' ) ) {
	/**
	 * Calculates the width of the content area without a sidebar, gutters and container paddings (left and right).
	 * Used in frontend.
	 *
	 * @return float|int
	 */
	function mateo_get_display_width() {

		$container_width = mateo_get_content_width();
		$padding         = mateo_get_site_padding();
		$content_width   = $container_width - $padding['left'] - $padding['right'];

		if ( ! mateo_is_layout_one_column() && mateo_is_active_sidebar() ) {
			$content_width = $content_width - mateo_get_sidebar_width() - $padding['right'];
		}

		return $content_width;
	}
}
if ( ! function_exists( 'mateo_get_background_image_css' ) ) {
	/**
	 * Returns CSS rules related to the background image generated by native WordPress function.
	 *
	 * @return string
	 */
	function mateo_get_background_image_css() {
		$background = set_url_scheme( get_background_image() );
		if ( ! $background ) {
			return '';
		}
		ob_start();
		_custom_background_cb();
		$style  = ob_get_clean();
		$status = preg_match( '/[^\{]*\{([^\}]*)\}/', $style, $rules );
		if ( 1 !== $status ) {
			return '';
		}

		return 'body {' . $rules[1] . '}';
	}
}
if ( ! function_exists( 'mateo_relative_to_pixel' ) ) {
	/**
	 * Converts relative size (rem,em) to pixel according to base font setting.
	 *
	 * @param int $size Relative size to convert.
	 *
	 * @return int Size in pixels.
	 */
	function mateo_relative_to_pixel( $size ) {
		static $font_size_base;
		if ( ! $font_size_base ) {
			$font_size_base = (int) get_theme_mod( 'font-size', 16 );
		}
		return $size * $font_size_base;
	}
}
if ( ! function_exists( 'mateo_get_site_padding' ) ) {
	/**
	 * Gets customized paddings.
	 *
	 * @return array|float[]|int[] Array of site content padding.
	 */
	function mateo_get_site_padding() {
		$result = array();

		$padding_unit     = get_theme_mod( 'mateo-content-pad-unit', 'rem' );
		$result['left']   = get_theme_mod( 'mateo-content-pad-left', 3 );
		$result['right']  = get_theme_mod( 'mateo-content-pad-right', 3 );
		$result['top']    = get_theme_mod( 'mateo-content-pad-top', 3 );
		$result['bottom'] = get_theme_mod( 'mateo-content-pad-bottom', 3 );

		// Convert in pixels.
		if ( 'px' !== $padding_unit ) {
			$result = array_map( 'mateo_relative_to_pixel', $result );
		}
		return $result;
	}
}

if ( ! function_exists( 'mateo_editor_custom_css' ) ) {
	/**
	 * Generates editor CSS with user chosen colors and other settings.
	 */
	function mateo_editor_custom_css() {
		global $post;
		$css               = '';
		$one_column_layout = mateo_get_theme_layout() === 1;
		$layer_width       = mateo_get_content_width();
		$title_width       = get_theme_mod( 'title-width', MATEO_MAX_CONTENT_WIDTH );
		$padding           = mateo_get_site_padding();
		$content_width     = $layer_width - $padding['left'] - $padding['right'];
		// Declare CSS variables used in editor styles.
		$css .= <<<DOC
			:root{
				--mateo-layer-width: ${layer_width}px;
				--mateo-content-width: ${content_width}px;
				--mateo-title-width: ${title_width}px;
			}
DOC;

		if ( ! $one_column_layout ) {
			// Remove sidebar width from the content width.
			$unit          = ( $layer_width - $padding['left'] - $padding['right'] + MATEO_CSS_GUTTERS ) / 12;
			$content_width = $unit * 9 - MATEO_CSS_GUTTERS;
			$layer_width   = $content_width + $padding['left'] + $padding['right'];
			$css          .= <<<DOC
				:root{
					--mateo-layer-width-small:  ${layer_width}px;
					--mateo-column-width: ${content_width}px;
				}
DOC;
		}

		if ( $post ) {
			$page_bgr = (string) get_post_meta( $post->ID, ThemeOptionsMetabox::META_PAGE_BGR, true );
			if ( $page_bgr ) {
				$css .= <<<DOC
				:root{
					--mateo-custom-page-bgr:  $page_bgr;
				}
DOC;
			}
		}

		// Custom colors.
		$theme_color = new ThemeColor( mateo_get_color_scheme() );
		$theme_color->init_user_settings();

		$css .= $theme_color->get_css( false, true );

		// CSS for the background image if set up.
		$css .= mateo_get_background_image_css();

		// Custom fonts.
		$custom_fonts = mateo_custom_fonts();
		if ( $custom_fonts ) {
			$css .= $custom_fonts['css'];
		}

		// Custom sizing.
		$css .= mateo_custom_css_vars();

		return mateo_get_css( $css );
	}
}

if ( ! function_exists( 'mateo_post_custom_css' ) ) {
	/**
	 * Generates and returns CSS rules defined for some post.
	 * Used in front end.
	 *
	 * @return string CSS rules or empty string.
	 */
	function mateo_post_custom_css() {

		global $post;
		$css = '';
		if ( ! is_singular() ) {
			return $css;
		}
		if ( $post ) {
			$page_bgr_color = sanitize_hex_color( get_post_meta( $post->ID, ThemeOptionsMetabox::META_PAGE_BGR, true ) );
			if ( $page_bgr_color ) {
				$css .= <<<DOC
				body{
					background-color : $page_bgr_color;
				}
DOC;
			}
		}

		return $css;
	}
}

if ( ! function_exists( 'mateo_is_active_breadcrumb' ) ) {
	/**
	 * Check if breadcrumb.
	 *
	 * @return bool TRUE is active FALSE if not.
	 */
	function mateo_is_active_breadcrumb() {
		return get_theme_mod( 'mateo-breadcrumb', false );
	}
}

